#include "curl.h"
#include <wx/sstream.h>
#include <wx/wfstream.h>
#include <wx/stopwatch.h>
#include <wx/textfile.h>

// timeouts on loading data
#define TIMEOUT 10000
#define SLEEP 100

// threshold for distinguishing error messages from real data
#define MAX_ERROR_SIZE 1024

//-----------------------------------------------------------------------------

cUrl::cUrl(const wxString& url)
{
  m_UrlName = url;
}

//-----------------------------------------------------------------------------

cUrl::~cUrl()
{
}

//-----------------------------------------------------------------------------
// Store an error in m_Error
// Always returns false

bool cUrl::ReportError(const wxString& msg)
{
  m_Error = msg;
  wxLogDebug(msg);
  return false;
}

//-----------------------------------------------------------------------------
// Load a file from an URL into a string
// - data = receives the file contents
// Returns success of operation

bool cUrl::ReadToString(wxString* data)
{

  wxASSERT(data != NULL);
  wxASSERT(data->IsEmpty());

  // set up output stream to write to
  wxStringOutputStream strOut(data);

  return ReadData(&strOut);
}

//-----------------------------------------------------------------------------
// Load a file from an URL into a local file
// - fileName = receives the file contents
// Returns success of operation

bool cUrl::ReadToFile(const wxString& fileName)
{
  // set up output stream to write to
  wxFFileOutputStream fileOut(fileName, "wb");
  if (!fileOut.IsOk()) {
    ReportError(wxString::Format(_T("Failed to create file '%s'"), fileName.c_str()));
    return false;
  }

  if (!ReadData(&fileOut)) return false;

  // check if it's an error message from LFS World
  if (fileOut.TellO() > MAX_ERROR_SIZE) return true; // OK

  // read the error from the file, then delete the file
  fileOut.Close();
  wxTextFile errorFile(fileName);
  errorFile.Open();
  if (errorFile.GetLineCount() > 0) m_Error = errorFile.GetLine(0);
  errorFile.Close();
  wxRemoveFile(fileName);

  return false;
}

//-----------------------------------------------------------------------------
// Do the actual reading

bool cUrl::ReadData(wxOutputStream* outStream)
{
  bool result = true;

  // open the URL
  m_URL.SetURL(m_UrlName);
  if (!m_URL.IsOk()) {
    ReportError(wxString::Format(_T("Failed to open URL (code %d)"), long(m_URL.GetError())));
    return false;
  }

  // set up input stream to read from
  wxInputStream* inStream = m_URL.GetInputStream();
  if (inStream == NULL) {
    ReportError(wxString::Format(_T("Error reading from URL (failed to get input stream)")));
    return false;
  }

  // read the data
  wxStopWatch timer;
  while (inStream->IsOk() && outStream->IsOk()) {
    if (inStream->CanRead()) {
      // read available data
      inStream->Read(*outStream);
      timer.Start(); // reset timeout
    }
    else {
      // wait a while, then check timeout
      wxMilliSleep(SLEEP);
      if (timer.Time() > TIMEOUT) {
        ReportError(wxString::Format(_T("Timeout reading from URL '%s'"), m_UrlName.c_str()));
        result = false;
        break;
      }
    }
  }

  // handle errors (the only acceptable error is wxSTREAM_EOF on the input)
  if (!outStream->IsOk()) {
    ReportError(wxString::Format(_T("Error writing to output stream from URL (code %d)"), long(outStream->GetLastError())));
    result = false;
  }
  if (!inStream->Eof()) {
    ReportError(wxString::Format(_T("Error reading from URL (code %d)"), long(inStream->GetLastError())));
    result = false;
  }

  delete inStream;
  return result;
}
