#include "cgraph_y.h"
#include "cgraphview.h"
#include "ccurve_y.h"
#include "cruler.h"
#include "cmgr.h"
#include "capp.h"
#include <wx/filedlg.h>
#include <wx/dcclient.h>

//-----------------------------------------------------------------------------
// Event table

BEGIN_EVENT_TABLE(cGraphY, cGraph)
  EVT_MOTION(cGraphY::OnMouseMove)
  EVT_ENTER_WINDOW(cGraphY::OnMouseEntering)
  EVT_LEAVE_WINDOW(cGraphY::OnMouseLeaving)
  EVT_PAINT(cGraphY::OnPaint)
  EVT_SIZE(cGraphY::OnSize)
END_EVENT_TABLE()

//-----------------------------------------------------------------------------

cGraphY::cGraphY(cGraphView* parent)
: cGraph(parent, GRAPHTYPE_Y)
{
  m_FitDimensions = wxVERTICAL;

  m_Logtype = LOGTYPE_FIRST;
  m_Wheel = WHEELTYPE_FIRST;
  m_DeltaLap = NULL;
  m_DeltaMode = false;

  AdjustBbox(); // init bounding box
}

//-----------------------------------------------------------------------------

cGraphY::~cGraphY()
{
}

//-----------------------------------------------------------------------------
// Get the rbb mode to start select-dragging (RBBMODE_NONE = veto)
// - event = event that signals start of drag

rbbmode_t cGraphY::GetRbbMode(wxMouseEvent& event)
{
  return (event.ShiftDown()) ? RBBMODE_BOTH : RBBMODE_XONLY;
}

//-----------------------------------------------------------------------------
// Add a curve to the graph
// - lap = lap for which to add a curve

void cGraphY::AddCurve(cLap* lap)
{
  CHECK_PTR(lap);

  // restore delta-mode
  if (m_DeltaMode && (m_Curves.GetCount() == 0)) m_DeltaLap = lap;

  // add the curve
  cCurveY* curve = new cCurveY(lap);
  curve->SetLogtype(m_Logtype, m_Wheel);
  curve->SetDeltaLap(m_DeltaLap);
  m_Curves.Add(curve);

  // update display
  SetZoomY(1.0f);
  AdjustBbox();
  Refresh();
}

//-----------------------------------------------------------------------------
// Delete a curve
// - lap = lap whose curve is to be deleted

void cGraphY::DeleteCurve(cLap* lap)
{
  CHECK_PTR(lap);

  // find the curve for this lap
  for (size_t c = 0; c < m_Curves.GetCount(); c++) {
    if (m_Curves[c].GetLap() == lap) {
      // found it
      m_Curves.RemoveAt(c);
      if (lap == m_DeltaLap) SetDeltaLap(NULL);
      if (m_Curves.IsEmpty()) {
        SetZoomX(1.0f);
        SetZoomY(1.0f);
      }
      AdjustBbox();
      Refresh();
      return;
    }
  }
}

//-----------------------------------------------------------------------------
// Delete all curves

void cGraphY::DeleteAllCurves()
{
  m_Curves.Clear();
  m_DeltaLap = NULL;
  ZoomReset();
  AdjustBbox();
}

//-----------------------------------------------------------------------------

void cGraphY::OnSize(wxSizeEvent& event)
{
  RemoveTrackCursor();
  event.Skip();
}

//-----------------------------------------------------------------------------
// Plot the graph

void cGraphY::OnPaint(wxPaintEvent& WXUNUSED(event))
{
  wxPaintDC dc(this);

  int screenX, screenY;
  dc.GetSize(&screenX, &screenY);
  if ((screenX <= 0) || (screenY <= 0)) return;

  // plot the grid
  dc.SetPen(cGraph::s_GridPen);
  float value;
  wxCoord coord;
  // X grid (vertical lines)
  value = GetGridStartX();
  value -= fmod(value, m_GridStepX);
  value -= m_GridStepX;
  coord = 0;
  while (coord <= screenX) {
    coord = Value2CoordX(value);
    dc.DrawLine(coord, 0, coord, screenY);
    value += m_GridStepX;
  }
  // Y grid (horizontal lines)
  value = GetGridEndY();
  value -= fmod(value, m_GridStepY);
  value += m_GridStepY;
  coord = 0;
  while (coord <= screenY) {
    if ((fabs(value) < EPSILON) || (m_DeltaLap != NULL)) {
      // plot Y axis
      dc.SetPen(cGraph::s_AxisPen);
    }
    else {
      // plot normal grid line
      dc.SetPen(cGraph::s_GridPen);
    }
    coord = Value2CoordY(value);
    dc.DrawLine(0, coord, screenX, coord);
    value -= m_GridStepY;
  }

  // plot each curve
  float minX = Coord2ValueX(0);
  float maxX = Coord2ValueX(screenX);
  for (size_t c = 0; c < m_Curves.GetCount(); c++) {
    m_Curves[c].Plot(dc, *this, minX, maxX);
  }

  // update the rulers
  if (m_RulerX != NULL) m_RulerX->Refresh();
  if (m_RulerY != NULL) m_RulerY->Refresh();

  // plot the crosshair (if it was present)
  if (m_CrossHairVisible) {
    dc.SetPen(*wxBLACK_PEN);
    dc.CrossHair(m_CrossHairPos.x, m_CrossHairPos.y);
  }
}

//-----------------------------------------------------------------------------

void cGraphY::OnMouseMove(wxMouseEvent& event)
{
  DrawTrackCursor();
  event.Skip();
}

//-----------------------------------------------------------------------------
// Set the type of data that is shown
// - log = type of log-data
// - wheel = index of the wheel for which the data must be shown

void cGraphY::SetLogtype(int log, int wheel)
{
  wxASSERT(log >= LOGTYPE_FIRST);
  wxASSERT(log < LOGTYPE_LAST);
  wxASSERT(wheel >= WHEELTYPE_FIRST);
  wxASSERT(wheel < WHEELTYPE_LAST);

  if ((m_Logtype == log) && (m_Wheel == wheel)) return; // no change
  m_Logtype = log;
  m_Wheel = wheel;

  // set it for each curve
  for (size_t c = 0; c < m_Curves.GetCount(); c++) {
    ((cCurveY&)m_Curves[c]).SetLogtype(log, wheel);
  }

  // update display
  SetZoomY(1.0);
  AdjustBbox();
  Refresh();
}

//-----------------------------------------------------------------------------
// Set the delta-lap
// - lap = the new delta-lap, or NULL if there is no delta lap

void cGraphY::SetDeltaLap(cLap* lap)
{
  wxASSERT((lap == NULL) || (MGR->FindLap(lap) >= 0));
  CHECK_PTR_NULL(lap);

  if (m_DeltaLap == lap) return; // no change
  m_DeltaLap = lap;

  // set it for each curve
  for (size_t c = 0; c < m_Curves.GetCount(); c++) {
    ((cCurveY&)m_Curves[c]).SetDeltaLap(m_DeltaLap);
  }

  // update display
  SetZoomY(1.0);
  AdjustBbox();
  Refresh();
}

//-----------------------------------------------------------------------------
// Draw the cursor (= crosshair) at a new position
// - distance = distance_in_lap (in m) that corresponds to the cursor position
// - type = type of cursor-set command
// If distance < 0 then cursor is removed

void cGraphY::DoSetTrackCursorPos(cGraphView* WXUNUSED(view), float distance, int type)
{
  if (m_RbbMode != RBBMODE_NONE) return; // ignore calls while dragging
  if (!m_CrossHairMode && (type == CURSORSET_FOLLOW)) return;

  RemoveCrossHair();
  if (distance < 0) return;

  // calc new position
  wxPoint pos = ScreenToClient(wxGetMousePosition());
  if ((pos.x < 0) || (pos.y < 0) ||
      (pos.x > GetClientSize().GetWidth()) ||
      (pos.y > GetClientSize().GetHeight())) {
    // mouse is not over window - do not draw the horizontal line
    pos.y = -1;
  }
  pos.x = Value2CoordX(distance);

  if (type == CURSORSET_PLAYBACK) pos.y = -1; // during playback: only draw a vertical line
  if (!m_CrossHairMode) pos.y = -1; // when crosshair is off: only draw a vertical line

  // draw at new position
  DrawCrossHair(pos, (type != CURSORSET_FOLLOW));
}

//-----------------------------------------------------------------------------
// Set the visible ('selected') part of the lap

void cGraphY::DoSetTrackSelection(float start, float end)
{
  if (SetZoomX((m_MaxX - m_MinX) / (end - start))) {
    AdjustDimensions();
    Refresh();
  }
  SetGraphOriginX(start);
}

//-----------------------------------------------------------------------------
// Sync'ing

void cGraphY::DoSync()
{
  ::SetTrackSelection_Send(GetGridStartX(), GetGridEndX());
}

//-----------------------------------------------------------------------------

wxString cGraphY::GetCurveValueAt(cLap* lap, float distance)
{
  CHECK_PTR(lap);
  CHECK_PTR_NULL(m_DeltaLap);

  // calculate value
  float value = lap->GetLogDataAt(distance, m_Logtype, m_Wheel);
  if (m_DeltaLap != NULL) {
    if (m_DeltaLap == lap) {
      // in rare cases, a diff with itself does not yield zero, so we set it explicitly
      value = 0.0f;
    }
    else {
      // calculate difference with other lap
      value -= m_DeltaLap->GetLogDataAt(distance, m_Logtype, m_Wheel);
    }
  }

  return FormatValueY(value);
}

//-----------------------------------------------------------------------------
// Loading and saving the configuration settings

wxString cGraphY::SaveSettings() const
{
  wxString settings;

  settings += wxString::Format(_T("data=%s;"), cCarState::GetLogtypeCode(GetLogtype()).c_str());
  settings += wxString::Format(_T("wheel=%s;"), cCarState::GetWheeltypeCode(GetWheeltype()).c_str());
  settings += wxString::Format(_T("delta=%d;"), MGR->FindLap(GetDeltaLap()));

  return settings;
}


void cGraphY::LoadSettings(const wxString& settings)
{
  // get current settings
  int data = GetLogtype();
  int wheel = GetWheeltype();
  int delta = MGR->FindLap(GetDeltaLap());

  // parse new settings
  wxArrayString keys;
  wxArrayString values;
  ::DecodeKeyValueList(settings, keys, values);
  for (size_t k = 0; k < keys.GetCount(); k++) {
    int val = wxAtoi(values[k]);
    if (keys[k] == _T("data")) {
      data = cCarState::FindLogtypeCode(values[k]);
      if (data == wxNOT_FOUND) data = val; // fallback for format1: identify logtype by index
    }
    else if (keys[k] == _T("wheel")) {
      wheel = cCarState::FindWheeltypeCode(values[k]);
      if (wheel == wxNOT_FOUND) wheel = val; // fallback for format1: identify wheeltype by index
    }
    else if (keys[k] == _T("delta")) {
      SetDeltaMode(val >= 0);
      if (val < (int)(MGR->GetLapCount())) delta = val;
    }
  }

  // put new settings
  SetLogtype(data, wheel);
  cLap* lap = NULL;
  if (delta >= 0) lap = MGR->GetLap(delta);
  SetDeltaLap(lap);
  UpdateAll();
}

//-----------------------------------------------------------------------------

void cGraphY::OnMouseEntering(wxMouseEvent& WXUNUSED(event))
{
  DrawTrackCursor();
}

void cGraphY::OnMouseLeaving(wxMouseEvent& WXUNUSED(event))
{
  RemoveTrackCursor();
}

//-----------------------------------------------------------------------------
// Handle a left-click + release at the same position
// - pos = logical position of mouse

void cGraphY::OnMouseClickRelease(wxPoint pos)
{
  ::SetTrackCursorPos_Send(Coord2ValueX(pos.x), CURSORSET_FORCE);
}

//-----------------------------------------------------------------------------
// Draw or remove the track cursor

void cGraphY::DrawTrackCursor()
{
  wxPoint pos = ScreenToClient(wxGetMousePosition());
  ::SetTrackCursorPos_Send(Coord2ValueX(pos.x), CURSORSET_FOLLOW);
}

void cGraphY::RemoveTrackCursor()
{
  ::SetTrackCursorPos_Send(-1, CURSORSET_FOLLOW);
}

//-----------------------------------------------------------------------------
//

void cGraphY::ExportGraph()
{
  // open file dialog
  wxString caption = _TT(ID_TXT_GR_EXPORT_CAPTION, "Export graph to CSV");
  wxString wildcard = _TT(ID_TXT_GR_EXPORT_WILDCARD , "CSV files") + _T(" (*.csv)|*.csv");
  wxFileDialog fileDlg(NULL, caption, APP->GetExportFolder(), wxEmptyString, wildcard,
      (wxFD_SAVE | wxFD_OVERWRITE_PROMPT | wxCHANGE_DIR));
  if (fileDlg.ShowModal() != wxID_OK) return;

  // get user choices
  APP->SetExportFolder(fileDlg.GetDirectory());
  wxString fileName = fileDlg.GetPath();

  fileDlg.Close();

  // create file
  wxFile file;
  file.Create(fileName, true);
  if (!file.IsOpened()) return;

  // write header line
  wxString line;
  line = _T("\"d\"");
  for (size_t c = 0; c < m_Curves.GetCount(); c++) {
    if (!m_Curves[c].IsShown()) continue;
    line += wxString::Format(_T(",\"%s\""), m_Curves[c].GetLap()->GetName().c_str());
  }
  file.Write(line + "\n");

  // write curve data
  float step = 0.5f; // export with interval of 50cm
  float start = GetSelectionStart();
  start -= fmod(start, step);
  for (float d = start; d <= GetSelectionEnd(); d += step) {
    line.Printf(_T("%.1f"), d);
    for (size_t c = 0; c < m_Curves.GetCount(); c++) {
      if (!m_Curves[c].IsShown()) continue;
      line += wxString::Format(_T(",\"%f\""), m_Curves[c].GetLap()->GetLogDataAt(d, m_Logtype, m_Wheel));
    }
    file.Write(line + "\n");
  }

  // done
  file.Close();
}
