package lfsqualifyingticker;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.Toolkit;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;

import javax.swing.Box;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JScrollPane;
import javax.swing.JSplitPane;
import javax.swing.SwingConstants;
import javax.swing.Timer;

import net.sf.jinsim.response.RaceStartResponse;
import net.sf.jinsim.response.ResultResponse;
import net.sf.jinsim.response.SplitTimeResponse;
import lfsqualifyingticker.models.TimingModel;
import lfsqualifyingticker.models.TrackModel;
import lfsqualifyingticker.structures.Lap;
import lfsqualifyingticker.structures.MiniCompCar;
import lfsqualifyingticker.structures.PlayerDisplayRow;
import lfsqualifyingticker.structures.Helper;
import lfsqualifyingticker.structures.PlayerInfo;
import lfsqualifyingticker.structures.QualiTickerMessagePanel;
import lfsqualifyingticker.structures.QualiTickerPrefs;
import lfsqualifyingticker.structures.SpeedTrapReading;
import lfsqualifyingticker.structures.TimingModelListener;
import lfsqualifyingticker.structures.TrackCanvas;
import lfsqualifyingticker.structures.TrackData;
import lfsqualifyingticker.structures.TrackMap;
import lfsqualifyingticker.structures.PlayerDisplayRow.DriverStatus;

/**
 * The GUI for the LFS Qualifying Ticker application. Listens to the timing model
 * for important events. Allows the user to change cameras to follow a particular
 * car
 * @author Gus
 *
 */

public class LFSQualifyingTickerGUI extends JFrame implements TimingModelListener, ActionListener {
	/**
	 * 
	 */
	private static final long serialVersionUID = 7937271955242460031L;

	// Absolute widths for labels (not flexible but time constraints...)
	public static final int NAME_LABEL_WIDTH = 140;
	public static final int IN_SESSION_LABEL_WIDTH = 70;
	public static final int ON_HOTLAP_LABEL_WIDTH = 70;
	
	// ActionCommand used to aid in the detection of camera change requests
	public static final String VIEW_PLAYER_COMMAND = "ViewPlayer";
	
	public static final Color BACKGROUND_COLOUR = new Color(200, 200, 200);

	private final TimingModel timingModel;
	
	private Box mainBox, playersBox;
	
	private JSplitPane horizontalSplitPane;

	private TrackMap trackMap;
	
	private ActionListener listener;
	
	private JLabel nameLabel, inSessionLabel, onHotlapLabel, fastestOverallLapField;
	private JLabel[] bestSplitLabels = new JLabel[4];
	
	private HashMap<Byte, PlayerDisplayRow> displayRows = new HashMap<Byte, PlayerDisplayRow>(40);
	
	private QualiTickerMessagePanel messagePanel;
	
	private HashMap<String, Timer> waitingBestSpeedTrapMessageTimers = 
		new HashMap<String, Timer>();
	
	/**
	 * Create a new GUI with the given timing model and action listener
	 * @param timingModel
	 * @param listener
	 */
	public LFSQualifyingTickerGUI(TimingModel timingModel, ActionListener listener) {
		this.timingModel = timingModel;
		
		timingModel.addTimingModelListener(this);
		
		this.listener = listener;
		
		setupGUI();
	}
	
	private void setupGUI() {
		this.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
		
		this.setTitle(QualiTickerPrefs.getApplicationName()+" "+
				QualiTickerPrefs.getVersion());
		
		this.setLayout(new BorderLayout());

		Box headerLabelBox = Box.createHorizontalBox();

		nameLabel = new JLabel("Name");
		inSessionLabel = new JLabel("In Session");
		onHotlapLabel = new JLabel("On Hotlap");

		// Sizes
		Helper.setPrefSizeSizeAndMaxSizeOnComponent(nameLabel, NAME_LABEL_WIDTH);
		Helper.setPrefSizeSizeAndMaxSizeOnComponent(inSessionLabel, IN_SESSION_LABEL_WIDTH);
		Helper.setPrefSizeSizeAndMaxSizeOnComponent(onHotlapLabel, ON_HOTLAP_LABEL_WIDTH);
		
		// Alignment
		nameLabel.setHorizontalAlignment(SwingConstants.CENTER);
		inSessionLabel.setHorizontalAlignment(SwingConstants.CENTER);
		onHotlapLabel.setHorizontalAlignment(SwingConstants.CENTER);

		headerLabelBox.add(Box.createHorizontalStrut(5));
		headerLabelBox.add(nameLabel);
		headerLabelBox.add(Box.createHorizontalStrut(5));
		headerLabelBox.add(inSessionLabel);
		headerLabelBox.add(Box.createHorizontalStrut(5));
		headerLabelBox.add(onHotlapLabel);
		headerLabelBox.add(Box.createHorizontalStrut(5));
		headerLabelBox.add(Box.createHorizontalGlue());
		
		playersBox = Box.createVerticalBox();

		mainBox = Box.createVerticalBox();
		mainBox.add(headerLabelBox);
		mainBox.add(Box.createVerticalStrut(2));
		mainBox.add(playersBox);
		mainBox.add(Box.createVerticalGlue());
		
		mainBox.setAlignmentX(0);
		
		JScrollPane mainBoxScrollPane = new JScrollPane();
		mainBoxScrollPane.getViewport().add(mainBox);
		
		horizontalSplitPane = new JSplitPane(JSplitPane.HORIZONTAL_SPLIT);
		
		horizontalSplitPane.setLeftComponent(mainBoxScrollPane);

		horizontalSplitPane.setDividerLocation(mainBox.getPreferredSize().width);

		// If the user moves the split pane divider repaint the screen
		horizontalSplitPane.addPropertyChangeListener(new PropertyChangeListener() {
			public void propertyChange(PropertyChangeEvent evt) {
				if(evt.getPropertyName().equals(JSplitPane.DIVIDER_LOCATION_PROPERTY)) {
					splitPaneDividerLocationChanged(evt);
				}
			}
		});
		
		// Track map gets additional size
		horizontalSplitPane.setResizeWeight(0);
		
		// Initially set the split pane divider really small, but expand if necessary
		horizontalSplitPane.setDividerSize(0);
		
		Box fastestLapBox = Box.createHorizontalBox();
		
		JLabel fastestOverallLapLabel = new JLabel("Fastest Lap In Session: ");
		fastestOverallLapField = new JLabel("No Laps Completed");

		fastestLapBox.add(fastestOverallLapLabel);
		fastestLapBox.add(Box.createHorizontalStrut(5));
		fastestLapBox.add(fastestOverallLapField);
		fastestLapBox.add(Box.createHorizontalGlue());
		
		Box fastestSplitsBox = Box.createHorizontalBox();

		fastestSplitsBox.add(new JLabel("Best Splits:"));
		fastestSplitsBox.add(Box.createHorizontalStrut(5));
		
		for(int i=0; i<bestSplitLabels.length; i++) {
			bestSplitLabels[i] = new JLabel();
			
			if(i < bestSplitLabels.length-1) {
				fastestSplitsBox.add(bestSplitLabels[i]);
				fastestSplitsBox.add(Box.createHorizontalStrut(5));
			} else {
				fastestSplitsBox.add(bestSplitLabels[i]);
				fastestSplitsBox.add(Box.createHorizontalGlue());
			}
		}
		
		// Create and add the message panel if required
		if(QualiTickerPrefs.getDisplayMessagePanel()) {
			messagePanel = new QualiTickerMessagePanel();
			
			JSplitPane verticalSplitPane = new JSplitPane(JSplitPane.VERTICAL_SPLIT);
			
			verticalSplitPane.setTopComponent(horizontalSplitPane);
			verticalSplitPane.setBottomComponent(messagePanel);
			
			verticalSplitPane.setResizeWeight(1);

			verticalSplitPane.setDividerSize(6);
			
			this.add(verticalSplitPane, BorderLayout.CENTER);
			
			if(QualiTickerPrefs.getDrawTrackMap()) {
				this.setSize(1000, 900);
			} else {
				this.setSize(800, 900);
			}
		} else {
			this.add(horizontalSplitPane, BorderLayout.CENTER);

			if(QualiTickerPrefs.getDrawTrackMap()) {
				this.setSize(1000, 800);
			} else {
				this.setSize(800, 800);
			}
		}

		Box fastestInformationBox = Box.createVerticalBox();
		
		fastestInformationBox.add(fastestLapBox);
		fastestInformationBox.add(fastestSplitsBox);
		fastestInformationBox.add(Box.createVerticalGlue());
		
		this.add(fastestInformationBox, BorderLayout.SOUTH);

		// Centre on screen
		Dimension screenResolution = Toolkit.getDefaultToolkit().getScreenSize();
		
		int frameXCo = (int) ((screenResolution.getWidth()/2)-(this.getSize().getWidth()/2));
		int frameYCo = (int) ((screenResolution.getHeight()/2)-(this.getSize().getHeight()/2));
		this.setLocation(frameXCo, frameYCo);

		mainBox.setOpaque(true);
		mainBox.setBackground(BACKGROUND_COLOUR);
		this.setBackground(BACKGROUND_COLOUR);
		horizontalSplitPane.setOpaque(true);
		horizontalSplitPane.setBackground(Color.white);
		fastestInformationBox.setOpaque(true);
		fastestInformationBox.setBackground(BACKGROUND_COLOUR);
		
		headerLabelBox.setMaximumSize(new Dimension(headerLabelBox.getMaximumSize().width, 
				headerLabelBox.getMinimumSize().height));

		this.setVisible(true);
	}

	private synchronized void updateDisplay() {
		playersBox.removeAll();

		Collection<PlayerInfo> currentPlayersCollection = timingModel.getAllCurrentPlayers();

		PlayerInfo[] currentPlayersArray = currentPlayersCollection.toArray(
				new PlayerInfo[currentPlayersCollection.size()]);

		// Just compare the players by player name
		Arrays.sort(currentPlayersArray, new Comparator<PlayerInfo>() {
			public int compare(PlayerInfo o1, PlayerInfo o2) {
				String o1Name = o1.getPlayerName() != null ? 
						Helper.getPlainPlayerName(o1.getPlayerName()) : "";
				String o2Name = o2.getPlayerName() != null ? 
						Helper.getPlainPlayerName(o2.getPlayerName()) : "";
						
				return o1Name.compareTo(o2Name);
			}
		});

		for(PlayerInfo player : currentPlayersArray) {
			//print("updateDisplay, playerName: "+player.getPlayerName()+
			//", ID: "+player.getPlayerId());
			
			// Determine if this player is on a hotlap or not
			byte playerID = player.getPlayerID();
			
			DriverStatus driverStatus = timingModel.getDriverStatus(playerID);
			
			// If there's an existing display row just add it back in, otherwise create one
			if(displayRows.containsKey(playerID)) {
				PlayerDisplayRow displayRow = displayRows.get(playerID);
				
				displayRow.setDriverStatus(driverStatus);
				
				playersBox.add(displayRow);
				playersBox.add(Box.createVerticalStrut(2));
			} else {
				PlayerDisplayRow displayRow = new PlayerDisplayRow(player, 
						this, driverStatus);

				playersBox.add(displayRow);
				playersBox.add(Box.createVerticalStrut(2));

				displayRows.put(playerID, displayRow);
			}
		}

		playersBox.add(Box.createVerticalGlue());

		playersBox.revalidate();
		playersBox.repaint();
	}
	
	private void splitPaneDividerLocationChanged(PropertyChangeEvent evt) {
		if(trackMap != null) {
			trackMap.updateTrackSizeBasedOnSplitPaneLocation();
		
			requestRepaint();
		}
	}
	
	private void requestRepaint() {
		this.repaint();
	}

	private void print(String msg) {
		LFSQualifyingTickerStart.print(this.getClass().getSimpleName()+" - "+msg);
	}
	
	public void raceStartResponseReceived(RaceStartResponse raceStartResponse) {
		removeStateFromPreviousSession();
		
		if(QualiTickerPrefs.getDrawTrackMap()) {
			// Draw the new track map
			TrackData trackData = TrackModel.getDataForOneTrack(raceStartResponse.getTrack());

			if(trackData != null) {
				trackMap = new TrackMap(trackData, this);

				horizontalSplitPane.setRightComponent(trackMap);

				horizontalSplitPane.setDividerSize(5);

				Dimension displayRowMinimumSize = PlayerDisplayRow.preferredSizeForAllDisplayRows;
				
				if(displayRowMinimumSize != null) {
					horizontalSplitPane.setDividerLocation(
						displayRowMinimumSize.width+20);
				} else {
					// Hackety hackety hack
					int width = NAME_LABEL_WIDTH+IN_SESSION_LABEL_WIDTH+ON_HOTLAP_LABEL_WIDTH+230;
					
					horizontalSplitPane.setDividerLocation(width+20);
				}
				
				trackMap.updateTrackSizeBasedOnSplitPaneLocation();
				
				this.repaint();
			} else {
				trackMap = null;

				horizontalSplitPane.setDividerSize(1);
			}
		}
	}

	private void removeStateFromPreviousSession() {
		fastestOverallLapField.setText("No Laps Completed");
		
		playersBox.removeAll();
		displayRows.clear();
		
		trackMap = null;

		if(messagePanel != null) {
			messagePanel.clearMessagePanel();
		}
		
		// Clear the best split fields
		for(JLabel bestSplitLabel : bestSplitLabels) {
			bestSplitLabel.setText("");
		}
		
		updateDisplay();
	}
	
	private void updateFastestLapInformation(Lap fastestLap) {
		if(fastestLap != null && fastestLap.getLaptime() != null) {
			StringBuilder fastestLapString = new StringBuilder();
			
			if(fastestLap.getPlayer() != null) {
				fastestLapString.append(Helper.getPlainPlayerName(
						fastestLap.getPlayer().getPlayerName())+" - ");
			}

			fastestLapString.append(Helper.getTimeString(fastestLap.getLaptime()));
			
			Collection<SplitTimeResponse> splits = fastestLap.getAllSplits();
			
			if(splits != null && splits.size() > 0) {
				SplitTimeResponse[] splitsArray = splits.toArray(
						new SplitTimeResponse[splits.size()]);
				
				fastestLapString.append(" (");
				
				for(int i=0; i<splitsArray.length; i++) {
					if(i < splits.size()-1) {
						fastestLapString.append("Split "+splitsArray[i].getSplit()+": "+
								Helper.getTimeString(splitsArray[i].getTime())+", ");
					} else {
						fastestLapString.append("Split "+splitsArray[i].getSplit()+": "+
								Helper.getTimeString(splitsArray[i].getTime()));
					}
				}
				
				fastestLapString.append(") ");
				
				fastestOverallLapField.setText(fastestLapString.toString());
			}
		} else {
			fastestOverallLapField.setText("");
		}
	}
	
	private void removePlayerFromTrackMap(byte playerID) {
		if(trackMap != null) {
			trackMap.removePlayerFromTrackMap(playerID);
		}
	}

	/*
	 * (non-Javadoc)
	 * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
	 */
	public void actionPerformed(ActionEvent e) {
		if(e.getSource() instanceof PlayerDisplayRow) {
			// Pass on events from the display rows to the main class
			listener.actionPerformed(e);
		} else if(e.getSource() instanceof TrackCanvas) {
			// Pass on events from the track map
			listener.actionPerformed(e);
		} else {
			print("actionPerformed - unrecognised source class: "+
					e.getSource().getClass().getSimpleName());
		}
	}
	
	public Dimension getMinimumSize() {
		if(displayRows != null && displayRows.size() > 0) {
			for(PlayerDisplayRow displayRow : displayRows.values()) {
				Dimension displayRowMinSize = displayRow.getMinimumSize();
				Dimension mainBoxMinSize = mainBox.getMinimumSize();
				
				//print("getMinimumSize - returning from (1): "+new Dimension(
				//		displayRowMinSize.width, mainBoxMinSize.height).toString());
				
				return new Dimension(displayRowMinSize.width, mainBoxMinSize.height);
			}
			
			// Won't be executed
			return null;
		} else {
			Dimension mainBoxMinSize = mainBox.getMinimumSize();

			//print("getMinimumSize - returning from (1): "+new Dimension(
			//		mainBoxMinSize.width+20, mainBoxMinSize.height+10).toString());
			
			return new Dimension(mainBoxMinSize.width+20, mainBoxMinSize.height+10);
		}
	}
	
	private void speedTrapTimerMessageFire(ActionEvent e) {
		Timer sourceTimer = (Timer) e.getSource();
		sourceTimer.stop();
	}
	
	private void addSpeedTrapReadingMessage(SpeedTrapReading speedTrapReading) {
		messagePanel.addMessage("Speed Trap - "+speedTrapReading.toString());
	}
	
	// Start of TimingModelListener methods
	/*
	 * (non-Javadoc)
	 * @see lfsqualifyingticker.structures.TimingModelListener#
	 * playerJoinedSession(net.sf.jinsim.response.NewPlayerResponse)
	 */
	public void playerJoinedSession(byte playerIDJoining) {
		if(displayRows.containsKey(playerIDJoining)) {
			DriverStatus status = timingModel.getDriverStatus(playerIDJoining);
			
			displayRows.get(playerIDJoining).setDriverStatus(status);
		} else {
			// If the display row wasn't found update the full display
			updateDisplay();
		}
	}

	/*
	 * (non-Javadoc)
	 * @see lfsqualifyingticker.structures.TimingModelListener#
	 * playerLeftSession(net.sf.jinsim.response.PlayerLeavingResponse)
	 */
	public void playerLeftSession(byte playerIDLeaving) {
		if(displayRows.containsKey(playerIDLeaving)) {
			displayRows.remove(playerIDLeaving);
		}
			
		removePlayerFromTrackMap(playerIDLeaving);
		
		updateDisplay();
	}

	/*
	 * (non-Javadoc)
	 * @see lfsqualifyingticker.structures.TimingModelListener#
	 * playerPitted(net.sf.jinsim.response.PlayerPitsResponse)
	 */
	public void playerPitted(byte playerIDPitting) {
		if(displayRows.containsKey(playerIDPitting)) {
			displayRows.remove(playerIDPitting);
		}
			
		removePlayerFromTrackMap(playerIDPitting);
			
		updateDisplay();
	}

	/*
	 * (non-Javadoc)
	 * @see lfsqualifyingticker.structures.TimingModelListener#
	 * playerStartedHotlap(net.sf.jinsim.response.NewPlayerResponse)
	 */
	public void playerStartedHotlap(byte playerIDStartedHotlap) {
		if(displayRows.containsKey(playerIDStartedHotlap)) {
			DriverStatus status = timingModel.getDriverStatus(playerIDStartedHotlap);
			
			displayRows.get(playerIDStartedHotlap).setDriverStatus(status);
		} else {
			updateDisplay();
		}
	}
	
	/*
	 * (non-Javadoc)
	 * @see lfsqualifyingticker.structures.TimingModelListener#
	 * fastestLapUpdated(lfsqualifyingticker.structures.Lap)
	 */
	public void fastestLapUpdated(Lap fastestLap) {
		updateFastestLapInformation(fastestLap);
	}

	/*
	 * (non-Javadoc)
	 * @see lfsqualifyingticker.structures.TimingModelListener#
	 * updatePlayerPositionNodes(java.util.HashMap)
	 */
	public void updatePlayerPositionNodes(
			HashMap<PlayerInfo, MiniCompCar> playerPositions) {
		if(trackMap != null) {
			trackMap.updatePlayerPositionNodes(playerPositions);
		}
	}
	
	/*
	 * (non-Javadoc)
	 * @see lfsqualifyingticker.structures.TimingModelListener#
	 * resultReceived(net.sf.jinsim.response.ResultResponse)
	 */
	public void resultReceived(ResultResponse result) {
		if(QualiTickerPrefs.getDisplayMessagePanel()) {
			if(messagePanel != null) {
				/*
				 * Result position is -1 when the player completes a 
				 * lap that's slower than their fastest lap
				 */ 
				if(result.getResultPosition() != -1 && result.getResultPosition() != 255) {
					messagePanel.addMessage("Laptime: "+Helper.getPlainPlayerName(
							result.getNickname())+" - "+
							Helper.getTimeString(result.getBestLapTime())+
							" ("+Helper.getPositionStringForInt(
									result.getResultPosition()+1)+" position)");
				}
			}
		}
	}
	
	/*
	 * (non-Javadoc)
	 * @see lfsqualifyingticker.structures.TimingModelListener#
	 * bestSplitUpdated(lfsqualifyingticker.structures.PlayerInfo,
	 * net.sf.jinsim.response.SplitTimeResponse)
	 */
	public void bestSplitUpdated(PlayerInfo player, SplitTimeResponse split) {
		// Update the appropriate best split label
		bestSplitLabels[split.getSplit()-1].setText("Split "+split.getSplit()+
				": "+Helper.getTimeString(split)+" ("+
				Helper.getPlainPlayerName(player.getPlayerName())+")");
		
		if(QualiTickerPrefs.getDisplayMessagePanel()) {
			if(messagePanel != null) {
				if(player == null || split == null) {
					return;
				}
				
				messagePanel.addMessage("Best Split "+split.getSplit()+": "+
						Helper.getPlainPlayerName(player.getPlayerName()+" - "+
						Helper.getTimeString(split)));
			}
		}
	}
	
	/*
	 * (non-Javadoc)
	 * @see lfsqualifyingticker.structures.TimingModelListener#
	 * speedTrapReadingReceived(lfsqualifyingticker.
	 * structures.SpeedTrapReading)
	 */
	public void bestSpeedTrapReadingReceived(final SpeedTrapReading speedTrapReading) {
		if(speedTrapReading == null) {
			return;
		}
		
		/*
		 * Don't want to immediately print this speed trap reading because it's
		 * possible there will be multiple packets received for the given node
		 * so 2 or 3 best speed messages would be printed straight after each
		 * other. Therefore, the message display has to be postponed until
		 * the best one has been received
		 */
		
		// If there's an existing timer waiting to send the message stop it
		if(waitingBestSpeedTrapMessageTimers.get(
				speedTrapReading.getSpeedTrapName()) != null) {
			waitingBestSpeedTrapMessageTimers.get(
					speedTrapReading.getSpeedTrapName()).stop();
		}
		
		/*
		 * Make the delay for displaying message larger than the InSim buffer
		 * delay. If it's top speed make the delay larger again
		 */
		int delay = 0;
		
		if(speedTrapReading.getSpeedTrapName().equals(SpeedTrapReading.TOP_SPEED)) {
			delay = Math.max(3000, QualiTickerPrefs.getInSimBufferDelay()*3);
		} else {
			delay = Math.max(1500, QualiTickerPrefs.getInSimBufferDelay()*2);
		}
		
		Timer newMessageTimer = new Timer(delay, new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				addSpeedTrapReadingMessage(speedTrapReading);
				speedTrapTimerMessageFire(e);
			}
		});
		newMessageTimer.start();
		
		waitingBestSpeedTrapMessageTimers.put(speedTrapReading.getSpeedTrapName(), 
				newMessageTimer);
	}
	
	// End of TimingModelListener methods
}
