package lfsqualifyingticker;

import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;

import javax.swing.JOptionPane;
import javax.swing.Timer;

import lfsqualifyingticker.models.TimingModel;
import lfsqualifyingticker.structures.LFSDisplayMessage;
import lfsqualifyingticker.structures.QualiTickerPrefs;
import lfsqualifyingticker.structures.Lap;
import lfsqualifyingticker.structures.PlayerDisplayRow;
import lfsqualifyingticker.structures.QualifyingSessionInformation;
import lfsqualifyingticker.structures.ResponseAndTime;
import lfsqualifyingticker.structures.Helper;
import lfsqualifyingticker.structures.TrackCanvas;
import lfsqualifyingticker.structures.QualifyingSessionInformation.SessionType;
import net.sf.jinsim.Channel;
import net.sf.jinsim.SimpleClient;
import net.sf.jinsim.TCPChannel;
import net.sf.jinsim.Tiny;
import net.sf.jinsim.request.InSimRequest;
import net.sf.jinsim.request.InitRequest;
import net.sf.jinsim.request.SetCarCameraRequest;
import net.sf.jinsim.request.TinyRequest;
import net.sf.jinsim.response.CameraPositionResponse;
import net.sf.jinsim.response.ConnectionLeaveResponse;
import net.sf.jinsim.response.InSimListener;
import net.sf.jinsim.response.InSimResponse;
import net.sf.jinsim.response.LapTimeResponse;
import net.sf.jinsim.response.MultiCarInfoResponse;
import net.sf.jinsim.response.NewConnectionResponse;
import net.sf.jinsim.response.NewPlayerResponse;
import net.sf.jinsim.response.PlayerLeavingResponse;
import net.sf.jinsim.response.PlayerPitsResponse;
import net.sf.jinsim.response.RaceStartResponse;
import net.sf.jinsim.response.ResultResponse;
import net.sf.jinsim.response.SplitTimeResponse;
import net.sf.jinsim.response.StateResponse;

public class LFSQualifyingTicker implements InSimListener, ActionListener {
	// The timing model
	private TimingModel timingModel = new TimingModel();
	
	// The GUI for the LFS Qualifying Ticker application
	private LFSQualifyingTickerGUI gui;
	
	// The class which will actually update the displayed gap in LFS
	private LFSDisplayUpdater lfsDisplayUpdater;
	
	//	A global buffer to hold responses until they can be handled
	private ArrayList<ResponseAndTime> globalBuffer = new ArrayList<ResponseAndTime>(50);
	
	// The client that will communicate with LFS
	private SimpleClient client;

	// A Timer which will fire periodically to handle the response in the buffer
	private Timer bufferTimer;
	
	// A timer which will fire periodically to update the LFS display
	private Timer updateDisplayTimer;
	
	// A timer which will fire periodically and request the current camera position
	private Timer requestCameraPositionTimer;
	
	// The current camera position in LFS (used to determine viewed player)
	private CameraPositionResponse currentCameraPosition = null;

//	private HashMap<Class<?>, ArrayList<ResponseAndTime>> performanceTestingResponses = 
//		new HashMap<Class<?>, ArrayList<ResponseAndTime>>();
//	private int numPackets = 0;

	public LFSQualifyingTicker() {
		// Connect to LFS first
		connectToLFS();

		gui = new LFSQualifyingTickerGUI(timingModel, this);
		
		/* 
		 * Create the timers that will handle InSim packets, update the LFS display and 
		 * request camera position periodically
		 */
		bufferTimer = new Timer(QualiTickerPrefs.getInSimBufferDelay(), 
				new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				handleResponses();
			}
		});

		updateDisplayTimer = new Timer(QualiTickerPrefs.getUpdateLFSDisplayDelay(), 
				new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				updateLFSDisplay();
			}
		});

		requestCameraPositionTimer = new Timer(
				QualiTickerPrefs.getRequestCameraPositionDelay(), 
					new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				requestCameraPosition();
			}
		});

		// Create the display updater, passing in the client which was created in connectToLFS method
		lfsDisplayUpdater = new LFSDisplayUpdater(client);

		bufferTimer.start();
		updateDisplayTimer.start();
		requestCameraPositionTimer.start();

		print(QualiTickerPrefs.getApplicationName()+" "+
				QualiTickerPrefs.getVersion()+" is running\n");
	}
	
	private void connectToLFS() {
        try {
            client = new SimpleClient();

            client.addListener(this);

            String hostname = QualiTickerPrefs.getHostName();
            int hostPort = QualiTickerPrefs.getHostPort();
            String adminPassword = QualiTickerPrefs.getAdminPassword();

            Channel channel = new TCPChannel(hostname, hostPort);
            
            /*
             * Connect to LFS specifying the name of the application and that we want
             * to receive car information every 50 ms. Also specify the local flag
             * so buttons created by this application can be displayed at the same
             * time as buttons for another application
             */
            client.connect(channel, adminPassword, "Quali Ticker", 
            		(short) (InitRequest.RECEIVE_MULTI_CAR_INFO+InitRequest.LOCAL), 50, 0);

            // Request race start information (in case application is run while a race is in progress)
    		client.send(new TinyRequest(Tiny.RESTART));

    		client.send(new TinyRequest(Tiny.SEND_STATE_INFO));
        } catch(Exception e) {
        	e.printStackTrace();

        	LFSQualifyingTickerStart.print("Fatal exception starting application, must exit!");

        	JOptionPane.showMessageDialog(null, "Can't connect to LFS!\nMake sure you have enabled InSim " +
        			"on the correct port number in LFS", "Can't Connect To LFS", JOptionPane.ERROR_MESSAGE);

        	System.exit(1);
        }
	}
	
	/*
	 * Update the message currently being displayed in LFS. What the message gets updated to
	 * depends on a number of factors including whether or not anyone has posted a hotlap
	 * yet, whether the viewed player is on a hotlap etc
	 */
	private void updateLFSDisplay() {
		/* 
		 * Work out the message to send to LFS by using the viewed player and 
		 * information from the timing model
		 */
		//long startTime = System.nanoTime();
		
		LFSDisplayMessage messageToSend = getMessageToSendToLFS();
		
//		MessageType messageType = messageToSend.getMessageType();
//		
//		if(messageType.equals(MessageType.TIME_DIFFERENCE_AND_POSITION_MESSAGE) || 
//				messageType.equals(MessageType.TIME_DIFFERENCE_ONLY_MESSAGE)) {
//			print(Helper.getTimeTaken("get message to send to LFS", startTime));
//		}

		try {
			lfsDisplayUpdater.updateLFSDisplay(messageToSend);
		} catch(IOException ioe) {
			ioe.printStackTrace();
		}
	}
	
	// Returns an object representing what should be shown in LFS
	private LFSDisplayMessage getMessageToSendToLFS() {
		// Check if the session type is qualifying session
		QualifyingSessionInformation sessionInformation = timingModel.getSessionInformation();

		if(!sessionInformation.getSessionType().equals(SessionType.QUALIFYING_SESSION)) {
			// Not a qualifying session, send no message
			return new LFSDisplayMessage(LFSDisplayMessage.MessageType.NO_DISPLAY_MESSAGE);
		}
		
		Lap fastestLap = timingModel.getFastestOverallLap();
		
		if(fastestLap == null) {
			return new LFSDisplayMessage(LFSDisplayMessage.MessageType.NO_LAPS_COMPLETED_MESSAGE);
		} else {
			// Use the currently viewed player to determine live gap
			if(currentCameraPosition != null && currentCameraPosition.getPlayerIndex() == 0) {
				return new LFSDisplayMessage(LFSDisplayMessage.MessageType.NO_PLAYER_VIEWED);
			} else {
				if(currentCameraPosition == null) {
					return new LFSDisplayMessage(LFSDisplayMessage.
							MessageType.NO_DISPLAY_MESSAGE);
				} else {
					return timingModel.getLFSDisplayMessageForViewedPlayer(
							currentCameraPosition.getPlayerIndex());
				}
			}
		}
	}

	/*
	 * (non-Javadoc)
	 * @see net.sf.jinsim.response.InSimListener#packetReceived(
	 * net.sf.jinsim.response.InSimResponse)
	 */
	public synchronized void packetReceived(InSimResponse resp) {
		/*
		 * When a new packet is received add it to the global buffer and attach
		 * the time at which it was received.
		 */
		globalBuffer.add(new ResponseAndTime(resp, System.nanoTime()));
	}
	
	// Request the current camera position from LFS
	private void requestCameraPosition() {
		// Request the camera position from LFS (to get view playerID)
		sendRequestToLFS(new TinyRequest(Tiny.SEND_CAMERA_POSITION));
	}
	
	/*
	 * Handle the responses that are currently in the global buffer. This
	 * is called from the buffer timer which fires periodically
	 */
	private synchronized void handleResponses() {
		if(globalBuffer.size() > 0) {
			ResponseAndTime[] temporaryBuffer = globalBuffer.toArray(
					new ResponseAndTime[globalBuffer.size()]);
			
			globalBuffer.clear();
			
			// Sort the packets by priority
			Arrays.sort(temporaryBuffer, new Comparator<ResponseAndTime>() {
				public int compare(ResponseAndTime o1, ResponseAndTime o2) {
					Integer o1Priority = Helper.getInSimPacketPriority(
							o1.getResponse().getClass());
					Integer o2Priority = Helper.getInSimPacketPriority(
							o2.getResponse().getClass());
					
					if(o1Priority == null && o2Priority == null) {
						return 0;
					} else if(o1Priority == null) {
						return 1;
					} else if(o2Priority == null) {
						return -1;
					} else {
						return o1Priority.compareTo(o2Priority);
					}
				}
			});

			// Handle each of the responses in the temporary buffer
			for(ResponseAndTime responseAndTime : temporaryBuffer) {
				handleResponse(responseAndTime);
			}
		}
	}
	
	// Handle a given response
	private synchronized void handleResponse(ResponseAndTime responseAndTime) {
		InSimResponse response = responseAndTime.getResponse();

		//numPackets++;
		
		//long startNs = System.nanoTime();

		switch(response.getPacketType()) {
			case MULIT_CAR_INFO: {
				timingModel.handleMultiCarInfoResponse((MultiCarInfoResponse) response, 
						responseAndTime.getTimePacketReceivedNs());
				break;
			} case CAMERA_POSITION: {
				// Update the current camera position
				currentCameraPosition = (CameraPositionResponse) response;
				break;
			} case NEW_CONNECTION: {
				timingModel.addConnection((NewConnectionResponse) response);
				break;
			} case NEW_PLAYER: {
				timingModel.addPlayer((NewPlayerResponse) response);
				break;
			} case STATE: {
				timingModel.handleStateResponse((StateResponse) response);
				break;
			} case CONNECTION_LEFT: {
				timingModel.removeConnection((ConnectionLeaveResponse) response);
				break;
			} case PLAYER_LEAVE: {
				timingModel.removePlayer((PlayerLeavingResponse) response);
				break;
			} case LAP: {
				timingModel.handleLaptimeResponse((LapTimeResponse) response);
				break;
			} case SPLIT: {
				timingModel.handleSplitTimeResponse((SplitTimeResponse) response);
				break;
			} case PLAYER_PIT: {
				timingModel.handlePlayerPitsResponse((PlayerPitsResponse) response);
				break;
			} case RACE_START: {
				RaceStartResponse raceStartResponse = (RaceStartResponse) response;
				
				timingModel.handleRaceStartResponse(raceStartResponse);

				// GUI may not be used
				if(gui != null) {
					gui.raceStartResponseReceived(raceStartResponse);
				}

				try {
					// Send requests for players and connections
					client.send(new TinyRequest(Tiny.ALL_CONNECTIONS));
	    			client.send(new TinyRequest(Tiny.ALL_PLAYERS));
				} catch(IOException ioe) {
					ioe.printStackTrace();
				}
				
				break;
			} case CLOSE: {
				JOptionPane.showMessageDialog(null, "There has been a fatal error with the " +
						"connection to LFS! Quali Ticker must exit.\nThis may have been caused " +
						"by viewing a replay at faster than 1.0X speed\n or by LFS crashing.", 
						"Fatal Connection Error", JOptionPane.ERROR_MESSAGE);
				
				System.exit(1);
				break;
			} case RESULT_CONFIRMED: {
				timingModel.handleResultResponse((ResultResponse) response);
				
				break; 
			} 
			case MESSAGE: { break; } 
			case MESSAGE_EXTENDED: { break; } 
			case MESSAGE_OUT: { break; } 
			case MESSAGE_TO_CONNECTION: { break; } 
			case MESSAGE_TO_LOCAL: { break; } 
			case PIT_LANE: { break; } 
			case TINY: { break; } 
			case CAMERA_CHANGED: { break; } 
			case FLAG: { break; } 
			case PIT: { break; } 
			case PIT_FINISHED: { break; } 
			case CONNECTION_PLAYER_RENAMED: { break; } 
			case AUTOCROSS_LAYOUT: { break; } 
			case AUTOCROSS_HIT: { break; } 
			case REORDER: { break; } 
			case START_MULTIPLAYER: { break; } 
			case CAR_RESET: { break; } 
			case FINISHED_RACE: { break; } 
			case PENALTY: { break; } 
			case TAKE_OVER_CAR: { break; } 
			case VOTE_NOTIFICATION: { break; } 
			case OUT_GAUGE: { break; } 
			case NODE_LAP: { break; } 
			case OUT_SIM: { break; } 
			case SMALL: { break; } 
			case PLAYER_FLAGS: { break; } 
			case BUTTON: { break; } 
			case BUTTON_CLICKED: { break; } 
			case BUTTON_FUNCTION: { break; } 
			case BUTTON_TYPED: { break; } 
			case HIDDEN_MESSAGE: { break; } 
			case INSIM_INITIALIZE: { break; }
			default: {
				print(response.getClass().getSimpleName());
			}
		}
		
//		if(performanceTestingResponses.containsKey(response.getClass())) {
//			performanceTestingResponses.get(response.getClass()).add(new ResponseAndTime(
//					response, System.nanoTime()-startNs));
//		} else {
//			ArrayList<ResponseAndTime> newList = new ArrayList<ResponseAndTime>();
//			newList.add(new ResponseAndTime(response, System.nanoTime()-startNs));
//			performanceTestingResponses.put(response.getClass(), newList);
//		}
//		
//		if(numPackets % 1000 == 0) {
//			double slowestAverage = Integer.MIN_VALUE;
//			Class<?> slowestAverageClass = null;
//			long totalOverall = 0;
//			
//			for(List<ResponseAndTime> oneList : performanceTestingResponses.values()) {
//				long quickest = Long.MAX_VALUE, slowest = Long.MIN_VALUE, total=0;
//				
//				for(ResponseAndTime oneResponse : oneList) {
//					if(oneResponse.getTimePacketReceivedNs() < quickest) {
//						quickest = oneResponse.getTimePacketReceivedNs();
//					}
//					if(oneResponse.getTimePacketReceivedNs() > slowest) {
//						slowest = oneResponse.getTimePacketReceivedNs();
//					}
//					total += oneResponse.getTimePacketReceivedNs();
//					totalOverall += oneResponse.getTimePacketReceivedNs();
//				}
//
//				double averageMs = (total/(double) oneList.size())/1000000.0;
//
//				BigDecimal quickestBD = new BigDecimal(quickest/1000000.0);
//				BigDecimal slowestBD = new BigDecimal(slowest/1000000.0);
//				BigDecimal averageBD = new BigDecimal((total/(double) oneList.size())/1000000.0);
//				
//				quickestBD = quickestBD.setScale(2, BigDecimal.ROUND_HALF_UP);
//				slowestBD = slowestBD.setScale(2, BigDecimal.ROUND_HALF_UP);
//				averageBD = averageBD.setScale(2, BigDecimal.ROUND_HALF_UP);
//				
//				if(averageMs > slowestAverage) {
//					slowestAverage = averageMs;
//					slowestAverageClass = oneList.get(0).getResponse().getClass();
//				}
//
//				print(oneList.get(0).getResponse().getClass().getSimpleName()+
//						", slowest: "+slowestBD.toPlainString()+", quickest: "+quickestBD.toPlainString()+
//						", average: "+averageBD.toPlainString());
//			}
//			
//			BigDecimal slowestAverageBD = new BigDecimal(slowestAverage);
//			slowestAverageBD = slowestAverageBD.setScale(2, BigDecimal.ROUND_HALF_UP);
//			
//			BigDecimal overallAverage = new BigDecimal((totalOverall/(double)numPackets)/1000000.0);
//			overallAverage = overallAverage.setScale(2, BigDecimal.ROUND_HALF_UP);
//			
//			print("");
//			print("slowest average class: "+slowestAverageClass.getSimpleName()+", "+
//					slowestAverageBD.toPlainString()+"ms");
//			print("overall average: "+overallAverage.toPlainString()+"ms");
//		}
	}
	
	// Send the given request to LFS
	private void sendRequestToLFS(InSimRequest request) {
		try {
			client.send(request);
		} catch(Exception e) {
			e.printStackTrace();
		} 
	}

	private void print(String msg) {
		LFSQualifyingTickerStart.print(this.getClass().getSimpleName()+" - "+msg);
	}

	/*
	 * (non-Javadoc)
	 * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
	 */
	public void actionPerformed(ActionEvent e) {
		// Event from display row 
		if(e.getSource() instanceof PlayerDisplayRow) {
			// View player command (request for camera focus)
			if(e.getActionCommand().equals(LFSQualifyingTickerGUI.VIEW_PLAYER_COMMAND)) {
				// Use the ID as playerID to view
				byte playerIDToView = (byte) e.getID();
				

				// Send a new camera position to LFS with the playerID
				SetCarCameraRequest newCamPosition = new SetCarCameraRequest();
				newCamPosition.setUniqueId(playerIDToView);
				
				// If modifiers is set use it to determine camera type
				if(e.getModifiers() != Integer.MIN_VALUE) {
					byte cameraType = (byte) e.getModifiers();
				
					newCamPosition.setCameraType(cameraType);
				} else {
					if(currentCameraPosition != null) {
						newCamPosition.setCameraType(currentCameraPosition.getCameraType());
					}
				}

				sendRequestToLFS(newCamPosition);
				
				// Request the camera position from LFS
				requestCameraPosition();
			}
		} else if(e.getSource() instanceof TrackCanvas) {
			// Command came from a click on the track
			if(e.getActionCommand().equals(LFSQualifyingTickerGUI.VIEW_PLAYER_COMMAND)) {
				// Use the ID as playerID to view
				byte playerIDToView = (byte) e.getID();

				// Send a new camera position to LFS with the playerID
				SetCarCameraRequest newCamPosition = new SetCarCameraRequest();
				newCamPosition.setUniqueId(playerIDToView);
				
				// If modifiers is set use it to determine camera type
				if(e.getModifiers() != Integer.MIN_VALUE) {
					byte cameraType = (byte) e.getModifiers();
				
					newCamPosition.setCameraType(cameraType);
				} else {
					if(currentCameraPosition != null) {
						newCamPosition.setCameraType(currentCameraPosition.getCameraType());
					}
				}

				sendRequestToLFS(newCamPosition);
				
				// Request the camera position from LFS
				requestCameraPosition();
			}
		} else {
			print("actionPerformed - unrecognised source class: "+
					e.getSource().getClass().getSimpleName()+
					", actionCommand: "+e.getActionCommand());
		}
	}
}
