package lfsqualifyingticker.structures;

import java.io.Serializable;

import lfsqualifyingticker.LFSQualifyingTickerStart;

import net.sf.jinsim.Track;
import net.sf.jinsim.Weather;
import net.sf.jinsim.Wind;
import net.sf.jinsim.response.RaceStartResponse;
import net.sf.jinsim.response.StateResponse;

/**
 * Holds information about a race
 * @author Gus
 *
 */

public class QualifyingSessionInformation implements Serializable {
	/**
	 * 
	 */
	private static final long serialVersionUID = -788043296756459483L;
	
	public enum SessionType {UNKNOWN, NO_SESSION, RACE_SESSION, 
		QUALIFYING_SESSION, PRACTICE_SESSION};
	
	private Track track;
	private Wind wind;
	private Weather weather;
	
	private Boolean canReset;
	
	private int numLaps, numHours, qualifyingMinutes, 
		split1NodeIndex=Integer.MAX_VALUE, split2NodeIndex=Integer.MAX_VALUE, 
		split3NodeIndex=Integer.MAX_VALUE, 
		finishLineNodeIndex=Integer.MAX_VALUE, numNodes=Integer.MAX_VALUE;
	
	private SessionType sessionType = SessionType.UNKNOWN;

	public QualifyingSessionInformation() {
	}
	
	public QualifyingSessionInformation(RaceStartResponse raceInformation) {
		updateSessionInformation(raceInformation);
	}
	
	public void updateSessionInformation(RaceStartResponse raceInformation) {
		this.track = raceInformation.getTrack();
		this.wind = raceInformation.getWind();
		this.weather = raceInformation.getWeather();

		this.canReset = raceInformation.canReset();

		this.numLaps = raceInformation.getRaceLaps();
		this.numHours = raceInformation.getRaceHours();
		
		this.qualifyingMinutes = raceInformation.getQualifingMinutes();

		// Practice session if no laps and no hours set
		if(numLaps == 0 && numHours == 0 && qualifyingMinutes == 0) {
			sessionType = SessionType.PRACTICE_SESSION;
		} else {
			if(numLaps == 0) {
				sessionType = SessionType.QUALIFYING_SESSION;
			} else {
				sessionType = SessionType.RACE_SESSION;
			}
		}
		
		this.numNodes = raceInformation.getNumberOfNodes();
		this.split1NodeIndex = raceInformation.getSplit1NodeIndex();
		this.split2NodeIndex = raceInformation.getSplit2NodeIndex();
		this.split3NodeIndex = raceInformation.getSplit3NodeIndex();
		this.finishLineNodeIndex = raceInformation.getFinishLineNodeIndex();
		
		// print("numNodes: "+numNodes+", finish line: "+finishLineNodeIndex);
	}
	
	public void updateSessionInformation(StateResponse stateResponse) {
		this.track = stateResponse.getTrack();
		this.weather = Weather.getWeather(stateResponse.getWeatherType());
		this.wind = Wind.getWind(stateResponse.getWindType());

		this.numLaps = stateResponse.getNumLaps().getLaps();
		
		if(stateResponse.getNumLaps().isTimedRace()) {
			this.numHours = stateResponse.getNumLaps().getLaps();
		} else {
			this.numHours = 0;
		}

		this.qualifyingMinutes = stateResponse.getQualifyingLength();

		switch(stateResponse.getRaceInProgress()) {
			case 0: {
				sessionType = SessionType.NO_SESSION;
				
				break;
			} case 1: {
				if(stateResponse.getNumLaps().isPractice()) {
					sessionType = SessionType.PRACTICE_SESSION;
				} else {
					sessionType = SessionType.RACE_SESSION;
				}
				
				break;
			} case 2: {
				sessionType = SessionType.QUALIFYING_SESSION;
				
				break;
			}
		}
	}
	
	public void setTrack(Track track) {
		this.track = track;
	}
	
	public Track getTrack() {
		return track;
	}
	
	public void setWind(Wind wind) {
		this.wind = wind;
	}
	
	public Wind getWind() {
		return wind;
	}
	
	public void setWeather(Weather weather) {
		this.weather = weather;
	}
	
	public Weather getWeather() {
		return weather;
	}

	public void setCanReset(boolean canReset) {
		this.canReset = canReset;
	}
	
	public Boolean getCanReset() {
		return canReset;
	}
	
	public void setNumLaps(int numLaps) {
		this.numLaps = numLaps;
	}
	
	public int getNumLaps() {
		return numLaps;
	}
	
	public void setSessionType(SessionType sessionType) {
		this.sessionType = sessionType;
	}
	
	public SessionType getSessionType() {
		return sessionType;
	}
	
	public void setQualifyingMinutes(int qualifyingMinutes) {
		this.qualifyingMinutes = qualifyingMinutes;
	}
	
	public int getQualifyingMinutes() {
		return qualifyingMinutes;
	}
	
	public int getNumHours() {
		return numHours;
	}
	
	public int getSplit1NodeIndex() {
		return split1NodeIndex;
	}
	
	public int getSplit2NodeIndex() {
		return split2NodeIndex;
	}
	
	public int getSplit3NodeIndex() {
		return split3NodeIndex;
	}
	
	public int getFinishLineNodeIndex() {
		return finishLineNodeIndex;
	}
	
	public int getNumNodes() {
		return numNodes;
	}
	
	public int getSplitForNodeIndex(int nodeIndex) {
		if((nodeIndex > finishLineNodeIndex && nodeIndex <= numNodes) || 
				(nodeIndex >= 0 && nodeIndex <= split1NodeIndex)) {
			return 0;
		} else {
			if(split2NodeIndex != -1) {
				if(nodeIndex > split1NodeIndex && nodeIndex <= split2NodeIndex) {
					return 1;
				} else {
					if(split3NodeIndex != -1) {
						if(nodeIndex > split2NodeIndex && nodeIndex <= split3NodeIndex) {
							return 2;
						} else if(nodeIndex > split3NodeIndex && nodeIndex < finishLineNodeIndex) {
							return 3;
						} else {
							return Integer.MIN_VALUE;
						}
					} else {
						if(nodeIndex > split2NodeIndex && nodeIndex <= finishLineNodeIndex) {
							return 2;
						} else {
							return Integer.MIN_VALUE;
						}
					}
				}
			} else {
				if(nodeIndex > split1NodeIndex && nodeIndex <= finishLineNodeIndex) {
					return 1;
				} else {
					return Integer.MIN_VALUE;
				}
			}
		}
	}
	
	@SuppressWarnings("unused")
	private void print(String msg) {
		LFSQualifyingTickerStart.print(this.getClass().getSimpleName()+" : "+msg);
	}
}
