package lfsqualifyingticker.structures;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;

import lfsqualifyingticker.LFSQualifyingTickerStart;
import lfsqualifyingticker.models.TrackModel;

import net.sf.jinsim.Track;
import net.sf.jinsim.response.LapTimeResponse;
import net.sf.jinsim.response.SplitTimeResponse;

public class Lap implements Comparable<Lap> {
	// The connection information for the driver of this lap
	private ConnectionInfo connection;
	
	// The player information for the driver of this lap
	private PlayerInfo player;

	/*
	 * Timing data for each of the CompCar nodes received on this lap. 
	 * Key is the node index
	 */
	private HashMap<Integer, MiniCompCarAndTime> compCarTimes = 
		new HashMap<Integer, MiniCompCarAndTime>();
	
	// The latest CompCarAndTime received for this lap
	private MiniCompCarAndTime latestCompCarAndTime = null;
	
	// Splits on this lap
	private ArrayList<SplitTimeResponse> splits = 
		new ArrayList<SplitTimeResponse>(4);
	
	// The final laptime for this lap
	private LapTimeResponse laptime;
	
	// The short name of the track this lap was done on
	private String trackShortName;
	
	// The first CompCar on this lap
	private MiniCompCarAndTime firstCompCar;
	
	/**
	 * Construct a new Lap with the given player and connection 
	 * information
	 * @param connection
	 * @param player
	 */
	public Lap(ConnectionInfo connection, PlayerInfo player, 
			String trackShortName) {
		this.connection = connection;
		this.player = player;
		this.trackShortName = trackShortName;
	}

	/**
	 * Add the given TimingNode to this lap
	 * @param node
	 */
	public void addMiniCompCarAndTime(MiniCompCarAndTime compCar) {
		int node = compCar.getMiniCompCar().getNode();
		
		if(firstCompCar == null) {
			// Need to make sure the first comp car is after the finish line
			TrackData trackData = TrackModel.getDataForOneTrack(
					Track.getTrackByShortName(trackShortName));
			
			if(trackData != null) {
				int finishNode = trackData.getFinishNodeIndex();
				int nodeCount = trackData.getNodeCount();
				int thisNode = compCar.getMiniCompCar().getNode();
				
				if(finishNode  < nodeCount) {
					if(thisNode > finishNode) {
						firstCompCar = compCar;
					}
				} else {
					if(thisNode >= 0) {
						firstCompCar = compCar;
					}
				}		
			}
		}
		
		latestCompCarAndTime = compCar;
		
		if(compCarTimes.containsKey(node)) {
			//compCarTimes.get(node).add(compCar);
			// Only store one CompCar per node
		} else {
			compCarTimes.put(node, compCar);
		}
	}
	
	/**
	 * Returns the CompCarAndTime packet for the first CompCar received at 
	 * the given node or null if no CompCar have been received for the given 
	 * node index
	 * @param nodeIndex
	 * @return
	 */
	public MiniCompCarAndTime getFirstMiniCompCarPacketForNode(int nodeIndex) {
		if(compCarTimes.containsKey(nodeIndex)) {
			return compCarTimes.get(nodeIndex);
		} else {
			return null;
		}
	}
	
	/**
	 * Set the final laptime on this Lap
	 * @param laptime
	 */
	public void setLaptime(LapTimeResponse laptime) {
		this.laptime = laptime;
	}
	
	/**
	 * Get the laptime for this lap
	 * @return
	 */
	public LapTimeResponse getLaptime() {
		return laptime;
	}
	
	/**
	 * Get the player from this Lap
	 * @return
	 */
	public PlayerInfo getPlayer() {
		return player;
	}
	
	/**
	 * Get the connection from this lap
	 * @return
	 */
	public ConnectionInfo getConnection() {
		return connection;
	}
	
	/**
	 * Add the given split time to this lap
	 * @param splitTime
	 */
	public void addSplitTime(SplitTimeResponse splitTime) {
		splits.add(splitTime);
	}
	
	/**
	 * Returns the latest CompCarAndTime object received for this 
	 * lap or null if none have been received yet
	 * @return
	 */
	public MiniCompCarAndTime getLatestMiniCompCarAndTime() {
		return latestCompCarAndTime;
	}

	/**
	 * Compares the laptimes of the two given laps
	 * @param o1
	 * @param o2
	 * @return
	 */
	public int compare(Lap o1, Lap o2) {
		if(o1.getLaptime() != null && o2.getLaptime() != null) {
			int o1TimeMillis = o1.getLaptime().getTime().getTime();
			int o2TimeMillis = o2.getLaptime().getTime().getTime();
			
			if(o1TimeMillis < o2TimeMillis) {
				return -1;
			} else if(o1TimeMillis > o2TimeMillis) {
				return 1;
			} else {
				return 0;
			}
		} else {
			return 1;
		}
	}

	/**
	 * Compares the laptimes of this lap against the given lap if available
	 */
	public int compareTo(Lap o2) {
		return compare(this, o2);
	}
	
	/**
	 * Returns the split time at the given split on this lap if present 
	 * or null if there are no splits on this lap
	 * @param splitNumber
	 * @return
	 */
	public SplitTimeResponse getSplitNumber(byte splitNumber) {
		if(splits == null || splits.size() == 0) {
			return null;
		} else {
			for(SplitTimeResponse oneSplit : splits) {
				if(oneSplit.getSplit() == splitNumber) {
					return oneSplit;
				}
			}
			
			return null;
		}
	}
	
	/**
	 * Returns a Collection containing all of the splits on this lap or an 
	 * empty Collection if there are no splits attached to 
	 * this lap
	 * @return
	 */
	public Collection<SplitTimeResponse> getAllSplits() {
		return splits;
	}
	
	/**
	 * Returns the last completed split on this lap or null if no splits
	 * have been recorded
	 * @return
	 */
	public SplitTimeResponse getLastCompletedSplit() {
		if(splits == null || splits.size() == 0) {
			return null;
		} else {
			return splits.get(splits.size()-1);
		}
	}
	
	/**
	 * Get the track this lap was done on
	 * @return
	 */
	public String getTrackShortName() {
		return trackShortName;
	}
	
	/**
	 * Get the first CompCarAndTime received for this lap. May return 
	 * null if no comp cars have been received yet.
	 * @return
	 */
	public MiniCompCarAndTime getFirstMiniCompCarInLap() {
		return firstCompCar;
	}
	
	@SuppressWarnings("unused")
	private void print(String msg) {
		LFSQualifyingTickerStart.print(this.getClass().
				getSimpleName()+" - "+msg);
	}
}
