package lfsqualifyingticker.models;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;

import lfsqualifyingticker.LFSQualifyingTickerStart;
import lfsqualifyingticker.structures.ConnectionInfo;
import lfsqualifyingticker.structures.MiniCompCar;
import lfsqualifyingticker.structures.MiniCompCarAndTime;
import lfsqualifyingticker.structures.LFSDisplayMessage;
import lfsqualifyingticker.structures.PlayerInfo;
import lfsqualifyingticker.structures.QualiTickerPrefs;
import lfsqualifyingticker.structures.Lap;
import lfsqualifyingticker.structures.QualifyingSessionInformation;
import lfsqualifyingticker.structures.Helper;
import lfsqualifyingticker.structures.TimingModelListener;
import lfsqualifyingticker.structures.TrackData;
import lfsqualifyingticker.structures.PlayerDisplayRow.DriverStatus;
import lfsqualifyingticker.structures.QualifyingSessionInformation.SessionType;

import net.sf.jinsim.Track;
import net.sf.jinsim.response.ConnectionLeaveResponse;
import net.sf.jinsim.response.LapTimeResponse;
import net.sf.jinsim.response.MultiCarInfoResponse;
import net.sf.jinsim.response.NewConnectionResponse;
import net.sf.jinsim.response.NewPlayerResponse;
import net.sf.jinsim.response.PlayerLeavingResponse;
import net.sf.jinsim.response.PlayerPitsResponse;
import net.sf.jinsim.response.RaceStartResponse;
import net.sf.jinsim.response.ResultResponse;
import net.sf.jinsim.response.SplitTimeResponse;
import net.sf.jinsim.response.StateResponse;
import net.sf.jinsim.types.CompCar;

/**
 * The main model in the LFS Qualifying Ticker application. Holds information
 * including all the connections and players currently on the LFS server,
 * completed lap and current lap information.
 * @author Gus
 *
 */

public class TimingModel implements TimingModelListener {
	/* 
	 * A map containing all of the connections currently on the LFS server,
	 * key is the unique connectionID
	 */
	private Map<Byte, ConnectionInfo> currentConnections = 
			new HashMap<Byte, ConnectionInfo>(50);
	
	/* 
	 * A map containing all of the players currently in the session,
	 * key is the unique playerID
	 */
	private Map<Byte, PlayerInfo> currentPlayers = 
		new HashMap<Byte, PlayerInfo>(50);

	/*
	 * All the laps in this session. Key is the player who did these laps 
	 * (use the player here rather than playerID so players who are not on
	 * the server can still be shown)
	 */
	private Map<PlayerInfo, ArrayList<Lap>> allLaps = 
		new HashMap<PlayerInfo, ArrayList<Lap>>(50);
	
	// The fastest lap done by anyone in this session
	private Lap fastestLap = null;
	
	// Information for the current session (track, wind etc)
	private QualifyingSessionInformation sessionInformation = 
		new QualifyingSessionInformation();

	// Listeners for updates to this model (including the GUI)
	private ArrayList<TimingModelListener> listeners = 
		new ArrayList<TimingModelListener>(2);
	
	// Players who are in the session and hotlaps
	private HashSet<Byte> playersOnHotlap = new HashSet<Byte>(50);
	private HashSet<Byte> playersInSession = new HashSet<Byte>(50);
	
	// A cached version of the fastest lap each driver has done in this session
	private Lap[] fastestLapForEachDriverDuplicatesRemoved = null;
	
	// The best splits recorded in this session
	private SplitTimeResponse[] bestSplits = new SplitTimeResponse[4];

	/**
	 * Construct a new, empty TimingModel
	 */
	public TimingModel() {}
	
	/* ************************************************************************
	 * ** Start public methods in TimingModel *********************************
	 * ********************************************************************* */
	/**
	 * Add the given connection to the current connections on the server
	 * @param connection
	 */
	public void addConnection(NewConnectionResponse connection) {
//		print("addConnection: ID: "+connection.getConnectionId()+
//				", S2: "+connection.getUsername());

		if(!isQualifyingSession()) {
			return;
		}
		
		currentConnections.put(connection.getConnectionId(), new ConnectionInfo(connection));
	}
	
	/**
	 * Add the given player to the current players in the session
	 * @param player
	 */
	public void addPlayer(NewPlayerResponse player) {
//		print("addPlayer: "+Helper.getPlainPlayerName(player.getPlayerName())+
//				", playerID: "+player.getPlayerId()+", connectionID: "+player.getConnectionId());

		if(!isQualifyingSession()) {
			return;
		}
		
		byte playerID = player.getPlayerId();

		currentPlayers.put(playerID, new PlayerInfo(player));

		addPlayerToPlayersInSession(playerID);

		// Fire event on listeners
		playerJoinedSession(playerID);
	}
	
	/**
	 * Remove the given connection from the connections currently on the server
	 * @param connectionLeave
	 */
	public void removeConnection(ConnectionLeaveResponse connectionLeave) {
		if(currentConnections.containsKey(connectionLeave.getConnectionId())) {
			currentConnections.remove(connectionLeave.getConnectionId());
		} else {
			print("removeConnection - Could not find connectionID "+
					connectionLeave.getConnectionId()+" in currentConnections map");
		}
	}
	
	/**
	 * Remove the given player from the current players in the session
	 * @param playerLeaving
	 */
	public void removePlayer(PlayerLeavingResponse playerLeaving) {
		byte playerID = playerLeaving.getPlayerId();

		if(currentPlayers.containsKey(playerID)) {
//			NewPlayerResponse player = currentPlayers.get(playerLeaving.getPlayerId());
//
//			print("player leaving: "+player.getPlayerName()+", playerID: "+
//					playerLeaving.getPlayerId()+", connectionID: "+player.getConnectionId());
			
			currentPlayers.remove(playerID);
		} else {
			print("removePlayer - Could not find playerID "+
					playerID+" in currentPlayers map");
		}
		
		removeLapInProgressForPlayer(playerID);
		
		removePlayerFromPlayersInSession(playerID);
		removePlayerFromPlayersOnHotlapList(playerID);
		
		// Fire event on listeners
		playerLeftSession(playerID);
	}
	
	/**
	 * Handle the given multi car info response
	 * @param mci
	 */
	public void handleMultiCarInfoResponse(MultiCarInfoResponse mci, long packetTimeNs) {
		if(!isQualifyingSession()) {
			return;
		}
		
		// Map used to store the player's positions
		HashMap<PlayerInfo, MiniCompCar> playerPositions = new HashMap
			<PlayerInfo, MiniCompCar>(mci.getCarInfoList().size());
		
		for(CompCar compCar : mci.getCarInfoList()) {
			byte playerID = compCar.getPlayerId();
			
			// If the player is currently on a lap add this node index to that lap
			Lap playersCurrentLap = getLapInProgressForPlayer(playerID);
			
			if(playersCurrentLap != null) {
				playersCurrentLap.addMiniCompCarAndTime(
						new MiniCompCarAndTime(compCar, packetTimeNs));
			}

			// Add player to in session and hotlap if appropriate
			addPlayerToPlayersInSession(playerID);
			
			if(compCar.getLap() > 1) {
				addPlayerToPlayersOnHotlapList(playerID);
			}
			
			if(currentPlayers.containsKey(playerID)) {
				playerPositions.put(currentPlayers.get(playerID), new MiniCompCar(compCar));
			}
		}
		
		updatePlayerPositionNodes(playerPositions);
	}

	/**
	 * Handle the given laptime response
	 * @param laptimeResponse
	 */
	public void handleLaptimeResponse(LapTimeResponse laptimeResponse) {
		if(!isQualifyingSession()) {
			return;
		}
		
		byte playerID = laptimeResponse.getPlayerId();
		
		if(currentPlayers.containsKey(playerID)) {
			PlayerInfo player = currentPlayers.get(playerID);

			/*
			 *  Get the lap in progress for this player (there may be none if they just 
			 *  completed an outlap)
			 */
			Lap lapInProgress = getLapInProgressForPlayer(playerID);
			
			if(lapInProgress != null) {
				// laptime is 1 hour, remove this lap from all laps
				if(laptimeResponse.getTime() != null && 
						laptimeResponse.getTime().getHours() == 1) {
					ArrayList<Lap> allDriversLaps = allLaps.get(player);
					
					if(allDriversLaps != null && allDriversLaps.contains(lapInProgress)) {
						allDriversLaps.remove(lapInProgress);
					}
				} else {
					// Set the final laptime on the lap in progress
					lapInProgress.setLaptime(laptimeResponse);

					/*
					 * If this lap is not the fastest lap this driver
					 * has completed remove it from their laps
					 */
					Lap fastestLapForPlayer = getFastestLapForPlayer(player);
					
					if(fastestLapForPlayer != null) {
						if(fastestLapForPlayer != lapInProgress) {
							allLaps.get(player).remove(lapInProgress);
						}
					}
					
					// Add this lap to completed laps
					lapCompleted(lapInProgress);
				}
			}

			addPlayerToPlayersInSession(playerID);
			addPlayerToPlayersOnHotlapList(playerID);

			// Add a new lap for this driver
			if(currentConnections.containsKey(player.getConnectionID())) {
				ConnectionInfo connection = currentConnections.get(
						player.getConnectionID());

				addNewLap(new Lap(connection, player, 
						sessionInformation.getTrack().getShortname()));
			} else {
				print("!!! handleLaptimeResponse - could not find connectionID "+
						player.getConnectionID()+" in currentConnections map");
			}
		} else {
			print("!! handleLaptimeResponse - currentPlayers map doesn't " +
					"contain an entry for playerID "+playerID);
		}
	}
	
	/**
	 * Add the given split time to the player's lap in progress
	 * @param splitResponse
	 */
	public void handleSplitTimeResponse(SplitTimeResponse splitResponse) {
		if(!isQualifyingSession()) {
			return;
		}
		
		byte playerID = splitResponse.getPlayerId();
		
		addPlayerToPlayersInSession(playerID);
		
		Lap playersCurrentLap = getLapInProgressForPlayer(playerID);

		if(playersCurrentLap != null) {
			playersCurrentLap.addSplitTime(splitResponse);
		} /*else {
			print("!! Could not find a lap in progress for playerID "+
				splitResponse.getPlayerId());
		} */
		
		// Update the best split if applicable
		int splitNum = splitResponse.getSplit()-1;

		if(bestSplits[splitNum] == null || 
				bestSplits[splitNum].getTime().getTime() > 
				splitResponse.getTime().getTime()) {
			if(splitResponse.getTime().getHours() != 1) {
				//print("bestSplits["+splitNum+"] is null or slower than " +
				//		"new split: "+Helper.getTimeString(splitResponse));

				bestSplits[splitNum] = splitResponse;

				PlayerInfo player = getPlayerForID(playerID);

				if(player != null) {
					bestSplitUpdated(player, splitResponse);
				}
			}
		}
	}

	/**
	 * Update the session information
	 * @param stateResponse
	 */
	public void handleStateResponse(StateResponse stateResponse) {
		sessionInformation.updateSessionInformation(stateResponse);
	}
	
	/**
	 * Handle the given race start response
	 * @param raceStartResponse
	 */
	public void handleRaceStartResponse(RaceStartResponse raceStartResponse) {
		removeStateFromPreviousSession();

		sessionInformation.updateSessionInformation(raceStartResponse);
	}
	
	/**
	 * Handle the given ResultResponse
	 * @param resultResponse
	 */
	public void handleResultResponse(ResultResponse resultResponse) {
		if(!isQualifyingSession()) {
			return;
		}
		
		resultReceived(resultResponse);
	}

	/**
	 * Returns the lap currently in progress for a driver or null if the driver is
	 * not currently on a lap
	 * @param playerID
	 * @return
	 */
	public Lap getLapInProgressForPlayer(byte playerID) {
		//print("getLapInProgressForPlayer called with playerID "+playerID);
		PlayerInfo player = getPlayerForID(playerID);
		
		if(player != null) {
			if(allLaps.containsKey(player)) {
				for(Lap oneLap : allLaps.get(player)) {
					// Return the first lap that has no laptime (not completed)
					if(oneLap.getLaptime() == null) {
						return oneLap;
					}
				}

				return null;
			} else {
				return null;
			}
		} else {
			return null;
		}
	}
	
	/**
	 * Returns the player for the given playerID or null if it doesn't exist
	 * @param playerID
	 * @return
	 */
	public PlayerInfo getPlayerForID(byte playerID) {
		if(currentPlayers.containsKey(playerID)) {
			return currentPlayers.get(playerID);
		} else {
			return null;
		}
	}
	
	/**
	 * Returns the fastest lap done in this session or null if no laps have 
	 * been done
	 * @return
	 */
	public Lap getFastestOverallLap() {
		return fastestLap;
	}
	
	/**
	 * Returns the fastest lap for the given playerID or null if they haven't 
	 * yet completed a lap
	 * @param playerID
	 * @return
	 */
	public Lap getFastestLapForPlayer(PlayerInfo player) {
		//print("getFastestLapForPlayer called with playerID: "+playerID);
		if(player != null) {
			if(allLaps.containsKey(player)) {
				Lap fastestLap = null;

				for(Lap oneLap : allLaps.get(player)) {
					if(oneLap.getLaptime() != null) {
						if(fastestLap == null || 
								oneLap.getLaptime().getTime().getTime() < 
								fastestLap.getLaptime().getTime().getTime()) {
							fastestLap = oneLap;
						}
					}
				}

				return fastestLap;
			} else {
				return null;
			}
		} else {
			return null;
		}
	}
	
	/**
	 * Returns an array of laps containing the fastest lap for each player. 
	 * The array is NOT ordered in any particular way, the caller should 
	 * handle this. 
	 * @return
	 */
	public Lap[] getFastestLapForEachDriver() {
		ArrayList<Lap> fastestLaps = new ArrayList<Lap>(allLaps.size());
		
		for(PlayerInfo player : allLaps.keySet()) {
			Lap playerFastestLap = getFastestLapForPlayer(player);
			
			if(playerFastestLap != null) {
				fastestLaps.add(playerFastestLap);
			}
		}
		
		return fastestLaps.toArray(new Lap[fastestLaps.size()]);
	}
	
	/**
	 * Returns an array of laps containing the fastest lap driven for each
	 * driver. If multiple laps are found which have been driven
	 * by a player with the same racername only the 
	 * quickest will be returned. The laps in the returned array are
	 * ordered by laptime (fastest first)
	 * @return
	 */
	public Lap[] getFastestLapForEachDriverRemoveProbableDuplicates() {
		if(fastestLapForEachDriverDuplicatesRemoved != null) {
			return fastestLapForEachDriverDuplicatesRemoved;
		} else {
			HashMap<String, Lap> fastestLaps = new HashMap<String, Lap>();

			for(PlayerInfo player : allLaps.keySet()) {
				Lap playerFastestLap = getFastestLapForPlayer(player);

				if(playerFastestLap != null) {
					// There is an existing lap for this driver
					if(fastestLaps.containsKey(player.getPlayerName())) {
						Lap existingLap = fastestLaps.get(player.getPlayerName());

						/*
						 * If the new lap is quicker than the existing one remove
						 * the existing one and add the new one.
						 */
						if(playerFastestLap.getLaptime().getTime().getTime() < 
								existingLap.getLaptime().getTime().getTime()) {
							fastestLaps.remove(player.getPlayerName());

							fastestLaps.put(player.getPlayerName(), playerFastestLap);
						}
					} else {
						// No existing lap for same driver name, just add
						fastestLaps.put(player.getPlayerName(), playerFastestLap);
					}
				}
			}

			if(fastestLaps.size() > 0) {
				fastestLapForEachDriverDuplicatesRemoved = fastestLaps.values().
					toArray(new Lap[fastestLaps.size()]);
				
				// Sort array
				if(fastestLapForEachDriverDuplicatesRemoved.length > 1) {
					// Sort the laps by ultimate laptime
					Arrays.sort(fastestLapForEachDriverDuplicatesRemoved, 
							new Comparator<Lap>() {
						public int compare(Lap o1, Lap o2) {
							LapTimeResponse o1Lap = o1.getLaptime();
							LapTimeResponse o2Lap = o2.getLaptime();

							if(o1Lap == null) {
								return 1;
							}
							if(o2Lap == null) {
								return -1;
							}
							if(o1Lap.getTime().getTime() < o2Lap.getTime().getTime()) {
								return -1;
							} else if(o1Lap.getTime().getTime() > o2Lap.getTime().getTime()) {
								return 1;
							} else {
								return 0;
							}
						}
					});
				}
			} else {
				fastestLapForEachDriverDuplicatesRemoved = null;
			}

			return fastestLapForEachDriverDuplicatesRemoved;
		}
	}
	
	/**
	 * Returns all laps the player has done in this session or null if none have
	 * been done
	 * @param playerID
	 * @return
	 */
	public ArrayList<Lap> getAllLapsForPlayerID(byte playerID) {
		//print("getAllLapsForPlayerID called with playerID "+playerID);
		PlayerInfo player = getPlayerForID(playerID);
		
		if(player != null) {
			if(allLaps.containsKey(player)) {
				return allLaps.get(player);
			} else {
				return null;
			}
		} else {
			return null;
		}
	}
	
	/**
	 * Handle the given player pits response. If the player is currently on a lap
	 * remove it from their laps
	 * @param playerPits
	 */
	public void handlePlayerPitsResponse(PlayerPitsResponse playerPits) {
		if(!isQualifyingSession()) {
			return;
		}
		
		byte playerID = playerPits.getPlayerId();
		
		// If the player is currently on a lap remove it
		removeLapInProgressForPlayer(playerID);
		
		removePlayerFromPlayersInSession(playerID);
		removePlayerFromPlayersOnHotlapList(playerID);
		
		// Fire an event to let listeners know
		playerPitted(playerPits.getPlayerId());
	}
	
	/**
	 * Returns all the players currently in the session
	 * @return
	 */
	public Collection<PlayerInfo> getAllCurrentPlayers() {
		return currentPlayers.values();
	}

	/**
	 * Get the current status of the given player
	 * @param playerID
	 * @return
	 */
	public DriverStatus getDriverStatus(byte playerID) {
		boolean driverInSession = isPlayerInSession(playerID);
		boolean driverOnHotlap = isPlayerOnAHotlap(playerID);
		
		if(driverInSession && driverOnHotlap) {
			return DriverStatus.IN_SESSION_ON_HOTLAP;
		} else if(driverInSession && !driverOnHotlap) {
			return DriverStatus.IN_SESSION_NOT_ON_HOTLAP;
		} else {
			return DriverStatus.NOT_IN_SESSION;
		}
	}

	/**
	 * Get the session information 
	 * @return
	 */
	public QualifyingSessionInformation getSessionInformation() {
		return sessionInformation;
	}
	
	/**
	 * Returns the message that should be displayed to LFS. Depending on a number
	 * of conditions this message can be to display nothing, to display a time
	 * gap, a time gap with predicted position, a time gap with a split difference
	 * etc
	 * @param playerID
	 * @return
	 */
	public LFSDisplayMessage getLFSDisplayMessageForViewedPlayer(byte playerID) {
		//long startTime = System.nanoTime();

		PlayerInfo viewedPlayer = getPlayerForID(playerID);

		if(viewedPlayer == null) {
			//print("viewedPlayer is null");
			return new LFSDisplayMessage(LFSDisplayMessage.
					MessageType.NO_DISPLAY_MESSAGE);
		}

		//long startGetCurrentLap = System.nanoTime();

		Lap currentLap = getLapInProgressForPlayer(viewedPlayer.getPlayerID());

		//print(Helper.getTimeTaken("get current lap", startGetCurrentLap));

		if(currentLap == null) {
			//print("currentLap is null");
			return new LFSDisplayMessage(LFSDisplayMessage.
					MessageType.PLAYER_NOT_ON_HOTLAP);
		}

		// Get the latest completed node for this player
		MiniCompCarAndTime currentLapLatestCompCar = currentLap.getLatestMiniCompCarAndTime();

		if(currentLapLatestCompCar == null) {
			// Missing data
			//print("latest comp car is null");
			return new LFSDisplayMessage(LFSDisplayMessage.
					MessageType.NO_DISPLAY_MESSAGE);
		}

		//long startGetTrackStuff = System.nanoTime();

		Track currentTrack = sessionInformation.getTrack();

		if(currentTrack == null) {
			//print("currentTrack is null");
			return new LFSDisplayMessage(LFSDisplayMessage.
					MessageType.NO_DISPLAY_MESSAGE);
		}

		TrackData trackData = TrackModel.getDataForOneTrack(currentTrack);

		if(trackData == null) {
			//print("trackData is null");
			return new LFSDisplayMessage(LFSDisplayMessage.
					MessageType.NO_DISPLAY_MESSAGE);
		}

		//print(Helper.getTimeTaken("get track stuff", startGetTrackStuff));

		//long startGetLapPercCompleted = System.nanoTime();

		// Only calculate a time if the player is at least 5% into lap
		Double percentageLapCompleted = trackData.getPercentageOfLapCompletedAtPosition(
				currentLapLatestCompCar.getMiniCompCar());

		if(percentageLapCompleted == null || percentageLapCompleted < 
				QualiTickerPrefs.getPercentageOfLapToCompleteBeforeDisplayingGap()) {
			//print("not enough of the lap completed - "+percentageLapCompleted+" completed, "+
			//	QualiTickerPrefs.getPercentageOfLapToCompleteBeforeDisplayingGap()+" required");
			return new LFSDisplayMessage(LFSDisplayMessage.
					MessageType.NO_DISPLAY_MESSAGE);
		}

		//print(Helper.getTimeTaken("get perc lap completed", startGetLapPercCompleted));

		int startNode = currentLapLatestCompCar.getMiniCompCar().getNode();

		//long startGetGapCurrentLap = System.nanoTime();

		Long gapCurrentLap = getTimeDifferenceBetweenLapsNS(
				fastestLap, currentLap, 
				QualiTickerPrefs.getNumberOfNodesForAverageGap());

		if(gapCurrentLap == null) {
			//print("gap current lap is null");
			return new LFSDisplayMessage(LFSDisplayMessage.
					MessageType.NO_DISPLAY_MESSAGE);
		}

		//print(Helper.getTimeTaken("get current lap", startGetGapCurrentLap));

		//long startGetFastestLaps = System.nanoTime();

		/* 
		 * Determine the relative position of this player's lap using the fastest
		 * completed laps of the other people in qualifying
		 */
		Lap[] fastestCompletedLaps = getFastestLapForEachDriverRemoveProbableDuplicates();

		//print(Helper.getTimeTaken("get fastest laps for drivers", 
		//startGetFastestLaps));

		if(fastestCompletedLaps != null) {
			//long startGetPositionForCurrentLap = System.nanoTime();

			/* 
			 * Now see where the time elapsed on the current lap fits in with the 
			 * order of the completed laps
			 */
			int position = Integer.MAX_VALUE;

			/*
			 * Use the average gap of the current lap compared to the average
			 * gaps of the other completed laps to determine predicted position
			 */
			if(gapCurrentLap != null) {
				for(int i=0; i<fastestCompletedLaps.length; i++) {
					Lap thisLap = fastestCompletedLaps[i];

					Long gapThisPlayersFastestLap = getTimeDifferenceBetweenLapsNSFromStartNode(
							fastestLap, thisLap, 
							QualiTickerPrefs.getNumberOfNodesForAverageGap(),
							startNode);

					if(gapThisPlayersFastestLap == null || 
							gapCurrentLap.longValue() < gapThisPlayersFastestLap.longValue()) {
						// i+1 because index 0 is actually 1st pos
						position = i+1;

						break;
					}
				}
			}

			if(position == Integer.MAX_VALUE) {
				position = fastestCompletedLaps.length+1;
			}

			//print(Helper.getTimeTaken("get position for current lap", startGetPositionForCurrentLap));

			//print(Helper.getTimeTaken("return display message", startTime));

			return new LFSDisplayMessage(LFSDisplayMessage.MessageType.
					TIME_DIFFERENCE_AND_POSITION_MESSAGE, 
					currentLap, fastestLap, gapCurrentLap, position);
		} else {
			return new LFSDisplayMessage(LFSDisplayMessage.MessageType.
					TIME_DIFFERENCE_ONLY_MESSAGE, 
					currentLap, fastestLap, gapCurrentLap, Integer.MAX_VALUE);
		}
	}
	
	/**
	 * Add the given TimingModelListener to this TimingModel
	 * @param listener
	 */
	public void addTimingModelListener(TimingModelListener listener) {
		if(!listeners.contains(listener)) {
			listeners.add(listener);
		}
	}
	
	/**
	 * Remove the given TimingModelListener to this TimingModel
	 * @param listener
	 */
	public void removeTimingModelListener(TimingModelListener listener) {
		if(listeners.contains(listener)) {
			listeners.remove(listener);
		}
	}
	
	/* ************************************************************************
	 * ** End of public methods in TimingModel ********************************
	 * ********************************************************************* */
	
	/* ************************************************************************
	 * ** Start of private methods in TimingModel *****************************
	 * ********************************************************************* */
	
	private Long getTimeDifferenceBetweenLapsNSFromStartNode(Lap fastestLap, 
			Lap currentLap, int numNodes, int startNode) {
		// print("getTimeDifferenceBetweenLaps called");
		MiniCompCarAndTime currentLapCompCarAtStartNode = currentLap.
			getFirstMiniCompCarPacketForNode(startNode);
		
		if(currentLapCompCarAtStartNode == null) {
			//print("current lap last comp car is null");
			
			return null;
		}
		
		int numNodesInTrack = sessionInformation.getNumNodes();

		MiniCompCarAndTime currentLapFirstCompCarAfterStartLine = currentLap.
			getFirstMiniCompCarInLap();
		
		if(currentLapFirstCompCarAfterStartLine == null) {
			return null;
		}
		
		MiniCompCarAndTime fastestLapFirstCompCarAfterStartLine = fastestLap.
			getFirstMiniCompCarInLap();
		
		if(fastestLapFirstCompCarAfterStartLine == null) {
			return null;
		}
		
		// Standardise the lap started times across current lap and fastest lap
		long currentLapStartTimeNS = Long.MIN_VALUE, fastestLapStartTimeNS = Long.MIN_VALUE;

		Track currentTrack = sessionInformation.getTrack();
		
		if(currentTrack == null) {
			return null;
		}
		
		TrackData trackData = TrackModel.getDataForOneTrack(currentTrack);
		
		if(trackData == null) {
			return null;
		}
		
		Double currentLapMetresAtFirstCompCar = trackData.getLapLengthToPositionInMetres(
				currentLapFirstCompCarAfterStartLine.getMiniCompCar());
		
		if(currentLapMetresAtFirstCompCar == null) {
			return null;
		}
		
		//print("currentLapMetres: "+currentLapMetres);
		
		Double fastestLapMetresAtFirstCompCar = trackData.getLapLengthToPositionInMetres(
				fastestLapFirstCompCarAfterStartLine.getMiniCompCar());
		
		if(fastestLapMetresAtFirstCompCar == null) {
			return null;
		}

		if(currentLapMetresAtFirstCompCar == fastestLapMetresAtFirstCompCar) {
			currentLapStartTimeNS = currentLapFirstCompCarAfterStartLine.getTime();
			fastestLapStartTimeNS = fastestLapFirstCompCarAfterStartLine.getTime();
		} else if(currentLapMetresAtFirstCompCar > fastestLapMetresAtFirstCompCar) {
			short fastestLapSpeedRaw = fastestLapFirstCompCarAfterStartLine.getMiniCompCar().getSpeed();
			double fastestLapSpeedMS = Helper.getSpeedInMetresPerSecond(fastestLapSpeedRaw);
			
			long nsTakenToReachCurrentLap = (long) (((currentLapMetresAtFirstCompCar - 
					fastestLapMetresAtFirstCompCar) / fastestLapSpeedMS) * 1000000000.0);

			currentLapStartTimeNS = currentLapFirstCompCarAfterStartLine.getTime();
			fastestLapStartTimeNS = fastestLapFirstCompCarAfterStartLine.getTime() + 
				nsTakenToReachCurrentLap;
		} else {
			//print("current lap metres is greater than fastest lap metres");
			
			short currentLapSpeedRaw = currentLapFirstCompCarAfterStartLine.getMiniCompCar().getSpeed();
			double currentLapSpeedMS = Helper.getSpeedInMetresPerSecond(currentLapSpeedRaw);
			
			long nsTakenToReachFastestLap = (long) (((fastestLapMetresAtFirstCompCar - 
					currentLapMetresAtFirstCompCar) / currentLapSpeedMS) * 1000000000.0);
			
			//print(nsTakenToReachFastestLap+"ns required at "+currentLapSpeedMS+
			//		" m/s to reach fastest lap position");
			
			fastestLapStartTimeNS = fastestLapFirstCompCarAfterStartLine.getTime();
			currentLapStartTimeNS = currentLapFirstCompCarAfterStartLine.getTime() + 
				nsTakenToReachFastestLap;
		}
		// End standardising start time for current and fastest laps
		
		ArrayList<Long> allDifferences = new ArrayList<Long>();

		int loopCounter = 0;
		int currentNode = startNode;

		for(loopCounter=0; loopCounter<numNodes; loopCounter++) {
			if(currentNode == -1) {
				currentNode = numNodesInTrack;
			}
			
			MiniCompCarAndTime currentLapComp = currentLap.getFirstMiniCompCarPacketForNode(currentNode);

			if(currentLapComp != null) {
				MiniCompCarAndTime fastestLapComp = fastestLap.getFirstMiniCompCarPacketForNode(currentNode);
				
				if(fastestLapComp != null) {
					Long diff = getTimeDifferenceBetweenCompCarAndTimes(
							currentLapStartTimeNS, currentLapComp, 
							fastestLapStartTimeNS, fastestLapComp, 
							trackData);
					
					if(diff != null) {
						allDifferences.add(diff);
						
						// print("difference added, size: "+allDifferences.size());
					} else {
						print("diff is null, not adding");
					}
				}
			}
			
			currentNode--;
		}
		
		// Work out the average difference using all the differences
		long totalDifferenceNs = 0;
		
		for(Long oneDiff : allDifferences) {
			totalDifferenceNs += oneDiff;
		}
		
		if(allDifferences.size() > 0) {
			//print("allDifferences.size(): "+allDifferences.size());
			
			return (long) (totalDifferenceNs / allDifferences.size());
		} else {
			//print("no differences found, lots of missing data");
			
			return null;
		}
	}

	private Long getTimeDifferenceBetweenLapsNS(Lap fastestLap, Lap currentLap, 
			int numNodes) {
		// print("getTimeDifferenceBetweenLaps called");
		MiniCompCarAndTime currentLapLastCompCar = currentLap.getLatestMiniCompCarAndTime();
		
		if(currentLapLastCompCar == null) {
			//print("current lap last comp car is null");
			return null;
		}
		
		int numNodesInTrack = sessionInformation.getNumNodes();
		
		MiniCompCarAndTime currentLapFirstCompCarAfterStartLine = currentLap.
			getFirstMiniCompCarInLap();
		
		if(currentLapFirstCompCarAfterStartLine == null) {
			//print("current lap first comp car after sf line is null");
			return null;
		}
		
		MiniCompCarAndTime fastestLapFirstCompCarAfterStartLine = fastestLap.
			getFirstMiniCompCarInLap();
		
		if(fastestLapFirstCompCarAfterStartLine == null) {
			//print("fastest lap first comp car after sf line is null");
			return null;
		}
		
		// Standardise the lap started times across current lap and fastest lap
		long currentLapStartTimeNS = Long.MIN_VALUE, fastestLapStartTimeNS = Long.MIN_VALUE;

		Track currentTrack = sessionInformation.getTrack();
		
		if(currentTrack == null) {
			//print("current track is null");
			return null;
		}
		
		TrackData trackData = TrackModel.getDataForOneTrack(currentTrack);
		
		if(trackData == null) {
			//print("track data is null");
			return null;
		}
		
		Double currentLapMetresAtFirstCompCar = trackData.getLapLengthToPositionInMetres(
				currentLapFirstCompCarAfterStartLine.getMiniCompCar());
		
		if(currentLapMetresAtFirstCompCar == null) {
			//print("current lap metres at first compcar is null");
			return null;
		}

		Double fastestLapMetresAtFirstCompCar = trackData.getLapLengthToPositionInMetres(
				fastestLapFirstCompCarAfterStartLine.getMiniCompCar());
		
		if(fastestLapMetresAtFirstCompCar == null) {
			//print("fastest lap metres at first compcar is null");
			return null;
		}

		if(currentLapMetresAtFirstCompCar == fastestLapMetresAtFirstCompCar) {
			currentLapStartTimeNS = currentLapFirstCompCarAfterStartLine.getTime();
			fastestLapStartTimeNS = fastestLapFirstCompCarAfterStartLine.getTime();
		} else if(currentLapMetresAtFirstCompCar > fastestLapMetresAtFirstCompCar) {
			short fastestLapSpeedRaw = fastestLapFirstCompCarAfterStartLine.getMiniCompCar().getSpeed();
			double fastestLapSpeedMS = Helper.getSpeedInMetresPerSecond(fastestLapSpeedRaw);
			
			long nsTakenToReachCurrentLap = (long) (((currentLapMetresAtFirstCompCar - 
					fastestLapMetresAtFirstCompCar) / fastestLapSpeedMS) * 1000000000.0);

			currentLapStartTimeNS = currentLapFirstCompCarAfterStartLine.getTime();
			fastestLapStartTimeNS = fastestLapFirstCompCarAfterStartLine.getTime() + 
				nsTakenToReachCurrentLap;
		} else {
			//print("current lap metres is greater than fastest lap metres");
			
			short currentLapSpeedRaw = currentLapFirstCompCarAfterStartLine.getMiniCompCar().getSpeed();
			double currentLapSpeedMS = Helper.getSpeedInMetresPerSecond(currentLapSpeedRaw);
			
			long nsTakenToReachFastestLap = (long) (((fastestLapMetresAtFirstCompCar - 
					currentLapMetresAtFirstCompCar) / currentLapSpeedMS) * 1000000000.0);

			fastestLapStartTimeNS = fastestLapFirstCompCarAfterStartLine.getTime();
			currentLapStartTimeNS = currentLapFirstCompCarAfterStartLine.getTime() + 
				nsTakenToReachFastestLap;
		}
		// End standardising start time for current and fastest laps
		
		ArrayList<Long> allDifferences = new ArrayList<Long>();
		
		int nodeIndex = currentLapLastCompCar.getMiniCompCar().getNode();
		
		int loopCounter = 0;
		int currentNode = nodeIndex;

		for(loopCounter=0; loopCounter<numNodes; loopCounter++) {
			// Reset current node to max node number if it's less than 0
			if(currentNode == -1) {
				currentNode = numNodesInTrack;
			}
			
			MiniCompCarAndTime currentLapComp = currentLap.getFirstMiniCompCarPacketForNode(currentNode);

			if(currentLapComp != null) {
				MiniCompCarAndTime fastestLapComp = fastestLap.getFirstMiniCompCarPacketForNode(currentNode);
				
				if(fastestLapComp != null) {
					Long diff = getTimeDifferenceBetweenCompCarAndTimes(
							currentLapStartTimeNS, currentLapComp, 
							fastestLapStartTimeNS, fastestLapComp, 
							trackData);
					
					if(diff != null) {
						allDifferences.add(diff);
						
						// print("difference added, size: "+allDifferences.size());
					} else {
						// print("diff is null, not adding");
					}
				}
			}
			
			currentNode--;
		}
		
		// Work out the average difference using all the differences
		long totalDifferenceNs = 0;
		
		for(Long oneDiff : allDifferences) {
			totalDifferenceNs += oneDiff;
		}
		
		if(allDifferences.size() > 0) {
			//print("allDifferences.size(): "+allDifferences.size());
			return (long) (totalDifferenceNs / allDifferences.size());
		} else {
			//print("no differences found, lots of missing data");
			return null;
		}
	}

	// Returns true if the player is currently on a hotlap, false otherwise
	private boolean isPlayerOnAHotlap(byte playerID) {
		return playersOnHotlap.contains(playerID);
	}
	
	// Returns true if the player is in the session, false otherwise
	private boolean isPlayerInSession(byte playerID) {
		return playersInSession.contains(playerID);
	}
	
	/*
	 * Returns the gap between the two given CompCarAndTimes from the given
	 * lap start times
	 */
	private Long getTimeDifferenceBetweenCompCarAndTimes(long currentLapStartTimeNs, 
			MiniCompCarAndTime currentLapComp, long fastestLapStartTimeNs, 
			MiniCompCarAndTime fastestLapComp, TrackData trackData) {
		if(currentLapComp == null || fastestLapComp == null) {
			return null;
		}
		
		Double currentLapMetres = trackData.getLapLengthToPositionInMetres(
				currentLapComp.getMiniCompCar());
		
		if(currentLapMetres == null) {
			return null;
		}
		
		Double fastestLapMetres = trackData.getLapLengthToPositionInMetres(
				fastestLapComp.getMiniCompCar());
		
		if(fastestLapMetres == null) {
			return null;
		}
		
		long timeElapsedOnCurrentLapToLastCompCarNs = currentLapComp.getTime()-
			currentLapStartTimeNs;
		long timeElapsedOnFastestLapToSameNodeNs = fastestLapComp.getTime()-
			fastestLapStartTimeNs;

		if(currentLapMetres == fastestLapMetres) {
			return timeElapsedOnCurrentLapToLastCompCarNs - 
				timeElapsedOnFastestLapToSameNodeNs;
		} else if(currentLapMetres > fastestLapMetres) {
			/*
			 * Make up the fastest lap metres to the same as the current lap metres
			 * using the speed from the comp car object. Add on the time taken to close
			 * the gap to the elapsed time
			 */

			short fastestLapSpeedRaw = fastestLapComp.getMiniCompCar().getSpeed();
			double fastestLapSpeedMS = Helper.getSpeedInMetresPerSecond(fastestLapSpeedRaw);
			
			long nsTakenToReachCurrentLap = (long) (((currentLapMetres - fastestLapMetres) / 
					fastestLapSpeedMS) * 1000000000.0);

			timeElapsedOnFastestLapToSameNodeNs += nsTakenToReachCurrentLap;
			
			return timeElapsedOnCurrentLapToLastCompCarNs - 
				timeElapsedOnFastestLapToSameNodeNs;
		} else {
			short currentLapSpeedRaw = currentLapComp.getMiniCompCar().getSpeed();
			double currentLapSpeedMS = Helper.getSpeedInMetresPerSecond(currentLapSpeedRaw);
			
			long nsTakenToReachFastestLap = (long) (((fastestLapMetres - currentLapMetres) / 
					currentLapSpeedMS) * 1000000000.0);

			timeElapsedOnFastestLapToSameNodeNs += nsTakenToReachFastestLap;
			
			return timeElapsedOnCurrentLapToLastCompCarNs - 
				timeElapsedOnFastestLapToSameNodeNs;
		}
	}
	
	/*
	 * If the lap that has just been completed is faster than the current fastest lap overall
	 * update it.
	 */
	private void lapCompleted(Lap completedLap) {
		/*
		 * Only set this lap as fastest lap if no lap exists or this is quicker than 
		 * existing lap and it's not the first lap (i.e. with a time of 1 hour)
		 */
		if(completedLap.getLaptime().getTime().getHours() != 1) {
			if(fastestLap == null || 
					completedLap.getLaptime().getTime().getTime() < 
					fastestLap.getLaptime().getTime().getTime()) {
				fastestLap = completedLap;
				
				fastestLapUpdated(fastestLap);
			}
		}
		
		// Nullify fastest laps cache so it needs to be recalculated
		fastestLapForEachDriverDuplicatesRemoved = null;
	}

	// Add the given lap to the laps for this driver in the allLaps Map.
	private void addNewLap(Lap newLap) {
		if(newLap.getPlayer() != null) {
			PlayerInfo player = newLap.getPlayer();
			byte playerID = player.getPlayerID();
			
			// Add this lap to the existing laps for driver
			if(allLaps.containsKey(player)) {
				allLaps.get(player).add(newLap);
			} else {
				// Create a new vector and add this first lap to it
				ArrayList<Lap> newList = new ArrayList<Lap>();
				newList.add(newLap);
				
				allLaps.put(player, newList);
			}
			
			// Add this player to hotlap list
			addPlayerToPlayersOnHotlapList(playerID);
			
			// Add this player to players in session if not already there
			addPlayerToPlayersInSession(playerID);

			// Fire event on listeners
			playerStartedHotlap(playerID);
		} else {
			print("!!! addNewLap - player in newLap is null");
		}
	}
	
	// Add the given player to the list of players currently on a hotlap
	private void addPlayerToPlayersOnHotlapList(byte playerID) {
		if(!playersOnHotlap.contains(playerID)) {
			playersOnHotlap.add(playerID);
		}
	}
	
	// Remove the given player from the list of players currently on a hotlap
	private void removePlayerFromPlayersOnHotlapList(Byte playerID) {
		playersOnHotlap.remove(playerID);
	}
	
	// Add the given player to the list of players currently in the session
	private void addPlayerToPlayersInSession(byte playerID) {
		if(!playersInSession.contains(playerID)) {
			playersInSession.add(playerID);
		}
	}
	
	// Remove the player from the list of players currently in the session
	private void removePlayerFromPlayersInSession(Byte playerID) {
		playersInSession.remove(playerID);
	}
	
	// Remove the current lap for the driver
	private void removeLapInProgressForPlayer(byte playerID) {
		//print("removeLapInProgressForPlayer called with playerID: "+playerID);
		PlayerInfo player = getPlayerForID(playerID);
		
		if(player != null) {
			if(allLaps.containsKey(player)) {
				Lap lapInProgress = getLapInProgressForPlayer(playerID);

				if(lapInProgress != null) {
					allLaps.get(player).remove(lapInProgress);
				}
			}
		}
	}
	
	// Remove state data from the last session when a new session is started
	private void removeStateFromPreviousSession() {
		currentConnections.clear();
		currentPlayers.clear();
		
		allLaps.clear();
		
		playersOnHotlap.clear();
		playersInSession.clear();
		
		fastestLap = null;
		
		fastestLapForEachDriverDuplicatesRemoved = null;
		
		bestSplits = new SplitTimeResponse[4];
	}
	
	// Returns true if the current session is a qualifying session, false otherwise
	private boolean isQualifyingSession() {
		SessionType sessionType = sessionInformation.getSessionType();
		
		if(sessionType != null && 
				sessionType.equals(SessionType.QUALIFYING_SESSION)) {
			return true;
		} else {
			return false;
		}
	}
	
	private void print(String msg) {
		LFSQualifyingTickerStart.print(this.getClass().getSimpleName()+" - "+msg);
	}
	
	/* ************************************************************************
	 * ** End of private methods in TimingModel *******************************
	 * ********************************************************************* */
	
	/* ************************************************************************
	 * ** Start of listener methods in TimingModel ****************************
	 * ********************************************************************* */

	/*
	 * (non-Javadoc)
	 * @see lfsqualifyingticker.structures.TimingModelListener#playerJoinedSession(byte)
	 */
	public void playerJoinedSession(byte playerIDJoining) {
		for(TimingModelListener listener : listeners) {
			listener.playerJoinedSession(playerIDJoining);
		}
	}

	/*
	 * (non-Javadoc)
	 * @see lfsqualifyingticker.structures.TimingModelListener#
	 * playerLeftSession(byte)
	 */
	public void playerLeftSession(byte playerIDLeaving) {
		for(TimingModelListener listener : listeners) {
			listener.playerLeftSession(playerIDLeaving);
		}
	}

	/*
	 * (non-Javadoc)
	 * @see lfsqualifyingticker.structures.TimingModelListener#
	 * playerPitted(byte)
	 */
	public void playerPitted(byte playerIDPitting) {
		for(TimingModelListener listener : listeners) {
			listener.playerPitted(playerIDPitting);
		}
	}

	/*
	 * (non-Javadoc)
	 * @see lfsqualifyingticker.structures.TimingModelListener#
	 * playerStartedHotlap(byte)
	 */
	public void playerStartedHotlap(byte playerIDStartedHotlap) {
		for(TimingModelListener listener : listeners) {
			listener.playerStartedHotlap(playerIDStartedHotlap);
		}
	}
	
	/*
	 * (non-Javadoc)
	 * @see lfsqualifyingticker.structures.TimingModelListener#
	 * fastestLapUpdated(lfsqualifyingticker.structures.Lap)
	 */
	public void fastestLapUpdated(Lap fastestLap) {
		for(TimingModelListener listener : listeners) {
			listener.fastestLapUpdated(fastestLap);
		}
	}
	
	/*
	 * (non-Javadoc)
	 * @see lfsqualifyingticker.structures.TimingModelListener#
	 * resultReceived(net.sf.jinsim.response.ResultResponse)
	 */
	public void resultReceived(ResultResponse result) {
		for(TimingModelListener listener : listeners) {
			listener.resultReceived(result);
		}
	}

	/*
	 * (non-Javadoc)
	 * @see lfsqualifyingticker.structures.TimingModelListener#
	 * updatePlayerPositionNodes(java.util.HashMap)
	 */
	public void updatePlayerPositionNodes(
			HashMap<PlayerInfo, MiniCompCar> playerPositions) {
		for(TimingModelListener listener : listeners) {
			listener.updatePlayerPositionNodes(playerPositions);
		}
	}
	
	/*
	 * (non-Javadoc)
	 * @see lfsqualifyingticker.structures.TimingModelListener#
	 * bestSplitUpdated(lfsqualifyingticker.structures.PlayerInfo, 
	 * net.sf.jinsim.response.SplitTimeResponse)
	 */
	public void bestSplitUpdated(PlayerInfo player, SplitTimeResponse split) {
		for(TimingModelListener listener : listeners) {
			listener.bestSplitUpdated(player, split);
		}
	}
	
	/* ************************************************************************
	 * ** End of listener methods in TimingModel ******************************
	 * ********************************************************************* */
}
