package lfsqualifyingticker.structures;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Point;
import java.awt.Polygon;
import java.awt.Rectangle;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.util.HashMap;

import javax.swing.JPanel;

import lfsqualifyingticker.LFSQualifyingTickerGUI;
import lfsqualifyingticker.LFSQualifyingTickerStart;

public class TrackCanvas extends JPanel {
	/**
	 * 
	 */
	private static final long serialVersionUID = -2806741103405871329L;

	private TrackData trackData;
	
	private int maximumX=Integer.MIN_VALUE, maximumY, minimumX, minimumY, trackWidth, trackHeight;
	
	private Polygon trackCentrePolygon;

	private Dimension lastSize, preferredSize;
	
	private HashMap<PlayerInfo, MiniCompCar> playerPositions = 
		new HashMap<PlayerInfo, MiniCompCar>();
	
	private final int ovalDiameter = 10;
	
	private HashMap<Byte, Point> playersDrawnOnMap = new HashMap<Byte, Point>();
	
	private ActionListener listener;

	public TrackCanvas(TrackData trackData, ActionListener listener) {
		this.trackData = trackData;
		
		this.listener = listener;

		setupTrackCanvas();
	}
	
	private void setupTrackCanvas() {
		this.addMouseListener(new MouseAdapter() {
			public void mouseClicked(MouseEvent e) {
				handleMouseClickOnMap(e);
			}
		});

		createTrackPolygon();
	}

	public void createTrackPolygon() {
		//print("createTrackPolygon called");

		OneNode[] allNodes = trackData.getAllNodes();
		
		// Maximum x hasn't been set yet, calculate these values now
		if(maximumX == Integer.MIN_VALUE) {
			for(OneNode oneNode : allNodes) {
				int centreX = (int) (oneNode.getCentreX()/65536.0);
				int centreY = (int) (oneNode.getCentreY()/65536.0);

				if(centreX > maximumX) {
					maximumX = centreX;
				}
				if(centreX < minimumX) {
					minimumX = centreX;
				}

				if(centreY > maximumY) {
					maximumY = centreY;
				}
				if(centreY < minimumY) {
					minimumY = centreY;
				}
			}

			trackWidth = maximumX - minimumX;
			trackHeight = maximumY - minimumY;
		}
		
		double scaleFactor = 1;

		Dimension targetSize = this.getSize();

		// Scale down each of the points of the track according to the preferred size
		if(trackWidth > trackHeight) {
			scaleFactor = (trackWidth / targetSize.getWidth() * 1.0);
		} else if(trackHeight > trackWidth) {
			scaleFactor = (trackHeight / targetSize.getHeight() * 1.0);
		} else {
			scaleFactor = (trackHeight / targetSize.getHeight() * 1.0);
		}

		int[] trackCentreXCoords = new int[allNodes.length];
		int[] trackCentreYCoords = new int[allNodes.length];

		for(OneNode oneNode : allNodes) {
			int nodeIndex = oneNode.getNodeIndex();

			int centreX = (int) (oneNode.getCentreX()/65536.0);
			int centreY = (int) (oneNode.getCentreY()/65536.0);

			int xPoint = (int) (centreX / scaleFactor);
			int yPoint = (int) (centreY / scaleFactor);

			trackCentreXCoords[nodeIndex] = xPoint;
			trackCentreYCoords[nodeIndex] = yPoint;
		}

		trackCentrePolygon = new Polygon(trackCentreYCoords, trackCentreXCoords, allNodes.length);
		
		// Shift the track up to the top left of the canvas area
		Rectangle polyBounds = trackCentrePolygon.getBounds();
		
		trackCentrePolygon.translate(-(polyBounds.x)+30, -(polyBounds.y)+30);

		lastSize = targetSize;
	}
	
	/*
	 * (non-Javadoc)
	 * @see javax.swing.JComponent#paintComponent(java.awt.Graphics)
	 */
	public synchronized void paintComponent(Graphics g) {
		// If the size has changed recreate the polygon
		Dimension currentSize = this.getPreferredSize() != null ? 
				this.getPreferredSize() : this.getSize();
		
		//print("paintComponent, trackCanvas.size(): "+currentSize.toString());
		
		if(lastSize == null || 
				currentSize.height != lastSize.height || 
				currentSize.width != lastSize.width) {
			createTrackPolygon();
		}

		if(trackCentrePolygon != null) {
			g.setColor(Color.black);
			g.drawPolygon(trackCentrePolygon);
		}

		playersDrawnOnMap.clear();

		// Now add the player's current positions
		for(PlayerInfo player : playerPositions.keySet()) {
			byte playerID = player.getPlayerID();

			MiniCompCar compCar = playerPositions.get(player);

			if(compCar != null && compCar.getLap() > 0) {
				if(trackCentrePolygon != null) {
					int xCoord = trackCentrePolygon.xpoints[compCar.getNode()];
					int yCoord = trackCentrePolygon.ypoints[compCar.getNode()];

					if(xCoord < 0) {
						xCoord += ovalDiameter/2.0;
					} else {
						xCoord -= ovalDiameter/2.0;
					}

					if(yCoord < 0) {
						yCoord += ovalDiameter/2.0;
					} else {
						yCoord -= ovalDiameter/2.0;
					}

					playersDrawnOnMap.put(playerID, new Point(xCoord, yCoord));

					// If the player is on an outlap draw them in blue
					if(compCar.getLap() == 1) {
						g.setColor(Color.blue);
					} else {
						g.setColor(Color.black);
					}

					g.drawOval(xCoord, yCoord, ovalDiameter, ovalDiameter);
					g.drawString(Helper.getPlainPlayerName(player.getPlayerName()), 
							xCoord, yCoord);
				}
			}
		}

//		Rectangle polyBounds = trackCentrePolygon.getBounds();

//		g.setColor(Color.orange);
//		g.drawRect(polyBounds.x, polyBounds.y, polyBounds.width, polyBounds.height);

		g.setColor(Color.red);

		// Add text for the start/finish line
//		if(trackData.getFinishNodeIndex() != -1) {
//			int xCoord = trackCentrePolygon.xpoints[trackData.getFinishNodeIndex()];
//			int yCoord = trackCentrePolygon.ypoints[trackData.getFinishNodeIndex()];
//
//			// Move the text a little away from the track to make it easier to read
//			if(xCoord >= 0) {
//				xCoord = (int) (xCoord * 0.99);
//			} else {
//				xCoord = (int) (xCoord * 1.01);
//			}
//
//			if(yCoord >= 0) {
//				yCoord = (int) (yCoord * 0.99);
//			} else {
//				yCoord = (int) (yCoord * 1.01);
//			}
//
//			print("draw string: "+"S/F"+", x: "+xCoord+", y: "+yCoord);
//
//			g.drawString("S/F", xCoord, yCoord);
//		}
	}

//	g.setColor(Color.red);
//	g.drawRect(0,0, this.getSize().width, this.getSize().width);

//	if(getPreferredSize() != null) {
//	g.setColor(Color.blue);
//	g.drawRect(0,0, this.getPreferredSize().width, this.getPreferredSize().width);
//	}
	
	/*
	 * (non-Javadoc)
	 * @see javax.swing.JComponent#getPreferredSize()
	 */
	public Dimension getPreferredSize() {
//		if(preferredSize == null) {
//			print("returning null from getPreferredSize");
//		} else {
//			print("returning from getPreferredSize: "+preferredSize);
//		}
		
		return preferredSize;
	}
	
	public void setPreferredSize(Dimension preferredSize) {
//		if(preferredSize != null) {
//			print("setPreferredSize called with size: "+preferredSize);
//		} else {
//			print("setPreferredSize called with null size");
//		}
		
		this.preferredSize = preferredSize;
	}
	
	/**
	 * Update the positions of the players on the map according to the new values
	 * @param playerPositions
	 */
	public void updatePlayerPositions(HashMap<PlayerInfo, MiniCompCar> newPlayerPositions) {
		/*
		 * Cannot just overwrite the player positions map because there are a maximum of 8
		 * players per MCI, so need to add these to the existing map instead
		 */
		for(PlayerInfo player : newPlayerPositions.keySet()) {
			// If the lap on the CompCar is 0 remove this player
			MiniCompCar thisMiniCompCar = newPlayerPositions.get(player);
			
			if(thisMiniCompCar.getLap() == 0) {
				playerPositions.remove(player);
			} else {
				this.playerPositions.put(player, newPlayerPositions.get(player));
			}
		}
	}
	
	/**
	 * Remove the given player from the track map
	 * @param playerID
	 */
	public void removePlayerFromTrackMap(byte playerID) {
		PlayerInfo playerToRemove = null;
		
		for(PlayerInfo player : playerPositions.keySet()) {
			if(player.getPlayerID() == playerID) {
				playerToRemove = player;
			}
		}
		
		if(playerToRemove != null) {
			playerPositions.remove(playerToRemove);
		}
	}
	
	private void handleMouseClickOnMap(MouseEvent e) {
		Point clickPoint = e.getPoint();
		
		int closestX = Integer.MAX_VALUE, closestY = Integer.MAX_VALUE;
		byte closestPlayerID = Byte.MAX_VALUE;
		
		// Find the player closest to the click
		for(Byte playerID : playersDrawnOnMap.keySet()) {
			Point playerPoint = playersDrawnOnMap.get(playerID);
			
			int diffX = Math.abs(clickPoint.x - playerPoint.x);
			int diffY = Math.abs(clickPoint.y - playerPoint.y);
			
			if(diffX < closestX && diffY < closestY) {
				closestX = diffX;
				closestY = diffY;
				closestPlayerID = playerID;
			}
		}
		
		if(closestPlayerID != Byte.MAX_VALUE) {
			if(closestX < (ovalDiameter*2.0) && closestY < (ovalDiameter*2.0)) {
				/* 
				 * If the click is close enough to a player send an event, pass int 
				 * min value as modifiers to keep the current camera type
				 */
				listener.actionPerformed(new ActionEvent(this, closestPlayerID, 
						LFSQualifyingTickerGUI.VIEW_PLAYER_COMMAND, Integer.MIN_VALUE));
			}
		}
	}
	
	@SuppressWarnings("unused")
	private void print(String msg) {
		LFSQualifyingTickerStart.print(this.getClass().getSimpleName()+" - "+msg);
	}
}
