package lfsqualifyingticker.structures;

import java.util.Comparator;

import lfsqualifyingticker.LFSQualifyingTickerStart;

import net.sf.jinsim.types.InSimVector;

/**
 * Represents one node on a track
 * @author Gus
 *
 */

public class OneNode implements Comparator<OneNode>, Comparable<OneNode> {
	// The index of this node in a track
	private int nodeIndex;
	
	// The centre point of this node
	private int centreX, centreY, centreZ; 
	
	// The direction of this node (heading)
	private float directionX, directionY, directionZ;
	
	// The limit of the track area (including grass) to left and right of this node
	private float limitLeft, limitRight;
	
	// The limit of the track area (excluding grass) to left and right of this node
	private float driveLeft, driveRight;
	
//	// The coordinates for the left and right side of the track
//	private int driveLeftX=Integer.MAX_VALUE, driveLeftY=Integer.MAX_VALUE, 
//		driveLeftZ=Integer.MAX_VALUE, driveRightX=Integer.MAX_VALUE, 
//		driveRightY=Integer.MAX_VALUE, driveRightZ=Integer.MAX_VALUE;
//	
//	private int limitLeftX=Integer.MAX_VALUE, limitLeftY=Integer.MAX_VALUE, 
//		limitLeftZ=Integer.MAX_VALUE, limitRightX=Integer.MAX_VALUE, 
//		limitRightY=Integer.MAX_VALUE, limitRightZ=Integer.MAX_VALUE;
	
	// An InSimVector to the centre of this node
	private InSimVector centrePosition;

	/**
	 * Construct a new node with the given parameters
	 * @param nodeIndex
	 * @param centreX
	 * @param centreY
	 * @param centreZ
	 * @param directionX
	 * @param directionY
	 * @param directionZ
	 * @param limitLeft
	 * @param limitRight
	 * @param driveLeft
	 * @param driveRight
	 */
	public OneNode(int nodeIndex, 
			int centreX, int centreY, int centreZ, 
			float directionX, float directionY, float directionZ, 
			float limitLeft, float limitRight, 
			float driveLeft, float driveRight) {
		this.nodeIndex = nodeIndex;
		
		this.centreX = centreX;
		this.centreY = centreY;
		this.centreZ = centreZ;
		
		this.directionX = directionX;
		this.directionY = directionY;
		this.directionZ = directionZ;
		
		this.limitLeft = limitLeft;
		this.limitRight = limitRight;
		
		this.driveLeft = driveLeft;
		this.driveRight = driveRight;
	}
	
	/**
	 * The index of this node in the PTH file
	 * @return
	 */
	public int getNodeIndex() {
		return nodeIndex;
	}
	
	/**
	 * The x co-ordinate for the centre of this node
	 * @return
	 */
	public int getCentreX() {
		return centreX;
	}

	/**
	 * The y co-ordinate for the centre of this node
	 * @return
	 */
	public int getCentreY() {
		return centreY;
	}

	/**
	 * The z co-ordinate for the centre of this node
	 * @return
	 */
	public int getCentreZ() {
		return centreZ;
	}

	/**
	 * The x component for the direction of this node
	 * @return
	 */
	public float getDirectionX() {
		return directionX;
	}

	/**
	 * The y component for the direction of this node
	 * @return
	 */
	public float getDirectionY() {
		return directionY;
	}

	/**
	 * The z component for the direction of this node
	 * @return
	 */
	public float getDirectionZ() {
		return directionZ;
	}

	/**
	 * The limit of the area to the left of this node
	 * @return
	 */
	public float getLimitLeft() {
		return limitLeft;
	}

	/**
	 * The limit of the area to the right of this node
	 * @return
	 */
	public float getLimitRight() {
		return limitRight;
	}

	/**
	 * The limit of the track to the left of this node
	 * @return
	 */
	public float getDriveLeft() {
		return driveLeft;
	}

	/**
	 * The limit to the track to the right of this node
	 * @return
	 */
	public float getDriveRight() {
		return driveRight;
	}
	
	/**
	 * Returns an InSimVector describing the centre of this node
	 * @return
	 */
	public InSimVector getCentrePositionVector() {
		if(centrePosition != null) {
			return centrePosition;
		} else {
			centrePosition = new InSimVector(centreX, centreY, centreZ);
			
			return centrePosition;
		}
	}
	
//	public int getDriveLeftXCoordinate() {
//		return driveLeftX;
//	}
//	
//	public int getDriveLeftYCoordinate() {
//		return driveLeftY;
//	}
//	
//	public int getDriveLeftZCoordinate() {
//		return driveLeftZ;
//	}
//	
//	public int getDriveRightXCoordinate() {
//		return driveRightX;
//	}
//
//	public int getDriveRightYCoordinate() {
//		return driveRightY;
//	}
//
//	public int getDriveRightZCoordinate() {
//		return driveRightZ;
//	}
//
//	public int getLimitLeftXCoordinate() {
//		return limitLeftX;
//	}
//	
//	public int getLimitLeftYCoordinate() {
//		return limitLeftY;
//	}
//	
//	public int getLimitLeftZCoordinate() {
//		return limitLeftZ;
//	}
//	
//	public int getLimitRightXCoordinate() {
//		return limitRightX;
//	}
//
//	public int getLimitRightYCoordinate() {
//		return limitRightY;
//	}
//
//	public int getLimitRightZCoordinate() {
//		return limitRightZ;
//	}
//
//	/**
//	 * Calculate the coordinates of the drive left and drive right parts
//	 * of this node (i.e. calculate where the track stops and the grass
//	 * starts)
//	 * http://www.lfsforum.net/showthread.php?t=59999
//	 * @param trackCentreX
//	 * @param trackCentreY
//	 * @param trackCentreZ
//	 */
//	public void calculateDriveLeftAndRightCoordinates(double trackCentreX, 
//			double trackCentreY, double trackCentreZ) {
//	}
//	
//	public void calculateLimitLeftAndRightCoordinates(double trackCentreX, 
//			double trackCentreY, double trackCentreZ) {
//	}
	
	/*
	 * (non-Javadoc)
	 * @see java.util.Comparator#compare(java.lang.Object, java.lang.Object)
	 */
	public int compare(OneNode o1, OneNode o2) {
		if(o1.getNodeIndex() < o2.getNodeIndex()) {
			return -1;
		} else if(o1.getNodeIndex() > o2.getNodeIndex()) {
			return 1;
		} else {
			return 0;
		}
	}
	
	/*
	 * For testing purposes. Highlights the differences between this node and 
	 * the given node
	 */
	public String getDifferenceComparedToAnotherNode(OneNode comp) {
		StringBuilder diff = new StringBuilder();
		
		diff.append("Differences between node "+this.getNodeIndex()+
				" and node "+comp.getNodeIndex()+"\n");
		
		if(this.centreX != comp.getCentreX()) {
			diff.append("centreX: "+((this.getCentreX()-comp.getCentreX())/65536.0)+"\n");
		}
		if(this.centreY != comp.getCentreY()) {
			diff.append("centreY: "+((this.getCentreY()-comp.getCentreY())/65536.0)+"\n");
		}
		if(this.centreZ != comp.getCentreZ()) {
			diff.append("centreZ: "+((this.getCentreZ()-comp.getCentreZ())/65536.0)+"\n");
		}
		
		if(this.directionX != comp.getDirectionX()) {
			diff.append("directionX: "+(this.getDirectionX()-comp.getDirectionX())+"\n");
		}
		if(this.directionY != comp.getDirectionY()) {
			diff.append("directionY: "+(this.getDirectionY()-comp.getDirectionY())+"\n");
		}
		if(this.directionZ != comp.getDirectionZ()) {
			diff.append("directionZ: "+(this.getDirectionZ()-comp.getDirectionZ())+"\n");
		}

		if(this.limitLeft != comp.getLimitLeft()) {
			diff.append("limitLeft: "+(this.getLimitLeft()-comp.getLimitLeft())+"\n");
		}
		if(this.limitRight != comp.getLimitRight()) {
		diff.append("limitRight: "+(this.getLimitRight()-comp.getLimitRight())+"\n");
		}
		
		if(this.driveLeft != comp.getDriveLeft()) {
			diff.append("driveLeft: "+(this.getDriveLeft()-comp.getDriveLeft())+"\n");
		}
		if(this.driveRight != comp.getDriveRight()) {
			diff.append("driveRight: "+(this.getDriveRight()-comp.getDriveRight())+"\n");
		}

		return diff.toString();
	}

	/*
	 * (non-Javadoc)
	 * @see java.lang.Comparable#compareTo(java.lang.Object)
	 */
	public int compareTo(OneNode o) {
		return compare(this, o);
	}
	
	/*
	 * (non-Javadoc)
	 * @see java.lang.Object#toString()
	 */
	public String toString() {
		return "Node: "+nodeIndex+", centre: ("+centreX+","+centreY+","+centreZ+"), " +
				"dir: ("+directionX+","+directionY+","+directionZ+"), " +
				"lim (L: "+limitLeft+", R: "+limitRight+"), " +
				"drive (L: "+driveLeft+", R: "+driveRight+")";
	}
	
	@SuppressWarnings("unused")
	private void print(String msg) {
		LFSQualifyingTickerStart.print(this.getClass().getSimpleName()+" - "+msg);
	}
}
