package lfsqualifyingticker.structures;

import java.awt.Point;
import java.util.Arrays;
import java.util.HashMap;

import lfsqualifyingticker.LFSQualifyingTickerStart;

import net.sf.jinsim.Track;
import net.sf.jinsim.types.InSimVector;

/**
 * Represents one track from LFS. Contains all of the node information
 * found in the track's PTH file
 * @author Gus
 *
 */

public class TrackData {
	private Track track;
	
	// The nodes that make up this track (key is node index)
	private HashMap<Integer, OneNode> trackNodes = new HashMap<Integer, OneNode>();
	
	/*
	 * A map containing the length of track from the start/finish line to the centre
	 * of each node index in metres. Key is the node index
	 */
	private HashMap<Integer, Double> lapLengthToCentreOfNode = new HashMap<Integer, Double>();
	
	// The node index of the finish line for this track
	private int finishNodeIndex;
	
	// The length of this track in metres
	private double trackLength = Double.MIN_VALUE;
	
	// The centre point co-ordinates for this track
	private Double centreXCoordinate, centreYCoordinate, centreZCoordinate;
	
	/**
	 * Create a track with the given name
	 * @param trackName - the track name (e.g. AS5)
	 */
	public TrackData(Track track) {
		this.track = track;
	}
	
	/**
	 * Perform the initial calculations (populating track data points)
	 * for this track. 
	 */
	public void performInitialCalculations() {
		populateLapLengthAtNodesMap();
		
		//calculateDriveLeftAndRightCoordinates();
		
		//calculateLimitLeftAndRightCoordinates();
	}
	
	/**
	 * Set the finish line node index for this track
	 * @param finishNode
	 */
	public void setFinishNodeIndex(int finishNode) {
		this.finishNodeIndex = finishNode;
	}
	
	/**
	 * Get the finish line node index for this track
	 * @return
	 */
	public int getFinishNodeIndex() {
		return finishNodeIndex;
	}
	
	/**
	 * Add one node to this track
	 * @param node
	 */
	public void addOneNode(OneNode node) {
		trackLength = Double.MIN_VALUE;
		
		trackNodes.put(node.getNodeIndex(), node);
	}
	
	/**
	 * Get an ordered array (ordered by node index) of all the nodes 
	 * in this track.
	 * @return
	 */
	public OneNode[] getAllNodes() {
		if(trackNodes != null && trackNodes.size() > 0) {
			OneNode[] nodes = trackNodes.values().toArray(
					new OneNode[trackNodes.size()]);
			
			Arrays.sort(nodes);
			
			return nodes;
		} else {
			return null;
		}
	}
	
	/**
	 * Get track
	 * @return
	 */
	public Track getTrack() {
		return track;
	}
	
	/**
	 * Get the number of nodes in this track
	 * @return
	 */
	public int getNodeCount() {
		return trackNodes.size();
	}
	
	/**
	 * Returns the length of this track in metres.
	 * Credit to avetere from here 
	 * http://www.lfsforum.net/showthread.php?p=1173222#post1173222
	 * @return
	 */
	public double getTrackLength() {
		if(trackLength != Double.MIN_VALUE) {
			return trackLength;
		} else {
			double length = 0;

			for(int i=0; i<trackNodes.size(); i++) {
				if(i<trackNodes.size()-1) {
					OneNode thisNode = trackNodes.get(i);
					OneNode nextNode = trackNodes.get(i+1);

					length += getDistanceBetweenVectors(
							thisNode.getCentrePositionVector(), 
							nextNode.getCentrePositionVector());
				} else {
					OneNode thisNode = trackNodes.get(i);
					OneNode firstNode = trackNodes.get(0);

					length += getDistanceBetweenVectors(
							firstNode.getCentrePositionVector(), 
							thisNode.getCentrePositionVector());
				}
			}

			trackLength = length;

			return trackLength;
		}
	}
	
	// Returns the distance between the two given vectors in metres
	private double getDistanceBetweenVectors(InSimVector firstPosition, 
			InSimVector secondPosition) {
		double diffX = Math.pow((secondPosition.getX() - firstPosition.getX())/65536.0, 2);
		double diffY = Math.pow((secondPosition.getY() - firstPosition.getY())/65536.0, 2);
		double diffZ = Math.pow((secondPosition.getZ() - firstPosition.getZ())/65536.0, 2);

		return Math.sqrt(diffX + diffY + diffZ);
	}
	
	/**
	 * Returns a Point representing the centre of this track
	 * @return
	 */
	public Point getCentrePoint() {
		double totalX = 0, totalY = 0;
		
		for(OneNode oneNode : trackNodes.values()) {
			totalX += (oneNode.getCentreX()/65536.0);
			totalY += (oneNode.getCentreY()/65536.0);
		}

		int centreX = (int) (totalX / (double) trackNodes.size());
		int centreY = (int) (totalY / (double) trackNodes.size());
		
		return new Point(centreX, centreY);
	}
	
	/**
	 * Returns the node with the given index if it's available or null if it's not
	 * @param nodeIndex
	 * @return
	 */
	public OneNode getNodeAtIndex(int nodeIndex) {
		if(trackNodes.containsKey(nodeIndex)) {
			return trackNodes.get(nodeIndex);
		} else {
			return null;
		}
	}
	
	/**
	 * Returns the distance from the start of the lap to the driver's current position
	 * in metres. May return null if there is no track data or incomplete track data
	 * @param position
	 * @return
	 */
	public Double getLapLengthToPositionInMetres(MiniCompCar miniCompCar) {
		if(miniCompCar == null) {
			return null;
		}
		
		int nodeIndex = miniCompCar.getNode();

		if(lapLengthToCentreOfNode.containsKey(nodeIndex)) {
			double lapLengthToNode = lapLengthToCentreOfNode.get(nodeIndex);

			OneNode nodeData = getNodeAtIndex(nodeIndex);
			
			if(nodeData == null) {
				// Node data missing, return null
				return null;
			}
			
			// Add/subtract the distance from centre of node to current position
			lapLengthToNode += getDistanceBetweenVectors(
					nodeData.getCentrePositionVector(), miniCompCar.getPosition());
			
			return lapLengthToNode;
		} else {
			return null;
		}
	}

	/**
	 * Returns the percentage of the lap length completed at the position in the given
	 * CompCar. May return null if there is missing track length data
	 * @param compCar
	 * @return
	 */
	public Double getPercentageOfLapCompletedAtPosition(MiniCompCar miniCompCar) {
		if(miniCompCar == null) {
			return null;
		}
		
		Double lapLengthCompleted = getLapLengthToPositionInMetres(miniCompCar);
		
		if(lapLengthCompleted != null) {
			double trackLength = getTrackLength();
			
			if(trackLength != 0) {
				return (lapLengthCompleted / trackLength) * 100;
			} else {
				return null;
			}
		} else {
			return null;
		}
	}
	
	/**
	 * Get the X coordinate for the centre of this track
	 * @return
	 */
	public Double getCentreXCoordinate() {
		if(centreXCoordinate != null) {
			return centreXCoordinate;
		} else {
			calculateCentreCoordinates();
			
			return centreXCoordinate;
		}
	}

	/**
	 * Get the Y coordinate for the centre of this track
	 * @return
	 */
	public Double getCentreYCoordinate() {
		if(centreYCoordinate != null) {
			return centreYCoordinate;
		} else {
			calculateCentreCoordinates();
			
			return centreYCoordinate;
		}
	}
	
	/**
	 * Get the Z coordinate for the centre of this track
	 * @return
	 */
	public Double getCentreZCoordinate() {
		if(centreZCoordinate != null) {
			return centreZCoordinate;
		} else {
			calculateCentreCoordinates();
			
			return centreZCoordinate;
		}
	}
	
	/*
	 * Populate the map containing the track length at each node. These lengths
	 * are used in the process of calculating time gaps between drivers
	 */
	private void populateLapLengthAtNodesMap() {
		if(trackNodes != null && trackNodes.size() > 0) {
			if(finishNodeIndex != trackNodes.size()-1) {
				// First add in the distance from the finish line to first node after
				OneNode finishNode = trackNodes.get(finishNodeIndex);
				OneNode firstNodeAfterFinish = trackNodes.get(finishNodeIndex+1);

				if(finishNode != null && firstNodeAfterFinish != null) {
					Double distanceToFirstNode = getDistanceBetweenVectors(finishNode.
							getCentrePositionVector(), 
							firstNodeAfterFinish.getCentrePositionVector());

					if(distanceToFirstNode != null) {
						lapLengthToCentreOfNode.put(firstNodeAfterFinish.getNodeIndex(), 
								distanceToFirstNode);
					}
				} 
			} else {
				// Finish node and max node are the same, add in zero for the max node
				lapLengthToCentreOfNode.put(finishNodeIndex, 0.0);
			}

			// Start with the node after finish node and work round
			for(int i=finishNodeIndex+1; i<trackNodes.size()-1; i++) {
				OneNode firstNode = trackNodes.get(i);
				OneNode secondNode = trackNodes.get(i+1);

				if(firstNode != null && secondNode != null) {
					/*
					 * Get the distance to the first node then add on the distance
					 * between these nodes
					 */
					Double distanceToFirstNode = lapLengthToCentreOfNode.get(
							firstNode.getNodeIndex());

					if(distanceToFirstNode != null) {
						Double distance = getDistanceBetweenVectors(firstNode.
								getCentrePositionVector(), 
								secondNode.getCentrePositionVector());

						if(distance != null) {
							lapLengthToCentreOfNode.put(secondNode.getNodeIndex(), 
									distanceToFirstNode+distance);
						}
					} 
				}
			}

			// Now add in the distance from the last node (node with max value) to 0 node
			OneNode maxNode = trackNodes.get(trackNodes.size()-1);
			OneNode zeroNode = trackNodes.get(0);

			double distanceFromLastNodeToNodeZero = 0;

			if(maxNode != null && zeroNode != null) {
				Double distanceToMaxNode = lapLengthToCentreOfNode.get(trackNodes.size()-1);
				
				Double distanceToZero = getDistanceBetweenVectors(maxNode.
						getCentrePositionVector(), 
						zeroNode.getCentrePositionVector());

				if(distanceToZero != null && distanceToMaxNode != null) {
					distanceFromLastNodeToNodeZero = distanceToZero;

					lapLengthToCentreOfNode.put(zeroNode.getNodeIndex(), 
							distanceToZero+distanceToMaxNode);
				}
			}
			// End special node (max -> zero)

			// now go from 0 back up to the finish line
			for(int i=0; i<finishNodeIndex; i++) {
				OneNode firstNode = trackNodes.get(i);
				OneNode secondNode = trackNodes.get(i+1);

				if(firstNode != null && secondNode != null) {
					/*
					 * Get the distance to the first node then add on the distance
					 * between these nodes
					 */
					Double distanceToFirstNode = lapLengthToCentreOfNode.get(firstNode.getNodeIndex());

					if(distanceToFirstNode != null) {
						Double distance = getDistanceBetweenVectors(firstNode.
								getCentrePositionVector(), secondNode.getCentrePositionVector());

						if(distance != null) {
							if(i == 1) {
								// if this is the first node after zero add in the zero distance
								lapLengthToCentreOfNode.put(secondNode.getNodeIndex(), 
										distanceFromLastNodeToNodeZero+distanceToFirstNode+
										distance);
							} else {
								lapLengthToCentreOfNode.put(secondNode.getNodeIndex(), 
										distanceToFirstNode+distance);
							}
						}
					}
				}
			}
		}
	}
	
//	private void calculateDriveLeftAndRightCoordinates() {
//		if(centreXCoordinate == null) {
//			calculateCentreCoordinates();
//		}
//		
//		for(OneNode oneNode : trackNodes.values()) {
//			oneNode.calculateDriveLeftAndRightCoordinates(centreXCoordinate, 
//					centreYCoordinate, centreZCoordinate);
//		}
//	}
//	
//	private void calculateLimitLeftAndRightCoordinates() {
//		if(centreXCoordinate == null) {
//			calculateCentreCoordinates();
//		}
//		
//		for(OneNode oneNode : trackNodes.values()) {
//			oneNode.calculateLimitLeftAndRightCoordinates(centreXCoordinate, 
//					centreYCoordinate, centreZCoordinate);
//		}
//	}
	
	private void calculateCentreCoordinates() {
		double totalX = 0;
		double totalY = 0;
		double totalZ = 0;
		
		for(OneNode oneNode : trackNodes.values()) {
			totalX += oneNode.getCentreX();
			totalY += oneNode.getCentreY();
			totalZ += oneNode.getCentreZ();
		}
		
		centreXCoordinate = (totalX / (double) trackNodes.size());
		centreYCoordinate = (totalY / (double) trackNodes.size());
		centreZCoordinate = (totalZ / (double) trackNodes.size());
	}
	
	@SuppressWarnings("unused")
	private void print(String msg) {
		if(track != null) {
			LFSQualifyingTickerStart.print(this.getClass().getSimpleName()+
					" - "+track.getShortname()+" - "+msg);
		} else {
			LFSQualifyingTickerStart.print(this.getClass().getSimpleName()+
					" - "+msg);
		}
	}
}