package lfsqualifyingticker.structures;

import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.math.BigDecimal;
import java.util.HashMap;

import javax.swing.Box;
import javax.swing.JLabel;

import lfsqualifyingticker.LFSQualifyingTickerGUI;
import lfsqualifyingticker.LFSQualifyingTickerStart;

import net.sf.jinsim.Colors;
import net.sf.jinsim.response.CameraPositionResponse;
import net.sf.jinsim.response.ConnectionCloseResponse;
import net.sf.jinsim.response.ConnectionLeaveResponse;
import net.sf.jinsim.response.LapTimeResponse;
import net.sf.jinsim.response.MessageResponse;
import net.sf.jinsim.response.MultiCarInfoResponse;
import net.sf.jinsim.response.NewConnectionResponse;
import net.sf.jinsim.response.NewPlayerResponse;
import net.sf.jinsim.response.NodeLapInfoResponse;
import net.sf.jinsim.response.PlayerLeavingResponse;
import net.sf.jinsim.response.RaceStartResponse;
import net.sf.jinsim.response.SplitTimeResponse;
import net.sf.jinsim.response.StateResponse;
import net.sf.jinsim.response.TinyResponse;
import net.sf.jinsim.types.InSimTime;

/**
 * Collection of little helper methods for the LFS Qualifying Ticker application
 * @author Gus
 *
 */

public class Helper {
	/* 
	 * A cache for use with the getPlainPlayerName function. Key is the raw player name 
	 * string, value is the formatted string
	 */
	private static HashMap<String, String> playerNameCache = new HashMap<String, String>(50);
	
	/*
	 * A cache for use with the get colourful player name box method. Key is the raw
	 * player name string, value is the box containing the player name
	 */
	private static HashMap<String, Box> playerNameBoxCache = new HashMap<String, Box>(40);
	
	/**
	 * Returns a priority integer for certain types of InSimResponse. This is used to 
	 * handle certain packets before others (e.g. NewConnectionResponse should be handled
	 * before a SplitResponse)
	 * @param inSimClass
	 * @return
	 */
	public static Integer getInSimPacketPriority(Class<?> inSimClass) {
		if(inSimClass.equals(ConnectionCloseResponse.class)) {
			return 0;
		} else if(inSimClass.equals(NewConnectionResponse.class) || 
				inSimClass.equals(ConnectionLeaveResponse.class)) {
			return 1;
		} else if(inSimClass.equals(NewPlayerResponse.class) || 
				inSimClass.equals(PlayerLeavingResponse.class)) {
			return 2;
		} else if(inSimClass.equals(RaceStartResponse.class)) {
			return 3;
		} else if(inSimClass.equals(StateResponse.class)) {
			return 4;
		} else if(inSimClass.equals(NodeLapInfoResponse.class)) {
			return 5;
		} else if(inSimClass.equals(MultiCarInfoResponse.class)) {
			return 6;
		} else if(inSimClass.equals(TinyResponse.class)) {
			return 7;
		} else if(inSimClass.equals(CameraPositionResponse.class)) {
			return 20;
		} else if(inSimClass.equals(MessageResponse.class)) {
			return 21;
		} else {
			return Integer.MAX_VALUE;
		}
	}
	
	/**
	 * Returns a String representation of the given position (e.g. 
	 * an input of 1 returns "1st", 2 returns "2nd")
	 * @param position
	 * @return
	 */
	public static String getPositionStringForInt(int position) {
		// There must be 'nicer' ways of doing this but probably not as quick
		switch(position) {
			case 1: { return position+"st"; } 
			case 2: { return position+"nd"; } 
			case 3: { return position+"rd"; } 
			case 4: { return position+"th"; } 
			case 5: { return position+"th"; } 
			case 6: { return position+"th"; } 
			case 7: { return position+"th"; } 
			case 8: { return position+"th"; } 
			case 9: { return position+"th"; } 
			case 10: { return position+"th"; } 
			case 11: { return position+"th"; } 
			case 12: { return position+"th"; } 
			case 13: { return position+"th"; } 
			case 14: { return position+"th"; } 
			case 15: { return position+"th"; } 
			case 16: { return position+"th"; } 
			case 17: { return position+"th"; } 
			case 18: { return position+"th"; } 
			case 19: { return position+"th"; } 
			case 20: { return position+"th"; } 
			case 21: { return position+"st"; } 
			case 22: { return position+"nd"; } 
			case 23: { return position+"rd"; } 
			case 24: { return position+"th"; } 
			case 25: { return position+"th"; } 
			case 26: { return position+"th"; } 
			case 27: { return position+"th"; } 
			case 28: { return position+"th"; } 
			case 29: { return position+"th"; } 
			case 30: { return position+"th"; } 
			case 31: { return position+"st"; } 
			case 32: { return position+"nd"; } 
			case 33: { return position+"rd"; } 
			case 34: { return position+"th"; } 
			case 35: { return position+"th"; } 
			case 36: { return position+"th"; } 
			case 37: { return position+"th"; } 
			case 38: { return position+"th"; } 
			case 39: { return position+"th"; } 
			case 40: { return position+"th"; } 
			case 41: { return position+"st"; } 
			case 42: { return position+"nd"; } 
			case 43: { return position+"rd"; } 
			case 44: { return position+"th"; } 
			case 45: { return position+"th"; } 
			case 46: { return position+"th"; } 
			case 47: { return position+"th"; } 
			case 48: { return position+"th"; } 
			case 49: { return position+"th"; } 
			case 50: { return position+"th"; } 
			default: { 
				print("getPositionStringForInt - " +
					"returning ? for position: "+position);
				return "?"; 
			}
		}
	}
	
	/**
	 * Set the preferred size, size and max size of the given component to the given size
	 * @param comp
	 * @param size
	 */
	public static void setPrefSizeSizeAndMaxSizeOnComponent(Component comp, Dimension size) {
		comp.setPreferredSize(size);
		comp.setSize(size);
		comp.setMaximumSize(size);
	}
	
	/**
	 * 
	 * @param comp
	 * @param width
	 */
	public static void setPrefSizeSizeAndMaxSizeOnComponent(Component comp, int width) {
		int height = 1;
		
		if(comp.getMinimumSize() != null) {
			height = comp.getMinimumSize().height;
		} else if(comp.getPreferredSize() != null) {
			height = comp.getPreferredSize().height;
		} else if(comp.getSize() != null) {
			height = comp.getSize().height;
		}
		
		setPrefSizeSizeAndMaxSizeOnComponent(comp, new Dimension(width, height));
	}
	
	/**
	 * Returns a plain player name (without colour codes and other formatting) from
	 * the raw player name
	 * @param rawPlayerName
	 * @return
	 */
	public static String getPlainPlayerName(String rawPlayerName) {
		if(playerNameCache.containsKey(rawPlayerName)) {
			return playerNameCache.get(rawPlayerName);
		} else {
			// If the cache is too big trim it back down
			if(playerNameCache.size() > 100) {
				playerNameCache.clear();
			}
			
			String nameToReturn = rawPlayerName;
			
			if(nameToReturn != null) {
				// Look out for the vertical bar (pipe) before colour codes. ^v is the pipe
				if(nameToReturn.contains("^v")) {
					nameToReturn = nameToReturn.replaceAll("\\^v", "|");
				}

				while(nameToReturn.contains("^")) {
					nameToReturn = nameToReturn.substring(0, nameToReturn.indexOf("^"))+
					nameToReturn.substring(nameToReturn.indexOf("^")+2);
				}
				
				playerNameCache.put(rawPlayerName, nameToReturn);
				
				return nameToReturn;
			} else {
				return "";
			}
		}
	}
	
	/**
	 * Returns a plain string (with colour codes and other formatting removed)
	 * for the given player
	 * @param player
	 * @return
	 */
	public static String getPlainPlayerNameFromPlayer(NewPlayerResponse player) {
		if(player != null && player.getPlayerName() == null) {
			return getPlainPlayerName(player.getPlayerName());
		} else {
			return null;
		}
	}
	
	/**
	 * Returns a formatted string for the given time
	 * @param time
	 * @return
	 */
	public static String getTimeString(SplitTimeResponse splitTime) {
		return getTimeString(splitTime.getTime());
	}
	
	/**
	 * Returns a formatted string for the given time
	 * @param time
	 * @return
	 */
	public static String getTimeString(LapTimeResponse laptime) {
		return getTimeString(laptime.getTime());
	}
	
	/**
	 * Returns a formatted string for the given time
	 * @param time
	 * @return
	 */
	public static String getTimeString(int time) {
		return getTimeString(new InSimTime(time));
	}
	
	/**
	 * Returns a formatted string for the given time
	 * @param time
	 * @return
	 */
	public static String getTimeString(InSimTime time) {
		StringBuffer timeString = new StringBuffer("");
		
		if(time.getHours() > 0) {
			timeString.append(time.getHours()+":");
		}
		
		if(time.getMinutes() > 0) {
			timeString.append(time.getMinutes()+":");
		}
		
		if(time.getSeconds() < 10) {
			if(time.getMinutes() > 0) {
				timeString.append("0"+time.getSeconds()+".");
			} else {
				timeString.append(time.getSeconds()+".");
			}
		} else {
			timeString.append(time.getSeconds()+".");
		}
		
		if(time.getThousandths() < 10) {
			if(time.getThousandths() == 0) {
				timeString.append("00");
			} else {
				timeString.append("00"+time.getThousandths());
			}
		} else if(time.getThousandths() < 100) {
			if(time.getThousandths() % 10 == 0) {
				timeString.append("0"+(time.getThousandths()/10));
			} else {
				timeString.append("0"+time.getThousandths());
			}
		} else {
			if(time.getThousandths() % 10 == 0) {
				timeString.append(time.getThousandths()/10);
			} else {
				timeString.append(time.getThousandths());
			}
		}
		
		return timeString.toString();
	}
	
	/**
	 * Method used to help with performance benchmarking algorithms
	 * @param taskName
	 * @param startTime
	 * @return
	 */
	public static String getTimeTaken(String taskName, long startTime) {
		return getTimeTaken(taskName, startTime, System.nanoTime(), 2);
	}
	
	/**
	 * Method used to help with performance benchmarking algorithms
	 * @param taskName
	 * @param startTime
	 * @param endTime
	 * @return
	 */
	public static String getTimeTaken(String taskName, long startTime, long endTime) {
		return getTimeTaken(taskName, startTime, endTime, 2);
	}
	
	/**
	 * Method used to help with performance benchmarking algorithms
	 * @param taskName
	 * @param startTime
	 * @param endTime
	 * @param decimalPlaces
	 * @return
	 */
	public static String getTimeTaken(String taskName, long startTime, long endTime, int decimalPlaces) {
		BigDecimal timeTaken = new BigDecimal((endTime - startTime) / 1000000.0);
		timeTaken = timeTaken.setScale(decimalPlaces, BigDecimal.ROUND_HALF_UP);
		
		return timeTaken.toPlainString()+"ms to "+taskName;
	}
	
	/**
	 * Returns a horizontal Box containing the player's name coloured in similar colours
	 * to that of LFS
	 * @param rawPlayerName
	 * @return
	 */
	public static Box getColourfulPlayerNameBox(String rawPlayerName) {
		if(playerNameBoxCache.containsKey(rawPlayerName)) {
			return playerNameBoxCache.get(rawPlayerName);
		} else {
			// Clear out the cache if it's too big
			if(playerNameBoxCache.size() > 75) {
				playerNameBoxCache.clear();
			}
			
			Box playerNameBox = Box.createHorizontalBox();
			
			// Split the raw text into parts based on colour
			String[] parts = rawPlayerName.split("\\^");

			for(String part : parts) {
				Color thisPartForegroundColour = QualiTickerPrefs.LFSTEXTDEFAULTCOLOR;

				if(part.length() > 0 && Character.isDigit(part.charAt(0))) {
					String colourCode = "^"+part.charAt(0);
					part = part.substring(1);

					if(colourCode.equals(Colors.BLACK)) {
						thisPartForegroundColour = QualiTickerPrefs.LFSTEXTBLACKCOLOR;
					} else if(colourCode.equals(Colors.WHITE)) {
						thisPartForegroundColour = QualiTickerPrefs.LFSTEXTWHITECOLOR;
					} else if(colourCode.equals(Colors.BLUE)) {
						thisPartForegroundColour = QualiTickerPrefs.LFSTEXTBLUECOLOR;
					} else if(colourCode.equals(Colors.GREEN)) {
						thisPartForegroundColour = QualiTickerPrefs.LFSTEXTGREENCOLOR;
					} else if(colourCode.equals(Colors.LIGHT_BLUE)) {
						thisPartForegroundColour = QualiTickerPrefs.LFSTEXTLIGHTBLUECOLOR;
					} else if(colourCode.equals(Colors.RED)) {
						thisPartForegroundColour = QualiTickerPrefs.LFSTEXTREDCOLOR;
					} else if(colourCode.equals(Colors.VIOLET)) {
						thisPartForegroundColour = QualiTickerPrefs.LFSTEXTVIOLETCOLOR;
					} else if(colourCode.equals(Colors.YELLOW)) {
						thisPartForegroundColour = QualiTickerPrefs.LFSTEXTYELLOWCOLOR;
					} else if(colourCode.equals("^9")) {
						// The ^9 string is used to indicate a change back to 
						// the default colour from another colour
					}
				} else if(part.length() > 0 && part.charAt(0) == 'v') {
					// The string ^v is used to signify the vertical bar (pipe) in the name
					part = part.replaceAll("v", "|");

					// If there's a label before use the same colour as that label
					if(playerNameBox.getComponentCount() > 0) {
						JLabel lastLabel = (JLabel) playerNameBox.getComponent(
								playerNameBox.getComponentCount()-1);

						thisPartForegroundColour = lastLabel.getForeground();
					}
				}

				JLabel label = new JLabel(part);
				label.setForeground(thisPartForegroundColour);
				label.setOpaque(true);
				label.setBackground(LFSQualifyingTickerGUI.BACKGROUND_COLOUR);

				playerNameBox.add(label);
			}
			
			playerNameBox.add(Box.createHorizontalGlue());
			
			// Add to cache
			playerNameBoxCache.put(rawPlayerName, playerNameBox);

			return playerNameBox;
		}
	}
	
	/**
	 * Round the given BigDecimal to 2 decimal places using the 
	 * BigDecimal.ROUND_HALF_UP rounding mode
	 * @param bigDecimal
	 * @return
	 */
	public static BigDecimal roundBigDecimalTo2DP(BigDecimal bigDecimal) {
		return roundBigDecimal(bigDecimal, 2);
	}
	
	/**
	 * Round the given BigDecimal to the given number of decimal 
	 * places using the BigDecimal.ROUND_HALF_UP rounding mode
	 * @param bigDecimal
	 * @param numPlaces
	 * @return
	 */
	public static BigDecimal roundBigDecimal(BigDecimal bigDecimal, int numPlaces) {
		return bigDecimal.setScale(numPlaces, BigDecimal.ROUND_HALF_UP);
	}
	
	/**
	 * Returns the speed in metres per second for the raw speed input
	 * @param rawSpeed
	 * @return
	 */
	public static double getSpeedInMetresPerSecond(short rawSpeed) {
		// 32768 = 100 metres per second
		return (rawSpeed / 32768.0) * 100;
	}
	
	public static double getLengthOf3DVector(double x, double y, double z) {
		return Math.sqrt((x*x) + (y*y) + (z*z));
	}
	
	@SuppressWarnings("unused")
	private static void print(String msg) {
		LFSQualifyingTickerStart.print(msg);
	}
}
