package lfsqualifyingticker.structures;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.Rectangle;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.beans.PropertyChangeEvent;
import java.math.BigDecimal;
import java.util.HashMap;

import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.Timer;

import net.sf.jinsim.response.NewPlayerResponse;
import net.sf.jinsim.types.CompCar;

import lfsqualifyingticker.LFSQualifyingTickerGUI;
import lfsqualifyingticker.LFSQualifyingTickerStart;

/**
 * Provides an interactive 2D track map for a given LFS track
 * @author Gus
 *
 */

public class TrackMap extends JPanel {
	/**
	 * 
	 */
	private static final long serialVersionUID = -7630282012006986213L;

	private TrackCanvas trackCanvas;
	
	private JScrollPane trackScrollPane;
	
	// Timer which will be used to redraw the track map
	private Timer updateMapTimer;

	public TrackMap(TrackData trackData, ActionListener listener) {
		trackCanvas = new TrackCanvas(trackData, listener);

		updateMapTimer = new Timer(QualiTickerPrefs.getUpdateTrackMapDelay(), 
				new ActionListener() {
			public void actionPerformed(ActionEvent e) { 
				requestRepaint();
			}
		});
		
		updateMapTimer.start();
		
		StringBuilder trackDetailsString = new StringBuilder();
		
		trackDetailsString.append(" "+trackData.getTrack().getName()+" ("+
				trackData.getTrack().getShortname()+")");
		
		Double trackLengthMetres = trackData.getTrackLength();
		
		if(trackLengthMetres != null) {
			BigDecimal trackLengthKilometresBD = new BigDecimal(trackLengthMetres/1000.0);
			trackLengthKilometresBD = Helper.roundBigDecimal(trackLengthKilometresBD, 3);
			
			BigDecimal trackLengthMilesBD = new BigDecimal(trackLengthMetres/1609.33);
			trackLengthMilesBD = Helper.roundBigDecimal(trackLengthMilesBD, 3);
			
			trackDetailsString.append(" - "+trackLengthMilesBD.toPlainString()+
					" miles ("+trackLengthKilometresBD.toPlainString()+" km)");
		}
		
		JLabel trackDetailsLabel = new JLabel(trackDetailsString.toString());
		trackDetailsLabel.setOpaque(true);
		trackDetailsLabel.setBackground(LFSQualifyingTickerGUI.BACKGROUND_COLOUR);
		
		this.setLayout(new BorderLayout());
		
		trackScrollPane = new JScrollPane();
		trackScrollPane.getViewport().add(trackCanvas);
		
		trackScrollPane.setOpaque(true);
		trackScrollPane.setBackground(Color.white);
		
		this.add(trackScrollPane, BorderLayout.CENTER);
		this.add(trackDetailsLabel, BorderLayout.SOUTH);
		
		this.setBackground(Color.white);
	}
	
	private void requestRepaint() {
		if(trackScrollPane != null && trackCanvas != null) {
			Rectangle visibleTrackRect = trackScrollPane.getVisibleRect();
		
			trackCanvas.setPreferredSize(new Dimension(visibleTrackRect.width, 
				visibleTrackRect.height));
		}
		
		this.repaint();
	}

	/**
	 * Update the position of players on the track map
	 * @param playerPositions
	 */
	public void updatePlayerPositionNodes(HashMap<NewPlayerResponse, CompCar> playerPositions) {
		trackCanvas.updatePlayerPositions(playerPositions);
	}
	
	/**
	 * Update the size of the track based on the new position of the split pane
	 */
	public void updateTrackSizeBasedOnSplitPane(PropertyChangeEvent evt) {
		if(trackScrollPane != null) {
			//print("oldValue: "+evt.getOldValue().toString());
			//print("newValue: "+evt.getNewValue().toString());
			
			//print("updateTrackSizeBasedOnSplitPane - scrollPane size: "+
			//		trackScrollPane.getVisibleRect().toString());
			
			trackCanvas.setPreferredSize(new Dimension(trackScrollPane.getVisibleRect().width, 
					trackScrollPane.getVisibleRect().height));
			
			requestRepaint();
		}
	}
	
	/**
	 * Remove the given player from the track map
	 * @param playerID
	 */
	public void removePlayerFromTrackMap(byte playerID) {
		trackCanvas.removePlayerFromTrackMap(playerID);
	}

	@SuppressWarnings("unused")
	private void print(String msg) {
		LFSQualifyingTickerStart.print(this.getClass().getSimpleName()+" - "+msg);
	}
}
