package lfsqualifyingticker.structures;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Point;
import java.awt.Polygon;
import java.awt.Rectangle;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.util.HashMap;

import javax.swing.JPanel;

import net.sf.jinsim.response.NewPlayerResponse;
import net.sf.jinsim.types.CompCar;

import lfsqualifyingticker.LFSQualifyingTickerGUI;
import lfsqualifyingticker.LFSQualifyingTickerStart;

public class TrackCanvas extends JPanel {
	/**
	 * 
	 */
	private static final long serialVersionUID = -2806741103405871329L;

	private TrackData trackData;
	
	private int maximumX=Integer.MIN_VALUE, maximumY, minimumX, minimumY, trackWidth, trackHeight;
	
	private Polygon trackCentrePolygon, driveLeftPolygon, driveRightPolygon, limitLeftPolygon, limitRightPolygon;
	
	private Dimension lastSize, preferredSize;
	
	private HashMap<NewPlayerResponse, CompCar> playerPositions = 
		new HashMap<NewPlayerResponse, CompCar>();
	
	private final int ovalDiameter = 15;
	
	private HashMap<Byte, Point> playersDrawnOnMap = new HashMap<Byte, Point>();
	
	private ActionListener listener;

	public TrackCanvas(TrackData trackData, ActionListener listener) {
		this.trackData = trackData;
		
		this.listener = listener;
		
		setupTrackCanvas();
	}
	
	private void setupTrackCanvas() {
		this.addMouseListener(new MouseAdapter() {
			public void mouseClicked(MouseEvent e) {
				handleMouseClickOnMap(e);
			}
		});

		createTrackPolygon();
	}

	public void createTrackPolygon() {
		print("createTrackPolygon called");
		
		OneNode[] allNodes = trackData.getAllNodes();
		
		// Maximum x hasn't been set yet, calculate these values now
		if(maximumX == Integer.MIN_VALUE) {
			for(OneNode oneNode : allNodes) {
				int centreX = (int) (oneNode.getCentreX()/65536.0);
				int centreY = (int) (oneNode.getCentreY()/65536.0);

				if(centreX > maximumX) {
					maximumX = centreX;
				}
				if(centreX < minimumX) {
					minimumX = centreX;
				}

				if(centreY > maximumY) {
					maximumY = centreY;
				}
				if(centreY < minimumY) {
					minimumY = centreY;
				}
			}

			trackWidth = maximumX - minimumX;
			trackHeight = maximumY - minimumY;
		}
		
		double scaleFactor = 1;

		Dimension targetSize = this.getSize();

		// Scale down each of the points of the track according to the preferred size
		if(trackWidth > trackHeight) {
			scaleFactor = (trackWidth / targetSize.getWidth() * 1.20);
		} else if(trackHeight > trackWidth) {
			scaleFactor = (trackHeight / targetSize.getHeight() * 1.20);
		} else {
			scaleFactor = (trackHeight / targetSize.getHeight() * 1.20);
		}

		int[] trackCentreXCoords = new int[allNodes.length];
		int[] trackCentreYCoords = new int[allNodes.length];
		
		int[] driveLeftXCoords = new int[allNodes.length];
		int[] driveLeftYCoords = new int[allNodes.length];
		
		int[] driveRightXCoords = new int[allNodes.length];
		int[] driveRightYCoords = new int[allNodes.length];
		
		int[] limitLeftXCoords = new int[allNodes.length];
		int[] limitLeftYCoords = new int[allNodes.length];
		
		int[] limitRightXCoords = new int[allNodes.length];
		int[] limitRightYCoords = new int[allNodes.length];
		
		for(OneNode oneNode : allNodes) {
			int centreX = (int) (oneNode.getCentreX()/65536.0);
			int centreY = (int) (oneNode.getCentreY()/65536.0);
			
			trackCentreXCoords[oneNode.getNodeIndex()] = (int) (centreX / scaleFactor);
			trackCentreYCoords[oneNode.getNodeIndex()] = (int) (centreY / scaleFactor);

			int driveLeftX = (int) (oneNode.getDriveLeftXCoordinate()/65536.0);
			int driveLeftY = (int) (oneNode.getDriveLeftYCoordinate()/65536.0);
			
			driveLeftXCoords[oneNode.getNodeIndex()] = (int) (driveLeftX / scaleFactor);
			driveLeftYCoords[oneNode.getNodeIndex()] = (int) (driveLeftY / scaleFactor);
			
			int driveRightX = (int) (oneNode.getDriveRightXCoordinate()/65536.0);
			int driveRightY = (int) (oneNode.getDriveRightYCoordinate()/65536.0);
			
			driveRightXCoords[oneNode.getNodeIndex()] = (int) (driveRightX / scaleFactor);
			driveRightYCoords[oneNode.getNodeIndex()] = (int) (driveRightY / scaleFactor);
			
			int limitLeftX = (int) (oneNode.getLimitLeftXCoordinate()/65536.0);
			int limitLeftY = (int) (oneNode.getLimitLeftYCoordinate()/65536.0);
			
			limitLeftXCoords[oneNode.getNodeIndex()] = (int) (limitLeftX / scaleFactor);
			limitLeftYCoords[oneNode.getNodeIndex()] = (int) (limitLeftY / scaleFactor);
			
			int limitRightX = (int) (oneNode.getLimitRightXCoordinate()/65536.0);
			int limitRightY = (int) (oneNode.getLimitRightYCoordinate()/65536.0);
			
			limitRightXCoords[oneNode.getNodeIndex()] = (int) (limitRightX / scaleFactor);
			limitRightYCoords[oneNode.getNodeIndex()] = (int) (limitRightY / scaleFactor);
		}
		
//		for(int i=0; i<driveLeftXCoords.length; i++) {
//			print("driveLeftXCoords ["+i+"]: "+driveLeftXCoords[i]);
//			print("driveLeftYCoords ["+i+"]: "+driveLeftYCoords[i]);
//		}
		
		trackCentrePolygon = new Polygon(trackCentreYCoords, trackCentreXCoords, allNodes.length);
		driveLeftPolygon = new Polygon(driveLeftYCoords, driveLeftXCoords, allNodes.length);
		driveRightPolygon = new Polygon(driveRightYCoords, driveRightXCoords, allNodes.length);
		limitLeftPolygon = new Polygon(limitLeftYCoords, limitLeftXCoords, allNodes.length);
		limitRightPolygon = new Polygon(limitRightYCoords, limitRightXCoords, allNodes.length);

		// Shift the track up to the top left of the canvas area
		Rectangle polyBounds = trackCentrePolygon.getBounds();
			
		trackCentrePolygon.translate(-(polyBounds.x)+30, -(polyBounds.y)+30);
		driveLeftPolygon.translate(-(polyBounds.x)+30, -(polyBounds.y)+30);
		driveRightPolygon.translate(-(polyBounds.x)+30, -(polyBounds.y)+30);
		limitLeftPolygon.translate(-(polyBounds.x)+30, -(polyBounds.y)+30);
		limitRightPolygon.translate(-(polyBounds.x)+30, -(polyBounds.y)+30);
		
		lastSize = targetSize;
	}
	
	/*
	 * (non-Javadoc)
	 * @see javax.swing.JComponent#paintComponent(java.awt.Graphics)
	 */
	public synchronized void paintComponent(Graphics g) {
		// If the size has changed recreate the polygon
		Dimension currentSize = this.getPreferredSize() != null ? 
				this.getPreferredSize() : this.getSize();
		
		print("paintComponent, trackCanvas.size(): "+currentSize.toString());
		
		if(lastSize == null || 
				currentSize.height != lastSize.height || 
				currentSize.width != lastSize.width) {
			createTrackPolygon();
		}
		
		if(trackCentrePolygon != null) {
			//g.setColor(Color.black);
			//g.drawPolygon(trackCentrePolygon);
			
			g.setColor(Color.red);
			g.drawPolygon(driveLeftPolygon);
			
			g.setColor(Color.blue);
			g.drawPolygon(driveRightPolygon);
			
			//g.setColor(Color.magenta);
			//g.drawPolygon(limitLeftPolygon);
			
			//g.setColor(Color.black);
			//g.drawPolygon(limitRightPolygon);
			
			playersDrawnOnMap.clear();
			
			// Now add the player's current positions
			for(NewPlayerResponse player : playerPositions.keySet()) {
				byte playerID = player.getPlayerId();
				
				CompCar compCar = playerPositions.get(player);

				if(compCar != null && compCar.getLap() > 0) {
					int xCoord = trackCentrePolygon.xpoints[compCar.getNode()];
					int yCoord = trackCentrePolygon.ypoints[compCar.getNode()];

					if(xCoord < 0) {
						xCoord += ovalDiameter/2.0;
					} else {
						xCoord -= ovalDiameter/2.0;
					}

					if(yCoord < 0) {
						yCoord += ovalDiameter/2.0;
					} else {
						yCoord -= ovalDiameter/2.0;
					}

					playersDrawnOnMap.put(playerID, new Point(xCoord, yCoord));

					// If the player is on an outlap draw them in blue
					if(compCar.getLap() == 1) {
						g.setColor(Color.blue);
					} else {
						g.setColor(Color.black);
					}

//					if(player.getPlayerName().contains("amp")) {
//						print(Helper.getPlainPlayerName(player.getPlayerName())+", direction: "+compCar.getDirection()+
//							", direction (degrees): "+compCar.getDirectionInDegrees());
//					}
					
					g.drawOval(xCoord, yCoord, ovalDiameter, ovalDiameter);
					g.drawString(Helper.getPlainPlayerName(player.getPlayerName()), 
							xCoord, yCoord);
					
					Rectangle polyBounds = trackCentrePolygon.getBounds();
					
					g.setColor(Color.orange);
					g.drawRect(polyBounds.x, polyBounds.y, polyBounds.width, polyBounds.height);
				}
			}
		}
	}
	
	/*
	 * (non-Javadoc)
	 * @see javax.swing.JComponent#getPreferredSize()
	 */
	public Dimension getPreferredSize() {
		return preferredSize;
		
//		if(trackPolygon == null) {
//			return new Dimension(10,10);
//		} else {
//			Rectangle polyBounds = trackPolygon.getBounds();
//			
//			return new Dimension(polyBounds.width+20, polyBounds.height+20);
//		}
		
//		if(lastSize != null) {
//			//print("returning from branch 1 last size: "+lastSize.toString());
//			
//			return lastSize;
//		} else {
//			//print("returning from branch 2 preferred size: "+lastSize.toString());
//		
//			return preferredSize;
//		}
	}
	
	public void setPreferredSize(Dimension preferredSize) {
		this.preferredSize = preferredSize;
	}
	
	/**
	 * Update the positions of the players on the map according to the new values
	 * @param playerPositions
	 */
	public void updatePlayerPositions(HashMap<NewPlayerResponse, CompCar> newPlayerPositions) {
		/*
		 * Cannot just overwrite the player positions map because there are a maximum of 8
		 * players per MCI, so need to add these to the existing map instead
		 */
		for(NewPlayerResponse player : newPlayerPositions.keySet()) {
			// If the lap on the CompCar is 0 remove this player
			CompCar thisCompCar = newPlayerPositions.get(player);
			
			if(thisCompCar.getLap() == 0) {
				playerPositions.remove(player);
			} else {
				this.playerPositions.put(player, newPlayerPositions.get(player));
			}
		}
	}
	
	/**
	 * Remove the given player from the track map
	 * @param playerID
	 */
	public void removePlayerFromTrackMap(byte playerID) {
		NewPlayerResponse playerToRemove = null;
		
		for(NewPlayerResponse player : playerPositions.keySet()) {
			if(player.getPlayerId() == playerID) {
				playerToRemove = player;
			}
		}
		
		if(playerToRemove != null) {
			playerPositions.remove(playerToRemove);
		}
	}
	
	private void handleMouseClickOnMap(MouseEvent e) {
		Point clickPoint = e.getPoint();
		
		int closestX = Integer.MAX_VALUE, closestY = Integer.MAX_VALUE;
		byte closestPlayerID = Byte.MAX_VALUE;
		
		// Find the player closest to the click
		for(Byte playerID : playersDrawnOnMap.keySet()) {
			Point playerPoint = playersDrawnOnMap.get(playerID);
			
			int diffX = Math.abs(clickPoint.x - playerPoint.x);
			int diffY = Math.abs(clickPoint.y - playerPoint.y);
			
			if(diffX < closestX && diffY < closestY) {
				closestX = diffX;
				closestY = diffY;
				closestPlayerID = playerID;
			}
		}
		
		if(closestPlayerID != Byte.MAX_VALUE) {
			if(closestX < (ovalDiameter*1.5) && closestY < (ovalDiameter*1.5)) {
				/* 
				 * If the click is close enough to a player send an event, pass int 
				 * min value as modifiers to keep the current camera type
				 */
				listener.actionPerformed(new ActionEvent(this, closestPlayerID, 
						LFSQualifyingTickerGUI.VIEW_PLAYER_COMMAND, Integer.MIN_VALUE));
			}
		}
	}
	
	@SuppressWarnings("unused")
	private void print(String msg) {
		LFSQualifyingTickerStart.print(this.getClass().getSimpleName()+" - "+msg);
	}
}
