package lfsqualifyingticker.structures;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;

import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.JButton;
import javax.swing.JLabel;
import javax.swing.SwingConstants;

import lfsqualifyingticker.LFSQualifyingTickerGUI;
import lfsqualifyingticker.LFSQualifyingTickerStart;

import net.sf.jinsim.response.NewPlayerResponse;
import net.sf.jinsim.response.StateResponse;

/**
 * Each PlayerDisplayRow represents one driver in a qualifying session. 
 * Details about the driver and camera controls for this driver
 * are displayed
 * @author Gus
 *
 */

public class PlayerDisplayRow extends Box {
	/**
	 * 
	 */
	private static final long serialVersionUID = 9176632848953566649L;

	public enum DriverStatus {IN_SESSION_ON_HOTLAP, IN_SESSION_NOT_ON_HOTLAP, NOT_IN_SESSION};
	
	private NewPlayerResponse player;
	
	private Box playerNameBox;
	private JLabel inSessionField, onHotlapField;
	
	private JButton driverViewButton, customViewButton, heliViewButton, 
		tvViewButton, chaseViewButton;

	private final ActionListener listener;
	
	private DriverStatus status;
	
	public PlayerDisplayRow(NewPlayerResponse player, ActionListener listener, 
			DriverStatus status) {
		super(BoxLayout.X_AXIS);
		
		this.player = player;
		this.listener = listener;
		this.status = status;
		
		setupGUI();
	}
	
	private void setupGUI() {
		inSessionField = new JLabel();
		onHotlapField = new JLabel();
		
		// Alignment
		inSessionField.setHorizontalAlignment(SwingConstants.CENTER);
		onHotlapField.setHorizontalAlignment(SwingConstants.CENTER);

		// Use the driver's status to fill the session and hotlap fields
		displayDriverStatus();

		playerNameBox = Helper.getColourfulPlayerNameBox(player.getPlayerName());
		
		driverViewButton = new JButton("Driver");
		driverViewButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				requestCameraFocus(StateResponse.DRIVER_VIEW);
			}
		});
		
		customViewButton = new JButton("Custom");
		customViewButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				requestCameraFocus(StateResponse.CUSTOM_VIEW);
			}
		});
		
		heliViewButton = new JButton("Heli");
		heliViewButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				requestCameraFocus(StateResponse.HELICOPTER_VIEW);
			}
		});
		
		tvViewButton = new JButton("TV");
		tvViewButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				requestCameraFocus(StateResponse.TV_VIEW);
			}
		});
		
		chaseViewButton = new JButton("Chase");
		chaseViewButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				requestCameraFocus(StateResponse.ARCADE_VIEW);
			}
		});

		// Set sizes
		Helper.setPrefSizeSizeAndMaxSizeOnComponent(playerNameBox, 
				LFSQualifyingTickerGUI.NAME_LABEL_WIDTH);
		Helper.setPrefSizeSizeAndMaxSizeOnComponent(inSessionField, 
				LFSQualifyingTickerGUI.IN_SESSION_LABEL_WIDTH);
		Helper.setPrefSizeSizeAndMaxSizeOnComponent(onHotlapField, 
				LFSQualifyingTickerGUI.ON_HOTLAP_LABEL_WIDTH);
		
		// Decrease margins
		Insets buttonInsets = new Insets(2,2,2,2);
		
		tvViewButton.setMargin(buttonInsets);
		driverViewButton.setMargin(buttonInsets);
		customViewButton.setMargin(buttonInsets);
		chaseViewButton.setMargin(buttonInsets);
		heliViewButton.setMargin(buttonInsets);

		this.add(Box.createHorizontalStrut(5));
		this.add(playerNameBox);
		this.add(Box.createHorizontalStrut(5));
		this.add(inSessionField);
		this.add(Box.createHorizontalStrut(5));
		this.add(onHotlapField);
		this.add(Box.createHorizontalStrut(10));
		this.add(tvViewButton);
		this.add(Box.createHorizontalStrut(5));
		this.add(driverViewButton);
		this.add(Box.createHorizontalStrut(5));
		this.add(customViewButton);
		this.add(Box.createHorizontalStrut(5));
		this.add(chaseViewButton);
		this.add(Box.createHorizontalStrut(5));
		this.add(heliViewButton);
		this.add(Box.createHorizontalGlue());

		this.setMaximumSize(new Dimension(this.getMaximumSize().width, 
				this.getMinimumSize().height));
	}
	
	/**
	 * Get the player in this row
	 * @return
	 */
	public NewPlayerResponse getPlayer() {
		return player;
	}
	
	/*
	 * Updates the on hotlap display field to reflect whether the player is 
	 * on a hotlap or not
	 */
	private void setPlayerOnHotlap(boolean playerOnHotlap) {
		if(playerOnHotlap) {
			onHotlapField.setText("Yes");
			onHotlapField.setForeground(QualiTickerPrefs.LFSTEXTGREENCOLOR);
		} else {
			onHotlapField.setText("No");
			onHotlapField.setForeground(Color.red);
		}
	}

	/*
	 * Updates the in session display field to reflect whether the player is 
	 * in the session or not
	 */
	private void setPlayerInSession(boolean playerInSession) {
		if(playerInSession) {
			inSessionField.setText("Yes");
			inSessionField.setForeground(QualiTickerPrefs.LFSTEXTGREENCOLOR);
		} else {
			inSessionField.setText("No");
			inSessionField.setForeground(Color.red);
		}
	}
	
	/**
	 * Set the status of the driver in this display row. If necessary this will update
	 * the in session and in hotlap displays
	 * @param status
	 */
	public void setDriverStatus(DriverStatus newStatus) {
		if(!status.equals(newStatus)) {
			// Update the status and the fields
			this.status = newStatus;
			
			displayDriverStatus();
		}
	}
		
	// Update the in session and on hotlap labels to reflect the driver's status
	private void displayDriverStatus() {
		// If else branch is slightly quicker
		if(status.equals(DriverStatus.IN_SESSION_NOT_ON_HOTLAP)) {
			setPlayerInSession(true);
			setPlayerOnHotlap(false);
		} else if(status.equals(DriverStatus.IN_SESSION_ON_HOTLAP)) {
			setPlayerInSession(true);
			setPlayerOnHotlap(true);
		} else if(status.equals(DriverStatus.NOT_IN_SESSION)) {
			setPlayerInSession(false);
			setPlayerOnHotlap(false);
		} else {
			print("displayDriverStatus - unrecognised driver status: "+status.toString());
		}
		
//		switch(status) {
//			case NOT_IN_SESSION: {
//				setPlayerInSession(false);
//				setPlayerOnHotlap(false);
//	
//				break;
//			} case IN_SESSION_NOT_ON_HOTLAP: {
//				setPlayerInSession(true);
//				setPlayerOnHotlap(false);
//	
//				break;
//			} case IN_SESSION_ON_HOTLAP: {
//				setPlayerInSession(true);
//				setPlayerOnHotlap(true);
//	
//				break;
//			} default: {
//				print("undefined driver status: "+status.toString());
//			}
//		}
	}
	
	private void fireActionPerformed(ActionEvent e) {
		listener.actionPerformed(e);
	}
	
	private void requestCameraFocus(byte cameraTypeByte) {
		// Use the modifiers on the ActionEvent to communicate view type
		fireActionPerformed(new ActionEvent(this, player.getPlayerId(), 
				LFSQualifyingTickerGUI.VIEW_PLAYER_COMMAND, cameraTypeByte));
	}
	
	/*
	 * (non-Javadoc)
	 * @see java.lang.Object#equals(java.lang.Object)
	 */
	public boolean equals(Object o) {
		if(o instanceof PlayerDisplayRow) {
			PlayerDisplayRow compRow = (PlayerDisplayRow) o;
			
			if(this.getPlayer() != null && compRow.getPlayer() != null) {
				return this.getPlayer().getPlayerId() == compRow.getPlayer().getPlayerId();
			} else {
				return false;
			}
		} else {
			return false;
		}
	}
	
	private void print(String msg) {
		LFSQualifyingTickerStart.print(this.getClass().getSimpleName()+" - "+msg);
	}
}
