package lfsqualifyingticker.structures;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;

import lfsqualifyingticker.LFSQualifyingTickerStart;
import lfsqualifyingticker.models.TrackModel;

import net.sf.jinsim.Track;
import net.sf.jinsim.response.LapTimeResponse;
import net.sf.jinsim.response.NewConnectionResponse;
import net.sf.jinsim.response.NewPlayerResponse;
import net.sf.jinsim.response.SplitTimeResponse;

public class Lap implements Comparable<Lap> {
	// The connection information for the driver of this lap
	private NewConnectionResponse connection;
	
	// The player information for the driver of this lap
	private NewPlayerResponse player;
	
	// The time at which this lap was started
	private long startTimeNs;
	
	/*
	 * Timing data for each of the CompCar nodes received on this lap. Key
	 * is the node index
	 */
	private HashMap<Integer, ArrayList<CompCarAndTime>> compCarTimes = 
		new HashMap<Integer, ArrayList<CompCarAndTime>>();
	
	// The latest CompCarAndTime received for this lap
	private CompCarAndTime latestCompCarAndTime = null;
	
	// Splits on this lap
	private ArrayList<SplitTimeResponse> splits = new ArrayList<SplitTimeResponse>(4);
	
	// The final laptime for this lap
	private LapTimeResponse laptime;
	
	// The track this lap was done on
	private Track track;
	
	// The first CompCar on this lap
	private CompCarAndTime firstCompCar;
	
	/**
	 * Construct a new Lap with the given player and connection information
	 * @param connection
	 * @param player
	 */
	public Lap(NewConnectionResponse connection, NewPlayerResponse player, 
			long startTimeNs, Track track) {
		this.connection = connection;
		this.player = player;
		this.startTimeNs = startTimeNs;
		this.track = track;
	}
	
	/**
	 * Get the time at which this lap was started
	 * @return
	 */
	public long getStartTimeNs() {
		return startTimeNs;
	}
	
	/**
	 * Add the given TimingNode to this lap
	 * @param node
	 */
	public void addCompCarAndTime(CompCarAndTime compCar) {
		int node = compCar.getCompCar().getNode();
		
		if(firstCompCar == null) {
			// Need to make sure the first comp car is after the finish line
			TrackData trackData = TrackModel.getDataForOneTrack(track);
			
			if(trackData != null) {
				int finishNode = trackData.getFinishNodeIndex();
				int nodeCount = trackData.getNodeCount();
				int thisNode = compCar.getCompCar().getNode();
				
				if(finishNode  < nodeCount) {
					if(thisNode > finishNode) {
						firstCompCar = compCar;
					}
				} else {
					if(thisNode >= 0) {
						firstCompCar = compCar;
					}
				}		
			}
		}
		
		latestCompCarAndTime = compCar;
		
		if(compCarTimes.containsKey(node)) {
			//compCarTimes.get(node).add(compCar);
			// Only store one compcar per node
		} else {
			ArrayList<CompCarAndTime> newList = new ArrayList<CompCarAndTime>(10);
			newList.add(compCar);
			
			compCarTimes.put(node, newList);
		}
	}
	
	/**
	 * Get the elapsed laptime at this node (i.e. the time difference between
	 * the time at this node and the time at the start of the lap) in nanoseconds.
	 * If no time is found for this node null will be returned
	 * @param nodeIndex
	 * @return
	 */
	public Long getElapsedTimeAtNodeIndex(int nodeIndex) {
		if(compCarTimes.containsKey(nodeIndex)) {
			return compCarTimes.get(nodeIndex).get(0).getTime()-startTimeNs;
		} else {
			return null;
		}
	}
	
	/**
	 * Returns the CompCarAndTime packet for the first CompCar received at the given
	 * node or null if no CompCar have been received for the given node index
	 * @param nodeIndex
	 * @return
	 */
	public CompCarAndTime getFirstCompCarPacketForNode(int nodeIndex) {
		if(compCarTimes.containsKey(nodeIndex)) {
			return compCarTimes.get(nodeIndex).get(0);
		} else {
			return null;
		}
	}
	
	/**
	 * Set the final laptime on this Lap
	 * @param laptime
	 */
	public void setLaptime(LapTimeResponse laptime) {
		this.laptime = laptime;
	}
	
	/**
	 * Get the laptime for this lap
	 * @return
	 */
	public LapTimeResponse getLaptime() {
		return laptime;
	}
	
	/**
	 * Get the player from this Lap
	 * @return
	 */
	public NewPlayerResponse getPlayer() {
		return player;
	}
	
	/**
	 * Get the connection from this lap
	 * @return
	 */
	public NewConnectionResponse getConnection() {
		return connection;
	}
	
	/**
	 * Add the given split time to this lap
	 * @param splitTime
	 */
	public void addSplitTime(SplitTimeResponse splitTime) {
		splits.add(splitTime);
	}
	
	/**
	 * Returns the latest CompCarAndTime object received for this lap or null if none 
	 * have been received yet
	 * @return
	 */
	public CompCarAndTime getLatestCompCarAndTime() {
		return latestCompCarAndTime;
	}

	/**
	 * Compares the laptimes of the two given laps
	 * @param o1
	 * @param o2
	 * @return
	 */
	public int compare(Lap o1, Lap o2) {
		if(o1.getLaptime() != null && o2.getLaptime() != null) {
			int o1TimeMillis = o1.getLaptime().getTime().getTime();
			int o2TimeMillis = o2.getLaptime().getTime().getTime();
			
			if(o1TimeMillis < o2TimeMillis) {
				return -1;
			} else if(o1TimeMillis > o2TimeMillis) {
				return 1;
			} else {
				return 0;
			}
		} else {
			return 1;
		}
	}

	/**
	 * Compares the laptimes of this lap against the given lap if available
	 */
	public int compareTo(Lap o2) {
		return compare(this, o2);
	}
	
	/**
	 * Returns the split time at the given split on this lap if present or null 
	 * there are no splits on this lap
	 * @param splitNumber
	 * @return
	 */
	public SplitTimeResponse getSplitNumber(byte splitNumber) {
		if(splits == null || splits.size() == 0) {
			return null;
		} else {
			for(SplitTimeResponse oneSplit : splits) {
				if(oneSplit.getSplit() == splitNumber) {
					return oneSplit;
				}
			}
			
			return null;
		}
	}
	
	/**
	 * Returns a Collection containing all of the splits on this lap or an 
	 * empty Collection if there are no splits attached to 
	 * this lap
	 * @return
	 */
	public Collection<SplitTimeResponse> getAllSplits() {
		return splits;
	}
	
	/**
	 * Returns the last completed split on this lap or null if no splits
	 * have been recorded
	 * @return
	 */
	public SplitTimeResponse getLastCompletedSplit() {
		if(splits == null || splits.size() == 0) {
			return null;
		} else {
			return splits.get(splits.size()-1);
		}
	}
	
	/**
	 * Returns all of the CompCarAndTimes for the given node index if any are
	 * available or null if none are available
	 * @param nodeIndex
	 * @return
	 */
	public ArrayList<CompCarAndTime> getAllCompCarAndTimesForNode(int nodeIndex) {
		if(compCarTimes.containsKey(nodeIndex)) {
			return compCarTimes.get(nodeIndex);
		} else {
			return null;
		}
	}
	
	/**
	 * Get the track this lap was done on
	 * @return
	 */
	public Track getTrack() {
		return track;
	}
	
	/**
	 * Get the first CompCarAndTime received for this lap. May return null if no
	 * comp cars have been received yet
	 * @return
	 */
	public CompCarAndTime getFirstCompCarInLap() {
		return firstCompCar;
	}
	
	@SuppressWarnings("unused")
	private void print(String msg) {
		LFSQualifyingTickerStart.print(this.getClass().getSimpleName()+" - "+msg);
	}
}
