package lfsqualifyingticker.models;

import java.io.File;
import java.io.FileInputStream;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.channels.FileChannel;
import java.nio.channels.FileChannel.MapMode;
import java.util.Collection;
import java.util.HashMap;

import lfsqualifyingticker.LFSQualifyingTickerStart;
import lfsqualifyingticker.structures.OneNode;
import lfsqualifyingticker.structures.QualiTickerPrefs;
import lfsqualifyingticker.structures.TrackData;

import net.sf.jinsim.Track;

/**
 * Model containing data on all the tracks in LFS
 * @author Gus
 *
 */

public class TrackModel {
	private static HashMap<Track, TrackData> tracks = new HashMap<Track, TrackData>();

	private static File smxFolder;
	
	/**
	 * Load detailed track data for all PTH files
	 * @throws Exception
	 */
	public static void loadAllTrackData() throws Exception {
		// Check the location of the LFS folder is valud
		smxFolder = new File(QualiTickerPrefs.getLFSDirectoryPath()+"/data/smx");
		
		if(!smxFolder.exists()) {
			// LFS folder doesn't exist
			throw new RuntimeException("Cannot locate LFS directory at:\n"+
					QualiTickerPrefs.getLFSDirectoryPath()+"\nPlease correct this in " +
							"the config.cfg file and re-run the application.");
		}
		
		Track[] allTracks = Track.values();
		
		for(Track oneTrack : allTracks) {
			// No PTH file for BL3, AU1, AU2
			if(!oneTrack.equals(Track.BLACKWOOD_CAR_PARK) && 
					!oneTrack.equals(Track.AUTOCROSS) &&
					!oneTrack.equals(Track.SKIP_PAD)) {
				
				TrackData trackData = loadOneTrackData(oneTrack);

				if(trackData == null) {
					throw new RuntimeException("Error loading track data for "+
							oneTrack.getShortname());
				} else {
					// Calculate some values which will be required later
					trackData.performInitialCalculations();
					
					// Add this to map
					tracks.put(oneTrack, trackData);
				}
			}
		}
	}
	
	/*
	 * Load detailed information for the given track. Any problems loading data
	 * (e.g. missing file, corrupt file data) will cause an Exception to be
	 * thrown from this method to its caller (loadAllTrackData) and from there
	 * back to the caller of that method (which should be the main class). 
	 * Better to flag up invalid/missing data immediately than to handle it
	 * silently and continue executing with missing or incomplete data 
	 * (fail fast)
	 */
	private static TrackData loadOneTrackData(Track track) throws Exception {
		String trackCode = track.getShortname();
		
		// Construct the filename for this track's PTH file
		String filename = trackCode+".pth";
		
		// Open up a channel to read the file from disk
		FileChannel fc = new FileInputStream(smxFolder.getAbsolutePath()+
				File.separatorChar+filename).getChannel();
		ByteBuffer buff = fc.map(MapMode.READ_ONLY, 0, fc.size());
		
		// Set read ordering to little endian
		buff.order(ByteOrder.LITTLE_ENDIAN);

		// The total node count and finish node index for this track
		int nodeCount, finishNode;

		// Check for correct file version
		byte[] typeBytes = new byte[6];
		
		buff.get(typeBytes, 0, 6);
		
		String fileType = new String(typeBytes);

		if (!fileType.equals("LFSPTH")) {
			print("Invalid FileType map for track "+track.getName()+
					" ("+track.getShortname()+")");
			return null;
		} 

		if (buff.get() > 0) {
			print("Invalid Version for track "+track.getName()+" ("+
					track.getShortname()+")");
			return null;
		}

		if (buff.get() > 0) {
			print("Invalid Revision for track "+track.getName()+" ("+
					track.getShortname()+")");
			return null;
		}
		// End checking for correct file version

		/* 
		 * Now that pth file information is validated create the TrackData 
		 * object with the given Track information. The Track object 
		 * contains details like track name, track code, number of splits
		 */
		TrackData newTrack = new TrackData(track);
		
		// Read the node count and finish node index
		nodeCount = buff.getInt();
		finishNode = buff.getInt();

		/* 
		 * Set the node for the finish line of this track. The finish node index
		 * does not have to be (and hardly ever is) 0 or the max number of nodes,
		 * it's usually an arbitrary number
		 */
		newTrack.setFinishNodeIndex(finishNode);

		// Read all node data up to the node count
		for(int nodeIndex=0; nodeIndex<nodeCount; nodeIndex++) {
			int centreX = buff.getInt();
			int centreY = buff.getInt();
			int centreZ = buff.getInt();
			
			float directionX = buff.getFloat();
			float directionY = buff.getFloat();
			float directionZ = buff.getFloat();

			float limitLeft = buff.getFloat();
			float limitRight = buff.getFloat();

			float driveLeft = buff.getFloat();
			float driveRight = buff.getFloat();

			// Create a new node and add it to this track data
			newTrack.addOneNode(new OneNode(nodeIndex, centreX, centreY, centreZ, 
					directionX, directionY, directionZ, limitLeft, limitRight, 
					driveLeft, driveRight));
		}

		return newTrack;
	}
	
	/**
	 * Returns the TrackData information for the given track or null
	 * if there is no information found
	 * @param track
	 * @return
	 */
	public static TrackData getDataForOneTrack(Track track) {
		if(tracks.containsKey(track)) {
			return tracks.get(track);
		} else {
			return null;
		}
	}
	
	/**
	 * Returns TrackData for all of the tracks
	 * @return
	 */
	public static Collection<TrackData> getAllTrackData() {
		return tracks.values();
	}
	
	private static void print(String msg) {
		LFSQualifyingTickerStart.print("TrackModel - "+msg);
	}
}
