package lfsqualifyingticker;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.Toolkit;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.Arrays;
import java.util.Collection;
import java.util.Comparator;
import java.util.HashMap;

import javax.swing.Box;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JScrollPane;
import javax.swing.JSplitPane;
import javax.swing.SwingConstants;

import net.sf.jinsim.response.NewPlayerResponse;
import net.sf.jinsim.response.RaceStartResponse;
import net.sf.jinsim.response.SplitTimeResponse;
import net.sf.jinsim.types.CompCar;
import lfsqualifyingticker.models.TimingModel;
import lfsqualifyingticker.structures.Lap;
import lfsqualifyingticker.structures.PlayerDisplayRow;
import lfsqualifyingticker.structures.Helper;
import lfsqualifyingticker.structures.QualiTickerPrefs;
import lfsqualifyingticker.structures.TimingModelListener;
import lfsqualifyingticker.structures.TrackCanvas;
import lfsqualifyingticker.structures.TrackMap;
import lfsqualifyingticker.structures.PlayerDisplayRow.DriverStatus;

/**
 * The GUI for the LFS Qualifying Ticker application. Listens to the timing model
 * for important events
 * @author Gus
 *
 */

public class LFSQualifyingTickerGUI extends JFrame implements TimingModelListener, ActionListener {
	/**
	 * 
	 */
	private static final long serialVersionUID = 7937271955242460031L;

	// Absolute widths for labels (not flexible but time constraints...)
	public static final int NAME_LABEL_WIDTH = 145;
	public static final int IN_SESSION_LABEL_WIDTH = 70;
	public static final int ON_HOTLAP_LABEL_WIDTH = 70;
	
	public static final String VIEW_PLAYER_COMMAND = "ViewPlayer";
	
	public static final Color BACKGROUND_COLOUR = new Color(200, 200, 200);

	private final TimingModel timingModel;
	
	private Box mainBox, playersBox;
	
	private JSplitPane splitPane;

	private TrackMap trackMap;
	
	private ActionListener listener;
	
	private JLabel nameLabel, inSessionLabel, onHotlapLabel, fastestOverallLapField;
	
	private HashMap<Byte, PlayerDisplayRow> displayRows = new HashMap<Byte, PlayerDisplayRow>(40);
	
	/**
	 * Create a new GUI with the given timing model and action listener
	 * @param timingModel
	 * @param listener
	 */
	public LFSQualifyingTickerGUI(TimingModel timingModel, ActionListener listener) {
		this.timingModel = timingModel;
		
		timingModel.addTimingModelListener(this);
		
		this.listener = listener;
		
		setupGUI();
	}
	
	private void setupGUI() {
		this.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
		
		this.setLayout(new BorderLayout());

		Box headerLabelBox = Box.createHorizontalBox();

		nameLabel = new JLabel("Name");
		inSessionLabel = new JLabel("In Session");
		onHotlapLabel = new JLabel("On Hotlap");

		// Sizes
		Helper.setPrefSizeSizeAndMaxSizeOnComponent(nameLabel, NAME_LABEL_WIDTH);
		Helper.setPrefSizeSizeAndMaxSizeOnComponent(inSessionLabel, IN_SESSION_LABEL_WIDTH);
		Helper.setPrefSizeSizeAndMaxSizeOnComponent(onHotlapLabel, ON_HOTLAP_LABEL_WIDTH);
		
		// Alignment
		nameLabel.setHorizontalAlignment(SwingConstants.CENTER);
		inSessionLabel.setHorizontalAlignment(SwingConstants.CENTER);
		onHotlapLabel.setHorizontalAlignment(SwingConstants.CENTER);

		headerLabelBox.add(Box.createHorizontalStrut(5));
		headerLabelBox.add(nameLabel);
		headerLabelBox.add(Box.createHorizontalStrut(5));
		headerLabelBox.add(inSessionLabel);
		headerLabelBox.add(Box.createHorizontalStrut(5));
		headerLabelBox.add(onHotlapLabel);
		headerLabelBox.add(Box.createHorizontalStrut(5));
		headerLabelBox.add(Box.createHorizontalGlue());
		
		playersBox = Box.createVerticalBox();

		mainBox = Box.createVerticalBox();
		mainBox.add(headerLabelBox);
		mainBox.add(Box.createVerticalStrut(2));
		mainBox.add(playersBox);
		mainBox.add(Box.createVerticalGlue());
		
		mainBox.setAlignmentX(0);
		
		JScrollPane scrollPane = new JScrollPane();
		scrollPane.getViewport().add(mainBox);
		
		splitPane = new JSplitPane(JSplitPane.HORIZONTAL_SPLIT);
		
		splitPane.setLeftComponent(scrollPane);

		splitPane.setDividerLocation(mainBox.getPreferredSize().width);
		
		// If the user moves the split pane divider repaint the screen
		splitPane.addPropertyChangeListener(new PropertyChangeListener() {
			public void propertyChange(PropertyChangeEvent evt) {
				if(evt.getPropertyName().equals(JSplitPane.DIVIDER_LOCATION_PROPERTY)) {
					splitPaneDividerLocationChanged(evt);
				}
			}
		});
		
		// Track map gets additional size
		splitPane.setResizeWeight(0);
		
		// Initially set the split pane divider really small, but expand if necessary
		splitPane.setDividerSize(0);
		
		Box fastestLapBox = Box.createHorizontalBox();
		
		JLabel fastestOverallLapLabel = new JLabel("Fastest Lap In Session: ");
		fastestOverallLapField = new JLabel("No Laps Completed");

		fastestLapBox.add(Box.createHorizontalStrut(5));
		fastestLapBox.add(fastestOverallLapLabel);
		fastestLapBox.add(Box.createHorizontalStrut(5));
		fastestLapBox.add(fastestOverallLapField);
		fastestLapBox.add(Box.createHorizontalGlue());

		this.add(splitPane, BorderLayout.CENTER);
		this.add(fastestLapBox, BorderLayout.SOUTH);

		this.setTitle(QualiTickerPrefs.getApplicationName()+" "+
				QualiTickerPrefs.getVersion());
		
		this.setSize(600, 800);
		
		// Centre on screen
		Dimension screenResolution = Toolkit.getDefaultToolkit().getScreenSize();
		
		int frameXCo = (int) ((screenResolution.getWidth()/2)-(this.getSize().getWidth()/2));
		int frameYCo = (int) ((screenResolution.getHeight()/2)-(this.getSize().getHeight()/2));
		this.setLocation(frameXCo, frameYCo);

		headerLabelBox.setOpaque(true);
		headerLabelBox.setBackground(BACKGROUND_COLOUR);
		playersBox.setOpaque(true);
		playersBox.setBackground(BACKGROUND_COLOUR);
		mainBox.setOpaque(true);
		mainBox.setBackground(BACKGROUND_COLOUR);
		this.setBackground(BACKGROUND_COLOUR);
		fastestLapBox.setOpaque(true);
		fastestLapBox.setBackground(BACKGROUND_COLOUR);
		fastestOverallLapLabel.setOpaque(true);
		fastestOverallLapLabel.setBackground(BACKGROUND_COLOUR);
		fastestOverallLapField.setOpaque(true);
		fastestOverallLapField.setBackground(BACKGROUND_COLOUR);
		splitPane.setOpaque(true);
		splitPane.setBackground(Color.white);
		
		headerLabelBox.setMaximumSize(new Dimension(headerLabelBox.getMaximumSize().width, 
				headerLabelBox.getMinimumSize().height));

		this.setVisible(true);
	}

	private synchronized void updateDisplay() {
		playersBox.removeAll();

		Collection<NewPlayerResponse> currentPlayersCollection = timingModel.getAllCurrentPlayers();

		NewPlayerResponse[] currentPlayersArray = currentPlayersCollection.toArray(
				new NewPlayerResponse[currentPlayersCollection.size()]);

		// Just compare the players by player name
		Arrays.sort(currentPlayersArray, new Comparator<NewPlayerResponse>() {
			public int compare(NewPlayerResponse o1, NewPlayerResponse o2) {
				String o1Name = o1.getPlayerName() != null ? 
						Helper.getPlainPlayerName(o1.getPlayerName()) : "";
				String o2Name = o2.getPlayerName() != null ? 
						Helper.getPlainPlayerName(o2.getPlayerName()) : "";
						
				return o1Name.compareTo(o2Name);
			}
		});

		for(NewPlayerResponse player : currentPlayersArray) {
			//print("updateDisplay, playerName: "+player.getPlayerName()+", ID: "+player.getPlayerId());
			
			// Determine if this player is on a hotlap or not
			byte playerID = player.getPlayerId();
			
			DriverStatus driverStatus = timingModel.getDriverStatus(playerID);
			
			// If there's an existing display row just add it back in, otherwise create one
			if(displayRows.containsKey(playerID)) {
				PlayerDisplayRow displayRow = displayRows.get(playerID);
				
				displayRow.setDriverStatus(driverStatus);
				
				playersBox.add(displayRow);
				playersBox.add(Box.createVerticalStrut(2));
			} else {
				PlayerDisplayRow displayRow = new PlayerDisplayRow(player, 
						this, driverStatus);

				playersBox.add(displayRow);
				playersBox.add(Box.createVerticalStrut(2));

				displayRows.put(playerID, displayRow);
			}
		}

		playersBox.add(Box.createVerticalGlue());

		//splitPane.setDividerLocation(mainBox.getPreferredSize().width+10);

		// this.setMinimumSize(mainBox.getMinimumSize());
		
		playersBox.revalidate();
		playersBox.repaint();
	}
	
	private void splitPaneDividerLocationChanged(PropertyChangeEvent evt) {
		if(trackMap != null) {
			trackMap.updateTrackSizeBasedOnSplitPane(evt);
		
			requestRepaint();
		}
	}
	
	private void requestRepaint() {
		this.repaint();
	}

	private void print(String msg) {
		LFSQualifyingTickerStart.print(this.getClass().getSimpleName()+" - "+msg);
	}
	
	public void raceStartResponseReceived(RaceStartResponse raceStartResponse) {
		removeStateFromPreviousSession();
		
//		// Draw the new track map
//		TrackData trackData = TrackModel.getDataForOneTrack(raceStartResponse.getTrack());
//		
//		if(trackData != null) {
//			trackMap = new TrackMap(trackData, this);
//
//			splitPane.setRightComponent(trackMap);
//			
//			splitPane.setDividerSize(5);
//			
//			this.repaint();
//		} else {
//			trackMap = null;
//			
//			splitPane.setDividerSize(1);
//		}
	}

	private void removeStateFromPreviousSession() {
		fastestOverallLapField.setText("No Laps Completed");
		
		playersBox.removeAll();
		displayRows.clear();
		
		trackMap = null;

		updateDisplay();
	}
	
	private void updateFastestLapInformation(Lap fastestLap) {
		if(fastestLap != null && fastestLap.getLaptime() != null) {
			StringBuilder fastestLapString = new StringBuilder();
			
			if(fastestLap.getPlayer() != null) {
				fastestLapString.append(Helper.getPlainPlayerName(
						fastestLap.getPlayer().getPlayerName())+" - ");
			}

			fastestLapString.append(Helper.getTimeString(fastestLap.getLaptime()));
			
			Collection<SplitTimeResponse> splits = fastestLap.getAllSplits();
			
			if(splits != null && splits.size() > 0) {
				SplitTimeResponse[] splitsArray = splits.toArray(
						new SplitTimeResponse[splits.size()]);
				
				fastestLapString.append(" (");
				
				for(int i=0; i<splitsArray.length; i++) {
					if(i < splits.size()-1) {
						fastestLapString.append("Split "+splitsArray[i].getSplit()+": "+
								Helper.getTimeString(splitsArray[i].getTime())+", ");
					} else {
						fastestLapString.append("Split "+splitsArray[i].getSplit()+": "+
								Helper.getTimeString(splitsArray[i].getTime()));
					}
				}
				
				fastestLapString.append(") ");
				
				fastestOverallLapField.setText(fastestLapString.toString());
			}
		} else {
			fastestOverallLapField.setText("");
		}
	}
	
	private void removePlayerFromTrackMap(byte playerID) {
		if(trackMap != null) {
			trackMap.removePlayerFromTrackMap(playerID);
		}
	}

	/*
	 * (non-Javadoc)
	 * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
	 */
	public void actionPerformed(ActionEvent e) {
		if(e.getSource() instanceof PlayerDisplayRow) {
			// Pass on events from the display rows to the main class
			listener.actionPerformed(e);
		} else if(e.getSource() instanceof TrackCanvas) {
			// Pass on events from the track map
			listener.actionPerformed(e);
		} else {
			print("actionPerformed - unrecognised source class: "+
					e.getSource().getClass().getSimpleName());
		}
	}
	
	public Dimension getMinimumSize() {
		if(displayRows != null && displayRows.size() > 0) {
			for(PlayerDisplayRow displayRow : displayRows.values()) {
				Dimension displayRowMinSize = displayRow.getMinimumSize();
				Dimension mainBoxMinSize = mainBox.getMinimumSize();
				
				//print("getMinimumSize - returning from (1): "+new Dimension(
				//		displayRowMinSize.width, mainBoxMinSize.height).toString());
				
				return new Dimension(displayRowMinSize.width, mainBoxMinSize.height);
			}
			
			// Won't be executed
			return null;
		} else {
			Dimension mainBoxMinSize = mainBox.getMinimumSize();

			//print("getMinimumSize - returning from (1): "+new Dimension(
			//		mainBoxMinSize.width+20, mainBoxMinSize.height+10).toString());
			
			return new Dimension(mainBoxMinSize.width+20, mainBoxMinSize.height+10);
		}
	}
	
	// Start of TimingModelListener methods
	/*
	 * (non-Javadoc)
	 * @see lfsqualifyingticker.structures.TimingModelListener#
	 * playerJoinedSession(net.sf.jinsim.response.NewPlayerResponse)
	 */
	public void playerJoinedSession(byte playerIDJoining) {
		if(displayRows.containsKey(playerIDJoining)) {
			DriverStatus status = timingModel.getDriverStatus(playerIDJoining);
			
			displayRows.get(playerIDJoining).setDriverStatus(status);
		} else {
			// If the display row wasn't found update the full display
			updateDisplay();
		}
	}

	/*
	 * (non-Javadoc)
	 * @see lfsqualifyingticker.structures.TimingModelListener#
	 * playerLeftSession(net.sf.jinsim.response.PlayerLeavingResponse)
	 */
	public void playerLeftSession(byte playerIDLeaving) {
		if(displayRows.containsKey(playerIDLeaving)) {
			displayRows.remove(playerIDLeaving);
		}
			
		removePlayerFromTrackMap(playerIDLeaving);
		
		updateDisplay();
	}

	/*
	 * (non-Javadoc)
	 * @see lfsqualifyingticker.structures.TimingModelListener#
	 * playerPitted(net.sf.jinsim.response.PlayerPitsResponse)
	 */
	public void playerPitted(byte playerIDPitting) {
		if(displayRows.containsKey(playerIDPitting)) {
			displayRows.remove(playerIDPitting);
		}
			
		removePlayerFromTrackMap(playerIDPitting);
			
		updateDisplay();
	}

	/*
	 * (non-Javadoc)
	 * @see lfsqualifyingticker.structures.TimingModelListener#
	 * playerStartedHotlap(net.sf.jinsim.response.NewPlayerResponse)
	 */
	public void playerStartedHotlap(byte playerIDStartedHotlap) {
		if(displayRows.containsKey(playerIDStartedHotlap)) {
			DriverStatus status = timingModel.getDriverStatus(playerIDStartedHotlap);
			
			displayRows.get(playerIDStartedHotlap).setDriverStatus(status);
		} else {
			updateDisplay();
		}
	}
	
	/*
	 * (non-Javadoc)
	 * @see lfsqualifyingticker.structures.TimingModelListener#
	 * playerCompletedLap(byte)
	 */
	public void playerCompletedLap(byte playerID) {
		// Do nothing when a lap is completed, not displaying position any more
	}
	
	/*
	 * (non-Javadoc)
	 * @see lfsqualifyingticker.structures.TimingModelListener#
	 * fastestLapUpdated(lfsqualifyingticker.structures.Lap)
	 */
	public void fastestLapUpdated(Lap fastestLap) {
		updateFastestLapInformation(fastestLap);
	}

	/*
	 * (non-Javadoc)
	 * @see lfsqualifyingticker.structures.TimingModelListener#
	 * updatePlayerPositionNodes(java.util.HashMap)
	 */
	public void updatePlayerPositionNodes(
			HashMap<NewPlayerResponse, CompCar> playerPositions) {
		if(trackMap != null) {
			trackMap.updatePlayerPositionNodes(playerPositions);
		}
	}
	// End of TimingModelListener methods
}
