package lfsqualifyingticker;

import java.io.IOException;
import java.math.BigDecimal;
import java.util.ArrayList;

import lfsqualifyingticker.structures.LFSDisplayMessage;
import lfsqualifyingticker.structures.Helper;
import lfsqualifyingticker.structures.QualiTickerPrefs;

import net.sf.jinsim.Client;
import net.sf.jinsim.request.ButtonRequest;
import net.sf.jinsim.response.SplitTimeResponse;

/**
 * This class is responsible for sending the request to LFS to display the gap information
 * @author Gus
 *
 */

public class LFSDisplayUpdater {
	// The InSim client to send button requests to 
	private Client client;
	
	private final byte BUTTON_HEIGHT = 20, BUTTON_WIDTH = 60;

	private int numButtons = 1;

	/**
	 * Construct a new display updater with the specified client
	 * @param client
	 */
	public LFSDisplayUpdater(Client client) {
		this.client = client;
	}
	
	/**
	 * Update the LFS display with the given message
	 * @param msg
	 * @throws IOException
	 */
	public void updateLFSDisplay(LFSDisplayMessage msg) throws IOException {
		numButtons = 1;
		
		ArrayList<ButtonRequest> requestsToSend = new ArrayList<ButtonRequest>(2);

		// Use the message type to determine what to show to LFS (if anything)
		switch(msg.getMessageType()) {
			case NO_DISPLAY_MESSAGE: {
				// No message to display, clear any buttons created
				// Can't see a way to clear the buttons with JInSim 0.5 rc6
				ButtonRequest button1 = createNewEmptyButton();
				button1.setText("");

				requestsToSend.add(button1);
				
				break;
			} case NO_PLAYER_VIEWED: {
				ButtonRequest button1 = createNewEmptyButton();
				button1.setText(QualiTickerPrefs.getNoPlayerViewedString());
				
				requestsToSend.add(button1);
				
				break;
			} case NO_LAPS_COMPLETED_MESSAGE: {
				ButtonRequest button1 = createNewEmptyButton();
				button1.setText(QualiTickerPrefs.getNoLapsCompleted());
				
				requestsToSend.add(button1);

				break;
			} case PLAYER_NOT_ON_HOTLAP: {
				ButtonRequest button1 = createNewEmptyButton();
				button1.setText(QualiTickerPrefs.getPlayerNotOnHotlapString());
				
				requestsToSend.add(button1);
				
				break;
			} case TIME_DIFFERENCE_ONLY_MESSAGE: {
				Long gapNS = msg.getGapNS();
				
				if(gapNS != null) {
					BigDecimal secondsDifference = new BigDecimal(msg.getGapNS() / 1000000000.0);
					secondsDifference = Helper.roundBigDecimalTo2DP(secondsDifference);
					
					StringBuilder buttonString = new StringBuilder();
					
					ButtonRequest button1 = createNewEmptyButton();
					
					if(secondsDifference.doubleValue() > 0) {
						buttonString.append("+"+secondsDifference.toPlainString());
						button1.setButtonStyle(ButtonRequest.BUTTON_STYLE_RED);
					} else {
						buttonString.append(secondsDifference.toPlainString());
						button1.setButtonStyle(ButtonRequest.BUTTON_STYLE_GREEN);
					}
					
					if(msg.getCurrentLap() != null || msg.getFastestLap() != null) {
						// Now add in the gap at the last split if available
						SplitTimeResponse lastSplitCurrentLap = msg.getCurrentLap().
							getLastCompletedSplit();
						
						if(lastSplitCurrentLap != null) {
							SplitTimeResponse sameSplitFastestLap = msg.getFastestLap().
							getSplitNumber(lastSplitCurrentLap.getSplit());
							
							if(sameSplitFastestLap != null) {
								// Calculate difference between splits
								BigDecimal secondsDifferenceAtSplit = new BigDecimal(
										(lastSplitCurrentLap.getTime().getTime()-
										sameSplitFastestLap.getTime().getTime())/1000.0);
								secondsDifferenceAtSplit = Helper.roundBigDecimalTo2DP(
										secondsDifferenceAtSplit);
								
								// Append this to the string for the button
								if(secondsDifferenceAtSplit.doubleValue() > 0) {
									buttonString.append(" - Split "+lastSplitCurrentLap.getSplit()+
											": +"+secondsDifferenceAtSplit.toPlainString());
								} else {
									buttonString.append(" - Split "+lastSplitCurrentLap.getSplit()+
											": "+secondsDifferenceAtSplit.toPlainString());
								}
								
								// Expand the button width
								button1.setWidth((byte) (BUTTON_WIDTH*2));
							}
						}
					}

					button1.setText(buttonString.toString());
					
					requestsToSend.add(button1);
				} else {
					// No valid nodes to use for comparison
					ButtonRequest button1 = createNewEmptyButton();
					
					button1.setText("");
					
					requestsToSend.add(button1);
				}

				break;
			} case TIME_DIFFERENCE_AND_POSITION_MESSAGE: {
				Long gapNS = msg.getGapNS();
				
				if(gapNS != null) {
				
					BigDecimal secondsDifference = new BigDecimal(msg.getGapNS() / 1000000000.0);
					secondsDifference = Helper.roundBigDecimalTo2DP(secondsDifference);
					
					ButtonRequest button1 = createNewEmptyButton();
					
					StringBuilder buttonString = new StringBuilder();
					
					if(secondsDifference.doubleValue() > 0) {
						buttonString.append("+"+secondsDifference.toPlainString()+" ("+
								Helper.getPositionStringForInt(
										msg.getPredictedPosition())+")");
						button1.setButtonStyle(ButtonRequest.BUTTON_STYLE_RED);
					} else {
						buttonString.append(secondsDifference.toPlainString()+" ("+
								Helper.getPositionStringForInt(
										msg.getPredictedPosition())+")");
						button1.setButtonStyle(ButtonRequest.BUTTON_STYLE_GREEN);
					}
					
					if(msg.getCurrentLap() != null || msg.getFastestLap() != null) {
						// Now add in the gap at the last split if available
						SplitTimeResponse lastSplitCurrentLap = msg.getCurrentLap().
							getLastCompletedSplit();
						
						if(lastSplitCurrentLap != null) {
							SplitTimeResponse sameSplitFastestLap = msg.getFastestLap().
							getSplitNumber(lastSplitCurrentLap.getSplit());
							
							if(sameSplitFastestLap != null) {
								// Calculate difference between splits
								BigDecimal secondsDifferenceAtSplit = new BigDecimal(
										(lastSplitCurrentLap.getTime().getTime()-
										sameSplitFastestLap.getTime().getTime())/1000.0);
								secondsDifferenceAtSplit = Helper.roundBigDecimalTo2DP(secondsDifferenceAtSplit);
								
								// Append this to the string for the button
								if(secondsDifferenceAtSplit.doubleValue() > 0) {
									buttonString.append(" - Split "+lastSplitCurrentLap.getSplit()+
											": +"+secondsDifferenceAtSplit.toPlainString());
								} else {
									buttonString.append(" - Split "+lastSplitCurrentLap.getSplit()+
											": "+secondsDifferenceAtSplit.toPlainString());
								}
								
								// Expand the button width
								button1.setWidth((byte) (BUTTON_WIDTH*2));
							}
						}
					}
					
					button1.setText(buttonString.toString());
					
					requestsToSend.add(button1);
				} else {
					// No valid nodes to use for comparison
					ButtonRequest button1 = createNewEmptyButton();
					
					button1.setText("");
					
					requestsToSend.add(button1);
				}
				
				break;
			} default: {
				print("Undefined message type: "+msg.getMessageType());
			}
		}
		
		// Use the number of buttons to determine where to place on screen
		switch(requestsToSend.size()) {
			case 0: {
				// Nothing
				break;
			} case 1: {
				ButtonRequest firstButton = requestsToSend.get(0);
				
				firstButton.setTop((byte) 150);
				
				break;
			} case 2: {
				ButtonRequest firstButton = requestsToSend.get(0);
				
				firstButton.setTop((byte) 50);
				
				ButtonRequest secondButton = requestsToSend.get(1);
				
				secondButton.setTop((byte) (100));
				
				break;
			} default: {
				print("updateLFSDisplay - default case for button requests size: "+
						requestsToSend.size());
				
				break;
			}
		}
		
		// Send the button requests created above
		if(requestsToSend.size() > 0) {
			for(ButtonRequest req : requestsToSend) {
				client.send(req);
			}
		}
	}
	
	// Little helper method to get rid of some of the boilerplate button code
	private ButtonRequest createNewEmptyButton() {
		ButtonRequest newRequest = new ButtonRequest();
		
		newRequest.setConnectionId((byte) 0);
		newRequest.setRequestInfo((byte) numButtons);
		newRequest.setWidth(BUTTON_WIDTH);
		newRequest.setHeight(BUTTON_HEIGHT);
		newRequest.setLeft((byte) 40);
		
		numButtons++;
		
		return newRequest;
	}
	
	private void print(String msg) {
		LFSQualifyingTickerStart.print(this.getClass().getSimpleName()+
				" - "+msg);
	}
}
