package lfscustomreslauncher;

import java.awt.DisplayMode;
import java.awt.GraphicsDevice;
import java.awt.GraphicsEnvironment;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.IOException;
import java.io.PrintWriter;

import javax.swing.JOptionPane;

/**
 * This is a little application that allows a user to specify some startup parameters to control
 * the resolution, refresh rate, colour depth and windowed mode of Live For Speed. For more
 * details see the following thread on LFS Forum and the attached readme.txt file.
 * 
 * http://www.lfsforum.net/showthread.php?t=59245
 * 
 * @author Gus
 *
 */

public class LFSCustomResLauncherV1 {
	private int horizontalRes, verticalRes, refreshRate, colourDepth=Integer.MAX_VALUE;
	private Boolean windowed;
	
	// The command line arguments specified
	private final String[] args;
	
	// The name of the config file
	private final String CONFIG_FILENAME = "cfg.txt";
	
	// Identifiers for the relevant lines in the config file
	private final String SCREEN_MODE_LINE_IDENTIFIER = "Screen Info", WINDOW_MODE_LINE_IDENTIFIER = "Start Windowed";
	
	private final boolean debug = false;
	
	private LFSCustomResLauncherV1(String[] args) {
		this.args = args;
		
		if(debug) {
			print(this.getClass().getSimpleName()+" started with following arguments");
			
			if(args.length > 0) {
				for(String arg : args) {
					print(arg);
				}
			}
		}
		
		checkIfApplicationInLFSDirectory();
		
		// Parse the input arguments, make sure they're valid
		parseArguments();
		
		checkIfDisplayModeIsAllowed();
		
		outputDetailsToCFGFile();
		
		runLFS();
	}
	
	private void checkIfApplicationInLFSDirectory() {
		/*
		 * At the moment the application must be placed in the same directory as the LFS.exe file
		 * for the installation of LFS the settings apply for. If the LFS.exe file can't be found
		 * inform the user of this restriction and exit
		 */
		
		File lfsExecutableFile = new File("LFS.exe");
		
		if(!lfsExecutableFile.exists()) {
			if(debug) {
				print("Cannot find the LFS.exe file in the current directory: "+lfsExecutableFile.getAbsolutePath());
			}
			
			JOptionPane.showMessageDialog(null, 
					"The LFSCustomResLauncher jar file must be placed in the same folder as your LFS.exe file.\n" +
					"LFS.exe cannot be found at the current location of:\n" +
					lfsExecutableFile.getAbsolutePath(), "Cannot Find LFS.exe File", JOptionPane.ERROR_MESSAGE);
			
			System.exit(1);
		}
	}
	
	private void parseArguments() {
		if(debug) {
			print("parseArguments called with args.length: "+args.length);
		}
		
		// None or not enough arguments specified, inform user and exit
		if(args.length == 0 || args.length < 3) {
			displayUsageInformationToUser();
			
			System.exit(1);
		} else {
			// Attempt to parse first 3 arguments as the Horizontal res, vertical res and refresh rate
			horizontalRes = parseIntegerFromString(args[0]);
			verticalRes = parseIntegerFromString(args[1]);
			refreshRate = parseIntegerFromString(args[2]);

			if(debug) {
				print("Successfully parse the following arguments");
				print("Resolution: "+horizontalRes+"x"+verticalRes);
				print("Refresh rate: "+refreshRate+" Hz");
			}
			
			// Now check for additional arguments (colour depth or windowed mode)
			if(args.length == 4) {
				setAdditionalParameterFromString(args[3]);
			} else if(args.length == 5) {
				setAdditionalParameterFromString(args[3]);
				setAdditionalParameterFromString(args[4]);
			}
		}
	}
	
	private void checkIfDisplayModeIsAllowed() {
		if(debug) {
			print("checkIfResAndRefreshAreAllowed called");
			print("Resolution: "+horizontalRes+"x"+verticalRes);
			print("Refresh rate: "+refreshRate+" Hz");
		}

		GraphicsDevice graphicsDevice = GraphicsEnvironment.getLocalGraphicsEnvironment().getDefaultScreenDevice();

		DisplayMode[] allModes = graphicsDevice.getDisplayModes();
		
		/*
		 * Used to identify if the display mode is supported. There's room for doubt if the colour depth is not
		 * provided as a command line parameter because we're going to use the current one from the cfg.txt file
		 * in this case and I don't want to check the file at this point. 
		 */
		final String DISPLAY_MODE_SUPPORTED = "ModeSupported", 
			DISPLAY_MODE_ONE_COLOUR_DEPTH_SUPPORTED = "OneColourDepthSupported",
			DISPLAY_MODE_NOT_SUPPORTED = "DisplayModeNotSupported";
		
		String displayModeSupported = DISPLAY_MODE_NOT_SUPPORTED;
		
		for(DisplayMode mode : allModes) {
			if(debug) {
				print("Supported display mode: "+mode.getWidth()+"x"+mode.getHeight()+", "+
						mode.getRefreshRate()+"Hz, "+mode.getBitDepth()+" bit");
			}
			
			if(mode.getWidth() == horizontalRes && mode.getHeight() == verticalRes && 
					mode.getRefreshRate() == refreshRate) {
				if(colourDepth != Integer.MAX_VALUE && colourDepth == mode.getBitDepth()) {
					// Colour depth specified and matches with available mode
					displayModeSupported = DISPLAY_MODE_SUPPORTED;
					
					break;
				} else {
					if(displayModeSupported.equals(DISPLAY_MODE_NOT_SUPPORTED)) {
						// If no modes currently supported set support for one mode
						displayModeSupported = DISPLAY_MODE_ONE_COLOUR_DEPTH_SUPPORTED;
					} else if(displayModeSupported.equals(DISPLAY_MODE_ONE_COLOUR_DEPTH_SUPPORTED)) {
						// One mode is already supported, so now set both supported
						displayModeSupported = DISPLAY_MODE_SUPPORTED;
						
						break;
					}
				}
			}
		}

		if(displayModeSupported.equals(DISPLAY_MODE_SUPPORTED)) {
			if(debug) {
				print("Display mode is supported");
			}
			
			// Do nothing, all is well
		} else if(displayModeSupported.equals(DISPLAY_MODE_ONE_COLOUR_DEPTH_SUPPORTED)) {
			if(debug) {
				print("One colour depth is supported at the supplied res and refresh rate");
			}

			// Inform the user of the dilemma
			int selectedOption = JOptionPane.showConfirmDialog(null, "You have chosen not to specify a colour depth.\n" +
					"Your graphics card is reporting it only supports either 16 bit or 32 bit\n" +
					"colour depth at "+horizontalRes+"x"+verticalRes+" @ "+refreshRate+"Hz.\n" +
					"I suggest you choose not to continue and re-run the application with a colour depth specified.\n" +
					"If you specify the colour depth (by adding either 16 or 32 on the end of your command line)\n" +
					"the application will be able to determine if you can run the specified display mode." +
					"If you continue anyway things could go kablamo!\n\n" +
					"Do you wish to continue?",
					"Display Mode May Not Be Supported", 
					JOptionPane.YES_NO_OPTION, 
					JOptionPane.WARNING_MESSAGE);
			
			if(selectedOption == JOptionPane.NO_OPTION) {
				// User selected no, exit the application
				System.exit(1);
			}
			// Otherwise they wish to continue
		} else if(displayModeSupported.equals(DISPLAY_MODE_NOT_SUPPORTED)) {
			String displayMessage = null;
			
			if(colourDepth != Integer.MAX_VALUE) {
				displayMessage = "Your system is reporting it cannot display the specified display mode of\n" +
				colourDepth+" bit, "+horizontalRes+"x"+verticalRes+" @ "+refreshRate+"Hz." +
				"\n\n" +
				"Do you wish to continue anyway?";
			} else {
				displayMessage = "Your system is reporting it cannot display the specified display mode of\n" +
				horizontalRes+"x"+verticalRes+" @ "+refreshRate+"Hz." +
				"\n\n" +
				"Do you wish to continue anyway?";
			}
			
			int selectedOption = JOptionPane.showConfirmDialog(null, 
					displayMessage,
					"Display Mode Not Supported", 
					JOptionPane.YES_NO_OPTION, 
					JOptionPane.ERROR_MESSAGE);
			
			if(selectedOption == JOptionPane.NO_OPTION) {
				// User selected no, exit the application
				System.exit(1);
			}
		}
	}
	
	private void outputDetailsToCFGFile() {
		/*
		 * Output the details the user has specified to file. Create a temp file and
		 * to hold the contents of the updated cfg.txt file then delete the original
		 * cfg.txt file and rename the temp file to cfg.txt
		 */
		// The original config file
		File configFile = new File(CONFIG_FILENAME);
		FileReader configFileReader = null; 
		BufferedReader bufReader = null;
		
		try {
			configFileReader = new FileReader(configFile);
			bufReader = new BufferedReader(configFileReader);
		} catch(FileNotFoundException fnfe) {
			// Can't find the cfg.txt file, exit the application
			JOptionPane.showMessageDialog(null, "The "+CONFIG_FILENAME+" file cannot be found at the following location:\n" +
					configFile.getAbsolutePath()+"\n" +
					"Try running LFS on its own to create the "+CONFIG_FILENAME+" file then re-run this application.", 
					"Cannot Find Config File", 
					JOptionPane.ERROR_MESSAGE);
			
			System.exit(1);
		}
		
		// The temporary config file
		File newConfigFile = new File("temporary"+CONFIG_FILENAME);
		
		FileOutputStream newConfigOutputStream = null;
		PrintWriter writer = null;
		
		try {
			newConfigOutputStream = new FileOutputStream(newConfigFile);
			writer = new PrintWriter(newConfigOutputStream);
		} catch(FileNotFoundException fnfe) {
			fnfe.printStackTrace();
			
			JOptionPane.showMessageDialog(null, "The application cannot write the details of the new config file.\n" +
					"Please report this problem in the LFS Forum thread you downloaded the application from.\n" +
					"Include as many details as possible (including Operating System, Java version, " +
					"folder you're running this app from).\n" +
					"Thanks.", 
					"Cannot Write New File", 
					JOptionPane.ERROR_MESSAGE);
			
			System.exit(1);
		}
		
		/*
		 * Read files from the original config file. If they are not relevant to this application simply
		 * output them to the new config file. If they are relevant update the information as necessary
		 * and output it
		 */
		
		try {
			String inputLine = bufReader.readLine();
			
			while(inputLine != null) {
				if(inputLine.startsWith(SCREEN_MODE_LINE_IDENTIFIER)) {
					if(debug) {
						print("Found the screen info line in config file: "+inputLine);
					}
					
					// If the colour depth isn't set here it needs to be retained from current file
					if(colourDepth == Integer.MAX_VALUE) {
						// Get the details in the current screen info
						String currentDetails = inputLine.substring(inputLine.indexOf(SCREEN_MODE_LINE_IDENTIFIER)+
								SCREEN_MODE_LINE_IDENTIFIER.length()+1);
						
						String currentColourDepth = currentDetails.substring(0, 1);
						
						StringBuilder toOutput = new StringBuilder(SCREEN_MODE_LINE_IDENTIFIER+" ");
						
						// Colour depth
						toOutput.append(currentColourDepth+" ");
						
						// Refresh rate
						toOutput.append(refreshRate+" ");
						
						// Horizontal res
						toOutput.append(horizontalRes+" ");
						
						// Vertical res
						toOutput.append(verticalRes+" ");
						
						writer.println(toOutput.toString());
					} else {
						// Otherwise just output the new details
						StringBuilder toOutput = new StringBuilder(SCREEN_MODE_LINE_IDENTIFIER+" ");
						
						// Colour depth
						if(colourDepth == 16) {
							toOutput.append("1 ");
						} else if(colourDepth == 32) {
							toOutput.append("0 ");
						}
						
						// Refresh rate
						toOutput.append(refreshRate+" ");
						
						// Horizontal res
						toOutput.append(horizontalRes+" ");
						
						// Vertical res
						toOutput.append(verticalRes+" ");
						
						writer.println(toOutput.toString());
					}
				} else if(inputLine.startsWith(WINDOW_MODE_LINE_IDENTIFIER)) {
					if(debug) {
						print("Found the windowed mode line in config file: "+inputLine);
					}
					
					if(windowed == null) {
						// No windowed preference has been specified, just output the current line
						writer.println(inputLine);
					} else {
						if(windowed.booleanValue()) {
							writer.println(WINDOW_MODE_LINE_IDENTIFIER+" 1");
						} else {
							writer.println(WINDOW_MODE_LINE_IDENTIFIER+" 0");
						}
					}
				} else {
					writer.println(inputLine);
				}
				
				inputLine = bufReader.readLine();
			}
			
			writer.close();
			bufReader.close();
		} catch(IOException ioe) {
			ioe.printStackTrace();
			
			JOptionPane.showMessageDialog(null, "The application cannot read or write the config file.\n" +
					"Please report this problem in the LFS Forum thread you downloaded the application from.\n" +
					"Include as many details as possible (including Operating System, Java version, " +
					"folder you're running this app from).\n" +
					"Thanks.", 
					"Problem Reading Or Writing Config File", 
					JOptionPane.ERROR_MESSAGE);
			
			System.exit(1);
		}

		// Now the temporary config file contains the updated details delete the original file and rename new one
		boolean originalFileDeleted = configFile.delete();
		
		if(originalFileDeleted) {
			boolean newFileRenamed = newConfigFile.renameTo(configFile);
			
			if(!newFileRenamed) {
				JOptionPane.showMessageDialog(null, "The application cannot replace the original config file (2).\n" +
						"Please report this problem in the LFS Forum thread you downloaded the application from.\n" +
						"Include as many details as possible (including Operating System, Java version, " +
						"folder you're running this app from).\n" +
						"Thanks.", 
						"Problem Replacing Config File", 
						JOptionPane.ERROR_MESSAGE);
				
				System.exit(1);
			}
		} else {
			JOptionPane.showMessageDialog(null, "The application cannot replace the original config file (1).\n" +
					"Please report this problem in the LFS Forum thread you downloaded the application from.\n" +
					"Include as many details as possible (including Operating System, Java version, " +
					"folder you're running this app from).\n" +
					"Thanks.", 
					"Problem Replacing Config File", 
					JOptionPane.ERROR_MESSAGE);
			
			System.exit(1);
		}
	}
	
	private void runLFS() {
		try {
			/*
			 * Start LFS up. It should now load the details from the updated cfg.txt file.
			 */
			Runtime.getRuntime().exec("LFS.exe");
		} catch(IOException ioe) {
			ioe.printStackTrace();
			
			JOptionPane.showMessageDialog(null, "The application cannot run LFS.\n" +
					"Please report this problem in the LFS Forum thread you downloaded the application from.\n" +
					"Include as many details as possible (including Operating System, Java version, " +
					"folder you're running this app from).\n" +
					"Thanks.", 
					"Problem Running LFS", 
					JOptionPane.ERROR_MESSAGE);
			
			System.exit(1);
		}
	}
	
	private int parseIntegerFromString(String string) {
		if(debug) {
			print("parseIntegerFromString called with String: "+string);
		}
		
		try {
			return Integer.parseInt(string);
		} catch(NumberFormatException nfe) {
			// nfe.printStackTrace();
			
			// Error trying to parse the argument to integer, display usage info and exit
			displayUsageInformationToUser();
			System.exit(1);
		}
		
		// Never going to be returned
		return Integer.MIN_VALUE;
	}
	
	private void setAdditionalParameterFromString(String param) {
		if(debug) {
			print("setAdditionalParameterFromString called with String: "+param);
		}
		
		if(param.equals("16") || param.equals("32")) {
			// Found a colour depth
			if(debug) {
				print("Found a colour depth of "+param+" specified");
			}

			colourDepth = parseIntegerFromString(param);
		} else if(param.equals("true") || param.equals("false")) {
			// Found windowed mode param
			if(debug) {
				print("Found windowed mode parameter of "+param);
			}
			
			if(param.equals("true")) {
				windowed = true;
			} else {
				windowed = false;
			}
		} else {
			// Found an unrecognised parameter type, print usage info and exit
			displayUsageInformationToUser();
			System.exit(1);
		}
	}
	
	private void displayUsageInformationToUser() {
		JOptionPane.showMessageDialog(null, "You must launch "+this.getClass().getSimpleName()+
				"\nwith arguments of the following format:\n" +
				this.getClass().getSimpleName()+" <Horizontal Resolution> <Vertical Resolution> <Refresh Rate>\n" +
				"In addition to the above you may specify colour depth and/or windowed mode.\n" +
				"Colour depth must be specified as either 16 or 32 and windowed mode must be either true or false.\n" +
				"If no value is specified for colour depth the current value will be used.\n" +
				"If no value is specified for windowed mode the current value will be used.", 
				this.getClass().getSimpleName()+" - Invalid Arguments", 
				JOptionPane.ERROR_MESSAGE);
	}
	
	private void print(String msg) {
		System.out.println(msg);
	}
	
	public static void main(String[] args) {
		new LFSCustomResLauncherV1(args);
	}
}
