package tester;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.Toolkit;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseWheelEvent;
import java.awt.event.MouseWheelListener;
import java.awt.event.WindowEvent;
import java.awt.event.WindowAdapter;

import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.JButton;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JSpinner;
import javax.swing.JTextField;
import javax.swing.SpinnerNumberModel;
import javax.swing.SwingConstants;
import javax.swing.Timer;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

public class MultimediaTimerTester extends JFrame {
	/**
	 * 
	 */
	private static final long serialVersionUID = 1892399260834993540L;

	// The timer that fires when it's time to change the colour of the label
	private Timer changeLabelColourTimer;
	
	// The timer that fires when it's time to update the label informing the user of 
	// time to next colour change
	private Timer updateRemainingTimeToChangeLabelTimer;
	
	// Timer to update the fields displaying the time passed since the app started
	private Timer updateTimeSinceAppStartLabelsTimer;

	private JLabel randomColourLabel;
	private JTextField timeToNextChangeField, msSinceStartField, nsSinceStartField;
	private JButton startTimerButton, stopTimerButton;
	private JSpinner timerPeriodSpinner;
	private final int timerValueMaximum = 5000, timerValueMinimum = 10, timerValueIncrement = 10, 
		timerValueDefault = 500;
	
	// An array to hold all available colours
	private final Color[] colourArray;
	
	// The epoch time of the last time the label colour changed
	private long lastChangeColourTimerFire = Long.MIN_VALUE, programStartMs = System.currentTimeMillis(), 
		programStartNs = System.nanoTime();
	
	public MultimediaTimerTester() {
		
		
		// Setup GUI
		randomColourLabel = new JLabel("Hi LFS Forum!");
		randomColourLabel.setFont(new Font("", Font.BOLD, 48));
		randomColourLabel.setOpaque(true);
		randomColourLabel.setHorizontalAlignment(SwingConstants.CENTER);
		
		// Initialise colour array
		colourArray = new Color[12];
		colourArray[0] = Color.WHITE;
		colourArray[1] = Color.LIGHT_GRAY;
		colourArray[2] = Color.DARK_GRAY;
		colourArray[3] = Color.BLACK;
		colourArray[4] = Color.RED;
		colourArray[5] = Color.PINK;
		colourArray[6] = Color.ORANGE;
		colourArray[7] = Color.YELLOW;
		colourArray[8] = Color.GREEN;
		colourArray[9] = Color.MAGENTA;
		colourArray[10] = Color.CYAN;
		colourArray[11] = Color.BLUE;
		
		// Initially set the label a random colour
		changeRandomColourLabel();
		
		// Create a spinner to allow the user to alter the delay on the label colour change
		timerPeriodSpinner = new JSpinner(new SpinnerNumberModel(timerValueDefault, timerValueMinimum, 
				timerValueMaximum, timerValueIncrement));

		/*
		 * Add a mouse wheel listener to the spinner. When a user scrolls the mouse wheel over the delay field
		 * if the new value is legal update the field and the timer to this new value. If the new value is 
		 * outwith the bounds of the timer beep to let the user know and leave the values as before
		 */
		timerPeriodSpinner.addMouseWheelListener(new MouseWheelListener() {
			public void mouseWheelMoved(MouseWheelEvent e) {
				int scrollIncrements = e.getUnitsToScroll() / e.getScrollAmount();

				int currentValue = Integer.parseInt(timerPeriodSpinner.getValue().toString());

				int proposedValue = currentValue - (scrollIncrements * timerValueIncrement);
				
				if(proposedValue >= timerValueMinimum && proposedValue <= timerValueMaximum) {
					timerPeriodSpinner.setValue(proposedValue);
				} else {
					Toolkit.getDefaultToolkit().beep();
				}
			}
		});
		
		/*
		 * Listen for changes to the delay value in the spinner. When they occur (either by 
		 * scroll wheel above or by user typing in value) update the delay in the spinner
		 */
		timerPeriodSpinner.addChangeListener(new ChangeListener() {
			public void stateChanged(ChangeEvent e) {
				int newDelay = Integer.parseInt(timerPeriodSpinner.getValue().toString());
				
				updateTimerDelay(newDelay);
			}
		});
		
		timerPeriodSpinner.setMaximumSize(timerPeriodSpinner.getPreferredSize());
		
		startTimerButton = new JButton("Start Timer");
		startTimerButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				startTimerButton_actionPerformed();
			}
		});
		
		stopTimerButton = new JButton("Stop Timer");
		stopTimerButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				stopTimerButton_actionPerformed();
			}
		});

		JLabel timerDelayLabel = new JLabel("Delay (ms)");
		
		// Create the timer with the default wait value
		changeLabelColourTimer = new Timer(timerValueDefault, new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				changeLabelColourTimer_actionPerformed();
			}
		});

		updateRemainingTimeToChangeLabelTimer = new Timer(50, new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				updateRemainingTimeToChangeLabel_actionPerformed();
			}
		});

		updateTimeSinceAppStartLabelsTimer = new Timer(50, new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				updateTimeSinceAppStartLabels();
			}
		});

		JLabel timeToNextChangeLabel = new JLabel("Time To Next Change (ms):");
		
		// Initially set value of field to N/A because timer isn't running
		timeToNextChangeField = new JTextField("N/A", 4);
		timeToNextChangeField.setEditable(false);
		timeToNextChangeField.setMaximumSize(timeToNextChangeField.getPreferredSize());
		timeToNextChangeField.setHorizontalAlignment(SwingConstants.RIGHT);
		timeToNextChangeField.setBorder(BorderFactory.createEmptyBorder());
		timeToNextChangeField.setFont(timeToNextChangeLabel.getFont());
		
		setStartStopButtonsEnabled();
		
		Box topTimerBox = Box.createHorizontalBox();

		topTimerBox.add(Box.createHorizontalGlue());
		topTimerBox.add(startTimerButton);
		topTimerBox.add(Box.createHorizontalStrut(5));
		topTimerBox.add(stopTimerButton);
		topTimerBox.add(Box.createHorizontalStrut(5));
		topTimerBox.add(timerDelayLabel);
		topTimerBox.add(Box.createHorizontalStrut(5));
		topTimerBox.add(timerPeriodSpinner);
		topTimerBox.add(Box.createHorizontalStrut(5));
		topTimerBox.add(timeToNextChangeLabel);
		topTimerBox.add(Box.createHorizontalStrut(5));
		topTimerBox.add(timeToNextChangeField);
		topTimerBox.add(Box.createHorizontalGlue());
		
		Box bottomTimerBox = Box.createHorizontalBox();
		
		JLabel msSinceStartLabel = new JLabel("Millisecs Since App Start");
		JLabel nsSinceStartLabel = new JLabel("Nanosecs Since App Start");
		
		msSinceStartField = new JTextField(7);
		msSinceStartField.setFont(msSinceStartLabel.getFont());
		msSinceStartField.setHorizontalAlignment(SwingConstants.RIGHT);
		msSinceStartField.setMaximumSize(msSinceStartField.getPreferredSize());
		msSinceStartField.setEditable(false);
		
		nsSinceStartField = new JTextField(12);
		nsSinceStartField.setFont(nsSinceStartLabel.getFont());
		nsSinceStartField.setHorizontalAlignment(SwingConstants.RIGHT);
		nsSinceStartField.setMaximumSize(nsSinceStartField.getPreferredSize());
		nsSinceStartField.setEditable(false);

		bottomTimerBox.add(msSinceStartLabel);
		bottomTimerBox.add(Box.createHorizontalStrut(5));
		bottomTimerBox.add(msSinceStartField);
		bottomTimerBox.add(Box.createHorizontalGlue());
		bottomTimerBox.add(nsSinceStartLabel);
		bottomTimerBox.add(Box.createHorizontalStrut(5));
		bottomTimerBox.add(nsSinceStartField);
		
		Box mainTimerBox = Box.createVerticalBox();
		
		mainTimerBox.add(topTimerBox);
		mainTimerBox.add(Box.createVerticalStrut(5));
		mainTimerBox.add(bottomTimerBox);
		
		mainTimerBox.setBorder(BorderFactory.createEmptyBorder(5,5,5,5));
		
		this.setLayout(new BorderLayout());

		this.add(mainTimerBox, BorderLayout.SOUTH);
		this.add(randomColourLabel, BorderLayout.CENTER);
		
		// Start the timers to update the labels
		updateTimeSinceAppStartLabelsTimer.start();
		updateRemainingTimeToChangeLabelTimer.start();
		
		// Exit the program when the user closes the window
		this.addWindowListener(new WindowAdapter() {
			public void windowClosing(WindowEvent arg0) {
				System.exit(0);
			}
		});
		
		this.setTitle("Quick Timer Example");
		
		this.setSize(600, 450);
		
		// Set display to centre of screen
		Dimension screenResolution = Toolkit.getDefaultToolkit().getScreenSize();
		
		int frameXCo = (int) ((screenResolution.getWidth()/2)-(this.getSize().getWidth()/2));
		int frameYCo = (int) ((screenResolution.getHeight()/2)-(this.getSize().getHeight()/2));
		this.setLocation(frameXCo, frameYCo);
		
		this.setVisible(true);
	}

	private void startTimerButton_actionPerformed() {
		changeLabelColourTimer.start();

		lastChangeColourTimerFire = System.currentTimeMillis();
		
		setStartStopButtonsEnabled();
	}
	
	private void stopTimerButton_actionPerformed() {
		changeLabelColourTimer.stop();
		
		setStartStopButtonsEnabled();
	}
	
	private void changeLabelColourTimer_actionPerformed() {
		/*
		 * When the timer fires update the timestamp of the last timer fire to now
		 * and change the random colour background
		 */
		lastChangeColourTimerFire = System.currentTimeMillis();
		
		changeRandomColourLabel();
	}
	
	private void updateRemainingTimeToChangeLabel_actionPerformed() {
		if(changeLabelColourTimer.isRunning()) {
			/*
			 * Timer is currently running, use delay and last fire to work out time 
			 * to next fire
			 */
			 long msSinceLastFire = System.currentTimeMillis() - lastChangeColourTimerFire;
			 long msToNextFire = changeLabelColourTimer.getDelay()-msSinceLastFire;
			 
			 timeToNextChangeField.setText(""+msToNextFire);
			 
		} else {
			/*
			 * Timer isn't running, reset field to N/A
			 */
			timeToNextChangeField.setText("N/A");
		}
	}
	
	private void updateTimeSinceAppStartLabels() {
		long msSinceStart = System.currentTimeMillis() - programStartMs;
		long nsSinceStart = System.nanoTime() - programStartNs;
		
		msSinceStartField.setText(""+msSinceStart);
		nsSinceStartField.setText(""+nsSinceStart);
	}
	
	private void changeRandomColourLabel() {
		// Ensure a colour change
		Color currentColour = randomColourLabel.getBackground();
		
		int randomColourIndex = (int) (Math.random() * colourArray.length);
		Color newColour = colourArray[randomColourIndex];
		
		while(newColour.equals(currentColour)) {
			randomColourIndex = (int) (Math.random() * colourArray.length);
			newColour = colourArray[randomColourIndex];
		}
		
		// Set the text to be the negative colour of the background
		Color textColor = new Color((255-newColour.getRed()), (255-newColour.getGreen()), (255-newColour.getBlue()));
		
		randomColourLabel.setBackground(newColour);
		randomColourLabel.setForeground(textColor);
	}
	
	private void setStartStopButtonsEnabled() {
		/*
		 * Enable/disable start/stop buttons where appropriate
		 */
		
		if(changeLabelColourTimer.isRunning()) {
			startTimerButton.setEnabled(false);
			stopTimerButton.setEnabled(true);
		} else {
			startTimerButton.setEnabled(true);
			stopTimerButton.setEnabled(false);
		}
	}
	
	private void updateTimerDelay(int newDelay) {
		/* 
		 * Update both the initial delay and the normal delay. The initial delay 
		 * is the time between starting the timer and the first firing whereas
		 * the delay is the time between firings once it's up and running
		 * 
		 */
		changeLabelColourTimer.setInitialDelay(newDelay);
		changeLabelColourTimer.setDelay(newDelay);
		
		// Restart the timer with the new delay if currently running
		if(changeLabelColourTimer.isRunning()) {
			changeLabelColourTimer.restart();
			lastChangeColourTimerFire = System.currentTimeMillis();
		}
	}

	// Kick off this example
	public static void main(String[] args) {
		new MultimediaTimerTester();
	}
}
