#!/usr/bin/env python
#
# Copyright Alex McBride 2009.
#
# This file is part of pyinsim.
#
# pyinsim is free software: you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# pyinsim is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with pyinsim. If not, see <http://www.gnu.org/licenses/>.
#

"""An InSim module for the Python programming language.

"""

import socket
import threading
import struct
import math
import re
import os
from UserDict import UserDict
from xml.dom import minidom

# General constants.
_INSIM_BUFFER_SIZE = 1024
INSIM_VERSION = 4
INSIM_LOCALHOST = '127.0.0.1'

# Port and IP constants.
_IP_PATTERN = r'\b(25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\.(25[0-5]|2\
[0-4][0-9]|[01]?[0-9][0-9]?)\.(25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\.(25\
[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\b'
MAX_PORT = 65535
MIN_PORT = 1
ADMIN_LEN = 16

# Enum for packet-types
ISP_NONE = 0
ISP_ISI = 1
ISP_VER = 2
ISP_TINY = 3
ISP_SMALL = 4
ISP_STA = 5
ISP_SCH = 6
ISP_SFP = 7
ISP_SCC = 8
ISP_CPP = 9
ISP_ISM = 10
ISP_MSO = 11
ISP_III = 12
ISP_MST = 13
ISP_MTC = 14
ISP_MOD = 15
ISP_VTN = 16
ISP_RST = 17
ISP_NCN = 18
ISP_CNL = 19
ISP_CPR = 20
ISP_NPL = 21
ISP_PLP = 22
ISP_PLL = 23
ISP_LAP = 24
ISP_SPX = 25
ISP_PIT = 26
ISP_PSF = 27
ISP_PLA = 28
ISP_CCH = 29
ISP_PEN = 30
ISP_TOC = 31
ISP_FLG = 32
ISP_PFL = 33
ISP_FIN = 34
ISP_RES = 35
ISP_REO = 36
ISP_NLP = 37
ISP_MCI = 38
ISP_MSX = 39
ISP_MSL = 40
ISP_CRS = 41
ISP_BFN = 42
ISP_AXI = 43
ISP_AXO = 44
ISP_BTN = 45
ISP_BTC = 46
ISP_BTT = 47
ISP_RIP = 48
ISP_SSH = 49
# We make a special case for these.
ISP_OUTSIM = 251
ISP_OUTGAUGE = 252
ISP_ALL = 253
ISP_NODELAP = 254
ISP_COMPCAR = 255

# Enum for IS_TINY sub-type
TINY_NONE = 0
TINY_VER = 1
TINY_CLOSE = 2
TINY_PING = 3
TINY_REPLY = 4
TINY_VTC = 5
TINY_SCP = 6
TINY_SST = 7
TINY_GTH = 8
TINY_MPE = 9
TINY_ISM = 10
TINY_REN = 11
TINY_CLR = 12
TINY_NCN = 13
TINY_NPL = 14
TINY_RES = 15
TINY_NLP = 16
TINY_MCI = 17
TINY_REO = 18
TINY_RST = 19
TINY_AXI = 20
TINY_AXC = 21
TINY_RIP = 22

# Enum for IS_SMALL sub-type
SMALL_NONE = 0
SMALL_SSP = 1
SMALL_SSG = 2
SMALL_VTA = 3
SMALL_TMS = 4
SMALL_STP = 5
SMALL_RTP = 6
SMALL_NLI = 7

# Flags for ISI Flags
ISF_RES_0 = 1
ISF_RES_1 = 2
ISF_LOCAL = 4
ISF_MSO_COLS = 8
ISF_NLP = 16
ISF_MCI = 32

# Enum for User Values
MSO_SYSTEM = 0
MSO_USER = 1
MSO_PREFIX = 2
MSO_O = 3

# Enum for message sounds
SND_SILENT = 0
SND_MESSAGE = 1
SND_SYSMESSAGE = 2
SND_INVALIDKEY = 3
SND_ERROR = 4

# Enum for voting
VOTE_NONE = 0
VOTE_END = 1
VOTE_RESTART = 2
VOTE_QUALIFY = 3

# Enum for pitlane
PITLANE_EXIT = 0
PITLANE_ENTER = 1
PITLANE_NO_PURPOSE = 2
PITLANE_DT = 3
PITLANE_SG = 4

# Enum for view identifiers
VIEW_FOLLOW = 0
VIEW_HELI = 1
VIEW_CAM = 2
VIEW_DRIVER = 3
VIEW_CUSTOM = 4
VIEW_ANOTHER = 255

# Enum for leave reasons
LEAVR_DISCO = 0
LEAVR_TIMEOUT = 1
LEAVR_LOSTCONN = 2
LEAVR_KICKED = 3
LEAVR_BANNED = 4
LEAVR_SECURITY = 5

# Enum for Penalty values
PENALTY_NONE = 0
PENALTY_DT = 1
PENALTY_DT_VALID = 2
PENALTY_SG = 3
PENALTY_SG_VALID = 4
PENALTY_30 = 5
PENALTY_45 = 6

# Enum for Penalty reasons
PENR_UNKNOWN = 1
PENR_ADMIN = 2
PENR_WRONG_WAY = 3
PENR_FALSE_START = 4
PENR_SPEEDING = 5
PENR_STOP_SHORT = 6
PENR_STOP_LATE = 7

# Enum for Tyre compounds
TYRE_R1 = 0
TYRE_R2 = 1
TYRE_R3 = 2
TYRE_R4 = 3
TYRE_ROAD_SUPER = 4
TYRE_ROAD_NORMAL = 5
TYRE_HYBRID = 6
TYRE_KNOBBLY = 7
TYRE_NOT_CHANGED = 255

# ISS state flags
ISS_GAME = 1
ISS_REPLAY = 2
ISS_PAUSED = 4
ISS_SHIFTU = 8
ISS_SHIFTU_HIGH    = 16
ISS_SHIFTU_FOLLOW = 32
ISS_SHIFTU_NO_OPT = 64
ISS_SHOW_2D = 128
ISS_FRONT_END = 256
ISS_MULTI = 512
ISS_MPSPEEDUP =    1024
ISS_WINDOWED = 2048
ISS_SOUND_MUTE = 4096
ISS_VIEW_OVERRIDE = 8192
ISS_VISIBLE = 16384

# Pitwork flags
PSE_NOTHING = 1
PSE_STOP = 2
PSE_FR_DAM = 4
PSE_FR_WHL = 8
PSE_LE_FR_DAM = 16
PSE_LE_FR_WHL = 32
PSE_RI_FR_DAM = 64
PSE_RI_FR_WHL = 128
PSE_RE_DAM = 256
PSE_RE_WHL = 512
PSE_LE_RE_DAM = 1024
PSE_LE_RE_WHL = 2048
PSE_RI_RE_DAM = 4096
PSE_RI_RE_WHL = 8192
PSE_BODY_MINOR = 16384
PSE_BODY_MAJOR = 32768
PSE_SETUP = 65536
PSE_REFUEL = 131072
PSE_NUM = 262144

# Player flags
PIF_SWAPSIDE = 1
PIF_RESERVED_2 = 2
PIF_RESERVED_4 = 4
PIF_AUTOGEARS = 8
PIF_SHIFTER = 16
PIF_RESERVED_32 = 32
PIF_HELP_B = 64
PIF_AXIS_CLUTCH = 128
PIF_INPITS = 256
PIF_AUTOCLUTCH = 512
PIF_MOUSE = 1024
PIF_KB_NO_HELP = 2048
PIF_KB_STABILISED = 4096
PIF_CUSTOM_VIEW = 8192

# Confirmation flags
CONF_MENTIONED = 1
CONF_CONFIRMED = 2
CONF_PENALTY_DT = 4
CONF_PENALTY_SG = 8
CONF_PENALTY_30 = 16
CONF_PENALTY_45 = 32
CONF_DID_NOT_PIT = 64
CONF_DISQ = (CONF_PENALTY_DT | CONF_PENALTY_SG | CONF_DID_NOT_PIT)
CONF_TIME = (CONF_PENALTY_30 | CONF_PENALTY_45)

# Race flags
HOSTF_CAN_VOTE = 1
HOSTF_CAN_SELECT = 2
HOSTF_MID_RACE = 32
HOSTF_MUST_PIT = 64
HOSTF_CAN_RESET = 128
HOSTF_FCV = 256
HOSTF_CRUISE = 512

# Car info flags.
CCI_BLUE = 1
CCI_YELLOW = 2
CCI_LAG    = 32
CCI_FIRST = 64
CCI_LAST = 128

# Button Recommended Area
IS_X_MIN = 0
IS_X_MAX = 110
IS_Y_MIN = 30
IS_Y_MAX = 170

# Enum for BFN SubT
BFN_DEL_BTN = 0
BFN_CLEAR = 1
BFN_USER_CLEAR = 2
BFN_REQUEST = 3

INST_ALWAYS_ON = 128

# Flags for IS_BTN BStyle
ISB_C1 = 1
ISB_C2 = 2
ISB_C4 = 4
ISB_CLICK = 8
ISB_LIGHT = 16
ISB_DARK = 32
ISB_LEFT = 64
ISB_RIGHT = 128

# Flags for BTN CFlags
ISB_LMB    = 1
ISB_RMB = 2
ISB_CTRL = 4
ISB_SHIFT = 8

# Weather enum
WEA_BRIGHTCLEAR = 0
WEA_OVERCAST = 1
WEA_CLOUDYSUNSETDUSK = 2

# Wind enum
WND_NONE = 0
WND_LOW = 1
WND_HIGH = 2

# Flags enum
FLG_BLUE = 1
FLG_YELLOW = 2
FLG_ON = 1
FLG_OFF = 0

# Enum for replay errors
RIP_OK = 0
RIP_ALREADY = 1
RIP_DEDICATED = 2
RIP_WRONG_MODE = 3
RIP_NOT_REPLAY = 4
RIP_CORRUPTED = 5
RIP_NOT_FOUND = 6
RIP_UNLOADABLE = 7
RIP_DEST_OOB = 8
RIP_UNKNOWN = 9
RIP_USER = 10
RIP_OOS = 11

# Enum for replay options
RIPOPT_LOOP    = 1
RIPOPT_SKINS = 2

# Enum for screenshot errors
SSH_OK = 0
SSH_DEDICATED = 1
SSH_CORRUPTED = 2
SSH_NO_SAVE = 3

# SetF flags
SETF_SYMM_WHEELS = 1
SETF_TC_ENABLE = 2
SETF_ABS_ENABLE    = 4

# Colour constants.
COL_BLACK = '^0'
COL_RED = '^1'
COL_LIGHT_GREEN = '^2'
COL_YELLOW = '^3'
COL_BLUE = '^4'
COL_PURPLE = '^5'
COL_LIGHT_BLUE = '^6'
COL_WHITE = '^7'
COL_DARK_GREEN = '^8'
COL_ORIGINAL = '^9'

# OutGauge Flags
OG_SHIFTLIGHT = 1
OG_FULLBEAM = 2
OG_HANDBRAKE = 4
OG_PITSPEED = 8
OG_TC = 16
OG_HEADLIGHTS = 32
OG_SIGNAL_L = 64
OG_SIGNAL_R = 128
OG_REDLINE = 256
OG_OILWARN = 512
OG_1 = 1024
OG_2 = 2048
OG_3 = 4096
OG_4 = 8192
OG_KM = 16384
OG_BAR = 32768

# Default values for packet definitions.
_TYPE_DEFAULTS = {'B': 0, 'h': 0, 'H': 0, 's': '', 'f': 0.0, 'i': 0, 'I': 0, 'c': ' '}

# Packet definitions
_PACKET_DEFS = {}

# Initalisation
_PACKET_DEFS[ISP_ISI] = (('Size', 'B'),
                         ('Type', 'B'),
                         ('ReqI', 'B'),
                         ('Zero', 'B'),
                         ('UDPPort', 'H'),
                         ('Flags', 'H'),
                         ('Sp0', 'B'),
                         ('Prefix', 'c'),
                         ('Interval', 'H'),
                         ('Admin', '16s'),
                         ('IName', '16s'))

# General Purpose
_PACKET_DEFS[ISP_TINY] = (('Size', 'B'),
                          ('Type', 'B'),
                          ('ReqI', 'B'),
                          ('SubT', 'B'))

_PACKET_DEFS[ISP_SMALL] = (('Size', 'B'),
                           ('Type', 'B'),
                           ('ReqI', 'B'),
                           ('SubT', 'B'),
                           ('UVal', 'I'))

# Version Check
_PACKET_DEFS[ISP_VER] = (('Size', 'B'),
                         ('Type', 'B'),
                         ('ReqI', 'B'),
                         ('Zero', 'B'),
                         ('Version', '8s'),
                         ('Product', '6s'),
                         ('InSimVer', 'H'))

# State Reporting
_PACKET_DEFS[ISP_STA] = (('Size', 'B'),
                         ('Type', 'B'),
                         ('ReqI', 'B'),
                         ('Zero', 'B'),
                         ('ReplaySpeed', 'f'),
                         ('Flags', 'H'),
                         ('InGameCam', 'B'),
                         ('ViewPLID', 'B'),
                         ('NumP', 'B'),
                         ('NumConns', 'B'),
                         ('NumFinished', 'B'),
                         ('RaceInProg', 'B'),
                         ('QualMins', 'B'),
                         ('RaceLaps', 'B'),
                         ('Spare2', 'B'),
                         ('Spare3', 'B'),
                         ('Track', '6s'),
                         ('Weather', 'B'),
                         ('Wind', 'B'))

_PACKET_DEFS[ISP_SFP] = (('Size', 'B'),
                         ('Type', 'B'),
                         ('ReqI', 'B'),
                         ('Zero', 'B'),
                         ('Flag', 'H'),
                         ('OffOn', 'B'),
                         ('Sp3', 'B'))

# Screen Mode
_PACKET_DEFS[ISP_MOD] = (('Size', 'B'),
                         ('Type', 'B'),
                         ('ReqI', 'B'),
                         ('Zero', 'B'),
                         ('Bits16', 'i'),
                         ('RR', 'i'),
                         ('Width', 'i'),
                         ('Height', 'i'))

# Messages out (from LFS)
_PACKET_DEFS[ISP_MSO] = (('Size', 'B'),
                         ('Type', 'B'),
                         ('ReqI', 'B'),
                         ('Zero', 'B'),
                         ('UCID', 'B'),
                         ('PLID', 'B'),
                         ('UserType', 'B'),
                         ('TextStart', 'B'),
                         ('Msg', '128s'))

_PACKET_DEFS[ISP_III] = (('Size', 'B'),
                         ('Type', 'B'),
                         ('ReqI', 'B'),
                         ('Zero', 'B'),
                         ('UCID', 'B'),
                         ('PLID', 'B'),
                         ('Sp2', 'B'),
                         ('Sp3', 'B'),
                         ('Msg', '64s'))

# Messages in (to LFS)
_PACKET_DEFS[ISP_MST] = (('Size', 'B'),
                         ('Type', 'B'),
                         ('ReqI', 'B'),
                         ('Zero', 'B'),
                         ('Msg', '64s'))

_PACKET_DEFS[ISP_MSX] = (('Size', 'B'),
                         ('Type', 'B'),
                         ('ReqI', 'B'),
                         ('Zero', 'B'),
                         ('Msg', '96s'))

_PACKET_DEFS[ISP_MSL] = (('Size', 'B'),
                         ('Type', 'B'),
                         ('ReqI', 'B'),
                         ('Sound', 'B'),
                         ('Msg', '128s'))

_PACKET_DEFS[ISP_MTC] = (('Size', 'B'),
                         ('Type', 'B'),
                         ('ReqI', 'B'),
                         ('Zero', 'B'),
                         ('UCID', 'B'),
                         ('PLID', 'B'),
                         ('Sp2', 'B'),
                         ('Sp3', 'B'),
                         ('Msg', '64s'))

_PACKET_DEFS[ISP_SCH] = (('Size', 'B'),
                         ('Type', 'B'),
                         ('ReqI', 'B'),
                         ('Zero', 'B'),
                         ('CharB', 'B'),
                         ('Flags', 'B'),
                         ('Spare2', 'B'),
                         ('Spare3', 'B'))

# Multiplayer Notification.
_PACKET_DEFS[ISP_ISM] = (('Size', 'B'),
                         ('Type', 'B'),
                         ('ReqI', 'B'),
                         ('Zero', 'B'),
                         ('Host', 'B'),
                         ('Sp1', 'B'),
                         ('Sp2', 'B'),
                         ('Sp3', 'B'),
                         ('HName', '32s'))

# Voting
_PACKET_DEFS[ISP_VTN] = (('Size', 'B'),
                         ('Type', 'B'),
                         ('ReqI', 'B'),
                         ('Zero', 'B'),
                         ('UCID', 'B'),
                         ('Action', 'B'),
                         ('Spare2', 'B'),
                         ('Spare3', 'B'))

# Race Tracking
_PACKET_DEFS[ISP_RST] = (('Size', 'B'),
                         ('Type', 'B'),
                         ('ReqI', 'B'),
                         ('Zero', 'B'),
                         ('RaceLaps', 'B'),
                         ('QualMins', 'B'),
                         ('NumP', 'B'),
                         ('Spare', 'B'),
                         ('Track', '6s'),
                         ('Weather', 'B'),
                         ('Wind', 'B'),
                         ('Flags', 'H'),
                         ('NumNodes', 'H'),
                         ('Finish', 'H'),
                         ('Split1', 'H'),
                         ('Split2', 'H'),
                         ('Split3', 'H'))

_PACKET_DEFS[ISP_NCN] = (('Size', 'B'),
                         ('Type', 'B'),
                         ('ReqI', 'B'),
                         ('UCID', 'B'),
                         ('UName', '24s'),
                         ('PName', '24s'),
                         ('Admin', 'B'),
                         ('Total', 'B'),
                         ('Flags', 'B'),
                         ('Sp3', 'B'))

_PACKET_DEFS[ISP_CNL] = (('Size', 'B'),
                         ('Type', 'B'),
                         ('ReqI', 'B'),
                         ('UCID', 'B'),
                         ('Reason', 'B'),
                         ('Total', 'B'),
                         ('Sp2', 'B'),
                         ('Sp3', 'B'))

_PACKET_DEFS[ISP_CPR] = (('Size', 'B'),
                         ('Type', 'B'),
                         ('ReqI', 'B'),
                         ('UCID', 'B'),
                         ('PName', '24s'),
                         ('Plate', '8s'))

_PACKET_DEFS[ISP_NPL] = (('Size', 'B'),
                         ('Type', 'B'),
                         ('ReqI', 'B'),
                         ('PLID', 'B'),
                         ('UCID', 'B'),
                         ('PType', 'B'),
                         ('Flags', 'H'),
                         ('PName', '24s'),
                         ('Plate', '8s'),
                         ('CName', '4s'),
                         ('SName', '16s'),
                         ('Tyres1', 'B'),
                         ('Tyres2', 'B'),
                         ('Tyres3', 'B'),
                         ('Tyres4', 'B'),
                         ('H_Mass', 'B'),
                         ('H_TRes', 'B'),
                         ('Model', 'B'),
                         ('Pass', 'B'),
                         ('Spare', 'i'),
                         ('SetF', 'B'),
                         ('NumP', 'B'),
                         ('Sp2', 'B'),
                         ('Sp3', 'B'))

_PACKET_DEFS[ISP_PLP] = (('Size', 'B'),
                         ('Type', 'B'),
                         ('ReqI', 'B'),
                         ('PLID', 'B'))

_PACKET_DEFS[ISP_PLL] = (('Size', 'B'),
                         ('Type', 'B'),
                         ('ReqI', 'B'),
                         ('PLID', 'B'))

_PACKET_DEFS[ISP_CRS] = (('Size', 'B'),
                         ('Type', 'B'),
                         ('ReqI', 'B'),
                         ('PLID', 'B'))

_PACKET_DEFS[ISP_LAP] = (('Size', 'B'),
                         ('Type', 'B'),
                         ('ReqI', 'B'),
                         ('PLID', 'B'),
                         ('LTime', 'I'),
                         ('ETime', 'I'),
                         ('LapsDone', 'H'),
                         ('Flags', 'H'),
                         ('Sp0', 'B'),
                         ('Penalty', 'B'),
                         ('NumStops', 'B'),
                         ('Sp3', 'B'))

_PACKET_DEFS[ISP_SPX] = (('Size', 'B'),
                         ('Type', 'B'),
                         ('ReqI', 'B'),
                         ('PLID', 'B'),
                         ('STime', 'I'),
                         ('ETime', 'I'),
                         ('Split', 'B'),
                         ('Penalty', 'B'),
                         ('NumStops', 'B'),
                         ('Sp3', 'B'))

_PACKET_DEFS[ISP_PIT] = (('Size', 'B'),
                         ('Type', 'B'),
                         ('ReqI', 'B'),
                         ('PLID', 'B'),
                         ('LapsDone', 'H'),
                         ('Flags', 'H'),
                         ('Sp0', 'B'),
                         ('Penalty', 'B'),
                         ('NumStops', 'B'),
                         ('Sp3', 'B'),
                         ('Tyres1', 'B'),
                         ('Tyres2', 'B'),
                         ('Tyres3', 'B'),
                         ('Tyres4', 'B'),
                         ('Work', 'I'),
                         ('Spare', 'I'))

_PACKET_DEFS[ISP_PSF] = (('Size', 'B'),
                         ('Type', 'B'),
                         ('ReqI', 'B'),
                         ('PLID', 'B'),
                         ('STime', 'I'),
                         ('Spare', 'I'))

_PACKET_DEFS[ISP_PLA] = (('Size', 'B'),
                         ('Type', 'B'),
                         ('ReqI', 'B'),
                         ('PLID', 'B'),
                         ('Fact', 'B'),
                         ('Sp1', 'B'),
                         ('Sp2', 'B'),
                         ('Sp3', 'B'))

_PACKET_DEFS[ISP_CCH] = (('Size', 'B'),
                         ('Type', 'B'),
                         ('ReqI', 'B'),
                         ('PLID', 'B'),
                         ('Camera', 'B'),
                         ('Sp1', 'B'),
                         ('Sp2', 'B'),
                         ('Sp3', 'B'))

_PACKET_DEFS[ISP_PEN] = (('Size', 'B'),
                         ('Type', 'B'),
                         ('ReqI', 'B'),
                         ('PLID', 'B'),
                         ('OldPen', 'B'),
                         ('NewPen', 'B'),
                         ('Reason', 'B'),
                         ('Sp3', 'B'))

_PACKET_DEFS[ISP_TOC] = (('Size', 'B'),
                         ('Type', 'B'),
                         ('ReqI', 'B'),
                         ('PLID', 'B'),
                         ('OldUCID', 'B'),
                         ('NewUCID', 'B'),
                         ('Sp2', 'B'),
                         ('Sp3', 'B'))

_PACKET_DEFS[ISP_FLG] = (('Size', 'B'),
                         ('Type', 'B'),
                         ('ReqI', 'B'),
                         ('PLID', 'B'),
                         ('OffOn', 'B'),
                         ('Flag', 'B'),
                         ('CarBehind', 'B'),
                         ('Sp3', 'B'))

_PACKET_DEFS[ISP_PFL] = (('Size', 'B'),
                         ('Type', 'B'),
                         ('ReqI', 'B'),
                         ('PLID', 'B'),
                         ('Flags', 'H'),
                         ('Spare', 'H'))

_PACKET_DEFS[ISP_FIN] = (('Size', 'B'),
                         ('Type', 'B'),
                         ('ReqI', 'B'),
                         ('PLID', 'B'),
                         ('TTime', 'I'),
                         ('BTime', 'I'),
                         ('SpA', 'B'),
                         ('NumStops', 'B'),
                         ('Confirm', 'B'),
                         ('SpB', 'B'),
                         ('LapsDone', 'H'),
                         ('Flags', 'H'))

_PACKET_DEFS[ISP_RES] = (('Size', 'B'),
                         ('Type', 'B'),
                         ('ReqI', 'B'),
                         ('PLID', 'B'),
                         ('UName', '24s'),
                         ('PName', '24s'),
                         ('Plate', '8s'),
                         ('CName', '4s'),
                         ('TTime', 'I'),
                         ('BTime', 'I'),
                         ('SpA', 'B'),
                         ('NumStops', 'B'),
                         ('Confirm', 'B'),
                         ('SpB', 'B'),
                         ('LapsDone', 'H'),
                         ('Flags', 'H'),
                         ('ResultNum', 'B'),
                         ('NumRes', 'B'),
                         ('PSeconds', 'H'))

_PACKET_DEFS[ISP_REO] = (('Size', 'B'),
                         ('Type', 'B'),
                         ('ReqI', 'B'),
                         ('NumP', 'B'),
                         ('PLID', '32B'))

# Autocross
_PACKET_DEFS[ISP_AXI] = (('Size', 'B'),
                         ('Type', 'B'),
                         ('ReqI', 'B'),
                         ('Zero', 'B'),
                         ('AXStart', 'B'),
                         ('NumCP', 'B'),
                         ('NumO', 'H'),
                         ('LName', '32s'))

_PACKET_DEFS[ISP_AXO] = (('Size', 'B'),
                         ('Type', 'B'),
                         ('ReqI', 'B'),
                         ('PLID', 'B'))

# Car Position Tracking
_PACKET_DEFS[ISP_NLP] = (('Size', 'B'),
                         ('Type', 'B'),
                         ('ReqI', 'B'),
                         ('NumP', 'B'))

_PACKET_DEFS[ISP_NODELAP] = (('Node', 'H'),
                         ('Lap', 'H'),
                         ('PLID', 'B'),
                         ('Position', 'B'))

_PACKET_DEFS[ISP_MCI] = (('Size', 'B'),
                         ('Type', 'B'),
                         ('ReqI', 'B'),
                         ('NumC', 'B'))

_PACKET_DEFS[ISP_COMPCAR] = (('Node', 'H'),
                         ('Lap', 'H'),
                         ('PLID', 'B'),
                         ('Position', 'B'),
                         ('Info', 'B'),
                         ('Sp3', 'B'),
                         ('X', 'i'),
                         ('Y', 'i'),
                         ('Z', 'i'),
                         ('Speed', 'H'),
                         ('Direction', 'H'),
                         ('Heading', 'H'),
                         ('AngVel', 'h'))

# Camera Control
_PACKET_DEFS[ISP_SCC] = (('Size', 'B'),
                         ('Type', 'B'),
                         ('ReqI', 'B'),
                         ('Zero', 'B'),
                         ('ViewPLID', 'B'),
                         ('InGameCam', 'B'),
                         ('Sp2', 'B'),
                         ('Sp3', 'B'))

_PACKET_DEFS[ISP_CPP] = (('Size', 'B'),
                         ('Type', 'B'),
                         ('ReqI', 'B'),
                         ('Zero', 'B'),
                         ('Pos1', 'i'),
                         ('Pos2', 'i'),
                         ('Pos3', 'i'),
                         ('H', 'H'),
                         ('P', 'H'),
                         ('R', 'H'),
                         ('ViewPLID', 'B'),
                         ('InGameCam', 'B'),
                         ('FOV', 'f'),
                         ('Time', 'H'),
                         ('Flags', 'H'))

# Replay Controls
_PACKET_DEFS[ISP_RIP] = (('Size', 'B'),
                         ('Type', 'B'),
                         ('ReqI', 'B'),
                         ('Error', 'B'),
                         ('MPR', 'B'),
                         ('Paused', 'B'),
                         ('Options', 'B'),
                         ('Sp3', 'B'),
                         ('CTime', 'H'),
                         ('TTime', 'H'),
                         ('RName', '64s'))

# Screenshots
_PACKET_DEFS[ISP_SSH] = (('Size', 'B'),
                         ('Type', 'B'),
                         ('ReqI', 'B'),
                         ('Error', 'B'),
                         ('Sp0', 'B'),
                         ('Sp1', 'B'),
                         ('Sp2', 'B'),
                         ('Sp3', 'B'),
                         ('BMP', '32s'))

# Buttons
_PACKET_DEFS[ISP_BFN] = (('Size', 'B'),
                         ('Type', 'B'),
                         ('ReqI', 'B'),
                         ('SubT', 'B'),
                         ('UCID', 'B'),
                         ('ClickID', 'B'),
                         ('Inst', 'B'),
                         ('Sp3', 'B'))

_PACKET_DEFS[ISP_BTN] = (('Size', 'B'),
                         ('Type', 'B'),
                         ('ReqI', 'B'),
                         ('UCID', 'B'),
                         ('ClickID', 'B'),
                         ('Inst', 'B'),
                         ('BStyle', 'B'),
                         ('TypeIn', 'B'),
                         ('L', 'B'),
                         ('T', 'B'),
                         ('W', 'B'),
                         ('H', 'B'),
                         ('Text', '240s'))

_PACKET_DEFS[ISP_BTC] = (('Size', 'B'),
                         ('Type', 'B'),
                         ('ReqI', 'B'),
                         ('UCID', 'B'),
                         ('ClickID', 'B'),
                         ('Inst', 'B'),
                         ('CFlags', 'B'),
                         ('Sp3', 'B'))

_PACKET_DEFS[ISP_BTT] = (('Size', 'B'),
                         ('Type', 'B'),
                         ('ReqI', 'B'),
                         ('UCID', 'B'),
                         ('ClickID', 'B'),
                         ('Inst', 'B'),
                         ('TypeIn', 'B'),
                         ('Sp3', 'B'),
                         ('Text', '96s'))

# OutGauge
_PACKET_DEFS[ISP_OUTGAUGE] = (('Time', 'I'), 
                              ('Car', '4s'), 
                              ('Flags', 'H'),
                              ('Gear', 'B'), 
                              ('SpareB', 'B'), 
                              ('Speed', 'f'),
                              ('RPM', 'f'),
                              ('Turbo', 'f'),
                              ('EngTemp', 'f'),
                              ('Fuel', 'f'),
                              ('OilPress', 'f'),
                              ('Spare1', 'f'),
                              ('Spare2', 'f'),
                              ('Spare3', 'f'),
                              ('Throttle', 'f'),
                              ('Brake', 'f'),
                              ('Clutch', 'f'),
                              ('Display1', '16s'),
                              ('Display2', '16s'),
                              ('ID', 'i'))

#unsigned    Time;    // time in milliseconds (to check order)
#
#Vector    AngVel;        // 3 floats, angular velocity vector
#float    Heading;    // anticlockwise from above (Z)
#float    Pitch;        // anticlockwise from right (X)
#float    Roll;        // anticlockwise from front (Y)
#Vector    Accel;        // 3 floats X, Y, Z
#Vector    Vel;        // 3 floats X, Y, Z
#Vec        Pos;        // 3 ints   X, Y, Z (1m = 65536)
#
#int        ID;            // optional - only if OutSim ID is specified

# OutSim
_PACKET_DEFS[ISP_OUTSIM] = (('Time', 'I'),
                            ('AngVel1', 'f'),
                            ('AngVel2', 'f'),
                            ('AngVel3', 'f'),
                            ('Heading', 'f'),
                            ('Pitch', 'f'),
                            ('Roll', 'f'),
                            ('Accel1', 'f'),
                            ('Accel2', 'f'),
                            ('Accel3', 'f'),
                            ('Vel1', 'f'),
                            ('Vel2', 'f'),
                            ('Vel3', 'f'),
                            ('Pos1', 'i'),
                            ('Pos2', 'i'),
                            ('Pos3', 'i'),
                            ('ID', 'i'),)


_ENCODING_MAP = {'L': 'cp1252',
                 'G': 'iso8859_7',
                 'C': 'cp1251',
                 'J': 'Shift-JIS',
                 'E': 'iso8859_2',
                 'T': 'iso8859_9',
                 'B': 'iso8859_13',
                 'H': 'cp950',
                 'S': 'cp936',
                 'K': 'cp949'}

_ESCAPE_MAP = {'v': '|',
               'a': '*',
               'c': ':',
               'd': '\\',
               's': '/',
               'q': '?',
               't': '"',
               'l': '<',
               'r': '>',
               '^': '^'}

# Helper functions.
_packetSize   = lambda data: ord(data[0])
_packetType   = lambda data: ord(data[1])
_tinyType     = lambda data: ord(data[3])
_smallType    = lambda data: ord(data[3])
_eatNullChars = lambda str_: str_[:(str_.find('\0') if str_.find('\0') != -1 else None)]


def checkIP(ipStr):
    """Check if a string contains a valid IP address.

    Args:
        ipStr - The IP string (EG "127.0.0.1").

    Returns:
        True if the IP is correctly formated.

    """
    if ipStr.lower() == 'localhost':
        return True
    return re.match(_IP_PATTERN, ipStr)


def checkPort(portStr):
    """Check if a string contains a valid port number.

    Args:
        portStr - The port string (EG "29999").

    Returns:
        True if the port is a valid port number.

    """
    if portStr.isdigit():
        return int(portStr) >= MIN_PORT and int(portStr) <= MAX_PORT
    return False
    
    
def checkAdmin(adminStr):
    """Check if a string contains a valid admin password.

    Args:
        adminStr - The admin password str.

    Returns:
        True if the adminStr is a valid admin password.

    """    
    return len(adminStr) < ADMIN_LEN


def parseCommand(mso, prefix):
    """Parse a command string from an MSO message (EG '!admin DarkTimes') and
    return it as a tuple.

    Args:
        mso        - An IS_MSO packet, possibly containing a command.
        prefix     - The prefix (EG '!') the command begins with.

    Returns:
        A tuple, with the command as the first element, any following command
        string as the second element, or () if no command could be parsed.

    """
    msg = mso['Msg'][mso['TextStart']:]
    if msg.startswith(prefix):
        return msg[1:].split(' ', 1)
    return ()


def strToUnicode(str_, default='L', cols=True):
    """Convert a LFS encoded string to unicode.

    Args:
        str     - The LFS encoded string.
        default - The default encoding to start with (EG L, G, J etc..).
        cols    - Whether to keep colour codes in the string.

    Returns:
        A unicode string.

    """
    output = u''
    accum = ''
    codec = _ENCODING_MAP[default]
    ctrlChar = False
    for c in str_:
        if ctrlChar:
            if c in _ENCODING_MAP:
                codec = _ENCODING_MAP[c]
            elif c in _ESCAPE_MAP:
                accum += _ESCAPE_MAP[c]
            elif cols:
                accum += '^' + c
            ctrlChar = False
        else:
            if c == '^':
                ctrlChar = True
                output += accum.decode(codec)
                accum = ''
            else:
                accum += c
    output += accum.decode(codec)
    return output


def stripColours(lfsStr):
    """Strip colour codes (^1, ^3 etc..) from a string.

    Args:
        lfsStr - The string to strip.

    Returns:
        The string without colours.

    """
    return re.sub(re.compile('\^[0-9]'), '', lfsStr)


def speedToMps(speed):
    """Convert speed to meters per second.

    Args:
        speed - The speed to convert.

    Returns:
        Meters per second.

    """
    return speed / 327.68


def speedToKph(speed):
    """Convert speed to kilometers per hour.

    Args:
        speed - The speed to convert.

    Returns:
        Kilometers per hour.

    """
    return speed / 91.02
    
    
def mpsToKph(mps):
    return mps * 3.6


def speedToMph(speed):
    """Convert speed to miles per hour.

    Args:
        speed - The speed to convert.

    Returns:
        Miles per hour.

    """
    return speed / 146.486067
    
    
def mpsToMph(mps):
    return mps * 2.23
    
def metersToMiles(meters):
    return meters / 1609.344
    
def metersToKilometers(meters):
    return meters / 1000    

def directionToDegrees(direction):
    """Convert direction to degrees. 
    
    Args:
        direction - The direction to convert.
        
    Returns:
        Degrees ranging from 0 to 180.
    
    """
    return direction / 182.04

def lengthToMetres(length):
    """Convert length to meters.

    Args:
        length - The length in LFS world coordinate units.

    Returns:
        The length in meters.
        
    """
    return length / 65536.0
    
    
def distance(a = (0, 0, 0), b = (0, 0, 0)):
    """Calculate the distance between two points.
    
    Args:
        a = A tuple containing the points (X, Y, Z) coordinates.
        b = A tuple containing the points (X, Y, Z) coordinates.
        
    Returns:
        The distance between two points.
    
    """
    return math.sqrt((b[0] - a[0]) * (b[0] - a[0]) + (b[1] - a[1]) * 
                     (b[1] - a[1]) + (b[2] - a[2]) * (b[2] - a[2]))
    
    
def radiansToDegrees(radians):
    """Convert radians to degrees.
    
    Args:
        radians - The radians to convert.
        
    Returns:
        The radians as degrees.
    
    """
    return radians * (180.0 / math.pi);
    
def radiansToRpm(radians):
    """Convert radians to RPM.
    
    Args:
        radians - The radians per second to convert.
        
    Returns:
        The radians as RPM.
    
    """    
    return radiansPerSec * (60.0 / (2.0 * math.pi));

def packetXml(packetType, name=None, xmlPath=None, xmlStr=None):
    """Create a packet object from XML data. The following is an example of
    XML which this function will convert into a Packet object.

    <?xml version="1.0" encoding="utf-8"?>
    <pyinsim>
        <packet name="init">
            <ReqI>1</ReqI>
            <UDPPort>0</UDPPort>
            <Admin>pass</Admin>
            <IName>Pyinsim</IName>
            <Prefix>$</Prefix>
        </packet>
    </pyinsim>

    In order to create a Packet from this XML data, you would call the function
    with the following paramaters.

    isi = pyinsim.packetXml(pyinsim.ISP_ISI, "init", pathToXmlFile)

    Args:
        packetType   - The packet type from the ISP_* enumeration.
        name    - The name attribute of the packet in the XML.
        xmlPath - A path to an XML file or...
        xmlStr  - Alternatively a string of XML data.

    Returns:
        A packet object populated from the XML data.

    """
    # This function almost feels a little obsolete, it works well, but it
    # could be a lot better.
    if xmlPath is None:
        doc = minidom.parseString(xmlStr)
    else:
        doc = minidom.parse(xmlPath)

    packet = Packet(packetType)
    for baseNode in doc.getElementsByTagName('packet'):
        node = str(baseNode.getAttribute('name'))
        if node == name:
            for attribute in packet.iterkeys():
                for subNode in baseNode.getElementsByTagName(attribute):
                    data = subNode.firstChild.data
                    if type(packet[attribute]) == str:
                        packet[attribute] = str(data)
                    elif type(packet[attribute ]) == int:
                        packet[attribute] = int(data)
                    elif type(packet[attribute ]) == float:
                        packet[attribute] = float(data)
                    elif type(packet[attribute ]) == chr:
                        packet[attribute] = chr(data)
    return packet


def insimConnect(host='localhost', port=29999, **values):
    """This is a short-hand for initailising an InSim connection. It creates
    an Insim object, calls its connect() method, and sends an ISI packet.
    
    Example:
        insim = pyinsim.insimConnect('localhost', 29999, Admin='pass')
    
    Args:
        host   - The host to connect to
        port   - The port to connect to the host through.
        values - The values to initailise the ISI packet with.
        
    Returns:
        The initailised InSim object.
        
    """
    insim = InSim()    
    insim.connect(host, port)
    insim.send(ISP_ISI, **values)
    return insim
    
    
def timeStr(ms, hours=False):
    """Convert milliseconds into a formatted time string (EG h:m:s.t).
    
    Args:
        ms - The time in milliseconds.
        hours - Force the hours component of the time.
    
    """
    h = int(math.floor(ms / 3600000))
    m = int(math.floor(ms / 1000 / 60) % 60)
    s = int(math.floor(ms / 1000) % 60)
    t = int(ms % 1000)
    if h or hours:
        return '%d.%.2d:%.2d.%.3d' % (h, m, s, t)
    else:
        return '%d:%.2d.%.3d' % (m, s, t)   


class _Buffer:
    """Class to store the receive buffer data."""
    def __init__(self):
        """Initailise the buffer."""
        self.__buffer = ''

    def __len__(self):
        """Return the length of the buffer."""
        return len(self.__buffer)

    def append(self, data):
        """Append data to the buffer.

        Args:
            data - A binary formated string.

        """
        self.__buffer += data

    def packets(self):
        """Iterate through the packets currently in the buffer. Remove
        the next packet from the buffer and yield it.

        Returns:
            The next packet in the buffer.

        """
        while len(self) and len(self) >= _packetSize(self.__buffer):
            size = _packetSize(self.__buffer)
            packet = self.__buffer[:size]
            self.__buffer = self.__buffer[size:]
            yield packet


class InSim:
    """Class to manage an InSim connection with LFS.

    Manages the creation of an InSim connection, and provides methods and
    events for sending and recieving packets.

    """
    def __init__(self, name=None):
        """Init the InSim socket.
        
        Args:
            name - An optional name for the InSim connection.
        
        """
        self.__callbacks = {}
        self.__buffer = _Buffer()
        self.__sock = None
        self.__thread = None
        self.__connected = False
        self.__connLostCallbacks = []
        self.__threadErrorCallbacks = []
        self.__connectedCallbacks = []
        self.__name = name

    def getName(self):
        """Get the name of this InSim connection.
        
        Returns:
            The name.
        
        """
        return self.__name

    def isConnected(self):
        """Get if the socket is connected.

        Returns:
            True if the socket is connected.

        """
        return self.__connected

    def connect(self, host, port):
        """Connect to InSim and begin the packet receive thread.

        Args:
            host - The host IP address to connect to.
            port - The port to connect to the host through.

        """
        if not self.__connected:
            self.__sock = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
            self.__sock.connect((host, port))
            self.__connected = True
            self.__thread = threading.Thread(target=self.__receiveThread)            
            self.__thread.start()
            self.__raiseConnectedEvent()

    def close(self):
        """Close the InSim connection."""
        if self.__connected:
            self.__connected = False            
            self.send(ISP_TINY, SubT=TINY_CLOSE)
            self.__sock.close()

    def send(self, packetType, **values):
        """Send a packet to InSim.

        Example:

            InSim.send(pyinsim.ISP_MST, Msg='Hello, world!')

        Args:
            packetType    - The type of packet from the ISP_* enumeration.
            values - Default data to initailise the packet with.

        """
        self.sendB(Packet(packetType, **values).pack())

    def sendB(self, data):
        """Send binary data to InSim.

        Args:
            data - A binary packed string.

        """
        self.__sock.send(data)

    def sendP(self, *packets):
        """Send one or more packets to InSim.

        Args:
            packets - A list of packets to send.

        """
        [self.sendB(p.pack()) for p in packets]

    def __raiseConnectionLostEvent(self):
        """Raises connection lost event.

        """
        # TODO: Add disconnect reason enum to parameter (EG CONN_CLOSED_BY_USER,
        # CONN_CLOSED_BY_ERROR, CONN_CLOSED_AS_LFS_CLOSED etc..).
        [c(self) for c in self.__connLostCallbacks]

    def __raiseThreadErrorEvent(self, err):
        """Raises thread error event.

        Args:
            err - The error message.

        """
        [c(self, err) for c in self.__threadErrorCallbacks]

    def __raiseConnectedEvent(self):
        [c(self) for c in self.__connectedCallbacks]

    def __receiveThread(self, *args):
        """Thread to receive data from the socket.

        """
        try:
            while self.__connected:
                data = self.__sock.recv(_INSIM_BUFFER_SIZE)
                if data:
                    self.__buffer.append(data)
                    for packet in self.__buffer.packets():
                        self.__keepAlive(packet)
                        self.__raisePacketEvent(packet)
                else:
                    self.close()
                    self.__raiseConnectionLostEvent()
        except socket.error, err:
            self.close()
            self.__raiseThreadErrorEvent(err)

    def __keepAlive(self, data):
        """Respond to the InSim 'keep-alive' pulse.

        Args:
            data - The packet to check for a keepalive signal.

        """
        if _packetType(data) == ISP_TINY:
            if _tinyType(data) == TINY_NONE:
                self.sendB(data)

    def __raisePacketEvent(self, data):
        """Raise a packet event.

        Args:
            data - The packet data, as a binary formatted string.

        """
        packetType = _packetType(data)
        if self.isBound(packetType) or self.isBound(ISP_ALL):
            if packetType in (ISP_MCI, ISP_NLP):
                packet = _CarTrackPacket(data)
            else:
                packet = Packet(data=data)
            if self.isBound(packetType):
                [c(self, packet) for c in self.__callbacks[packetType]]
            if self.isBound(ISP_ALL):
                [c(self, packet) for c in self.__callbacks[ISP_ALL]]

    def bind(self, packetType, callback):
        """Bind a packet callback event-handler.
        
        Args:
            packetType    - The packet from the ISP_* enumeration.
            callback - The function to call once the packet has been received
        
        """
        if packetType not in self.__callbacks:
            self.__callbacks[packetType] = []
        if callback not in self.__callbacks[packetType]:
            self.__callbacks[packetType].append(callback)
        
    def unbind(self, packetType, callback=None):
        """Unbind a packet callback event-handler.
        
        Args:
            packetType    - The packet from the ISP_* enumeration.
            callback - The function to unbind from this event.
            
        """
        if callback:
            self.__callbacks[packetType].remove(callback)
        else:
            del self.__callbacks[packetType]
            
    def isBound(self, packetType, callback=None):
        """Check if a packet event-handler has been bound.
        
        Args:
            packetType    - The type of packet from the ISP_* enumeration.
            callback - The function check for this event.
        
        Returns:
            True if the packet is bound.
           
        """     
        if callback:
            return callback in self.__callbacks[packetType]
        return packetType in self.__callbacks

    def run(self):
        """Blocks the calling thread until pyinsim's internal receive thread
        has ended. This usful for stopping a console application from
        closing while the connection is still active.

        """
        self.__thread.join()

    def bindConnectionLost(self, callback):
        """Bind a callback to be triggered when the InSim connection is lost

        Args:
            callback - The function to call.

        """
        if callback not in self.__connLostCallbacks:
            self.__connLostCallbacks.append(callback)
        
    def unbindConnectionLost(self, callback=None):
        """Unbind a connection lost callback. If no callback is passed, then
        all will be unbound.

        Args:
            callback - The function to unbind.

        """
        if callback:
            self.__connLostCallbacks.remove(callback)
        else:
            self.__connLostCallbacks = []

    def bindThreadError(self, callback):
        """Bind a callback to be triggered when the receive thread throws an
        error.

        Args:
            callback - The function to call.

        """
        if callback not in self.__threadErrorCallbacks:
            self.__threadErrorCallbacks.append(callback)
        
    def unbindThreadError(self, callback=None):
        """Unbind a thread error callback. If no callback is passed, then
        all will be unbound.

        Args:
            callback - The function to unbind.

        """        
        if callback:
            self.__threadErrorCallbacks.remove(callback)
        else:
            self.__threadErrorCallbacks = []
            
    def bindConnected(self, callback):
        if callback not in self.__connectedCallbacks:
            self.__connectedCallbacks.append(callback)
            
    def unbindConnected(self, callback=None):
        if callback:
            self.__connectedCallbacks.remove(callback)
        else:
            self.__connectedCallbacks = []
        
    def raisePacketEvent(self, *packets):
        """Raise a packet event.
        
        Args:
            packets - A list of packets to raise events for.
        
        """
        [self.__raisePacketEvent(p.pack()) for p in packets]


class Packet(UserDict):
    """Class to handle an InSim packet. This is a generic Packet handling class
    which handles all packets, except for ISP_MCI and ISP_NLP, which are
    handled by _CarTrackPacket.

    A Packet is a Python dictionary, and the packet attributes are set and
    accessed by using the dictionary keys.

    """
    def __init__(self, packetType=None, data=None, **values):
        """Init the packet.

        To create a new Packet to send to InSim, set the packetType. For packets
        being received from InSim, set the data.

        Example (sending):
            mst = pyinsim.Packet(pyinsim.ISP_MST, Msg='Hello, world!')

        Example (receiving):
            packet = pyinsim.Packet(data=receivedData)

        Args:
            packetType - The type of packet to create.
            data       - A binary string containing packet data to unpack.
            values     - Default data to initailise the packet with.

        """
        self.__packStr = ''
        if data:          
            self.__decodePacket(data, values)
        else:
            self.__createPacket(packetType, values)
        UserDict.__init__(self, values)

    def __createPacket(self, packetType, values):
        """Create new packet. 

        """
        for k, v in _PACKET_DEFS[packetType]:
            self.__packStr += v
            if k not in values:
                values[k] = _TYPE_DEFAULTS[v[-1:]]
        values['Size'] = struct.calcsize(self.__packStr)
        values['Type'] = packetType

    def __decodePacket(self, data, values):
        """Decode packet from binary formatted string. 

        """
        packetType = _packetType(data)          
        for k, v in _PACKET_DEFS[packetType]:
            self.__packStr += v
        data = struct.unpack(self.__packStr, data)

        for i in range(len(_PACKET_DEFS[packetType])):
            k = _PACKET_DEFS[packetType][i][0]
            if isinstance(data[i], str):
                values[k] = _eatNullChars(data[i])
            else:
                values[k] = data[i]

    def pack(self):
        """Pack the current packet values into a binary string, which can then
        be sent to InSim.

        Returns:
          A binary formatted string.

        """
        values = []
        for packetDef in _PACKET_DEFS[self['Type']]:
            values.append(self[packetDef[0]])
        return struct.pack(self.__packStr, *values)        
   

class _OutSocket:
    """Class to manage UDP connections, such as OutGauge and OutSim.
    
    This is an abstract method, don't use it directly. See OutGauge and 
    OutSim to see how to inherit from it.
    
    """
    def __init__(self, timeout, name):
        """Init the _OutSocket.
        
        Args:
            timeout - The time to wait for a packet to arrive before timing out.
            name    - An optinal name for the connection.
            
        """
        self.__thread = threading.Thread(target=self.__receiveThread)
        self.__connected = False
        self.__callbacks = []
        self.__errorCallback = None
        self.__timeoutCallback = None
        self.__name = name
        self.__timeout = timeout
        
    def isConnected(self):
        """Get if the socket is connected.
        
        Returns:
            True if the socket is connected.
        
        """
        return self.__connected
        
    def getName(self):
        """Get the name of the connection.
        
        Returns:
            The name.
        
        """
        return self.__name
        
    def connect(self, host, port):
        """Connect to the specified host and port and begin the packet
        listener thread.
        
        Args:
            host - The host to connect to.
            port - The port to connect to the host through.
        
        """
        self.__sock = socket.socket(socket.AF_INET, socket.SOCK_DGRAM)
        self.__sock.settimeout(self.__timeout)        
        self.__sock.bind((host, port))
        self.__connected = True
        self.__thread.start()
        
    def close(self):
        """Close the connection.
        
        """
        self.__connected = False        
        self.__sock.close()   
        
    def __receiveThread(self):
        """UDP receive thread.
        
        """
        try:
            while self.__connected:
                data = self.__sock.recv(_INSIM_BUFFER_SIZE)
                if data:
                    self.__raisePacketEvent(data)
                else:
                    self.close()
        except Exception, err:
            self.close()
            if err[0] == 'timed out':
                self.__raiseTimeoutEvent(self.__timeout)
            else:
                self.__raiseErrorEvent(err)
        
    def __raisePacketEvent(self, data):
        """Raise a packet event.
        
        """
        if len(self.__callbacks):
            packet = self._getPacket(data)
            [c(self, packet) for c in self.__callbacks]
        
    def _getPacket(self, data):
        """Virtual method to get the correct packet.
        
        """
        pass
        
    def run(self):
        """Blocks the calling thread until pyinsim's internal receive thread
        has ended. This usful for stopping a console application from
        closing while the connection is still active.

        """
        self.__thread.join()
        
    def bind(self, callback):
        """Bind a packet callback event handler.
        
        Args:
            The function or method to call once a packet has been received.
        
        """
        if not self.isBound(callback):
            self.__callbacks.append(callback)
            
    def unbind(self, callback):
        """Unbind a packet callback event handler.
        
        Args:
            The function or method to remove.
        
        """
        if self.isBound(callback):
            self.__callbacks.remove(callback)
        
    def isBound(self, callback):
        """Check if an event-handler has been bound.
        
        Args:
            callback - The function to check.
        
        """
        return callback in self.__callbacks
        
    def __raiseErrorEvent(self, err):
        """Raises a thread error event."""
        if self.__errorCallback is not None:
            self.__errorCallback(self, err)
        
    def bindError(self, callback):
        """Bind a event-handler to be called if the internal receive thread
        encounters an error.
        
        Args:
            callback - The function to call.
        
        """
        self.__errorCallback = callback
        
    def __raiseTimeoutEvent(self, timeout):
        """Raise timeout event.
        
        """
        if self.__timeoutCallback is not None:
            self.__timeoutCallback(self, timeout)
        
    def bindTimeout(self, callback):
        """Bind a event-handler to be called if the connection times out.
        
        Args:
            callback - The function to call.
            
        """
        self.__timeoutCallback = callback
        

class OutGauge(_OutSocket):
    """Class to handle an OutGauge UDP connection with LFS.
    
    """
    def __init__(self, timeout=30.0, name=None):
        """Init the OutGauge object.
        
        Args:
            timeout - The number of seconds before the connection timesout.
            name    - An optional name for the OutGauge connection.
        
        """
        _OutSocket.__init__(self, timeout, name)
        
    def _getPacket(self, data):
        """Overwritten method to get the OutGauge packet.
        
        """
        return _OutGaugePacket(data)
               

class OutSim(_OutSocket):
    """Class to handle an OutSim UDP connection with LFS.
    
    """    
    def __init__(self, timeout=30.0, name=None):
        """Init the OutSim object.
        
        Args:
            timeout - The number of seconds before the connection timesout.
            name    - An optional name for the OutSim connection.
        
        """        
        _OutSocket.__init__(self, timeout, name)

    def _getPacket(self, data):
        """Overwritten method to get the OutSim packet.
        
        """        
        return _OutSimPacket(data)
 
 
class _CarTrackPacket(UserDict):
    """Class to handle car tracking packets, MCI and NLP.
    
    These packets need to be handled by a seperate class, as they both 
    contain variable sized arrays.
    
    """    
    def __init__(self, data):
        """Init the _CarTrackPacket
        
        Args:
            data - The packet data as a binary formatted string.
        
        """
        UserDict.__init__(self)    
        self.__data = data    
        
        # Unpack the header.
        headerPack = 'BBBB'
        offset = struct.calcsize(headerPack)     
        header = struct.unpack(headerPack, data[:offset])
        self['Size'] = header[0]
        self['Type'] = header[1]
        self['ReqI'] = header[2]
        
        packetType = _packetType(data)
        if packetType == ISP_NLP:
            self['NumP'] = header[3]
            self.__createNlp(data, offset)
        elif packetType == ISP_MCI:
            self['NumC'] = header[3]
            self.__createMci(data, offset)
        
    def __createNlp(self, data, offset):
        """Unpack an NLP packet.
        
        """
        self['NodeLaps'] = []        
        for i in range(self['NumP']):
            pack = 'HHBB'
            size = struct.calcsize(pack)                
            sub = struct.unpack(pack, data[offset:offset+size])
            nodelap = {}
            nodelap['Node'] = sub[0]
            nodelap['Lap'] = sub[1]
            nodelap['PLID'] = sub[2]
            nodelap['Position'] = sub[3]
            self['NodeLaps'].append(nodelap)
            offset += size       
    
    def __createMci(self, data, offset): 
        """Unpack an MCI packet.
        
        """
        self['CompCars'] = []
        for i in range(self['NumC']):              
            pack = 'HHBBBBiiiHHHh'
            size = struct.calcsize(pack)             
            sub = struct.unpack(pack, data[offset:offset+size])
            compcar = {}
            compcar['Node'] = sub[0]
            compcar['Lap'] = sub[1]
            compcar['PLID'] = sub[2]
            compcar['Position'] = sub[3]
            compcar['Info'] = sub[4]
            compcar['Sp3'] = sub[5]
            compcar['X'] = sub[6]
            compcar['Y'] = sub[7]
            compcar['Z'] = sub[8]
            compcar['Speed'] = sub[9]
            compcar['Direction'] = sub[10]
            compcar['Heading'] = sub[11]
            compcar['AngVel'] = sub[12]
            self['CompCars'].append(compcar)
            offset += size   
                 
    def pack(self):
        """Pack the current packet values into a binary string.

        Returns:
            A binary formatted string.

        """        
        return self.__data
        
 
class _OutGaugePacket(UserDict):
    """Class to handle an OutGauge packet.
    
    """
    def __init__(self, data):
        """Init the _OutGaugePacket.
        
        Args:
            data - The packet data as a binary formatted string.
        
        """
        UserDict.__init__(self)    
        self.__data = data    
        if len(data) == 96:
            data = struct.unpack('I4sHBBffffffffffff16s16si', data)
        elif len(data) == 92:
            data = struct.unpack('I4sHBBffffffffffff16s16s', data)
        self['Time'] = data[0]
        self['Car'] = _eatNullChars(data[1])
        self['Flags'] = data[2]
        self['Gear'] = data[3]
        self['SpareB'] = data[4]
        self['Speed'] = data[5]
        self['RPM'] = data[6]
        self['Turbo'] = data[7]
        self['EngTemp'] = data[8]
        self['Fuel'] = data[9]
        self['OilPress'] = data[10]
        self['Spare1'] = data[11]
        self['Spare2'] = data[12]
        self['Spare3'] = data[13]
        self['Throttle'] = data[14]
        self['Brake'] = data[15]
        self['Clutch'] = data[16]
        self['Display1'] = _eatNullChars(data[17])
        self['Display2'] = _eatNullChars(data[18])      
        if len(data) == 20:
            self['ID'] = data[19]

    def pack(self):
        """Pack the current packet values into a binary string.

        Returns:
            A binary formatted string.

        """        
        return self.__data


class _OutSimPacket(UserDict):
    """Class to handle an OutSim packet.
    
    """
    def __init__(self, data):
        """Init the _OutSimPacket.
        
        Args:
            data - The packet data as a binary formatted string.
        
        """
        UserDict.__init__(self)
        self.__data = data  
        if len(data) == 68:
            data = struct.unpack('Iffffffffffffiiii', data)
        elif len(data) == 64:
            data = struct.unpack('Iffffffffffffiii', data)
        self['Time'] = data[0]
        self['AngVel'] = (data[1], data[2], data[3])
        self['Heading'] = data[4]
        self['Pitch'] = data[5]
        self['Roll'] = data[6]
        self['Accel'] = (data[7], data[8], data[9])
        self['Vel'] = (data[10], data[11], data[12])
        self['Pos'] = (data[13], data[14], data[15])
        if len(data) == 17:
            self['ID'] = data[16]
            
    def pack(self):
        """Pack the current packet values into a binary string.

        Returns:
            A binary formatted string.

        """        
        return self.__data           
        




_HEADER = 'LFSPRF'
_VERSION = 1
_HEADER_STR = '6sBB'
_HEADER_SIZE = struct.calcsize(_HEADER_STR)


class LogError(Exception):
    pass
    

class LogWriter:
    def __init__(self, insim, addTimeStamps=False, dir=None, exe='race'):
        self.isLogging = False
        self.file = None
        self.addTimeStamps = addTimeStamps
        self.dir = dir
        self.exe = exe
        self.types = []
        self.path = None
        self.raceLoggedCallback = None
        self.logStartedCallback = None
        insim.bind(ISP_ALL, self.logPacket)
        insim.bindThreadError(self.threadError)
        insim.bindConnectionLost(self.connectionLost)    

    def bind(self, type_):
        if type_ not in self.types:
            self.types.append(type_)

    def unbind(self, type_):
        if type_ in self.types:
            self.types.remove(type_)

    def requestConnections(self, insim):
        insim.send(ISP_TINY, ReqI=1, SubT=TINY_NCN)

    def requestPlayers(self, insim):
        insim.send(ISP_TINY, ReqI=1, SubT=TINY_NPL)
        
    def requestHostInfo(self, insim):
        insim.send(ISP_TINY, ReqI=1, SubT=TINY_ISM)

    def logPacket(self, insim, packet):
        if packet['Type'] == ISP_RST:
            if self.isLogging:
                self.stopLogging()
            self.startLogging(self.createFileName(packet['Track']))
            self.requestHostInfo(insim)
            self.requestConnections(insim)
            self.requestPlayers(insim)
            self.writePacket(packet)
        elif packet['Type'] == ISP_TINY:
            if packet['SubT'] == TINY_REN:
                self.stopLogging()
        elif packet['Type'] in self.types:
            self.writePacket(packet)

    def startLogging(self, path):    
        if self.dir is not None:
            path = os.path.join(self.dir, path)
        self.file = open(path, 'wb')
        self.file.write(struct.pack(_HEADER_STR,
                                      _HEADER, _VERSION,
                                      int(self.addTimeStamps)))
        self.path = path
        self.isLogging = True
        if self.logStartedCallback is not None:
            self.logStartedCallback(self, path)

    def writePacket(self, packet):
        if self.isLogging:
            if self.addTimeStamps:
                self.file.write(str(int(time.time())))
            self.file.write(packet.pack())

    def stopLogging(self):
        if self.isLogging and self.file:
            self.writePacket(Packet(ISP_TINY, SubT=TINY_REN))
            self.file.close()
            self.isLogging = False
            if self.raceLoggedCallback is not None:
                self.raceLoggedCallback(self, self.path)

    def createFileName(self, track):
         return '%s_%s.%s' % (track,
                             datetime.datetime.now().strftime('%d%m%y_%H%M%S'),
                             self.exe)

    def threadError(self, insim, err):
        self.stopLogging()

    def connectionLost(self, insim):
        self.stopLogging()
        
    def bindLogStarted(self, callback):
        self.logStartedCallback = callback
        
    def bindLogEnded(self, callback):
        self.raceLoggedCallback = callback


class LogReader:
    def __init__(self):
        self.callbacks = {}
        self.logReadCallback = None

    def bind(self, type_, callback):
        self.callbacks[type_] = callback

    def unbind(self, type_):
        del self.callbacks[type_]

    def isBound(self, type_):
        return type_ in self.callbacks
        
    def bindLogRead(self, callback):
        self.logReadCallback = callback

    def peekByte(self, f):
        pos = f.tell()
        peek = ord(f.read(1))
        f.seek(pos)
        return peek

    def start(self, path):
        with open(path, 'rb') as f:
            data = struct.unpack(_HEADER_STR, f.read(_HEADER_SIZE))
            if data[0] != _HEADER:
                raise LogError('Invalid header')
            if data[1] != _VERSION:
                raise LogError('Invalid version')
            addTimeStamps = data[2] == 1

            while f.tell() < (os.path.getsize(path) - _HEADER_SIZE):
                size = self.peekByte(f)
                data = f.read(size)
                type_ = _packetType(data)
                if type_ in self.callbacks:
                    if type_ in (ISP_MCI, ISP_NLP):
                        packet = _CarTrackPacket(data)
                    else:
                        packet = Packet(type, data=data)
                    self.callbacks[type_](self, packet)
                    
            if self.logReadCallback is not None:
                self.logReadCallback(self, path)








if __name__ == '__main__':
    pass
