#
# This file is part of AutoMessage Redux.
# 
# AutoMessage Redux is free software: you can redistribute it and/or modify it 
# under the terms of the GNU General Public License as published by the Free 
# Software Foundation, either version 3 of the License, or (at your option) any 
# later version.
#
# AutoMessage Redux is distributed in the hope that it will be useful, but 
# WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or 
# FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more 
# details.
#
# You should have received a copy of the GNU General Public License along with 
# AutoMessage Redux. If not, see <http://www.gnu.org/licenses/>.
#

"""The Messenger class handles the sending and receiving of messages between
AutoMessage and InSim."""

# Dependencies.
import re
import time

# Libraries
import pyinsim
import info

# Constants.
HOST_ID = 0
REQI = 1
   
    
class Messenger:
    def __init__(self, insim, cfg):
        self.cfg = cfg
        insim.bind(pyinsim.ISP_NCN, self.onConnectionJoined)
        insim.bind(pyinsim.ISP_CNL, self.onConnectionLeft)
        insim.bind(pyinsim.ISP_NPL, self.onPlayerJoined)
        insim.bind(pyinsim.ISP_PLL, self.onPlayerLeft)
        insim.bind(pyinsim.ISP_ISM, self.onJoinedMultiplayer)
        insim.bind(pyinsim.ISP_RST, self.onRaceStart)
        insim.bind(pyinsim.ISP_TINY, self.onTinyReceived)
        insim.bind(pyinsim.ISP_MSO, self.onMessageReceived)    
        self.connections = {}
        self.players = {}        
        self.justJoined = True
        self.host = ''
        self.lastMessage = 0    
        
    def getConnection(self, packet):
        return self.connections[packet['UCID']]
        
    def getPlayer(self, packet):
        return self.players[packet['PLID']]    
        
    def onConnectionJoined(self, insim, ncn):
        self.connections[ncn['UCID']] = ncn
        if ncn['UCID'] != HOST_ID and ncn['ReqI'] != REQI:
            self.sendMessage(insim, self.cfg.playerConnects, ncn)
        
    def onConnectionLeft(self, insim, cnl):
        del self.connections[cnl['UCID']]
        
    def onPlayerJoined(self, insim, npl):
        self.players[npl['PLID']] = npl
        
    def onPlayerLeft(self, insim, pll):
        del self.players[pll['PLID']]
        
    def requestConnections(self, insim):
        insim.send(pyinsim.ISP_TINY, ReqI=REQI, SubT=pyinsim.TINY_NCN)
        
    def requestPlayers(self, insim):        
        insim.send(pyinsim.ISP_TINY, ReqI=REQI, SubT=pyinsim.TINY_NPL)
        
    def requestMultiplayer(self, insim):
        insim.send(pyinsim.ISP_TINY, ReqI=REQI, SubT=pyinsim.TINY_ISM)
        
    def requestInfo(self, insim):
        self.requestConnections(insim)
        self.requestPlayers(insim)
        self.requestMultiplayer(insim)
        
    def onJoinedMultiplayer(self, insim, ism):
        self.host = ism['HName']
        if ism['ReqI'] != REQI:
            self.sendMessage(insim, self.cfg.youConnect)
            self.requestConnections(insim)
            self.requestPlayers(insim)
        
    def onRaceStart(self, insim, rst):
        if self.justJoined:
            self.justJoined = False
        else:
            self.sendMessage(insim, self.cfg.raceStarts)
            
    def onTinyReceived(self, insim, tiny):
        if tiny['SubT'] == pyinsim.TINY_REN:
            self.sendMessage(insim, self.cfg.raceEnds)
        elif tiny['SubT'] == pyinsim.TINY_MPE:
            self.justJoined = False
            
    def onMessageReceived(self, insim, mso):
        if mso['UCID'] != HOST_ID and mso['UserType'] == pyinsim.MSO_USER:    
            messages = re.split('\W+', mso['Msg'][mso['TextStart']:].lower())
            for words, message in self.cfg.messages:
                tokens = words.lower().split(';')
                for token in tokens:
                    if token in messages:
                        self.sendMessage(insim, message, mso)
                        break
#        elif mso['UCID'] == HOST_ID and mso['UserType'] == pyinsim.MSO_SYSTEM:
#            if mso['Msg'].startswith('Fastest lap : '):
#                self.sendMessage(self.cfg.fastestLap)        
#            elif re.match('LFSW - new (\w+) PB by', mso['Msg']) is not None:
#                self.sendMessage(self.cfg.personalBest) 

    def parseMessage(self, message, packet):
        if packet:
            ncn = self.getConnection(packet)
            if '{player}' in message:
                    message = message.replace('{player}', ncn['PName'])
            if '{user}' in message:
                message = message.replace('{user}', ncn['UName'])
        if self.host and '{host}' in message:
            message = message.replace('{host}', self.host) 
        return message

    def sendMessage(self, insim, message, packet=None):
        if (time.time() - self.lastMessage) > self.cfg.interval:
            self.lastMessage = time.time()                            
            message = self.parseMessage(message, packet) 
            if info.DEBUG_MESSAGES:                    
                print message
            else:
                insim.send(pyinsim.ISP_MSX, Msg=message)    
                        
                        
if __name__ == '__main__':
    pass
    
                        