#
# This file is part of AutoMessage Redux.
# 
# AutoMessage Redux is free software: you can redistribute it and/or modify it 
# under the terms of the GNU General Public License as published by the Free 
# Software Foundation, either version 3 of the License, or (at your option) any 
# later version.
#
# AutoMessage Redux is distributed in the hope that it will be useful, but 
# WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or 
# FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more 
# details.
#
# You should have received a copy of the GNU General Public License along with 
# AutoMessage Redux. If not, see <http://www.gnu.org/licenses/>.
#

"""The MainFrame window of the program, which contains the main GUI."""

# Dependencies
import wx
import wx.lib.mixins.listctrl
import os

# Libraries
import info
import config
import menu
import dialog
import pyinsim
import messenger

MAX_MSG_LEN = 96

class ManagedListView(wx.ListView, wx.lib.mixins.listctrl.ListCtrlAutoWidthMixin):
    """Simple inherited ListView ctrl to support auto-column widths."""
    def __init__(self, *args, **kwds):
        wx.ListView.__init__(self, *args, **kwds)
        wx.lib.mixins.listctrl.ListCtrlAutoWidthMixin.__init__(self)
        

class MainFrame(wx.Frame):
    def __init__(self, *args, **kwds):
        wx.Frame.__init__(self, *args, **kwds)
        
        # Load config.
        self.cfg = config.load(info.CONFIG_PATH)
        
        # Initailise frame.
        self.SetIcon(wx.Icon(info.ICON_MAIN, wx.BITMAP_TYPE_ICO))
        self.taskBarIcon = menu.MainTaskBarIcon()
        self.menu = menu.MainMenuBar()
        self.SetMenuBar(self.menu)
        self.CreateStatusBar()
        self.SetStatusText('Ready')
        self.isClosing = False   
        self.SetBackgroundColour(wx.Color(255, 255, 255))           
        
        # Initailise layout.     
        sizer = wx.BoxSizer(wx.VERTICAL)
        self.SetSizer(sizer)
        
        # Messages section layout.        
        messagesSizer = wx.StaticBoxSizer(wx.StaticBox(self, wx.ID_ANY, 'Send message when'), wx.HORIZONTAL)
        sizer.Add(messagesSizer, 0, wx.EXPAND | wx.ALL, 7)         
        msgFlexGridSizer = wx.FlexGridSizer(4, 2)
        messagesSizer.Add(msgFlexGridSizer, 1, wx.EXPAND)
        
        msgFlexGridSizer.Add(wx.StaticText(self, wx.ID_ANY, 'You connect:'), 0, wx.TOP | wx.LEFT | wx.ALIGN_CENTER_VERTICAL, 7)
        youConnectTextCtrl = wx.TextCtrl(self, wx.ID_ANY, self.cfg.youConnect, size=(370, -1))
        youConnectTextCtrl.SetToolTipString('Message sent when you join a server')
        youConnectTextCtrl.SetMaxLength(MAX_MSG_LEN)
        msgFlexGridSizer.Add(youConnectTextCtrl, 0, wx.TOP | wx.LEFT | wx.ALIGN_CENTER_VERTICAL, 7)
        
        msgFlexGridSizer.Add(wx.StaticText(self, wx.ID_ANY, 'Player connects:'), 0, wx.TOP | wx.LEFT | wx.ALIGN_CENTER_VERTICAL, 7)
        playerConnectsTextCtrl = wx.TextCtrl(self, wx.ID_ANY, self.cfg.playerConnects)
        playerConnectsTextCtrl.SetToolTipString('Message sent when another player joins the server')
        playerConnectsTextCtrl.SetMaxLength(MAX_MSG_LEN)
        msgFlexGridSizer.Add(playerConnectsTextCtrl, 0, wx.EXPAND | wx.TOP | wx.LEFT | wx.ALIGN_CENTER_VERTICAL, 7)
        
        msgFlexGridSizer.Add(wx.StaticText(self, wx.ID_ANY, 'Race (re)starts:'), 0, wx.TOP | wx.LEFT | wx.ALIGN_CENTER_VERTICAL, 7)
        raceStartsTextCtrl = wx.TextCtrl(self, wx.ID_ANY, self.cfg.raceStarts)
        raceStartsTextCtrl.SetToolTipString('Message sent when a race starts or restarts')
        raceStartsTextCtrl.SetMaxLength(MAX_MSG_LEN)
        msgFlexGridSizer.Add(raceStartsTextCtrl, 0, wx.EXPAND | wx.TOP | wx.LEFT | wx.ALIGN_CENTER_VERTICAL, 7)
        
        msgFlexGridSizer.Add(wx.StaticText(self, wx.ID_ANY, 'Race ends:'), 0, wx.TOP | wx.LEFT | wx.ALIGN_CENTER_VERTICAL, 7)
        raceEndsTextCtrl = wx.TextCtrl(self, wx.ID_ANY, self.cfg.raceEnds)
        raceEndsTextCtrl.SetToolTipString('Message sent when a race ends (goes to track selection screen)')
        raceEndsTextCtrl.SetMaxLength(MAX_MSG_LEN)
        msgFlexGridSizer.Add(raceEndsTextCtrl, 0, wx.EXPAND | wx.TOP | wx.LEFT | wx.BOTTOM | wx.ALIGN_CENTER_VERTICAL, 7)
        
# TODO: Add messages for Fastest Lap and Personal best. Work in progress.
#        msgFlexGridSizer.Add(wx.StaticText(self, wx.ID_ANY, 'Fastest lap:'), 0, wx.TOP | wx.LEFT | wx.ALIGN_CENTER_VERTICAL, 7)
#        fastestLapTextCtrl = wx.TextCtrl(self, wx.ID_ANY, self.cfg.fastestLap)
#        fastestLapTextCtrl.SetToolTipString('Message sent when a players drives the fastest lap of a race')
#        msgFlexGridSizer.Add(fastestLapTextCtrl, 0, wx.EXPAND | wx.TOP | wx.LEFT | wx.ALIGN_CENTER_VERTICAL, 7)
#        
#        msgFlexGridSizer.Add(wx.StaticText(self, wx.ID_ANY, 'PB driven:'), 0, wx.TOP | wx.LEFT | wx.ALIGN_CENTER_VERTICAL, 7)
#        personalBestTextCtrl = wx.TextCtrl(self, wx.ID_ANY, self.cfg.personalBest)
#        personalBestTextCtrl.SetToolTipString('Message sent when a players drives a new PB (personal best) time')
#        msgFlexGridSizer.Add(personalBestTextCtrl, 0, wx.EXPAND | wx.TOP | wx.LEFT | wx.BOTTOM | wx.ALIGN_CENTER_VERTICAL, 7)        
        
        # Words section layout.
        wordsSizer = wx.StaticBoxSizer(wx.StaticBox(self, wx.ID_ANY, 'Words appear in chat'), wx.HORIZONTAL)
        sizer.Add(wordsSizer, 0, wx.EXPAND | wx.LEFT | wx.RIGHT | wx.BOTTOM, 7)         
    
        wordsBoxSizer = wx.BoxSizer(wx.VERTICAL)
        wordsSizer.Add(wordsBoxSizer, 1, wx.EXPAND)
        
        wordCtrlsBoxSizer = wx.BoxSizer(wx.HORIZONTAL)
        wordsBoxSizer.Add(wordCtrlsBoxSizer)
        wordCtrlsBoxSizer.Add(wx.StaticText(self, wx.ID_ANY, 'Words:'), 0, wx.TOP | wx.LEFT | wx.ALIGN_CENTER_VERTICAL, 7)
        self.wordsTextCtrl = wx.TextCtrl(self, wx.ID_ANY, size=(165, -1))
        self.wordsTextCtrl.SetToolTipString('List of words seperated by semi-colons')
        wordCtrlsBoxSizer.Add(self.wordsTextCtrl, 0, wx.TOP | wx.LEFT | wx.ALIGN_CENTER_VERTICAL, 7)
        wordCtrlsBoxSizer.Add(wx.StaticText(self, wx.ID_ANY, 'Message:'), 0, wx.TOP | wx.LEFT | wx.ALIGN_CENTER_VERTICAL, 7)
        self.messageTextCtrl = wx.TextCtrl(self, wx.ID_ANY, size=(165, -1))
        self.messageTextCtrl.SetToolTipString('Message when the words are detected')
        self.messageTextCtrl.SetMaxLength(MAX_MSG_LEN)
        wordCtrlsBoxSizer.Add(self.messageTextCtrl, 0, wx.TOP | wx.LEFT | wx.ALIGN_CENTER_VERTICAL, 7)
        self.addButton = wx.BitmapButton(self, wx.ID_ANY, wx.Bitmap(info.ICON_ADD), style=wx.NO_BORDER)
        self.addButton.SetToolTipString('Add new message')
        wordCtrlsBoxSizer.Add(self.addButton, 0, wx.TOP | wx.LEFT | wx.RIGHT | wx.ALIGN_CENTER_VERTICAL, 7)
    
        self.wordsListView = ManagedListView(self, wx.ID_ANY, size=(-1, 200))
        self.wordsListView.InsertColumn(0, 'Words', width=220)
        self.wordsListView.InsertColumn(1, 'Messages')
        for words, message in self.cfg.messages:
            item = self.wordsListView.InsertStringItem(self.wordsListView.GetItemCount(), words)
            self.wordsListView.SetStringItem(item, 1, message)
        wordsBoxSizer.Add(self.wordsListView, 1, wx.EXPAND | wx.ALL, 7)
    
        wordsEditBoxSizer = wx.BoxSizer(wx.HORIZONTAL)
        wordsBoxSizer.Add(wordsEditBoxSizer)
        self.editButton = wx.BitmapButton(self, wx.ID_ANY, wx.Bitmap(info.ICON_EDIT), style=wx.NO_BORDER)
        self.editButton.SetToolTipString('Edit the currently selected message')
        wordsEditBoxSizer.Add(self.editButton, 0, wx.LEFT | wx.BOTTOM | wx.ALIGN_CENTER_VERTICAL, 7)
        self.removeButton = wx.BitmapButton(self, wx.ID_ANY, wx.Bitmap(info.ICON_REMOVE), style=wx.NO_BORDER)
        self.removeButton.SetToolTipString('Remove the currently selected message')
        wordsEditBoxSizer.Add(self.removeButton, 0, wx.LEFT | wx.BOTTOM | wx.ALIGN_CENTER_VERTICAL, 7)
    
        # Initailise InSim.
        self.insim = pyinsim.InSim()
        self.insim.bindConnectionLost(self.onInSimConnectionLost)
        self.insim.bindThreadError(self.onInSimThreadError)
        self.insim.bind(pyinsim.ISP_VER, self.onInSimVersion)
        self.messenger = messenger.Messenger(self.insim, self.cfg)
        
        # Bind events.
        self.Bind(wx.EVT_CLOSE, self.onClose)
        self.Bind(wx.EVT_ICONIZE, self.onIconize)
        self.Bind(wx.EVT_MENU, self.onExit, id=info.ID_EXIT)
        self.Bind(wx.EVT_MENU, self.onOptions, id=info.ID_OPTIONS)
        self.Bind(wx.EVT_MENU, self.onConnect, id=info.ID_CONNECT)
        self.Bind(wx.EVT_MENU, self.onDisconnect, id=info.ID_DISCONNECT)
        self.Bind(wx.EVT_MENU, self.onLaunch, id=info.ID_LAUNCH)
        self.Bind(wx.EVT_MENU, self.onAbout, id=info.ID_ABOUT)
        self.Bind(wx.EVT_MENU, self.onEdit, id=info.ID_EDIT)
        self.Bind(wx.EVT_MENU, self.onRemove, id=info.ID_REMOVE)
        self.taskBarIcon.Bind(wx.EVT_MENU, self.onRestore, id=info.ID_RESTORE)
        self.taskBarIcon.Bind(wx.EVT_MENU, self.onExit, id=info.ID_EXIT)        
        youConnectTextCtrl.Bind(wx.EVT_TEXT, self.onYouConnect)
        playerConnectsTextCtrl.Bind(wx.EVT_TEXT, self.onPlayerConnects)
        raceStartsTextCtrl.Bind(wx.EVT_TEXT, self.onRaceStarts)
        raceEndsTextCtrl.Bind(wx.EVT_TEXT, self.onRaceEnds)
#        fastestLapTextCtrl.Bind(wx.EVT_TEXT, self.onFastestLap)
#        personalBestTextCtrl.Bind(wx.EVT_TEXT, self.onPersonalBest)
        self.addButton.Bind(wx.EVT_BUTTON, self.onAdd)
        self.editButton.Bind(wx.EVT_BUTTON, self.onEdit)
        self.removeButton.Bind(wx.EVT_BUTTON, self.onRemove)
        self.wordsListView.Bind(wx.EVT_LIST_ITEM_RIGHT_CLICK, self.onWordsRightClick)
        
        # Layout and show.
        self.SetFocus()
        self.Layout()
        self.Center()
        self.Show()
        
    def connected(self, connected):
        self.menu.connected(connected)
        title = info.PROG_NAME
        if connected:
            title += ' [Connected]'
        self.SetTitle(title) 
        
    def onClose(self, evt):
        self.isClosing = True
        config.save(self.cfg, info.CONFIG_PATH)
        self.insim.close()
        self.Destroy()    
        
    def onIconize(self, evt):
        if self.cfg.minimise and self.IsIconized():
            self.Hide()            
            self.taskBarIcon.SetIcon(wx.Icon(info.ICON_MAIN, wx.BITMAP_TYPE_ICO), info.PROG_NAME)  
        evt.Skip()   
        
    def onRestore(self, evt):
        self.Show()               
        self.taskBarIcon.RemoveIcon() 
        self.Iconize(False)   
        
    def onExit(self, evt):
        self.Close()

    def onOptions(self, evt):
        dlg = dialog.OptionsDialog(self, wx.ID_ANY)
        dlg.ShowModal()
        dlg.Destroy()
        
    def onConnect(self, evt):
        try:
            self.insim.connect(self.cfg.host, self.cfg.port)
            self.insim.send(pyinsim.ISP_ISI, ReqI=1, Admin=self.cfg.admin, IName=info.PROG_INAME)
            self.messenger.requestInfo(self.insim)
            self.connected(True)
        except pyinsim.socket.error, err:
            wx.MessageBox('Make sure LFS is running and configured to accept connections on the correct port', 'Couldn\'t connect to InSim', wx.OK | wx.ICON_ERROR)
                    
    def onDisconnect(self, evt):
        self.insim.close()
        self.connected(False)
        
    def onLaunch(self, evt):
        if os.path.exists(self.cfg.lfsExePath):
            import subprocess
            subprocess.Popen((self.cfg.lfsExePath, '/insim=%s' % self.cfg.port), cwd=os.path.dirname(self.cfg.lfsExePath))
            if self.cfg.autoConnect:
                self.onConnect(evt)
        else:
            wx.MessageBox('Make sure the path to LFS.exe is correctly set in the InSim options', 'Could not locate LFS.exe', wx.OK | wx.ICON_ERROR)
        
    def onAbout(self, evt):
        dlg = wx.AboutDialogInfo()
        dlg.SetName(info.PROG_NAME)
        dlg.SetVersion(info.PROG_VERSION)
        dlg.SetCopyright(info.PROG_COPYRIGHT)
        wx.AboutBox(dlg)
        
    def onInSimConnectionLost(self, insim):
        if hasattr(self, 'isClosing') and not self.isClosing:
            self.SetStatusText('InSim connection closed')
            self.connected(False)
        
    def onInSimThreadError(self, insim, err):
        wx.MessageBox(err, 'InSim has encountered an error', wx.OK | wx.ICON_ERROR)
        
    def onInSimVersion(self, insim, ver):
        if ver['InSimVer'] == pyinsim.INSIM_VERSION:
            self.SetStatusText('InSim Connected (LFS: %(Product)s %(Version)s InSim: %(InSimVer)d)' % ver)
        else:
            self.insim.close()
            wx.MessageBox('Got Insim version %d, but need version %d' % (ver['InSimVer'], pyinsim.INSIM_VERSION), 'Invalid InSim version detected', wx.OK | wx.ICON_ERROR)
            
    def onYouConnect(self, evt):
        self.cfg.youConnect = str(evt.GetString())
        
    def onPlayerConnects(self, evt):
        self.cfg.playerConnects = str(evt.GetString())
        
    def onRaceStarts(self, evt):
        self.cfg.raceStarts = str(evt.GetString())
    
    def onRaceEnds(self, evt):
        self.cfg.raceEnds = str(evt.GetString())
        
    def onFastestLap(self, evt):
        self.cfg.fastestLap = str(evt.GetString())
        
    def onPersonalBest(self, evt):
        self.cfg.personalBest = str(evt.GetString())
        
    def onAdd(self, evt):
        words = str(self.wordsTextCtrl.GetValue()).strip()
        message = str(self.messageTextCtrl.GetValue()).strip()
        if words and message:
            self.cfg.messages.append([words, message])
            item = self.wordsListView.InsertStringItem(self.wordsListView.GetItemCount(), words)
            self.wordsListView.SetStringItem(item, 1, message)
        
    def onEdit(self, evt):
        item = self.wordsListView.GetFirstSelected()
        if item > -1:
            words, message = self.cfg.messages[item]
            dlg = dialog.EditDialog(self, wx.ID_ANY)
            dlg.wordsTextCtrl.SetValue(words)
            dlg.messageTextCtrl.SetValue(message)
            dlg.messageTextCtrl.SetMaxLength(MAX_MSG_LEN)
            if dlg.ShowModal() == wx.ID_OK:
                words = str(dlg.wordsTextCtrl.GetValue())
                message = str(dlg.messageTextCtrl.GetValue())
                self.cfg.messages[item] = [words, message]
                self.wordsListView.SetStringItem(item, 0, words)
                self.wordsListView.SetStringItem(item, 1, message)
            dlg.Destroy()
        
    def onRemove(self, evt):
        item = self.wordsListView.GetFirstSelected()
        if item > -1:
            r = wx.MessageBox('Are you sure?', 'Remove message', wx.YES_NO | wx.ICON_QUESTION)
            if r == wx.YES:
                del self.cfg.messages[item]
                self.wordsListView.DeleteItem(item)
                
    def onWordsRightClick(self, evt):
        self.wordsListView.PopupMenu(menu.WordsPopupMenu(), evt.GetPosition())

        
