﻿using System;
using System.Windows.Forms;
using System.Collections;

using LFSReplayParser.Replays;

namespace LFSReplayViewer
{
    public partial class MainForm : Form
    {
        public MainForm()
        {
            InitializeComponent();
        }

        void exitToolStripMenuItem_Click(object sender, EventArgs e)
        {
            Application.Exit();
        }

        void addReplayToolStripMenuItem_Click(object sender, EventArgs e)
        {
            using (OpenFileDialog ofd = new OpenFileDialog())
            {
                ofd.Title = "Add replays";
                ofd.Filter = "LFS Multiplayer Replays (*.mpr)|*.mpr";
                ofd.Multiselect = true;

                if (ofd.ShowDialog() == DialogResult.OK)
                {
                    foreach (string filename in ofd.FileNames)
                    {
                        try
                        {
                            MultiplayerReplay replay = MultiplayerReplay.Parse(filename);

                            // Add replay to the replaysListView.
                            ListViewItem item = replaysListView.Items.Add(replay.Name);
                            item.SubItems.Add(replay.ShortTrackName);
                            item.SubItems.Add(replay.Laps.ToString());
                            item.SubItems.Add(string.Format("{0} / {1}", replay.NumFinished, replay.NumPlayers));
                            item.SubItems.Add(replay.Wind.ToString());
                            item.SubItems.Add(replay.Weather.ToString());
                            item.SubItems.Add(replay.StartTime.ToString());

                            // Store the replay object in the Tag.
                            item.Tag = (object)replay;
                        }
                        catch (Exception ex)
                        {
                            MessageBox.Show(
                                ex.Message,
                                "Error Reading Replay",
                                MessageBoxButtons.OK,
                                MessageBoxIcon.Error);
                        }
                    }
                }
            }
        }

        void replaysListView_SelectedIndexChanged(object sender, EventArgs e)
        {
            resultsListView.Items.Clear();
            if (replaysListView.SelectedItems.Count > 0)
            {
                // Cast replay object from the Tag.
                MultiplayerReplay replay = (MultiplayerReplay)replaysListView.SelectedItems[0].Tag;

                foreach (Result result in replay.Results)
                {
                    // Add result to resultsListView.
                    ListViewItem item = resultsListView.Items.Add(result.Position.ToString());
                    item.SubItems.Add(result.PlayerName);
                    item.SubItems.Add(result.LfsUserName);
                    item.SubItems.Add(result.ShortCarName);
                    item.SubItems.Add(result.BestLapTime.ToString());
                    // Replay.Results[0] is always the winner.
                    item.SubItems.Add(GetRaceTime(result, replay.Results[0])); 
                    item.SubItems.Add(result.Pitstops.ToString());
                }
            }
        }

        void removeReplaysToolStripMenuItem_Click(object sender, EventArgs e)
        {
            // Remove selected replays from the list.
            foreach (ListViewItem item in replaysListView.SelectedItems)
            {
                replaysListView.Items.Remove(item);
            }
        }

        void aboutToolStripMenuItem_Click(object sender, EventArgs e)
        {
            using (AboutDialog ad = new AboutDialog())
            {
                ad.ShowDialog();
            }
        }

        string GetRaceTime(Result result, Result winner)
        {
            if (result == winner)
            {
                // Winner - just return OverallTime.
                return result.OverallTime.ToString();
            }
            else if (result.LapsDone == winner.LapsDone)
            {
                // Not winner, but on same lap, return elapsed time.
                Time elapsed = result.OverallTime.Subtract(winner.OverallTime);
                return string.Format("+{0}", elapsed.ToString());
            }
            else
            {
                // Return laps behind winner.
                int laps = winner.LapsDone - result.LapsDone;
                return string.Format("+{0} laps", laps);
            }
        }
    }
}
