﻿/*
 *  Copyright Alex McBride 2008.
 * 
 *  This file is part of LFSReplayParser.
 *
 *  LFSReplayParser is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  LFSReplayParser is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with LFSReplayParser. If not, see <http://www.gnu.org/licenses/>.
 *  
 */

using System;
using System.Collections.Generic;
using System.IO;

using LFSReplayParser.Extensions;

namespace LFSReplayParser.Replays
{
    /// <summary>
    /// Represents a LFS multiplayer replay.
    /// </summary>
    public class MultiplayerReplay
    {
        const string Header = "LFSMPR";

        List<Result> results;

        /// <summary>
        /// Get if the player joined a race in progress.
        /// </summary>
        public bool ImmediateStart { get; private set; }

        /// <summary>
        /// Get the rules for the host.
        /// </summary>
        public HostRules Rules { get; private set; }

        /// <summary>
        /// Get the flags for the host.
        /// </summary>
        public HostFlags Flags { get; private set; }

        /// <summary>
        /// Get the laps byte, which can have various meanings. Use Laps or Hours properties instead.
        /// </summary>
        public int LapsByte { get; private set; }

        /// <summary>
        /// Get the difficulty of the host.
        /// </summary>
        public Skills Skill { get; private set; }

        /// <summary>
        /// Get the wind setting on the host.
        /// </summary>
        public Winds Wind { get; private set; }

        /// <summary>
        /// Get the number of players on the host.
        /// </summary>
        public int NumPlayers { get; private set; }

        /// <summary>
        /// Get the LFS version.
        /// </summary>
        public string LfsVersion { get; private set; }

        /// <summary>
        /// Get the short track name (EG BL1).
        /// </summary>
        public string ShortTrackName { get; private set; }

        /// <summary>
        /// Get the start time of the replay.
        /// </summary>
        public DateTime StartTime { get; private set; }

        /// <summary>
        /// Get the track name.
        /// </summary>
        public string TrackName { get; private set; }

        /// <summary>
        /// Get the track config.
        /// </summary>
        public int Config { get; private set; }

        /// <summary>
        /// Get if the track is reversed.
        /// </summary>
        public bool Reversed { get; private set; }

        /// <summary>
        /// Get the weather settings of the host.
        /// </summary>
        public Weathers Weather { get; private set; }

        /// <summary>
        /// Get the number of players who finished the race.
        /// </summary>
        public int NumFinished { get; private set; }

        /// <summary>
        /// The name of the replay file.
        /// </summary>
        public string FileName { get; private set; }

        /// <summary>
        /// Get the number of laps of the race.
        /// </summary>
        public int Laps
        {
            get
            {
                if ((LapsByte >= 1) && (LapsByte <= 99))
                {
                    return LapsByte;
                }
                else if ((LapsByte >= 100) && (LapsByte <= 190))
                {
                    return (LapsByte - 100) * 10 + 100;
                }
                return 0;
            }
        }

        /// <summary>
        /// Get the number of hours of the race.
        /// </summary>
        public int Hours
        {
            get
            {
                return ((LapsByte >= 191) && (LapsByte <= 238)) ? (LapsByte - 190) : 0;
            }
        }

        /// <summary>
        /// Get if the replay is for a race, practice or qualifying.
        /// </summary>
        public RaceTypes RaceType
        {
            get
            {
                if (LapsByte == 0)
                {
                    return RaceTypes.Practice;
                }
                else if ((results.Count > 0) && results[0].OverallTime.IsZero)
                {
                    // If the wining driver has no race time, it was a qualifying.
                    return RaceTypes.Qualifying;
                }
                return RaceTypes.Race;
            }
        }

        /// <summary>
        /// The name of the replay file, without path or extension.
        /// </summary>
        public string Name
        {
            get
            {
                return Path.GetFileNameWithoutExtension(FileName);
            }
        }

        /// <summary>
        /// Get an array of the race results.
        /// </summary>
        public Result[] Results
        {
            get
            {
                return results.ToArray();
            }
        }

        /// <summary>
        /// Create a new MultiplayerReplay object.
        /// </summary>
        public MultiplayerReplay()
        {
            results = new List<Result>();
        }

        /// <summary>
        /// Return a string representation of the MultiplayerReplay object.
        /// </summary>
        /// <returns>The replay Name.</returns>
        public override string ToString()
        {
            return Name;
        }

        /// <summary>
        /// Parse a MultiplayerReplay from file.
        /// </summary>
        /// <param name="path">The path to the .MPR replay to parse.</param>
        /// <returns>A MultiplayerReplay object populated with replay data.</returns>
        public static MultiplayerReplay Parse(string path)
        {
            MultiplayerReplay replay = new MultiplayerReplay() 
            { 
                FileName = path,
            };

            using (FileStream fs = File.Open(path, FileMode.Open))
            using (BinaryReader br = new BinaryReader(fs))
            {
                if (br.ReadString(6) != Header)
                {
                    throw new ReplayException("Invalid header in .MPR file '{0}'".FormatWith(path));
                }

                br.Skip(3);
                replay.ImmediateStart = br.ReadByteAsBool();
                br.Skip(2);
                replay.Rules = (HostRules)br.ReadInt32();
                replay.Flags = (HostFlags)br.ReadInt32();
                replay.LapsByte = br.ReadByte();
                replay.Skill = (Skills)br.ReadByte();
                replay.Wind = (Winds)br.ReadByte();
                replay.NumPlayers = br.ReadByte();
                replay.LfsVersion = br.ReadString(8);
                replay.ShortTrackName = br.ReadString(4);
                replay.StartTime = br.ReadTimeStamp();
                replay.TrackName = br.ReadString(32);
                replay.Config = br.ReadByte();
                replay.Reversed = br.ReadByteAsBool();
                replay.Weather = (Weathers)br.ReadByte();
                replay.NumFinished = br.ReadByte();
                br.Skip(4);

                for (int i = 0, position = 1; i < replay.NumFinished; i++, position++)
                {
                    replay.results.Add(Result.Parse(br, position));
                }
            }

            return replay;
        }

        /// <summary>
        /// Parses multiple MultiplayerReplays from files.
        /// </summary>
        /// <param name="paths">An IEnumerable collection of .MPR replay paths to parse.</param>
        /// <returns>An IEnumerable collection of MultiplayerReplay objects.</returns>
        public static IEnumerable<MultiplayerReplay> Parse(IEnumerable<string> paths)
        {
            foreach (string path in paths)
            {
                yield return MultiplayerReplay.Parse(path);
            }
        }
    }
}
