﻿/*
 *  Copyright Alex McBride 2008.
 * 
 *  This file is part of LFSReplayParser.
 *
 *  LFSReplayParser is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  LFSReplayParser is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with LFSReplayParser. If not, see <http://www.gnu.org/licenses/>.
 *  
 */

using System;
using System.Text;
using System.IO;

using LFSReplayParser.Extensions;

namespace LFSReplayParser.Replays
{
    /// <summary>
    /// Represents a LFS single player replay.
    /// </summary>
    public class SingleplayerReplay
    {
        const string Header = "LFSSPR";

        /// <summary>
        /// The length of the qualifying session.
        /// </summary>
        public int QualMinutes { get; private set; }

        /// <summary>
        /// Get the laps byte, which can have various meanings. Use Laps or Hours properties instead.
        /// </summary>
        public int LapsByte { get; private set; }

        /// <summary>
        /// Get the difficulty of the host.
        /// </summary>
        public Skills Skill { get; private set; }

        /// <summary>
        /// Get the wind settings.
        /// </summary>
        public Winds Wind { get; private set; }

        /// <summary>
        /// Get if the single-player replay was recorded in Hotlap mode.
        /// </summary>
        public HotlapMode HotlapMode { get; private set; }

        /// <summary>
        /// Get the version of LFS the replay was recorded on.
        /// </summary>
        public string LfsVersion { get; private set; }

        /// <summary>
        /// Get the short track name of the track (EG BL1).
        /// </summary>
        public string ShortTrackName { get; private set; }

        /// <summary>
        /// Get the players handicap mass (KG).
        /// </summary>
        public int HandicapMass { get; private set; }

        /// <summary>
        /// Get the players intake restriction (%).
        /// </summary>
        public int IntakeRestriction { get; private set; }

        /// <summary>
        /// Get the name of the track.
        /// </summary>
        public string TrackName { get; private set; }

        /// <summary>
        /// Get the players LFSWorld username.
        /// </summary>
        public string LfsUserName { get; private set; }

        /// <summary>
        /// Get the car name.
        /// </summary>
        public string CarName { get; private set; }

        /// <summary>
        /// Get the track config.
        /// </summary>
        public int Config { get; private set; }

        /// <summary>
        /// Get if the track is reversed.
        /// </summary>
        public bool Reversed { get; private set; }

        /// <summary>
        /// Get the weather settings.
        /// </summary>
        public Weathers Weather { get; private set; }

        /// <summary>
        /// Get the number of players on the host (player + AI).
        /// </summary>
        public int NumPlayers { get; private set; }

        /// <summary>
        /// Get the players flags.
        /// </summary>
        public PlayerFlags PlayerFlags { get; private set; }

        /// <summary>
        /// Get the lap number of the HLVC lap if in hotlap mode.
        /// </summary>
        public int HlvcBestLap { get; private set; }

        /// <summary>
        /// Get the number of splits on the track.
        /// </summary>
        public int NumberOfSplits { get; private set; }

        /// <summary>
        /// Get an array of the split times, including the final lap time.
        /// </summary>
        public Time[] Splits { get; private set; }

        /// <summary>
        /// Get the players name.
        /// </summary>
        public string PlayerName { get; private set; }

        /// <summary>
        /// Get the FileName of the replay.
        /// </summary>
        public string FileName { get; private set; }

        /// <summary>
        /// Get the number of laps of the race.
        /// </summary>
        public int Laps
        {
            get
            {
                if ((LapsByte >= 1) && (LapsByte <= 99))
                {
                    return LapsByte;
                }
                else if ((LapsByte >= 100) && (LapsByte <= 190))
                {
                    return (LapsByte - 100) * 10 + 100;
                }
                return 0;
            }
        }

        /// <summary>
        /// Get the number of hours of the race.
        /// </summary>
        public int Hours
        {
            get
            {
                return ((LapsByte >= 191) && (LapsByte <= 238)) ? (LapsByte - 190) : 0;
            }
        }

        /// <summary>
        /// Get the name of the replay, without path or extensions.
        /// </summary>
        public string Name
        {
            get
            {
                return Path.GetFileNameWithoutExtension(FileName);
            }
        }

        /// <summary>
        /// Create a new SinglePlayerReplay object.
        /// </summary>
        public SingleplayerReplay()
        {

        }

        /// <summary>
        /// Parse a SinglePlayerReplay object from file.
        /// </summary>
        /// <param name="path">The path to the .SPR replay to parse.</param>
        /// <returns>A SingleplayerReplay object populated with replay data.</returns>
        public static SingleplayerReplay Parse(string path)
        {
            SingleplayerReplay replay = new SingleplayerReplay()
            {
                FileName = path,
            };

            using(FileStream fs=File.Open(path, FileMode.Open))
            using (BinaryReader br = new BinaryReader(fs))
            {
                if (br.ReadString(6) != Header)
                {
                    throw new ReplayException("Invalid header in .SPR file '{0}'".FormatWith(path));
                }

                br.Skip(5);
                replay.QualMinutes = br.ReadByte();
                replay.LapsByte = br.ReadByte();
                replay.Skill = (Skills)br.ReadByte();
                replay.Wind = (Winds)br.ReadByte();
                replay.HotlapMode = (HotlapMode)br.ReadByte();
                replay.LfsVersion = br.ReadString(8);
                replay.ShortTrackName = br.ReadString(4);
                replay.HandicapMass = br.ReadByte();
                replay.IntakeRestriction = br.ReadByte();
                br.Skip(2);
                replay.TrackName = br.ReadString(32);
                replay.LfsUserName = br.ReadStringEncoded(32);
                replay.CarName = br.ReadString(32);
                replay.Config = br.ReadByte();
                replay.Reversed = br.ReadByteAsBool();
                replay.Weather = (Weathers)br.ReadByte();
                replay.NumPlayers = br.ReadByte();
                replay.PlayerFlags = (PlayerFlags)br.ReadInt16();
                replay.HlvcBestLap = br.ReadByte();
                replay.NumberOfSplits = br.ReadByte();

                byte[] split1 = br.ReadBytes(4);
                byte[] split2 = br.ReadBytes(4);
                byte[] split3 = br.ReadBytes(4);
                byte[] split4 = br.ReadBytes(4);
                replay.Splits = new Time[] 
                { 
                    new Time(split1[0], split1[1], split1[2], split1[3]), 
                    new Time(split2[0], split2[1], split2[2], split2[3]), 
                    new Time(split3[0], split3[1], split3[2], split3[3]), 
                    new Time(split4[0], split4[1], split4[2], split4[3]) 
                };

                br.Skip(4);
                replay.PlayerName = br.ReadStringEncoded(32);
            }

            return replay;
        }
    }
}
