#
# Copyright 2008 Alex McBride.
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the Lesser General Public License (LGPL) as
# published by the Free Software Foundation, either version 3 of the
# License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#

import Pyinsim
import sys
import ConfigParser
import time

CFG_FILE = 'Config.ini'
LAPS_FILE = 'Laps.txt'


class Player:
    def __init__(self, npl):
        self.PLID = npl['PLID']
        self.UCID = npl['UCID']
        self.PName = Pyinsim.StripColours(npl['PName'])
        self.Plate = npl['Plate']
        self.CName = npl['CName']
        self.Splits = [0, 0, 0]

    def AddSplit(self, spx):
        split = (spx['Split'] - 1)
        self.Splits[split] = spx['STime']

    def ResetSplits(self):
        self.Splits = [0, 0, 0]


# Init globals
insim = Pyinsim.InSim(Pyinsim.INSIM_TCP)
config = ConfigParser.ConfigParser()
connections = {}
players = {}
trackName = None


# Helper functions.
def SendMessage(msg):
    """Send message to LFS."""
    if len(msg) > 64:
        insim.SendP(Pyinsim.Packet(Pyinsim.ISP_MSX, Msg=msg))
    else:
        insim.SendP(Pyinsim.Packet(Pyinsim.ISP_MST, Msg=msg))


def RequestPlayersConns():
    """Request all players and connections to be sent."""
    insim.SendP(Pyinsim.Packet(Pyinsim.ISP_TINY, ReqI=1, SubT=Pyinsim.TINY_NCN))
    insim.SendP(Pyinsim.Packet(Pyinsim.ISP_TINY, ReqI=1, SubT=Pyinsim.TINY_NPL))


def GetConnection(ucid):
    """Get connection from UCID."""
    return connections[ucid]


def GetPlayer(plid):
    """Get player from PLID."""
    return players[plid]


def GetPlayerFromUcid(ucid):
    """Get player from UCID."""
    for player in players.itervalues():
        if player.UCID == ucid:
            return player
    return None


# TODO: Add more helper functions.


# Packet received events
def VersionCheck(ver):
    """Check the version."""
    if ver['InSimVer'] != Pyinsim.INSIM_VERSION:
        print 'Invalid InSim version detected.'
        sys.exit(0)


def HostJoined(ism):
    """Request all players and connections to be sent."""
    RequestPlayersConns()


def ConnectionJoined(ncn):
    """Add connection to connections dictionary."""
    connections[ncn['UCID']] = ncn


def ConnectionLeft(cnl):
    """Delete connection from connections dictionary."""
    del connections[cnl['UCID']]


def ConnectionRenamed(cpr):
    """Rename player in connections and players lists."""
    connection = GetConnection(cpr['UCID'])
    connection['PName'] = cpr['PName']
    player = GetPlayerFromUcid(cpr['UCID'])
    player.PName = Pyinsim.StripColours(cpr['PName'])
    player.Plate = cpr['Plate']


def PlayerJoined(npl):
    """Add player to players dictionary."""
    players[npl['PLID']] = Player(npl)


def PlayerLeft(pll):
    """Delete player from players dictionary."""
    del players[pll['PLID']]


def TookOverCar(toc):
    """Change UCID for player."""
    player = GetPlayer(toc['PLID'])
    player.UCID = toc['NewUCID']


def StateUpdated(sta):
    global trackName
    trackName = sta['Track']


def SplitCompleted(spx):
    player = GetPlayer(spx['PLID'])
    player.AddSplit(spx)


def LapCompleted(lap):
    player = GetPlayer(lap['PLID'])
    if player.PName == config.get('InSim', 'playername'):
        file = None
        try:
            file = open(LAPS_FILE, 'a')
        except IOError, (ex):
            print 'Error writing file: %s' % (ex.args[1])
        else:
            sp1 = Pyinsim.LFSTime(player.Splits[0])
            sp2 = Pyinsim.LFSTime(player.Splits[1])
            sp3 = Pyinsim.LFSTime(player.Splits[2])
            lapTime = Pyinsim.LFSTime(lap['LTime'])

            # Date, Track, Car, Sp1, Sp2, Sp3, Lap
            file.writelines('%s,%s,%s,%s,%s,%s,%s\n' % (
                            time.strftime("%Y.%m.%d %H:%M:%S"),
                            trackName,
                            player.CName,
                            sp1.GetLapTimeStr(),
                            sp2.GetLapTimeStr(),
                            sp3.GetLapTimeStr(),
                            lapTime.GetLapTimeStr()))
        finally:
            if file is not None:
                file.close()
    player.ResetSplits()


# TODO: Add more packet event handlers.


# Bind events.
insim.Bind({Pyinsim.ISP_VER: VersionCheck,
            Pyinsim.ISP_ISM: HostJoined,
            Pyinsim.ISP_NCN: ConnectionJoined,
            Pyinsim.ISP_CNL: ConnectionLeft,
            Pyinsim.ISP_NPL: PlayerJoined,
            Pyinsim.ISP_PLL: PlayerLeft,
            Pyinsim.ISP_CPR: ConnectionRenamed,
            Pyinsim.ISP_TOC: TookOverCar,
            Pyinsim.ISP_STA: StateUpdated,
            Pyinsim.ISP_SPX: SplitCompleted,
            Pyinsim.ISP_LAP: LapCompleted})


# Connection lost.
def ConnectionLost():
    print 'InSim connection lost.'
    sys.exit(0)

insim.ConnectionLost(ConnectionLost)


# Load config
file = None
try:
    file = open(CFG_FILE)
except IOError, (ex):
    print 'Error reading config: %s' % (ex.args[1])
    sys.exit(0)
else:
    config.readfp(file)
finally:
    if file is not None:
        file.close()


# Connect to InSim.
try:
    insim.Connect(config.get('InSim', 'host'), config.getint('InSim', 'port'))
except Pyinsim.socket.error, (ex):
    print 'Connection to InSim failed: %s' % (ex.args[1])
    sys.exit(0)
else:
    # Initailise InSim and request players/connections.
    insim.SendP(Pyinsim.Packet(Pyinsim.ISP_ISI, Admin=config.get('InSim', 'Admin'),
                               IName='^3LapStats', ReqI=1))
    RequestPlayersConns()

    # TODO: Add initailsation code here.
    print 'LapStats connected! Go drive some laps and check the file %s for \
the output.' % (LAPS_FILE)

    # Keep program thread alive.
    insim.Run()
finally:
    insim.Close()
