#
# Copyright 2008 Alex McBride.
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the Lesser General Public License (LGPL) as
# published by the Free Software Foundation, either version 3 of the
# License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#

import Pyinsim
import ConfigParser
import sys
import re


# Constants.
PROGRAM_NAME = 'FlagMessages'
PROGRAM_VERSION = 1

FLG_BLUE = 1
FLG_YELLOW = 2
FLG_ON = 1
FLG_OFF = 0
BTN_ID = 1

IP_PATTERN = r"\b(25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\.(25[0-5]|2\
[0-4][0-9]|[01]?[0-9][0-9]?)\.(25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\.(25\
[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\b"
MAX_PORT = 65535
MIN_PORT = 0

CFG_FILE_PATH = 'Config.ini'


# Init globals
insim = Pyinsim.InSim(Pyinsim.INSIM_TCP)
config = ConfigParser.ConfigParser()
connections = {}
players = {}


# Helper functions.
def SendMessage(msg):
    """Send message to LFS."""
    if len(msg) > 64:
        insim.SendP(Pyinsim.Packet(Pyinsim.ISP_MSX, Msg=msg))
    else:
        insim.SendP(Pyinsim.Packet(Pyinsim.ISP_MST, Msg=msg))


def SendMessageConn(msg, ucid=0, plid=0):
    """Send message to a specific connection or player."""
    insim.SendP(Pyinsim.Packet(Pyinsim.ISP_MTC, Msg=msg, UCID=ucid, PLID=plid))


def RequestPlayersConns():
    """Request all players and connections to be sent."""
    insim.SendP(Pyinsim.Packet(Pyinsim.ISP_TINY, ReqI=1, SubT=Pyinsim.TINY_NCN))
    insim.SendP(Pyinsim.Packet(Pyinsim.ISP_TINY, ReqI=1, SubT=Pyinsim.TINY_NPL))


def GetConnection(ucid):
    """Get connection from UCID."""
    return connections[ucid]


def GetPlayer(plid=None, ucid=None):
    """Get player from PLID."""
    return players[plid]


def GetPlayerFromUCID(ucid):
    """Get player from UCID."""
    for player in players.values():
        if player['UCID'] == ucid:
            return player
    return None


def SendButton(ucid, clickID, msg, top, left, width, height):
    """Send a button to InSim."""
    btn = Pyinsim.Packet(Pyinsim.ISP_BTN)
    btn["ReqI"] = 1
    btn["UCID"] = ucid
    btn['ClickID'] = clickID
    btn["T"] = top
    btn["L"] = left
    btn["W"] = width
    btn["H"] = height
    btn["Text"] = msg
    insim.SendP(btn)


def ClearButton(ucid, clickID):
    """Clear a specific button."""
    insim.SendP(Pyinsim.Packet(Pyinsim.ISP_BFN, SubT=Pyinsim.BFN_DEL_BTN,
                               UCID=ucid, ClickID=clickID))


def ClearAllButtons():
    """Clear all buttons."""
    for player in players.values():
        insim.SendP(Pyinsim.Packet(Pyinsim.ISP_BFN, SubT=Pyinsim.BFN_CLEAR,
                                   UCID=player['UCID']))


def CheckIP(ipStr):
    """Check for a valid IP address."""
    if re.match(IP_PATTERN, ipStr):
       return True
    return False


def CheckPort(portStr):
    """Check for a valid port number."""
    if portStr.isdigit() == True:
        if int(portStr) >= MIN_PORT and int(portStr) <= MAX_PORT:
            return True
    return False


# Packet recieved events
def VersionCheck(ver):
    """Check the InSim version."""
    if ver['InSimVer'] == Pyinsim.INSIM_VERSION:
        print 'Sucessfully connected to InSim...'
        print 'LFS Version: %s %s' % (ver['Product'], ver['Version'])
        print 'InSim Version: %s' % (ver['InSimVer'])
    else:
        print 'Invalid InSim version detected. This program requires InSim \
version %d.' % (Pyinsim.INSIM_VERSION)
        insim.Close()
        sys.exit(1)


def ConnectionJoined(ncn):
    """Add connection to connections dictionary."""
    connections[ncn['UCID']] = ncn


def ConnectionLeft(cnl):
    """Delete connection from connections dictionary."""
    del connections[cnl['UCID']]


def ConnectionRenamed(cpr):
    """Rename player in connections and players lists."""
    connection = GetConnection(cpr['UCID'])
    connection['PName'] = cpr['PName']
    player = GetPlayerFromUcid(cpr['UCID'])
    player['PName'] = cpr['PName']
    player['Plate'] = cpr['Plate']


def TookOverCar(toc):
    """Change UCID for player."""
    player = GetPlayer(toc['PLID'])
    player['UCID'] = toc['NewUCID']


def PlayerJoined(npl):
    """Add player to players dictionary."""
    players[npl['PLID']] = npl


def PlayerLeft(pll):
    """Delete player from players dictionary and remove button."""
    player = GetPlayer(pll['PLID'])
    ClearButton(player['UCID'], BTN_ID)
    del players[pll['PLID']]


def RaceStarted(rst):
    """Request all players and connections to be sent."""
    RequestPlayersConns()


def FlagCaused(flg):
    """Flag packet recieved. Send or clear button depending on
    whether it is a blue or yellow flag."""
    player = GetPlayer(flg['PLID'])
    if flg['OffOn'] == FLG_ON:
        if flg['Flag'] == FLG_BLUE:
            carBehind = GetPlayer(flg['CarBehind'])
            msg = config.get('BlueFlag', 'Msg')
            msg = msg.replace('{driver}', carBehind['PName'])
            SendButton(player['UCID'], BTN_ID, msg,
                       config.getint('BlueFlag', 'Top'),
                       config.getint('BlueFlag', 'Left'),
                       config.getint('BlueFlag', 'Width'),
                       config.getint('BlueFlag', 'Height'))
        elif flg['Flag'] == FLG_YELLOW:
            SendButton(player['UCID'], BTN_ID,
                       config.get('YellowFlag', 'Msg'),
                       config.getint('YellowFlag', 'Top'),
                       config.getint('YellowFlag', 'Left'),
                       config.getint('YellowFlag', 'Width'),
                       config.getint('YellowFlag', 'Height'))
    elif flg['OffOn'] == FLG_OFF:
        ClearButton(player['UCID'], BTN_ID)


def TinyRecieved(tiny):
    """Tiny recieved. If race had ended, clear all flag buttons on screen."""
    if tiny['SubT'] == Pyinsim.TINY_REN:
        ClearAllButtons()


def PlayerPits(plp):
    """Player pits, clear any flag buttons which might be on screen."""
    player = GetPlayer(plp['PLID'])
    ClearButton(player['UCID'], BTN_ID)


# Connection lost.
def ConnectionLost():
    print 'The InSim connection has been lost.'
    sys.exit(1)

# Bind events.
insim.Bind({Pyinsim.ISP_VER: VersionCheck,
            Pyinsim.ISP_NCN: ConnectionJoined,
            Pyinsim.ISP_CNL: ConnectionLeft,
            Pyinsim.ISP_NPL: PlayerJoined,
            Pyinsim.ISP_PLL: PlayerLeft,
            Pyinsim.ISP_RST: RaceStarted,
            Pyinsim.ISP_FLG: FlagCaused,
            Pyinsim.ISP_TINY: TinyRecieved,
            Pyinsim.ISP_PLP: PlayerPits})

insim.ConnectionLost(ConnectionLost)

# Welcome message.
print 'Welcome to %s V%d' % (PROGRAM_NAME, PROGRAM_VERSION)


# Load config
file = None
try:
    file = open(CFG_FILE_PATH)
except IOError, (ex):
    print 'Error: Could not read file %s.' % (CFG_FILE_PATH)
    sys.exit(1)
else:
    config.readfp(file)
finally:
    if file is not None:
        file.close()


# Check config
if CheckIP(config.get('InSim', 'HostAddress')) == False:
    print "Error: The host address '%s' is not a valid IP." \
           % config.get('InSim', 'HostAddress')
    sys.exit(1)

if CheckPort(config.get('InSim', 'InSimPort')) == False:
    print "Error: The port '%s' is not a valid port number between %d and %d." \
           % (config.get('InSim', 'InSimPort'), MIN_PORT, MAX_PORT)
    sys.exit(1)


# Connect to InSim.
try:
    insim.Connect(config.get('InSim', 'HostAddress'),
                  config.getint('InSim', 'InSimPort'))
except Pyinsim.socket.error, (ex):
    print 'Connection to InSim failed: %s. Make sure LFS is running and \
configured to accept connections on the correct port.' % (ex.args[1])
else:
    # Initailise InSim and request players/connections.
    insim.SendP(Pyinsim.Packet(Pyinsim.ISP_ISI,
                               Admin=config.get('InSim', 'AdminPass'),
                               IName=PROGRAM_NAME, ReqI=1))
    RequestPlayersConns()

    # Keep program thread alive.
    insim.Run()
finally:
    insim.Close()
