use clap::Parser;
use lfs_Packets::*;
use lfs_async_InSim as LFS;
use lfs_Packets::IS::npl::PType;
use std::{
   error::Error,
   time::{self, Duration},
   net::{SocketAddrV4, Ipv4Addr},
   path::PathBuf,
};
use std::io::Stdin;

#[derive(Debug, Parser)]
#[command(version, about, author, long_about, term_width = 100, help_template =
"{name}{tab}v{version} © {author}\n\
    {about}\n\n\
    {usage-heading}{tab}{usage}\n
    {tab}Example: wheel_breaker.exe -p 11000 \n\n\
    {all-args}\n"
) ]
/// Utility to make your wheel last bit longer.
///
/// It first reads values from given lfs config file or it's default location
/// than it overrides those values with command line supplied ones (if any),
/// afterwhich it will attempt to make a connection on a given port.
/// If successfull it will enable ForceFeedback on each race start
/// and disable it once the local driver finishes the race.
struct Cli {
   #[arg(long, short = 'p')]
   insim_port: u16,
   #[arg(long)]
   password: Option<String>,
   #[arg(long, default_value = r"C:\LFS\cfg.txt", name="PATH")]
   lfs_config: PathBuf,
   #[arg(long, name="INTEGER")]
   ffb: Option<u8>,
}

fn start_match_trim<'a>(text: &'a str, pattern: &str ) -> Option<&'a str> {
    if text.starts_with(pattern) {
        Some( text.trim_start_matches(pattern).trim_start() )
    } else { None }
}

#[tokio::main]
async fn main() -> Result<(), Box<dyn Error> > {
   let cli = Cli::try_parse().map_err( |e| {
      _ = e.print();
      eprintln!("Press enter to exit. . .");
      _ = std::io::stdin().read_line(&mut String::new());
      "Argument parsing failed."
   } )?;
   let ( mut password, mut ffb ) = Default::default();
   match tokio::fs::read_to_string( cli.lfs_config.clone() ).await {
      Ok(string) => { string.lines().for_each( |line| {
            if let Some(ver) = start_match_trim(&line, "Version") { println!("Detected config version: {ver}" ) }
            if let Some(force) = start_match_trim(&line, "FF Strength") {
            ffb = force.parse::<f32>().expect("Corrupted LFS config file at Force Strength") as u8;
            }
            if let Some(pass) = start_match_trim(&line, "Game Admin") {
            password = pass.to_string();
            }
         } );
      }
      Err(err) => {
         eprintln!("Can't read given LFS config file at \"{}\" due to \"{}\".", cli.lfs_config.to_str().unwrap(), err);
         if let (None, _) | (_, None) = (&cli.ffb, &cli.password) {
            eprintln!("No user provided values! Terminating.");
            return Err(err.into())
         };
      },
   }
   if let Some(force) = cli.ffb { ffb = force; } println!( "Using {ffb} ffb." );
   let ff_on_cmd = IS::MST::new( &format!("/ff {}", ffb) )?;
   let ff_off_cmd = IS::MST::new( "/ff 0" )?;
   if let Some(pass) = cli.password { password = pass; }

   let lfs_instance = LFS::Server {
      server_address: SocketAddrV4::new( Ipv4Addr::LOCALHOST, cli.insim_port),
      keep_alive: LFS::Keep_Alive::Auto,
   };
   println!("Connecting to LFS...");
   let mut lfs_instance = lfs_instance.initialise(
      IS::ISI::new()
          .set_ReqI( 1 )
          .set_Flags( &[ISF::LOCAL] )?
          .set_Admin( &password )?
          .set_IName( "wheel-breaker" )?
   ).await.inspect_err(|_|{eprintln!("Can't establish connection.")})?;
   
   let mut timeout_check = time::Instant::now();
   use Packet_from_bytes_Error;
   use ISP_Helper as ISPH;

   let mut my_plid = None;
   lfs_instance.packet_send( IS::TINY::new( 1, TINY::NPL ) ).await?;
   loop {
      match lfs_instance.packet_read().await {
         Err(error) => {
            if let Some(err @ Packet_from_bytes_Error::ConversionNotImplemented(_)) = error.downcast_ref::<Packet_from_bytes_Error>() {
               println!( "WARNING: {err}" );
               continue
            };
            return Err(error);
         },
         Ok(packet) => {
            match packet {
               Some(packet) => {
                  timeout_check = time::Instant::now();
                  match packet {
                     ISPH::VER(ver) => {
                        println!("{ver:?}");
                     },
                     ISPH::NPL(npl) => {
                        if (npl.PType & (PType::REMOTE | PType::AI)).0 == 0 {
                           println!( "Using PLID {}.", npl.PLID );
                           my_plid = Some(npl.PLID);
                        }
                     },
                     ISPH::RST(_) => {
                        lfs_instance.packet_send( ff_on_cmd ).await?;
                        println!( "Race start detected. Turning FF ON!" );
                     },
                     ISPH::FIN(fin) => {
                        let Some(plid) = my_plid else { continue };
                        if fin.PLID == plid {
                           lfs_instance.packet_send( ff_off_cmd ).await?;
                           println!( "Race finish detected. Turning FF OFF!" );
                        }
                     },
                     ISPH::CNL(cnl) => {
                        if cnl.UCID == 0 {
                           lfs_instance.packet_send( ff_on_cmd ).await?;
                           println!( "Local player left server. Restoring FF!" );
                        }
                     }
                     _ => {}
                  }
               },
               None => {
                  if let true = timeout_check.elapsed().as_secs() > 70
                     { panic!( "Something went wrong!" ) }
               },
            }
         },
      }
      tokio::time::sleep( Duration::from_millis( 100 ) ).await;
   }

   #[allow(unreachable_code)]
   Ok(())
}
