/*
    LFSLapper, Insim Race and qualification Manager for Live For Speed Game
    Copyright (C) 2007  Robert B. alias Gai-Luron and Monkster: lfsgailuron@free.fr

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
using System;
using WebCamService;
using System.Threading;
using System.Diagnostics;
using System.IO;


namespace LFS
{
    public class DriftPlayerStats
    {
        public string userName;
        public string nickName;
        public int personalBestLap;
        public string datePb;
        public string timePb;



        public DriftPlayerStats(string uN, string nN, string dPb, string tPb, int pb)
        {
            this.userName = uN;
            this.nickName = nN;
            this.datePb = dPb;
            this.timePb = tPb;
            this.personalBestLap = pb;
        }
    }

    public class DriftDb
    {

        System.Collections.Hashtable trackTable = new System.Collections.Hashtable();
        string filepath = null;
        string FtpServer = null;
        string FtpLogin = null;
        string FtpPasswd = null;
        string FtpRemotePath = null;
        ThreadUpload threadHandle;
        private Mutex myMutex;
        string curVersion = "USERNAME 01";


        public DriftDb( Mutex mut )
        {
//            Process myProcess = System.Diagnostics.Process.GetCurrentProcess();
//            this.stringMutex = "LFSDriftDb" + myProcess.Id.ToString();
//            this.myMutex = new Mutex(false, stringMutex);
            this.myMutex = mut;
        }
        public bool Load(string filepath, string FtpServer, string FtpLogin, string FtpPasswd, string FtpRemotePath)
        {
            this.filepath = filepath;
            this.FtpServer = FtpServer;
            this.FtpLogin = FtpLogin;
            this.FtpPasswd = FtpPasswd;
            this.FtpRemotePath = FtpRemotePath;

            this.threadHandle = new ThreadUpload(FtpServer, FtpRemotePath, FtpLogin, FtpPasswd, filepath, myMutex);

            try
            {
                while (updPbVersion(filepath) == true) ;
            }
            catch (System.Exception)
            {
                return false;
            }

            try
            {
                using (System.IO.StreamReader sr = new System.IO.StreamReader(filepath))
                {
                    string userName, nickName, datePb, timePb, time, carName, trackName;

                    string pbVersion = sr.ReadLine();
                    if (pbVersion.IndexOf("USERNAME") == -1) // Si pas fichier avec Username, on efface et on recrer
                        return false;

                    while (true)
                    {
                        userName = sr.ReadLine();
                        nickName = sr.ReadLine();
                        datePb = sr.ReadLine();
                        timePb = sr.ReadLine();

                        if (nickName == null)
                            return true;
                        carName = sr.ReadLine();
                        time = sr.ReadLine();
                        trackName = sr.ReadLine();

                        updateRow(userName, nickName,datePb,timePb, carName, int.Parse(time), trackName);
                    }
                }
            }
            catch (System.Exception)
            {
                return false;
            }
        }

        public bool Save()
        {
            bool retValue = false;
            try
            {
                //Exclude Concurrent access to same file when upload to FTP
                if (myMutex.WaitOne(-1, true))
                {
                    using (System.IO.StreamWriter sw = new System.IO.StreamWriter(filepath))
                    {
                        sw.WriteLine(curVersion);
                        System.Collections.IDictionaryEnumerator trackenum = trackTable.GetEnumerator();
                        while (trackenum.MoveNext()) //for each track
                        {
                            System.Collections.IDictionaryEnumerator carenum = (trackenum.Value as System.Collections.Hashtable).GetEnumerator();
                            while (carenum.MoveNext()) //for each car type
                            {
                                System.Collections.IDictionaryEnumerator entryenum = (carenum.Value as System.Collections.Hashtable).GetEnumerator();
                                while (entryenum.MoveNext())	//for each player
                                {
                                    sw.WriteLine(entryenum.Key.ToString());
                                    sw.WriteLine((entryenum.Value as DriftPlayerStats).nickName);
                                    sw.WriteLine((entryenum.Value as DriftPlayerStats).datePb);
                                    sw.WriteLine((entryenum.Value as DriftPlayerStats).timePb);
                                    sw.WriteLine(carenum.Key.ToString());
                                    sw.WriteLine((entryenum.Value as DriftPlayerStats).personalBestLap.ToString());
                                    sw.WriteLine(trackenum.Key.ToString());
                                }
                            }
                        }
                    }
                }
                retValue = true;
            }
            catch (Exception ex)
            {
                Console.WriteLine( "(" + filepath + ")" + ex.Message);   // Print the error message.
                return false;
            }
            finally
            {
                myMutex.ReleaseMutex();
            }
            if (retValue)
            {
                if (FtpServer != "")
                {
                    Thread ftpUpload = new Thread(new ThreadStart(threadHandle.UpLoad));
                    ftpUpload.Start();
                }
            }

            return retValue;
        }

        public void updateRow(LFSLapper.infoPlayer currInfoPlayer, string trackName, bool modeSync, string SyncID, string SyncDir, string[] SyncIDsToSync)
        {
            updateRow(currInfoPlayer.userName,
                        currInfoPlayer.nickName,
                        currInfoPlayer.DriftPBDate,
                        currInfoPlayer.DriftPBTime,
                        currInfoPlayer.CName,
                        (int)currInfoPlayer.bestdriftscore,
                        trackName
            );
        }
        public void updateRow(string userName, string nickName, string datePb, string timePb, string carName, int timeLap, string trackName)
        {

            // create new track table if it doesnt exist yet
            if (!trackTable.ContainsKey(trackName))
                trackTable[trackName] = new System.Collections.Hashtable();

            System.Collections.Hashtable carTable = trackTable[trackName] as System.Collections.Hashtable;

            // create new car table if it doesnt exist yet
            if (!carTable.ContainsKey(carName))
                carTable[carName] = new System.Collections.Hashtable();

            System.Collections.Hashtable playerTable = carTable[carName] as System.Collections.Hashtable;

            playerTable[userName.ToLower()] = new DriftPlayerStats(userName, nickName, datePb, timePb, timeLap);
        }

        public DriftPlayerStats retreiveRow(string userName, string carName, string trackName)
        {
            if (trackTable.ContainsKey(trackName))
            {
                System.Collections.Hashtable carTable = trackTable[trackName] as System.Collections.Hashtable;
                if (carTable.ContainsKey(carName))
                {
                    System.Collections.Hashtable playerTable = carTable[carName] as System.Collections.Hashtable;
                    if (playerTable.ContainsKey(userName.ToLower()))
                    {
                        return (DriftPlayerStats)playerTable[userName.ToLower()];
                    }
                }
            }
            return null;

        }

        public class DriverLapEntry : System.IComparable
        {
            public string userName;
            public DriftPlayerStats playerStats;
            public string carName;

            public DriverLapEntry(string user, DriftPlayerStats ps, string cnam)
            {
                userName = user;
                playerStats = ps;
                carName = cnam;
            }

            public int CompareTo(object x)
            {
                if ((x as DriverLapEntry).playerStats.personalBestLap<playerStats.personalBestLap)
                    return -1;
                else if ((x as DriverLapEntry).playerStats.personalBestLap>playerStats.personalBestLap)
                    return 1;
                else
                    return 0;
            }

        }

        string replace_group_car(string cars)
        {
            cars = cars.Replace("TBO", "XRT+RB4+FXO");
            cars = cars.Replace("LRF", "LX6+RAC+FZ5");
            cars = cars.Replace("GTR", "FXR+XRR+FZR");
            return cars;
        }

        public System.Collections.ArrayList GetTable(string trackName, string carName, string Filter, bool colorSensitive)
        {
            System.Collections.ArrayList list = new System.Collections.ArrayList();

            if (!trackTable.ContainsKey(trackName))
                return list;

            System.Collections.Hashtable carTable = trackTable[trackName] as System.Collections.Hashtable;
            carName = replace_group_car(carName);
            string[] lcarName = carName.Split('+');
//            Console.WriteLine("String->" + carName);
            for (int curr_idx_carName = 0; curr_idx_carName < lcarName.Length; curr_idx_carName++)
            {
                string curr_carName = lcarName[curr_idx_carName].Trim();

                if (!carTable.ContainsKey(curr_carName))
                    continue;

                System.Collections.Hashtable playerTable = carTable[curr_carName] as System.Collections.Hashtable;
                System.Collections.IDictionaryEnumerator entryenum = playerTable.GetEnumerator();

                while (entryenum.MoveNext())
                {
                    if (colorSensitive)
                    {
                        if (entryenum.Key.ToString().IndexOf(Filter) != -1)
                            list.Add(new DriverLapEntry(entryenum.Key.ToString(), entryenum.Value as DriftPlayerStats, curr_carName));
                    }
                    else
                    {
                        if (entryenum.Key.ToString().Replace("^0", "").Replace("^1", "").Replace("^2", "").Replace("^3", "").Replace("^4", "").Replace("^5", "").Replace("^6", "").Replace("^7", "").Replace("^8", "").IndexOf(Filter) != -1)
                            list.Add(new DriverLapEntry(entryenum.Key.ToString(), entryenum.Value as DriftPlayerStats, curr_carName));
                    }
                }
            }
            list.Sort();
            return list;
        }

        public System.Collections.ArrayList GetTable(string trackName, string carName)
        {
            System.Collections.ArrayList list = new System.Collections.ArrayList();

            if (!trackTable.ContainsKey(trackName))
                return list;

            System.Collections.Hashtable carTable = trackTable[trackName] as System.Collections.Hashtable;

            carName = replace_group_car(carName);
            string[] lcarName = carName.Split('+');
//            Console.WriteLine("String->" + carName);
            for (int curr_idx_carName = 0; curr_idx_carName < lcarName.Length; curr_idx_carName++)
            {
                string curr_carName = lcarName[curr_idx_carName].Trim();

                if (!carTable.ContainsKey(curr_carName))
                    continue;

                System.Collections.Hashtable playerTable = carTable[curr_carName] as System.Collections.Hashtable;
                System.Collections.IDictionaryEnumerator entryenum = playerTable.GetEnumerator();

                while (entryenum.MoveNext())
                {
                    list.Add(new DriverLapEntry(entryenum.Key.ToString(), entryenum.Value as DriftPlayerStats, curr_carName));
                }
            }
            list.Sort();
            return list;
        }

        /// <summary>
        /// Gets list of car names.
        /// </summary>
        /// <param name="trackName">Name of track.</param>
        /// <returns></returns>
        public System.Collections.ArrayList GetCars(string trackName)
        {
            System.Collections.ArrayList list = new System.Collections.ArrayList();
            System.Collections.IDictionaryEnumerator carTable = (trackTable[trackName] as System.Collections.Hashtable).GetEnumerator();
            while (carTable.MoveNext())
                list.Add(carTable.Key.ToString());

            list.Sort();
            return list;
        }
        private bool updPbVersion(string filePath)
        {
            // If it's not a new version, do nothing
            using (System.IO.StreamReader sr = new System.IO.StreamReader(filepath))
            {
                string pbVersion = sr.ReadLine();
                if (pbVersion.IndexOf("USERNAME") == -1) // Si pas fichier avec Username, on efface et on recrer
                    return false;
                if (pbVersion == curVersion)
                    return false;
            }
            string orig = Path.GetDirectoryName(filepath) + Path.DirectorySeparatorChar + "TMP_" + Path.GetFileName(filepath);
            File.Delete(orig);
            File.Move(filepath, orig);
            using (System.IO.StreamReader sr = new System.IO.StreamReader(orig))
            {
                using (System.IO.StreamWriter sw = new System.IO.StreamWriter(filepath, false))
                {
                    string userName, carName;

                    string pbVersion = sr.ReadLine();
                    switch (pbVersion)
                    {
                        case "USERNAME": // To version 01
                            sw.WriteLine("USERNAME 01");
                            Console.WriteLine("Convert file PB : USERNAME -> USERNAME 01");
                            while (true)
                            {
                                userName = sr.ReadLine();
                                if (userName == null)
                                {
                                    sw.Close();
                                    sr.Close();
                                    File.Delete(orig);
                                    return true;
                                }
                                sw.WriteLine(userName); //username
                                sw.WriteLine(sr.ReadLine()); //nickname
                                sw.WriteLine(sr.ReadLine()); //datePb
                                sw.WriteLine(sr.ReadLine()); //timepB
                                carName = sr.ReadLine();
                                switch (carName)
                                {
                                    case "UF 1000":
                                        carName = "UF1";
                                        break;
                                    case "XF GTI":
                                        carName = "XFG";
                                        break;
                                    case "XR GT":
                                        carName = "XRG";
                                        break;
                                    case "XR GT TURBO":
                                        carName = "XRT";
                                        break;
                                    case "RB4 GT":
                                        carName = "RB4";
                                        break;
                                    case "FXO TURBO":
                                        carName = "FXO";
                                        break;
                                    case "LX4":
                                        carName = "LX4";
                                        break;
                                    case "LX6":
                                        carName = "LX6";
                                        break;
                                    case "RA":
                                        carName = "RAC";
                                        break;
                                    case "FZ50":
                                        carName = "FZ5";
                                        break;
                                    case "MRT5":
                                        carName = "MRT";
                                        break;
                                    case "XF GTR":
                                        carName = "XFR";
                                        break;
                                    case "UF GTR":
                                        carName = "UFR";
                                        break;
                                    case "FORMULA XR":
                                        carName = "FOX";
                                        break;
                                    case "FORMULA V8":
                                        carName = "FO8";
                                        break;
                                    case "FXO GTR":
                                        carName = "FXR";
                                        break;
                                    case "XR GTR":
                                        carName = "XRR";
                                        break;
                                    case "FZ50 GTR":
                                        carName = "FZR";
                                        break;
                                    case "BMW SAUBER":
                                        carName = "BF1";
                                        break;
                                }
                                sw.WriteLine(carName);
                                sw.WriteLine(sr.ReadLine()); //time
                                sw.WriteLine(sr.ReadLine()); //trackName;
                            }
                        default:
                            break;
                    }
                    return false;
                }
            }
        }
    }
}