/*
 * Copyright (c) 2008, Cristbal Marco
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is furnished
 * to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
 * WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

/*
 * LFS Session Timing
 * ==================
 *
 * Version: 0.1
 * Author: Cristbal Marco (makakazo)
 *
 * "LFS Session Timing" is a basic InSim application written in C/C++ using the CInsim
 * library written by Cristbal Marco. The CInsim library uses WinSock2 for the socket
 * communication, so be sure to link to it.
 *
 * This has been written highly focused on being a tutorial on programming with InSim
 * and understanding it.
 *
 * PThreads-win32 was used for this project, so be sure to link to it when compiling,
 * and place the pthreads DLL in the working directory of the executable.PThreads-win32
 * is a library that provides a POSIX pthreads API for WIN32 environments. It is licensed
 * under the LGPL. You can find info and source @ http://sourceware.org/pthreads-win32/
 */

using namespace std;

#include <iostream>
#include "pthread.h"
#include "CInsim.h"

#define IS_PRODUCT_NAME "LFS Session Timing v0.1"
// The superadmin is the only person that can close the app with a chat command
// This is a USERNAME, so change it to your desire
#define SUPER_ADMIN "makakazo"
#define MAX_PLAYERS 32


// Global objects that must be accesed by all threads
CInsim insim;
pthread_mutex_t ismutex;


// Player struct. Contains all data related to a certain connected player.
struct player
{
    char UName[24];             // Username
    char PName[24];             // Player name
    byte UCID;                  // Connection ID
    byte PLID;                  // Player ID
    byte Admin;                 // 0 = not admin / 1 = is admin
    unsigned besttimes[5];      // Record of player's best lap: [0]=laptime / [1]=1st split / [2]=2nd split / etc.
    unsigned lasttimes[5];      // Record of player's last lap: [0]=laptime / [1]=1st split / [2]=2nd split / etc.
    byte lastsplit;             // Last split that the player crossed
    byte buttons;               // Button count for the help screen
    byte msg_off;               // Player setting for screen messages: 0 = ON / 1 = OFF
};


// Main struct than contains all info stored/used by the application
struct global_info
{
    struct player players[MAX_PLAYERS];     // Array of players
    unsigned sesbesttimes[5];               // Record of session best times for each sector (1-4) and lap (0)
    char UNamebesttimes[5][24];             // Usernames of players that have the current session best times
    char PNamebesttimes[5][24];             // As above, but player names
    unsigned sesbestlap[5];                 // Record of the session best lap
    char Track[6];                          // Current track
    byte permanent;                         // Global setting: 0 = times are permanent / 1 = times reset at each restart
};

/**
* Sends the help screen to the specified player "splayer"
* @param    splayer    Specified player
*/
void help_screen(struct player *splayer)
{
    // 100 was chosen as length, although IS buttons can take up to 200 characters.
    char rules_text[14][100];
    strncpy(rules_text[0], "LFS Session Timing v0.1",99);
    strncpy(rules_text[1], "^1CHAT COMMANDS",99);
    strncpy(rules_text[2], "^3 *) Type ^7!commands ^3to see the available chat commands (for timing info, etc.)",99);
    strncpy(rules_text[3], "^3 *) To bring this screen at any time type ^7!help",99);
    strncpy(rules_text[4], "^1DIFFERENCE TO BEST LAP",99);
    strncpy(rules_text[5], "^3 At each sector you will be shown the difference to the best lap at that point:",99);
    strncpy(rules_text[6], "^3 *) A ^2GREEN ^3difference means you are being faster than the current best lap",99);
    strncpy(rules_text[7], "^3 *) A ^1RED ^3difference means you are being slower than the current best lap",99);
    strncpy(rules_text[8], "^3 *) A ^7WHITE ^3difference (0.00) means you are doing the exact same time as the best lap",99);
    strncpy(rules_text[9], "^1SECTOR TIMES",99);
    strncpy(rules_text[10], "^3 You will be shown the time you have clocked in each sector (not aggregated, F1 style):",99);
    strncpy(rules_text[11], "^3 *) A ^5PURPLE ^3sector time means you did the best sector time of all drivers",99);
    strncpy(rules_text[12], "^3 *) A ^2GREEN ^3sector time means you did your personal best for that sector",99);
    strncpy(rules_text[13], "^3 *) A ^3YELLOW ^3sector time will be shown when none of the above applies",99);

    pthread_mutex_lock (&ismutex);

    // Create and fill the IS_BTN struct
    struct IS_BTN pack_btn;
    memset(&pack_btn, 0, sizeof(struct IS_BTN));
    pack_btn.Size = sizeof(struct IS_BTN);
    pack_btn.Type = ISP_BTN;
    pack_btn.ReqI = splayer->UCID;              // Must be non-zero, I'll just use UCID
    pack_btn.UCID = splayer->UCID;              // UCID of the player that will receive the button
    pack_btn.ClickID = splayer->buttons++;      // I use buttons from 0 onward for the help screen
    pack_btn.BStyle = ISB_DARK;                 // Dark frame for window title
    pack_btn.L = 0;
    pack_btn.T = 30;
    pack_btn.W = 110;
    pack_btn.H = 10;
    insim.send_packet(&pack_btn);

    // Re-fill and send the struct as many times as needed
    pack_btn.ClickID = splayer->buttons++;
    pack_btn.BStyle = ISB_LIGHT;                // Light frame for main text
    pack_btn.L = 0;
    pack_btn.T = 40;
    pack_btn.W = 110;
    pack_btn.H = 90;
    insim.send_packet(&pack_btn);

    pack_btn.ClickID = splayer->buttons++;
    // If no ISB_LIGHT or ISB_DARK used, then the background is transparent
    pack_btn.BStyle = ISB_LEFT + ISB_C2 + ISB_C4;   // Title text
    pack_btn.L = 0;
    pack_btn.T = 30;
    pack_btn.W = 110;
    pack_btn.H = 10;
    strcpy(pack_btn.Text, rules_text[0]);
    insim.send_packet(&pack_btn);

    int var_height = 0;
    for (int j=1; j<14; j++)
    {
        pack_btn.ClickID = splayer->buttons++;
        pack_btn.BStyle = ISB_LEFT;             // Main text lines
        pack_btn.L = 0;

        if ((j==4)||(j==9))
            var_height+=5;

        pack_btn.T = 40 + j*5 + var_height;
        pack_btn.W = 110;
        pack_btn.H = 5;
        strcpy(pack_btn.Text, rules_text[j]);
        insim.send_packet(&pack_btn);
    }

    pack_btn.ClickID = splayer->buttons++;                      // "Close" button
    pack_btn.BStyle = ISB_DARK + ISB_CLICK + ISB_C1 + ISB_C4;   // dark + clickable + red
    pack_btn.L = 94;
    pack_btn.T = 123;
    pack_btn.W = 15;
    pack_btn.H = 6;
    strcpy(pack_btn.Text, "Close");
    insim.send_packet(&pack_btn);

    pthread_mutex_unlock (&ismutex);
}

/**
* Shows the chat commands to the specified player "splayer"
* @param    splayer    Specified player
*/
void chat_commands(struct player *splayer)
{
    // 64 is the maximum text length allowed by IS_MTC packets
    char commands_text[11][64];
    strncpy(commands_text[0], "^1COMMANDS:",63);
    strncpy(commands_text[1], "^1=========",63);
    strncpy(commands_text[2], "^3 *) ^7!help ^3: Help screen",63);
    strncpy(commands_text[3], "^3 *) ^7!commands : ^3This list of chat commands",63);
    strncpy(commands_text[4], "^3 *) ^7!splits [player]: ^3Sector times analysis",63);
    strncpy(commands_text[5], "^3 ([player] is optional)",63);
    strncpy(commands_text[6], "^3 *) ^7!best : ^3Best global sector times",63);
    strncpy(commands_text[7], "^3 *) ^7!messages : ^3Toggle on/off screen messages",63);
    strncpy(commands_text[8], "^1Admin commands:",63);
    strncpy(commands_text[9], "^3 *) ^7!reset : ^3Resets all times stored",63);
    strncpy(commands_text[10], "^3 *) ^7!permanent : ^3Toggle permanent/reset times",63);

    // We use a "message to connection" packet
    struct IS_MTC pack_mtc;
    memset(&pack_mtc, 0, sizeof(struct IS_MTC));
    pack_mtc.Size = sizeof(struct IS_MTC);
    pack_mtc.Type = ISP_MTC;
    pack_mtc.UCID = splayer->UCID;

    pthread_mutex_lock (&ismutex);

    for (int j=0; j<8; j++){    // Messages for all players
        strcpy(pack_mtc.Msg, commands_text[j]);
        insim.send_packet(&pack_mtc);
    }
    if (splayer->Admin)         // Messages only for admins
    {
        for (int j=8; j<11; j++){
            strcpy(pack_mtc.Msg, commands_text[j]);
            insim.send_packet(&pack_mtc);
        }
    }

    pthread_mutex_unlock (&ismutex);
}

/**
* Clears splits and difference screen messages.
* This function is passed as a parameter when creating the cleaning threads.
* Sleeps for 5.5 seconds and then clears the messages.
* @param    pasUCID     UCID of the player whose messages will be cleared
*/
void *clear_msg(void *pasUCID)
{
    Sleep(5500);

    byte *UCID = (byte *)pasUCID;

    pthread_mutex_lock (&ismutex);

    struct IS_BFN pack_bfn;
    memset(&pack_bfn, 0, sizeof(struct IS_BFN));
    pack_bfn.Size = sizeof(struct IS_BFN);
    pack_bfn.Type = ISP_BFN;
    pack_bfn.SubT = BFN_DEL_BTN;
    pack_bfn.UCID = *UCID;
    pack_bfn.ClickID = 239;         // Fixed button ID
    insim.send_packet(&pack_bfn);

    pack_bfn.ClickID = 238;         // Fixed button ID
    insim.send_packet(&pack_bfn);

    pthread_mutex_unlock (&ismutex);

    pthread_exit(NULL);

    return 0;
}

/**
* Resets all times!
* @param    ginfo   Struct that contains all the info stored by the application
*/
void reset_times(struct global_info *ginfo)
{
    // Reset session times
    for(int k=0; k<5; k++)
    {
        ginfo->sesbesttimes[k] = 3600000;
        ginfo->sesbestlap[k] = 3600000;
    }

    // Reset player times
    for (int k=0; k<MAX_PLAYERS; k++)
    {
        for (int j=0; j<5; j++)
        {
            ginfo->players[k].besttimes[j] = 3600000;
            ginfo->players[k].lasttimes[j] = 3600000;
        }
    }
}

/**
* When a IS_REO packet arrives means that practice/qualifying/race is restarting.
* If times are not set to permanent, then times are reset.
* @param    ginfo   Struct that contains all the info stored by the application
*/
void case_reo (struct global_info *ginfo)
{
    cout << " * IS_REO (grid reorder)" << endl;
    if (ginfo->permanent == 0)
    {
        reset_times(ginfo);
        cout << "(All times reset!)" << endl;
    }
}

/**
* Resets times if the track was changed
* @param    ginfo   Struct that contains all the info stored by the application
*/
void case_sta (struct global_info *ginfo)
{
    cout << " * IS_STA (state info)" << endl;
    struct IS_STA *pack_sta = (struct IS_STA*)insim.get_packet();

    // ReqI != 0, so I asked for this packet at the beginning of the main, just to copy the track name
    if (pack_sta->ReqI != 0)
    {
        strcpy(ginfo->Track, pack_sta->Track);
        cout << "(Current track stored)" << endl;
        return;
    }

    // I didn't ask for this packet. If the track was changed then times must be reset
    if (strcmp(ginfo->Track, pack_sta->Track) != 0)
    {
        strcpy(ginfo->Track, pack_sta->Track);
        reset_times(ginfo);
        cout << "(All times reset!)" << endl;
    }
}

/**
* Actions to perform when a player crosses a split (but not when a lap is completed)
* @param    ginfo   Struct that contains all the info stored by the application
*/
void case_spx (struct global_info *ginfo)
{
    cout << " * IS_SPX (split time)" << endl;
    int i;

    struct IS_SPX *pack_spx = (struct IS_SPX*)insim.get_packet();

    // Find the player in the player list
    for (i=0; i < MAX_PLAYERS; i++)
    {
        if (ginfo->players[i].PLID == pack_spx->PLID)
        {
            cout << "ginfo->players[" << i << "].PLID found" << endl;
            cout << "UName: " << ginfo->players[i].UName << endl;
            cout << "UCID: " << (int)ginfo->players[i].UCID << endl;
            cout << "PLID: " << (int)ginfo->players[i].PLID << endl;
            cout << "Split: " << (int)pack_spx->STime << " ms" << endl;
            break;
        }
    }

    // C string vars that will be used later
    char strtime[13];
    char splittext[15];
    char diftext[15];
    char numsplit[2];

    // Perform all the stuff only if it's not a non-timed lap
    if (pack_spx->STime != 3600000)
    {
        cout << "(Timed lap, performing stuff...)" << endl;
        // Update the last split (number) for this player
        ginfo->players[i].lastsplit = pack_spx->Split;

        // Store the last split time into "lastsectortime". This is aggregated!
        unsigned lastsectortime = pack_spx->STime;

        // We must subtract the previous sector times to this split time to obtain the current sector time
        for (int j=ginfo->players[i].lastsplit-1; j>0; j--)
        {
            lastsectortime -= ginfo->players[i].lasttimes[j];
        }
        // Now we can store the current sector time
        ginfo->players[i].lasttimes[pack_spx->Split] = lastsectortime;

        // We are going to build up the split time button text into "splittext". By default we use yellow
        strcpy(splittext, "^3");

        // If you just did a personal best for this sector: update and use green
        if (lastsectortime < ginfo->players[i].besttimes[pack_spx->Split])
        {
            ginfo->players[i].besttimes[pack_spx->Split] = lastsectortime;
            strcpy(splittext, "^2");
            cout << "PERSONAL BEST SECTOR" << endl;

            // If you actually did the global best: update and use purple
            if (lastsectortime < ginfo->sesbesttimes[pack_spx->Split])
            {
                ginfo->sesbesttimes[pack_spx->Split] = lastsectortime;
                strcpy(ginfo->UNamebesttimes[pack_spx->Split], ginfo->players[i].UName);
                strcpy(ginfo->PNamebesttimes[pack_spx->Split], ginfo->players[i].PName);
                strcpy(splittext, "^5");
                cout << "GLOBAL BEST SECTOR!" << endl;
            }
        }

        // If the player has messages turned off then return, no need to send the button stuff
        if (ginfo->players[i].msg_off)
            return;

        // Var to calculate the difference to the best lap at this point.
        // 0 by default will be used if no lap has been set.
        long difference = 0;

        // Calculate the difference only if someone already set a time
        if (ginfo->sesbestlap[0] != 3600000)
        {
            long playerhere = 0;
            long besthere = 0;

            for (int j=1; j<pack_spx->Split + 1; j++)
            {
                besthere += ginfo->sesbestlap[j];
                playerhere += ginfo->players[i].lasttimes[j];
            }

            difference = playerhere - besthere;
            cout << "difference = " << playerhere << " - " << besthere << endl;
        }
        cout << "difference: " << difference << endl;

        pthread_mutex_lock (&ismutex);

        // This first button is the split time coloured
        struct IS_BTN pack_btn;
        memset(&pack_btn, 0, sizeof(struct IS_BTN));
        pack_btn.Size = sizeof(struct IS_BTN);
        pack_btn.Type = ISP_BTN;
        pack_btn.ReqI = ginfo->players[i].PLID;
        pack_btn.UCID = ginfo->players[i].UCID;
        pack_btn.ClickID = 239;     // Fixed click ID.
        pack_btn.L = 85;
        pack_btn.T = 60;
        pack_btn.W = 30;
        pack_btn.H = 10;
        strcat(splittext, "S");
        strcat(splittext, itoa(pack_spx->Split,numsplit,10));
        strcat(splittext, " - ");
        strcat(splittext, mstostr(ginfo->players[i].lasttimes[pack_spx->Split], strtime));
        strcpy(pack_btn.Text, splittext);
        insim.send_packet(&pack_btn);

        // The second button is the difference to the best lap at this point
        pack_btn.ClickID = 238;     // Fixed click ID.
        pack_btn.L = 85;
        pack_btn.T = 52;
        pack_btn.W = 30;
        pack_btn.H = 8;

        if (difference > 0)
            strcpy(diftext, "^1(+");
        else if (difference < 0)
            strcpy(diftext, "^2(");
        else
            strcpy(diftext, "^7(");

        strcat(diftext, mstostr(difference, strtime));
        strcat(diftext, ")");
        strcpy(pack_btn.Text, diftext);
        insim.send_packet(&pack_btn);

        pthread_mutex_unlock (&ismutex);

        // Create a thread to clear this buttons after 5.5 seconds. Will call clear_msg()
        int rc;
        pthread_t thread;
        rc = pthread_create(&thread, NULL, clear_msg, (void *)&ginfo->players[i].UCID);
        if (rc){
            printf("ERROR; return code from pthread_create() is %d\n", rc);
            exit(-1);
        }
    }
    else
        cout << "(non-timed lap, not doing anything...)" << endl;
}

/**
* Actions performed when someone types something in the chat (packet IS_MSO)
* @param    ginfo   Struct that contains all the info stored by the application
* @return   1 in normal cases, -1 if the super admin requested !exit
*/
int case_mso (struct global_info *ginfo)
{
    cout << " * IS_MSO (chat message)" << endl;
    int i;

    struct IS_MSO *pack_mso = (struct IS_MSO*)insim.get_packet();

    // The chat message is sent by the host, don't do anything
    if (pack_mso->UCID == 0)
    {
        cout << "(Chat message by host: " << pack_mso->Msg + ((unsigned char)pack_mso->TextStart) << ")" << endl;
        return 1;
    }

    // Find the player that wrote in the chat
    for (i=0; i < MAX_PLAYERS; i++)
    {
        if (ginfo->players[i].UCID == pack_mso->UCID)
        {
            cout << "ginfo->players[" << i << "].UCID found" << endl;
            cout << "UName: " << ginfo->players[i].UName << endl;
            cout << "UCID: " << (int)ginfo->players[i].UCID << endl;
            cout << "PLID: " << (int)ginfo->players[i].PLID << endl;
            cout << "Msg: " << pack_mso->Msg << endl;
            break;
        }
    }

    // !help
    if (strncmp(pack_mso->Msg + ((unsigned char)pack_mso->TextStart), "!help", 5) == 0)
    {
        cout << "(!help requested)" << endl;
        help_screen(&ginfo->players[i]);
        return 1;
    }

    // !commands
    if (strncmp(pack_mso->Msg + ((unsigned char)pack_mso->TextStart), "!commands", 9) == 0)
    {
        cout << "(!commands requested)" << endl;
        chat_commands(&ginfo->players[i]);
        return 1;
    }

    // !splits require more work
    if (strncmp(pack_mso->Msg + ((unsigned char)pack_mso->TextStart), "!splits", 7) == 0)
    {
        // We will store the UCID of the player requested in the !splits command
        // By default it's the same who typed in the command
        byte request = i;

        if (strlen(pack_mso->Msg + ((unsigned char)pack_mso->TextStart)) > 8)
        {
            for (int j=0; j<32; j++)
            {
                // We give the possiblity to enter either username or player name
                if (strcmp(pack_mso->Msg + ((unsigned char)pack_mso->TextStart) + 8, ginfo->players[j].UName) == 0)
                {
                    request = j;
                    break;
                }
                if (strcmp(pack_mso->Msg + ((unsigned char)pack_mso->TextStart) + 8, ginfo->players[j].PName) == 0)
                {
                    request = j;
                    break;
                }
            }
        }

        cout << "(!splits requested for player with UCID: " << (int)ginfo->players[request].UCID << ")" << endl;

        // Requested player hasn't completed a timed lap yet. No job done and return.
        if (ginfo->players[request].besttimes[0] == 3600000)
        {
            pthread_mutex_lock (&ismutex);
            struct IS_MTC pack_mtc;
            memset(&pack_mtc, 0, sizeof(struct IS_MTC));
            pack_mtc.Size = sizeof(struct IS_MTC);
            pack_mtc.Type = ISP_MTC;
            pack_mtc.UCID = ginfo->players[i].UCID;
            strcpy(pack_mtc.Msg, ginfo->players[request].PName);
            strcat(pack_mtc.Msg, "^3 hasnt' set any time yet");
            insim.send_packet(&pack_mtc);
            pthread_mutex_unlock (&ismutex);

            return 1;
        }

        // Message to connection for the guy who typed the command in
        pthread_mutex_lock (&ismutex);
        char strtime[13];
        struct IS_MTC pack_mtc;
        memset(&pack_mtc, 0, sizeof(struct IS_MTC));
        pack_mtc.Size = sizeof(struct IS_MTC);
        pack_mtc.Type = ISP_MTC;
        pack_mtc.UCID = ginfo->players[i].UCID;
        strcpy(pack_mtc.Msg, "^3Times by ");
        strcat(pack_mtc.Msg, ginfo->players[request].PName);
        insim.send_packet(&pack_mtc);

        strcpy(pack_mtc.Msg, "^3Best lap: ");

        // First we do the global lap time
        // We check if he has the session best to apply purple colour
        if (ginfo->players[request].besttimes[0] == ginfo->sesbesttimes[0])
        {
            strcat(pack_mtc.Msg, "^5");
            strcat(pack_mtc.Msg, mstostr(ginfo->players[request].besttimes[0], strtime));
        }
        else
        {
            strcat(pack_mtc.Msg, "^7");
            strcat(pack_mtc.Msg, mstostr(ginfo->players[request].besttimes[0], strtime));
            // We also show the difference to the best time
            strcat(pack_mtc.Msg, "^1 (+");
            strcat(pack_mtc.Msg, mstostr(ginfo->players[request].besttimes[0] - ginfo->sesbesttimes[0], strtime));
            strcat(pack_mtc.Msg, ")");
        }

        insim.send_packet(&pack_mtc);

        unsigned best_possible = 0;

        // Now we use a "for" loop for each sector
        for(int j=1; j<5; j++)
        {
            if (ginfo->players[request].besttimes[j] == 3600000)
                break;

            strcpy(pack_mtc.Msg, "^3Best Sector");
            itoa(j, pack_mtc.Msg+13, 10);
            strcat(pack_mtc.Msg, ": ");

            if (ginfo->players[request].besttimes[j] == ginfo->sesbesttimes[j])
            {
                strcat(pack_mtc.Msg, "^5");
                strcat(pack_mtc.Msg, mstostr(ginfo->players[request].besttimes[j], strtime));
            }
            else
            {
                strcat(pack_mtc.Msg, "^7");
                strcat(pack_mtc.Msg, mstostr(ginfo->players[request].besttimes[j], strtime));
                strcat(pack_mtc.Msg, "^1 (+");
                strcat(pack_mtc.Msg, mstostr(ginfo->players[request].besttimes[j] - ginfo->sesbesttimes[j], strtime));
                strcat(pack_mtc.Msg, ")");
            }

            insim.send_packet(&pack_mtc);

            best_possible += ginfo->players[request].besttimes[j];
        }

        // Finally we calculate this players best possible lap adding up his best sector times
        // We also show the difference to the best time
        char diftext[15];
        long difference = best_possible - ginfo->sesbesttimes[0];

        if (difference > 0)
            strcpy(diftext, "^1 (+");
        else if (difference < 0)
            strcpy(diftext, "^2 (");
        else
            strcpy(diftext, "^7 (");

        strcat(diftext, mstostr(difference, strtime));
        strcat(diftext, ")");

        strcpy(pack_mtc.Msg, "^3Best possible lap: ^7");
        strcat(pack_mtc.Msg, mstostr(best_possible, strtime));
        strcat(pack_mtc.Msg, diftext);
        insim.send_packet(&pack_mtc);

        pthread_mutex_unlock (&ismutex);
        return 1;
    }

    // !best shows the best sector and lap times
    if (strncmp(pack_mso->Msg + ((unsigned char)pack_mso->TextStart), "!best", 5) == 0)
    {
        cout << "(!best requested)" << endl;
        // No one has set a time yet. No job done and return.
        if (ginfo->sesbesttimes[0] == 3600000)
        {
            pthread_mutex_lock (&ismutex);
            struct IS_MTC pack_mtc;
            memset(&pack_mtc, 0, sizeof(struct IS_MTC));
            pack_mtc.Size = sizeof(struct IS_MTC);
            pack_mtc.Type = ISP_MTC;
            pack_mtc.UCID = ginfo->players[i].UCID;
            strcpy(pack_mtc.Msg, "^3No time has been set yet");
            insim.send_packet(&pack_mtc);
            pthread_mutex_unlock (&ismutex);

            return 1;
        }

        // First of all, the best lap, with player name included
        pthread_mutex_lock (&ismutex);
        char strtime[13];
        struct IS_MTC pack_mtc;
        memset(&pack_mtc, 0, sizeof(struct IS_MTC));
        pack_mtc.Size = sizeof(struct IS_MTC);
        pack_mtc.Type = ISP_MTC;
        pack_mtc.UCID = ginfo->players[i].UCID;
        strcpy(pack_mtc.Msg, "^3Best time: ^7");
        strcat(pack_mtc.Msg, ginfo->PNamebesttimes[0]);
        strcat(pack_mtc.Msg, " ^5");
        strcat(pack_mtc.Msg, mstostr(ginfo->sesbesttimes[0], strtime));
        insim.send_packet(&pack_mtc);

        // Define best possible lap here
        unsigned best_possible = 0;

        // Show best sectors and add them up to calculate the best possible lap
        for(int j=1; j<5; j++)
        {
            // Use this "break" to show only the number of sectors that the track contains
            if (ginfo->sesbesttimes[j] == 3600000)
                break;

            strcpy(pack_mtc.Msg, "^3Best Sector");
            itoa(j, pack_mtc.Msg+13, 10);
            strcat(pack_mtc.Msg, ": ^7");
            strcat(pack_mtc.Msg, ginfo->PNamebesttimes[j]);
            strcat(pack_mtc.Msg, " ^5");
            strcat(pack_mtc.Msg, mstostr(ginfo->sesbesttimes[j], strtime));
            insim.send_packet(&pack_mtc);

            best_possible += ginfo->sesbesttimes[j];
        }

        // Calculate and show the difference between the best possible and best actual lap
        char diftext[15];
        long difference = best_possible - ginfo->sesbesttimes[0];

        if (difference > 0)
            strcpy(diftext, "^1 (+");
        else if (difference < 0)
            strcpy(diftext, "^2 (");
        else
            strcpy(diftext, "^7 (");

        strcat(diftext, mstostr(difference, strtime));
        strcat(diftext, ")");

        strcpy(pack_mtc.Msg, "^3Best possible lap: ^7");
        strcat(pack_mtc.Msg, mstostr(best_possible, strtime));
        strcat(pack_mtc.Msg, diftext);
        insim.send_packet(&pack_mtc);

        pthread_mutex_unlock (&ismutex);
        return 1;
    }

    // !messages toggles between showing or not the splits and difference "messages"
    if (strcmp(pack_mso->Msg + ((unsigned char)pack_mso->TextStart), "!messages") == 0)
    {
        cout << "(!messages toggle requested)" << endl;
        pthread_mutex_lock (&ismutex);

        struct IS_MTC pack_mtc;
        memset(&pack_mtc, 0, sizeof(struct IS_MTC));
        pack_mtc.Size = sizeof(struct IS_MTC);
        pack_mtc.Type = ISP_MTC;
        pack_mtc.UCID = ginfo->players[i].UCID;
        strcpy(pack_mtc.Msg, "^3Screen messages ^7");

        // If it was to zero, toggle it to 1 (messages off) and clear all possible buttons showing right now
        if (ginfo->players[i].msg_off == 0)
        {
            ginfo->players[i].msg_off = 1;
            strcat(pack_mtc.Msg, "OFF");

            struct IS_BFN pack_bfn;
            memset(&pack_bfn, 0, sizeof(struct IS_BFN));
            pack_bfn.Size = sizeof(struct IS_BFN);
            pack_bfn.Type = ISP_BFN;
            pack_bfn.SubT = BFN_CLEAR;
            pack_bfn.UCID = ginfo->players[i].UCID;
            insim.send_packet(&pack_bfn);
        }
        else    // Toggle it to 0 (messages on)
        {
            ginfo->players[i].msg_off = 0;
            strcat(pack_mtc.Msg, "ON");
        }

        insim.send_packet(&pack_mtc);

        pthread_mutex_unlock (&ismutex);
        return 1;
    }

    // !exit can only be requested by the "super admin"
    if (strncmp(pack_mso->Msg + ((unsigned char)pack_mso->TextStart), "!exit", 6) == 0)
    {
        cout << "(!exit requested)" << endl;
        if (strcmp(ginfo->players[i].UName, SUPER_ADMIN) != 0)
            return 1;

        pthread_mutex_lock (&ismutex);
        struct IS_MST pack_mst;
        memset(&pack_mst, 0, sizeof(struct IS_MST));
        pack_mst.Size = sizeof(struct IS_MST);
        pack_mst.Type = ISP_MST;
        strcpy(pack_mst.Msg, "^3* Closing application (requested by admin) *");
        insim.send_packet(&pack_mst);

        pthread_mutex_unlock (&ismutex);
        return -1;
    }

    // This commands can be performed only by admins
    if (ginfo->players[i].Admin)
    {
        // !reset restes all times
        if (strncmp(pack_mso->Msg + ((unsigned char)pack_mso->TextStart), "!reset", 6) == 0)
        {
            cout << "(!reset requested by admin)" << endl;
            reset_times(ginfo);
            cout << "(All times were reset!)" << endl;

            pthread_mutex_lock (&ismutex);
            struct IS_MST pack_mst;
            memset(&pack_mst, 0, sizeof(struct IS_MST));
            pack_mst.Size = sizeof(struct IS_MST);
            pack_mst.Type = ISP_MST;
            strcpy(pack_mst.Msg, "^3* All times reset! *");
            insim.send_packet(&pack_mst);

            pthread_mutex_unlock (&ismutex);
            return 1;
        }

        // !permanent toggles between permanent times/times reset at each restart
        if (strncmp(pack_mso->Msg + ((unsigned char)pack_mso->TextStart), "!permanent", 10) == 0)
        {
            cout << "(!permanente toggle requested by admin)" << endl;
            // toggle var
            ginfo->permanent = (++ginfo->permanent) % 2;

            pthread_mutex_lock (&ismutex);
            struct IS_MST pack_mst;
            memset(&pack_mst, 0, sizeof(struct IS_MST));
            pack_mst.Size = sizeof(struct IS_MST);
            pack_mst.Type = ISP_MST;
            strcpy(pack_mst.Msg, "^3* Times: ");

            if (ginfo->permanent == 0)
                strcat(pack_mst.Msg, "^7RESET AT EACH RESTART");
            else
                strcat(pack_mst.Msg, "^7PERMANENT");

            insim.send_packet(&pack_mst);

            pthread_mutex_unlock (&ismutex);
            return 1;
        }
    }

    return 1;
}


/**
* Actions performed when a player completes a lap
* @param    ginfo   Struct that contains all the info stored by the application
*/
void case_lap (struct global_info *ginfo)
{
    cout << " * IS_LAP (lap completed)" << endl;
    int i;

    struct IS_LAP *pack_lap = (struct IS_LAP*)insim.get_packet();

    // Find the player
    for (i=0; i < MAX_PLAYERS; i++)
    {
        if (ginfo->players[i].PLID == pack_lap->PLID)
        {
            cout << "ginfo->players[" << i << "].PLID found" << endl;
            cout << "UName: " << ginfo->players[i].UName << endl;
            cout << "UCID: " << (int)ginfo->players[i].UCID << endl;
            cout << "PLID: " << (int)ginfo->players[i].PLID << endl;
            break;
        }
    }

    // Update his last lap time
    ginfo->players[i].lasttimes[0] = pack_lap->LTime;

    char strtime[13];
    char splittext[15];
    char diftext[15];
    char numsplit[2];

    // Perform these actions only if it's a timed lap
    if (pack_lap->LTime != 3600000)
    {
        cout << "(Timed lap, performing stuff...)" << endl;
        // Calculate the last sector time
        unsigned lastsectortime = pack_lap->LTime;

        for (int j=ginfo->players[i].lastsplit; j>0; j--)
        {
            lastsectortime -= ginfo->players[i].lasttimes[j];
        }
        ginfo->players[i].lasttimes[ginfo->players[i].lastsplit+1] = lastsectortime;

        // "splittext" used to store split text
        strcpy(splittext, "^3");

        // personal best split
        if (lastsectortime < ginfo->players[i].besttimes[ginfo->players[i].lastsplit+1])
        {
            ginfo->players[i].besttimes[ginfo->players[i].lastsplit+1] = lastsectortime;
            strcpy(splittext, "^2");
            cout << "PERSONAL BEST SPLIT" << endl;

            // Actually it was a global best split
            if (lastsectortime < ginfo->sesbesttimes[ginfo->players[i].lastsplit+1])
            {
                ginfo->sesbesttimes[ginfo->players[i].lastsplit+1] = lastsectortime;
                strcpy(ginfo->UNamebesttimes[ginfo->players[i].lastsplit+1], ginfo->players[i].UName);
                strcpy(ginfo->PNamebesttimes[ginfo->players[i].lastsplit+1], ginfo->players[i].PName);
                strcpy(splittext, "^5");
                cout << "GLOBAL BEST SPLIT!" << endl;
            }
        }

        long previous_best_lap = ginfo->sesbesttimes[0];

        // personal best lap
        if (pack_lap->LTime < ginfo->players[i].besttimes[0])
        {
            ginfo->players[i].besttimes[0] = pack_lap->LTime;

            // global best lap
            if (pack_lap->LTime < ginfo->sesbesttimes[0])
            {
                ginfo->sesbesttimes[0] = pack_lap->LTime;
                strcpy(ginfo->UNamebesttimes[0], ginfo->players[i].UName);
                strcpy(ginfo->PNamebesttimes[0], ginfo->players[i].PName);
                cout << "GLOBAL BEST LAP!" << endl;

                for (int j=0; j<5; j++)
                    ginfo->sesbestlap[j] = ginfo->players[i].lasttimes[j];
            }

            cout << "PERSONAL BEST LAP" << endl;
            cout << "PB by player \"" << ginfo->players[i].UName << "\": " << ginfo->players[i].besttimes[0] << " ms" << endl;

            // Send chat message to notify of new personal best for the session
            pthread_mutex_lock (&ismutex);
            char strtime[13];
            struct IS_MTC pack_mtc;
            memset(&pack_mtc, 0, sizeof(struct IS_MTC));
            pack_mtc.Size = sizeof(struct IS_MTC);
            pack_mtc.Type = ISP_MTC;
            pack_mtc.UCID = ginfo->players[i].UCID;
            strcpy(pack_mtc.Msg, "^3New PB for this session: ");

            // Will be shown in purple if it's the global best, green otherwise
            if(ginfo->sesbesttimes[0] != pack_lap->LTime)
                strcat(pack_mtc.Msg, "^7");
            else
                strcat(pack_mtc.Msg, "^5");

            strcat(pack_mtc.Msg, mstostr(pack_lap->LTime, strtime));
            insim.send_packet(&pack_mtc);
            pthread_mutex_unlock (&ismutex);
        }

        // If messages are turned off for this player, no buttons will be sent
        if (ginfo->players[i].msg_off)
            return;

        // Difference to the best lap. Default 0, used if no time has been set yet
        long difference = 0;

        // Calculate difference to the previous best lap
        if (previous_best_lap != 3600000)
        {
            difference = pack_lap->LTime - previous_best_lap;
            cout << "difference = " << pack_lap->LTime << " - " << previous_best_lap << endl;
        }
        cout << "difference: " << difference << endl;

        pthread_mutex_lock (&ismutex);

        // first button with split info (coloured)
        struct IS_BTN pack_btn;
        memset(&pack_btn, 0, sizeof(struct IS_BTN));
        pack_btn.Size = sizeof(struct IS_BTN);
        pack_btn.Type = ISP_BTN;
        pack_btn.ReqI = ginfo->players[i].PLID;
        pack_btn.UCID = ginfo->players[i].UCID;
        pack_btn.ClickID = 239;     // Fixed click ID.
        pack_btn.L = 85;
        pack_btn.T = 60;
        pack_btn.W = 30;
        pack_btn.H = 10;
        strcat(splittext, "S");
        strcat(splittext, itoa(ginfo->players[i].lastsplit+1,numsplit,10));
        strcat(splittext, " - ");
        strcat(splittext, mstostr(ginfo->players[i].lasttimes[ginfo->players[i].lastsplit+1], strtime));
        strcpy(pack_btn.Text, splittext);
        insim.send_packet(&pack_btn);

        // second button, difference to the previous best lap
        pack_btn.ClickID = 238;     // Fixed click ID.
        pack_btn.L = 85;
        pack_btn.T = 52;
        pack_btn.W = 30;
        pack_btn.H = 8;

        if (difference > 0)
            strcpy(diftext, "^1(+");
        else if (difference < 0)
            strcpy(diftext, "^2(");
        else
            strcpy(diftext, "^7(");

        strcat(diftext, mstostr(difference, strtime));
        strcat(diftext, ")");
        strcpy(pack_btn.Text, diftext);
        insim.send_packet(&pack_btn);

        pthread_mutex_unlock (&ismutex);

        // Create a thread to clear this buttons after 5.5 seconds. Will call clear_msg()
        int rc;
        pthread_t thread;
        rc = pthread_create(&thread, NULL, clear_msg, (void *)&ginfo->players[i].UCID);
        if (rc){
            printf("ERROR; return code from pthread_create() is %d\n", rc);
            exit(-1);
        }
    }
    else
        cout << "(Non-timed lap, not doing anything...)" << endl;

    cout << "Laptime: " << (int)ginfo->players[i].lasttimes[0] << " ms" << endl;
    cout << "Last sector: " << (int)ginfo->players[i].lasttimes[ginfo->players[i].lastsplit+1] << " ms" << endl;
}

/**
* Actions performed when a button is clicked.
* The only clickable button we have is the "close" button in the help screen, so the action
* +to perform is to close that window.
* @param    ginfo   Struct that contains all the info stored by the application
*/
void case_btc (struct global_info *ginfo)
{
    cout << " * IS_BTC (button click)" << endl;
    int i;

    struct IS_BTC *pack_btc = (struct IS_BTC*)insim.get_packet();

    // Find the player
    for (i=0; i < MAX_PLAYERS; i++)
    {
        if (ginfo->players[i].UCID == pack_btc->UCID)
        {
            cout << "ginfo->players[" << i << "].UCID found" << endl;
            cout << "UName: " << ginfo->players[i].UName << endl;
            cout << "UCID: " << (int)ginfo->players[i].UCID << endl;
            cout << "PLID: " << (int)ginfo->players[i].PLID << endl;
            break;
        }
    }

    // Create and fill an IS_BFN packet
    pthread_mutex_lock (&ismutex);
    struct IS_BFN pack_bfn;
    memset(&pack_bfn, 0, sizeof(struct IS_BFN));
    pack_bfn.Size = sizeof(struct IS_BFN);
    pack_bfn.Type = ISP_BFN;
    pack_bfn.SubT = BFN_DEL_BTN;
    pack_bfn.UCID = ginfo->players[i].UCID;

    // Clear all buttons from the help screen. This won't clear split or difference messages.
    cout << "(Clearing buttons from help screen...)" << endl;
    for(int j=ginfo->players[i].buttons; j>0; j--)
    {
        pack_bfn.ClickID = j-1;
        insim.send_packet(&pack_bfn);
    }

    pthread_mutex_unlock (&ismutex);

    ginfo->players[i].buttons = 0;
}

/**
* New connection. Add the player to the list and send welcome text and help screen.
* @param    ginfo   Struct that contains all the info stored by the application
*/
void case_ncn (struct global_info *ginfo)
{
    cout << " * IS_NCN (new connection)" << endl;
    int i;

    struct IS_NCN *pack_ncn = (struct IS_NCN*)insim.get_packet();

    // UCID == 0 means it's the host, don't add it to the array.
    // I use UCID=0 for empty slots within the player array.
    if (pack_ncn->UCID == 0)
    {
        cout << "(Host connected, not adding him to array...)" << endl;
        return;
    }

    // Stop when the first empty slot is found, checking UCID==0
    for (i=0; i<MAX_PLAYERS; i++)
        if (ginfo->players[i].UCID == 0)
            break;

    // If i reached MAX_PLAYERS then we have more players than what the application expects.
    // MAX_PLAYERS should be set to the actual maximum number of drivers that the server allows.
    if (i == MAX_PLAYERS)
    {
        cout << "NO MORE PEOPLE ALLOWED!!!" << endl;
        return;
    }

    // Copy all the player data we need into the ginfo->players[] array
    cout << "(Initializing player data into global struct)" << endl;
    strcpy(ginfo->players[i].UName, pack_ncn->UName);
    strcpy(ginfo->players[i].PName, pack_ncn->PName);
    ginfo->players[i].UCID = pack_ncn->UCID;
    ginfo->players[i].Admin = pack_ncn->Admin;

    // Initialize player times to max value (1 hour).
    for(int j=0; j<5; j++)
    {
        ginfo->players[i].besttimes[j] = 3600000;
        ginfo->players[i].lasttimes[j] = 3600000;
    }

    cout << "NEW player connected!" << endl;
    cout << "UName:" << ginfo->players[i].UName << endl;
    cout << "UCID: " << (int)ginfo->players[i].UCID << endl;
    cout << "PLID: " << (int)ginfo->players[i].PLID << endl;

    // Send a welcome message
    pthread_mutex_lock (&ismutex);
    struct IS_MTC pack_mtc;
    memset(&pack_mtc, 0, sizeof(struct IS_MTC));
    pack_mtc.Size = sizeof(struct IS_MTC);
    pack_mtc.Type = ISP_MTC;
    pack_mtc.UCID = ginfo->players[i].UCID;
    strcpy(pack_mtc.Msg, "^3Welcome! ^7LFS Session Timing v0.1 ^3is running...");
    insim.send_packet(&pack_mtc);
    pthread_mutex_unlock (&ismutex);

    // Send the help screen as the player connects
    help_screen(&ginfo->players[i]);
}

/**
* Player changed his pilot name (player name). Update the affected data.
* @param    ginfo   Struct that contains all the info stored by the application
*/
void case_cpr (struct global_info *ginfo)
{
    cout << " * IS_CPR (player rename)" << endl;
    int i;

    struct IS_CPR *pack_cpr = (struct IS_CPR*)insim.get_packet();

    // Find the player
    for (i=0; i < MAX_PLAYERS; i++)
    {
        if (ginfo->players[i].UCID == pack_cpr->UCID)
        {
            cout << "ginfo->players[" << i << "].PLID found" << endl;
            cout << "UName: " << ginfo->players[i].UName << endl;
            cout << "UCID: " << (int)ginfo->players[i].UCID << endl;
            cout << "PLID: " << (int)ginfo->players[i].PLID << endl;
            break;
        }
    }

    // Update the player name
    strcpy(ginfo->players[i].PName, pack_cpr->PName);
    cout << "(Player changed name to " << ginfo->players[i].PName << ")" << endl;

    // Check the "UNamebesttimes" to update the player names of the best times
    for (int j=0; j < 5; j++)
    {
        if (strcmp(ginfo->UNamebesttimes[j], ginfo->players[i].UName) == 0)
        {
            strcpy(ginfo->PNamebesttimes[j], ginfo->players[i].PName);
        }
    }
}

/**
* New player joining race or leaving pits
* @param    ginfo   Struct that contains all the info stored by the application
*/
void case_npl (struct global_info *ginfo)
{
    cout << " * IS_NPL (joining race or leaving pits)" << endl;
    int i;

    struct IS_NPL *pack_npl = (struct IS_NPL*)insim.get_packet();

    // Find player using UCID and update his PLID
    for (i=0; i < MAX_PLAYERS; i++)
    {
        if (ginfo->players[i].UCID == pack_npl->UCID)
        {
            cout << "ginfo->players[" << i << "].UCID found" << endl;
            cout << "UName: " << ginfo->players[i].UName << endl;
            cout << "UCID: " << (int)ginfo->players[i].UCID << endl;
            ginfo->players[i].PLID = pack_npl->PLID;
            cout << "PLID: " << (int)ginfo->players[i].PLID << " (new one if joining, or the same he had if exiting pits)" << endl;
            break;
        }
    }

    // We request this packets at the beginning of the main, to check all the players who were in the server when the application
    // + was started. We send this players to spectators so they start clean.
    if (pack_npl->ReqI == 1)
    {
        pthread_mutex_lock (&ismutex);
        struct IS_MST pack_mst;
        memset(&pack_mst, 0, sizeof(struct IS_MST));
        pack_mst.Size = sizeof(struct IS_MST);
        pack_mst.Type = ISP_MST;
        strcpy(pack_mst.Msg, "/spec ");
        strcpy(pack_mst.Msg + 6, pack_npl->PName);
        insim.send_packet(&pack_mst);

        pthread_mutex_unlock (&ismutex);
        cout << "(Player " << pack_npl->PName << " was in the grid, taken to spectators)" << endl;
    }
}

/**
* Player leaves race (spectates - loses slot)
* @param    ginfo   Struct that contains all the info stored by the application
*/
void case_pll (struct global_info *ginfo)
{
    cout << " * IS_PLL (player leaves race)" << endl;
    int i;

    struct IS_PLL *pack_pll = (struct IS_PLL*)insim.get_packet();

    // Find player and set his PLID to 0
    for (i=0; i < MAX_PLAYERS; i++)
    {
        if (ginfo->players[i].PLID == pack_pll->PLID)
        {
            cout << "ginfo->players[" << i << "].PLID found" << endl;
            cout << "UName: " << ginfo->players[i].UName << endl;
            cout << "UCID: " << (int)ginfo->players[i].UCID << endl;
            ginfo->players[i].PLID = 0;
            cout << "PLID: " << (int)ginfo->players[i].PLID << " (set to zero)" << endl;
            break;
        }
    }
}

/**
* Connection left
* @param    ginfo   Struct that contains all the info stored by the application
*/
void case_cnl (struct global_info *ginfo)
{
    cout << " * IS_CNL (connection left)" << endl;
    int i;

    struct IS_CNL *pack_cnl = (struct IS_CNL*)insim.get_packet();

    // Find player and set the whole player struct he was using to 0
    for (i=0; i < MAX_PLAYERS; i++)
    {
        if (ginfo->players[i].UCID == pack_cnl->UCID)
        {
            cout << "ginfo->players[" << i << "].UCID found: exiting" << endl;
            cout << "UName: " << ginfo->players[i].UName << endl;
            cout << "UCID: " << (int)ginfo->players[i].UCID << endl;
            cout << "PLID: " << (int)ginfo->players[i].PLID << endl;
            memset(&ginfo->players[i], 0, sizeof(ginfo->players[i]));
            cout << "ginfo->players[" << i << "].UCID found: CHECKING" << endl;
            cout << "UName: " << ginfo->players[i].UName << endl;
            cout << "UCID: " << (int)ginfo->players[i].UCID << endl;
            cout << "PLID: " << (int)ginfo->players[i].PLID << endl;
            break;
        }
    }
}

/**
* Main program. Receives arguments used in the connection to the server.
* @param    argc    Number of arguments in the application call
* @param    argv    Arguments received in the application call.
*/
int main (int argc, char **argv)
{
    // Initialize the mutex var
    pthread_mutex_init(&ismutex, NULL);

    // Arguments check
    if (argc < 4)
    {
        cerr << " * Not enough arguments! *" << endl;
        cerr << " Usage:\n LFS Session Timing (ip) (port) (admin_pass)" << endl;
        cerr << "\n PRESS RETURN KEY TO FINISH" << endl;
        getchar();
        return -1;
    }

    // Error check var used when calling next_packet()
    int error_ch;

    // Create the main global_info struct and clear it completely to zeroes.
    struct global_info ginfo;
    memset(&ginfo, 0, sizeof(struct global_info));

    // Now set the global times to max value (1 hour)
    for(int i=0; i<5; i++)
    {
        ginfo.sesbesttimes[i] = 3600000;
        ginfo.sesbestlap[i] = 3600000;
    }

    // We create an IS_VER packet to receive the response from insim.init()
    struct IS_VER pack_ver;

    // Initialize the connection to InSim:
    // argv[1] -> Host IP
    // argv[2] -> Host port (needs to be converted to an unsigned short (word))
    // argv[3] -> Admin password
    if (insim.init (argv[1], (word)atoi(argv[2]), IS_PRODUCT_NAME, argv[3], &pack_ver, '!') < 0)
    {
        cerr << "\n * Error during initialization * " << endl;
        cerr << "\n PRESS RETURN KEY TO FINISH" << endl;
        getchar();
        return -1;
    }

    cout << "\n********************************\n" << endl;
    cout << "LFS Version: " << pack_ver.Version << endl;
    cout << "InSim Version: " << pack_ver.InSimVer << endl;
    cout << "\n********************************\n" << endl;

    // Tihs IS_TINY packet is used to request several things from the server before the main loop
    struct IS_TINY pack_requests;
    memset(&pack_requests, 0, sizeof(struct IS_TINY));
    pack_requests.Size = sizeof(struct IS_TINY);
    pack_requests.Type = ISP_TINY;
    pack_requests.ReqI = 1;
    pack_requests.SubT = TINY_NCN;      // Request all connections to store their user data
    insim.send_packet(&pack_requests);
    cout << "Connections request packet sent!" << endl;

    pack_requests.SubT = TINY_NPL;      // Request all players in-grid to know their PLID
    insim.send_packet(&pack_requests);
    cout << "Player info request packet sent!" << endl;

    pack_requests.SubT = TINY_SST;      // Request server status to store the current track
    insim.send_packet(&pack_requests);
    cout << "Server status request packet sent!" << endl;

    // Main loop. When ok is 0 we'll exit this loop
    int ok = 1;
    while(ok > 0)
    {
        // Get next packet ready
        if (error_ch = insim.next_packet() < 0)
        {
            cerr << "\n * Error getting next packet * " << endl;
            cerr << "\n PRESS RETURN KEY TO FINISH" << endl;
            getchar();
            return error_ch;
        }

        // Peek the next packet to know its type, and call the matching function
        cout << " \n***********************************************************" << endl;
        cout << " * Received packet of type: " << (int)insim.peek_packet() << endl;
        switch(insim.peek_packet())
        {
            case ISP_MSO:
                ok = case_mso (&ginfo); // in case_mso the user can request to close the application
                break;
            case ISP_NPL:
                case_npl (&ginfo);
                break;
            case ISP_NCN:
                case_ncn (&ginfo);
                break;
            case ISP_PLL:
                case_pll (&ginfo);
                break;
            case ISP_CNL:
                case_cnl (&ginfo);
                break;
            case ISP_LAP:
                case_lap (&ginfo);
                break;
            case ISP_BTC:
                case_btc (&ginfo);
                break;
            case ISP_SPX:
                case_spx (&ginfo);
                break;
            case ISP_CPR:
                case_cpr (&ginfo);
                break;
            case ISP_STA:
                case_sta (&ginfo);
                break;
            case ISP_REO:
                case_reo (&ginfo);
                break;
        }
    }

    // The main loop ended: close connection to insim
    if (insim.isclose() < 0)
    {
        cerr << "\n * Error closing connection * " << endl;
        cerr << "\n PRESS RETURN KEY TO FINISH" << endl;
        getchar();
        return -1;
    }

    // Destroy the mutex var
    pthread_mutex_destroy(&ismutex);

    return 0;
}
