#include "cplayback.h"
#include "cmgr.h"
#include <wx/bitmap.h>
#include <wx/config.h>

// time (in ms) between timer events
#define TIMER_STEP 50

// include the graphics
// TODO: do it the Windows way = convert to .ico, add to resources
#include "img/play.xpm"
#include "img/pause.xpm"
#include "img/slow.xpm"
#include "img/fast.xpm"

//-----------------------------------------------------------------------------
// Event table

BEGIN_EVENT_TABLE(cPlayback, cToolbar)
  EVT_TIMER(wxID_ANY, cPlayback::OnTimer)
END_EVENT_TABLE()

//-----------------------------------------------------------------------------

cPlayback::cPlayback(wxWindow* parent)
: cToolbar(parent)
{
  m_Speed = 1.0f;
  m_Cursor = 0.0f;
  m_AutoScroll = true;

  m_Timer.SetOwner(this);

  m_TimeText = new wxStaticText(this, ID_CTRL_PB_TIME, wxEmptyString);
  m_SpeedText = new wxStaticText(this, ID_CTRL_PB_SPEED, wxEmptyString);

  SetToolBitmapSize(wxSize(16, 16));
  SetToolSeparation(10);

  // NB: the tool IDs are equal to the menu IDs, so events are sent to the same handler
  AddTool(ID_MENU_PLAYPAUSE, wxEmptyString, wxBitmap(play_xpm), wxEmptyString);
  AddControl(m_TimeText);
  AddSeparator();
  AddTool(ID_MENU_SLOWER, wxEmptyString, wxBitmap(slow_xpm), wxEmptyString);
  AddTool(ID_MENU_FASTER, wxEmptyString, wxBitmap(fast_xpm), wxEmptyString);
  AddControl(m_SpeedText);

  TranslateTexts();
  Realize();
  EnableTools();
  SetSpeed(1.0f);
  ShowTime();
}

//-----------------------------------------------------------------------------

cPlayback::~cPlayback()
{
  if (IsRunning()) m_Timer.Stop();
}

//-----------------------------------------------------------------------------

void cPlayback::TranslateTexts()
{
  SetToolShortHelp(ID_MENU_PLAYPAUSE, _TT(ID_CTRL_PB_PLAY, "Play/Pause"));
  SetToolLongHelp(ID_MENU_PLAYPAUSE, _TH(ID_CTRL_PB_PLAY, "Start or resume replaying"));

  SetToolShortHelp(ID_MENU_SLOWER, _TT(ID_CTRL_PB_SLOWER, "Slower"));
  SetToolLongHelp(ID_MENU_SLOWER, _TH(ID_CTRL_PB_SLOWER, "Decrease the replay speed"));

  SetToolShortHelp(ID_MENU_FASTER, _TT(ID_CTRL_PB_FASTER, "Faster"));
  SetToolLongHelp(ID_MENU_FASTER, _TH(ID_CTRL_PB_FASTER, "Increase the replay speed"));

  if (!m_Tooltips) {
    // disable tooltips
    SetToolShortHelp(ID_MENU_PLAYPAUSE, wxEmptyString);
    SetToolShortHelp(ID_MENU_SLOWER, wxEmptyString);
    SetToolShortHelp(ID_MENU_FASTER, wxEmptyString);
  }
}

//-----------------------------------------------------------------------------

void cPlayback::Reset()
{
  Pause();

  m_Cursor = 0.0f; // set cursor to start of lap

  if (m_AutoScroll) {
    // set cursor to start of selection
    cLap* lap = MGR->GetLeader();
    if (lap != NULL) m_Cursor = lap->GetTimeAt(m_SelStart);
  }

  EnableTools();
  ShowTime();
}

void cPlayback::Pause()
{
  if (!IsRunning()) return;

  m_Timer.Stop();
  EnableTools();
}

void cPlayback::Resume()
{
  if (IsRunning()) return;

  m_Timer.Start(TIMER_STEP, wxTIMER_CONTINUOUS);
  m_Sw.Start();
  EnableTools();
}

//-----------------------------------------------------------------------------

void cPlayback::SetSpeed(float factor)
{
  ::Maximise(factor, MIN_PLAYBACK_SPEED);
  ::Minimise(factor, MAX_PLAYBACK_SPEED);
  m_Speed = factor;

  m_SpeedText->SetLabel(wxString::Format(_T("%.3f x"), m_Speed));
  EnableTools();
}

//-----------------------------------------------------------------------------
// Perform the periodic action from the timer

void cPlayback::OnTimer(wxTimerEvent& WXUNUSED(event))
{
  cLap* lap = MGR->GetLeader();

  if (lap == NULL) {
    // no laps
    Reset();
    return;
  }

  // calculate new cursor position
  float newCursorTime = m_Cursor + m_Sw.Time() * 0.001 * m_Speed;
  float newCursorDist = lap->GetDistanceAt(newCursorTime);

  // check cursor
  if (m_AutoScroll) {
    // check for end of lap
    if (newCursorTime >= lap->GetLaptime()) {
      Reset();
      return;
    }
  }
  else {
    // check for start of selection
    if (newCursorDist < m_SelStart) {
      newCursorTime = lap->GetTimeAt(m_SelStart);
      newCursorDist = m_SelStart;
    }

    // check for end of selection
    if (newCursorDist >= m_SelEnd) {
      Reset();
      return;
    }
  }

  // update cursor
  wxASSERT(newCursorTime >= 0.0f);
  wxASSERT(newCursorTime <= lap->GetLaptime());
  m_Cursor = newCursorTime;
  m_Sw.Start(); // reset and restart stopwatch
  ShowTime();

  // move all track cursors
  ::SetTrackCursorPos_Send(newCursorDist, CURSORSET_PLAYBACK);
}

//-----------------------------------------------------------------------------
// Change the playback cursor

void cPlayback::DoSetTrackCursorPos(cGraphView* WXUNUSED(view), float distance, int type)
{
  if (type != CURSORSET_FORCE) return;  // only respond to forced cursor-settings
  if (distance < 0.0f) return;          // ignore "remove cursor" commands

  cLap* lap = MGR->GetLeader();
  if (lap == NULL) return;

  m_Cursor = lap->GetTimeAt(distance);
  ShowTime();
}

//-----------------------------------------------------------------------------

void cPlayback::DoSetTrackSelection(float start, float end)
{
  m_SelStart = start;
  m_SelEnd = end;
}

//-----------------------------------------------------------------------------
// Eanble/disable tools

void cPlayback::EnableTools()
{
  EnableTool(ID_MENU_SLOWER, CanGoSlower());
  EnableTool(ID_MENU_FASTER, CanGoFaster());
  EnableTool(ID_MENU_PLAYPAUSE, !MGR->IsEmpty());

  // change bitmap of play/pause button
  if (IsRunning()) {
    SetToolNormalBitmap(ID_MENU_PLAYPAUSE, wxBitmap(pause_xpm));
  }
  else {
    SetToolNormalBitmap(ID_MENU_PLAYPAUSE, wxBitmap(play_xpm));
  }
}

//-----------------------------------------------------------------------------

void cPlayback::UpdateAll()
{
  EnableTools();
}

//-----------------------------------------------------------------------------
// Display the value of the time cursor in the toolbar

void cPlayback::ShowTime()
{
  int min = m_Cursor / 60;
  int sec = (int)m_Cursor % 60;
  int tenths = (int)(10.0f * m_Cursor) % 10;

  wxString lbl;
  lbl.Printf(_T("%02d:%02d.%01d"), min, sec, tenths);

  if (lbl != m_TimeText->GetLabel()) m_TimeText->SetLabel(lbl);
}

//-----------------------------------------------------------------------------
// Loading and saving the configuration settings

void cPlayback::LoadConfig(wxRegConfig* config, const wxString& key)
{
  config->Read(key + _T("/autoscroll"), &m_AutoScroll);
}


void cPlayback::SaveConfig(wxRegConfig* config, const wxString& key)
{
  config->Write(key + _T("/autoscroll"), m_AutoScroll);
}
