#include "clegend.h"
#include "cmgr.h"
#include "cgraph.h"
#include "cgraphview.h"
#include "capp.h"
#include <wx/dcclient.h>
#include <wx/config.h>
#include <wx/utils.h>
#include <wx/filedlg.h>

// the width of the steering gauge
#define STEER_WIDTH 45

// the size of the marker in a symmetric gauge
#define GAUGE_MARKER_WIDTH 3

// the maximum for the steering gauge
#define MAX_STEER 10.0f

// the width of the pedal gauges
#define PEDAL_WIDTH 40

//-----------------------------------------------------------------------------
// Event table

BEGIN_EVENT_TABLE(cLegend, cPane)
  EVT_LEFT_DOWN(cLegend::OnMouseLeftClick)
  EVT_LEFT_DCLICK(cLegend::OnMouseLeftClick)
  EVT_MIDDLE_DOWN(cLegend::OnMouseMiddleClick)
  EVT_RIGHT_DOWN(cLegend::OnMouseRightClick)
  EVT_MENU(wxID_ANY, cLegend::OnMenuClick)
  EVT_ENTER_WINDOW(cLegend::OnMouseEntering)
  EVT_LEAVE_WINDOW(cLegend::OnMouseLeaving)
  EVT_MOTION(cLegend::OnMouseMove)
  EVT_MOUSEWHEEL(cLegend::OnMouseWheel)
  EVT_PAINT(cLegend::OnPaint)
END_EVENT_TABLE()

//-----------------------------------------------------------------------------
// Static members

wxPen cLegend::s_GridPen(wxColour(LIGHT_GREY, LIGHT_GREY, LIGHT_GREY));

//-----------------------------------------------------------------------------

cLegend::cLegend(wxWindow* parent)
: cPane(parent)
{
  m_MinWidth = 100;
  m_SelectedLine = -1;
  m_ShowDashboard = false;
  m_Graph = NULL;
  m_Distance = -1.0f;

  SetBackgroundColour(*wxWHITE);

  m_MenuNames = new wxMenu;
  cLang::AppendCheckItem(m_MenuNames, ID_MENU_LAPNAME_FILE, _T("Filename"));
  cLang::AppendCheckItem(m_MenuNames, ID_MENU_LAPNAME_TIME, _T("Laptime"));
  cLang::AppendCheckItem(m_MenuNames, ID_MENU_LAPNAME_TIME_DRIVER, _T("Laptime + driver"));
  cLang::AppendCheckItem(m_MenuNames, ID_MENU_LAPNAME_TIME_CAR_DRIVER, _T("Laptime + car + driver"));

  cLang::AppendMenuItem(&m_Context, ID_MENU_CLOSE_LAP, _T("Close file"));
  cLang::AppendMenuItem(&m_Context, ID_MENU_HIDE_LAP, _T("Hide"));
  cLang::AppendMenuItem(&m_Context, ID_MENU_LG_EXPORT, _T("Export to CSV..."));
  m_Context.AppendSeparator();
  cLang::AppendMenuItem(&m_Context, ID_MENU_CLOSE_OTHERS, _T("Close other files"));
  cLang::AppendMenuItem(&m_Context, ID_MENU_HIDE_OTHERS, _T("Hide other files"));
  cLang::AppendMenuItem(&m_Context, ID_MENU_SHOW_ALL, _T("Show all"));
  m_Context.AppendSeparator();
  cLang::AppendMenuItem(&m_Context, ID_MENU_LAPNAME, _T("Display files as"), m_MenuNames);
  cLang::AppendCheckItem(&m_Context, ID_MENU_DASHBOARD, _T("Show dashboard"));

  UpdateAll();
}

//-----------------------------------------------------------------------------

cLegend::~cLegend()
{
}

//-----------------------------------------------------------------------------
// Convert logical coordinates to device coordinates and vice versa

wxCoord cLegend::Logical2DeviceX(wxCoord logicalPosX)
{
  int devicePosX, devicePosY;
  CalcScrolledPosition(logicalPosX, 0, &devicePosX, &devicePosY);
  return devicePosX;
}

wxCoord cLegend::Device2LogicalX(wxCoord devicePosX)
{
  int logicalPosX, logicalPosY;
  CalcUnscrolledPosition(devicePosX, 0, &logicalPosX, &logicalPosY);
  return logicalPosX;
}

//-----------------------------------------------------------------------------
// Convert line number to window coordinate and vice versa
// NB line number = -1 for the header, 0 for the first lap, 1 for the second, etc.

wxCoord cLegend::Line2CoordY(int line) const
{
  int ppuX, ppuY;
  GetScrollPixelsPerUnit(&ppuX, &ppuY);
  int startX, startY;
  GetViewStart(&startX, &startY);
  return BASE_MARGIN + (line + 1) * m_LineSpacing - ppuY * startY;
}

int cLegend::Coord2LineY(wxCoord coord) const
{
  int ppuX, ppuY;
  GetScrollPixelsPerUnit(&ppuX, &ppuY);
  int startX, startY;
  GetViewStart(&startX, &startY);
  return (coord + ppuY * startY - BASE_MARGIN) / m_LineSpacing - 1;
}

//-----------------------------------------------------------------------------
// Return the line number that corresponds to a mouse position (-1 = none)
// NB line number = -1 for the header, 0 for the first lap, 1 for the second, etc.

int cLegend::DecodeMouse(wxMouseEvent& event)
{
  // get logical position of click
  wxClientDC dc(this);
  wxPoint pos = event.GetLogicalPosition(dc);

  if (pos.x < BASE_MARGIN) return -1;                           // click is to left of item
  if (pos.x > 2 * BASE_MARGIN + m_LabelY + m_LabelX) return -1; // click is to right of item

  int line = Coord2LineY(pos.y);
  if (line < 0) return -1;                                      // click is on header line
  if ((size_t)line >= MGR->GetLapCount()) return -1;            // not a valid line number

  return line;
}

//-----------------------------------------------------------------------------

void cLegend::UpdateAll()
{
  wxClientDC dc(this);
  dc.SetFont(*wxNORMAL_FONT);
  dc.GetTextExtent(_T("WWWW"), &m_LabelX, &m_LabelY); // default label size

  // determine max label size over all laps
  wxCoord sizeX, sizeY;
  for (size_t l = 0; l < MGR->GetLapCount(); l++) {
    dc.GetTextExtent(MGR->GetLap(l)->GetName(), &sizeX, &sizeY);
    ::Maximise(m_LabelX, sizeX);
    ::Maximise(m_LabelY, sizeY);
  }
  if (m_LabelY % 2 == 0) m_LabelY += 1; // always make label height odd-sized
  m_LineSpacing = m_LabelY + 2 * BASE_MARGIN;

  // determine start coordinates
  m_IconStart = 2 * BASE_MARGIN;
  m_LabelStart = m_IconStart + m_LabelY + BASE_MARGIN;
  m_LabelEnd = m_LabelStart + m_LabelX;

  dc.GetTextExtent(LARGEST_VALUE_TEXT _T("9"), &sizeX, &sizeY);
  m_CurveValueEnd = m_LabelEnd + 2 * BASE_MARGIN + sizeX;

  m_SteerStart = m_CurveValueEnd + 2 * BASE_MARGIN;
  m_PedalStart = m_SteerStart + STEER_WIDTH + 2 * BASE_MARGIN;

  dc.GetTextExtent(_TT(ID_TXT_LG_GEAR, "Gear"), &sizeX, &sizeY);
  m_GearMid = m_PedalStart + PEDAL_WIDTH + 2 * BASE_MARGIN + sizeX / 2;

  m_SpeedEnd = m_GearMid + sizeX / 2 + 2 * BASE_MARGIN;
  wxCoord size1X, size1Y;
  dc.GetTextExtent(_T("999.9"), &size1X, &size1Y);
  wxCoord size2X, size2Y;
  dc.GetTextExtent(_TT(ID_TXT_LG_SPEED, "Speed"), &size2X, &size2Y);
  ::Maximise(size1X, size2X);
  m_SpeedEnd += size1X;

  // change the minimum size
  if (m_ShowDashboard) {
    m_MinWidth = m_SpeedEnd + BASE_MARGIN; // width enough for all
  }
  else {
    m_MinWidth = m_LabelEnd + 2 * BASE_MARGIN; // width enough for lap names + selection rectangle
  }
  SetMinSize(wxSize(m_MinWidth, Line2CoordY(3))); // height enough for 2 laps

  // set scrolling
  sizeX = (m_ShowDashboard) ? m_SpeedEnd : m_LabelEnd;
  SetVirtualSize(sizeX  + BASE_MARGIN, Line2CoordY(MGR->GetLapCount()));
  SetScrollRate(m_LineSpacing, m_LineSpacing);

  Refresh();
}

//-----------------------------------------------------------------------------

void cLegend::OnPaint(wxPaintEvent& WXUNUSED(event))
{
  wxPaintDC dc(this);

  if (MGR->IsEmpty()) return; // nothing to paint

  dc.SetFont(*wxNORMAL_FONT);
  wxCoord y; // coordinate to plot at

  // plot header
  y = Line2CoordY(-1);
  dc.DrawText(_TT(ID_TXT_LG_FILE, "File"), Logical2DeviceX(m_IconStart), y);
  if (m_ShowDashboard) {
    ::DrawTextRightAligned(dc, _TT(ID_TXT_LG_GRAPH, "Graph"),
        Logical2DeviceX(m_CurveValueEnd), y);
    ::DrawTextCenteredH(dc, _TT(ID_TXT_LG_STEER, "Steer"),
        Logical2DeviceX(m_SteerStart + STEER_WIDTH / 2), y);
    dc.DrawText(_TT(ID_TXT_LG_PEDALS, "Pedals"),
        Logical2DeviceX(m_PedalStart), y);
    ::DrawTextCenteredH(dc, _TT(ID_TXT_LG_GEAR, "Gear"),
        Logical2DeviceX(m_GearMid), y);
    ::DrawTextRightAligned(dc, _TT(ID_TXT_LG_SPEED, "Speed"),
        Logical2DeviceX(m_SpeedEnd), y);
  }

  // plot horizontal line between header and laps
  int screenX, screenY;
  dc.GetSize(&screenX, &screenY);
  y = Line2CoordY(0) - 2 * BASE_MARGIN;
  dc.SetPen(cLegend::s_GridPen);
  dc.DrawLine(0, y, screenX, y);

  // plot data for each lap
  for (size_t i = 0; i < MGR->GetLapCount(); i++) {
    cLap* lap = MGR->GetLap(i); // lap to plot data from
    y = Line2CoordY(i);

    // draw rectangle in same colour as lap
    wxPen pen(lap->GetColour());
    dc.SetPen(pen);
    wxBrush brush(lap->GetColour());
    // open rectangle if lap is hidden, else a solid rectangle
    brush.SetStyle((lap->IsShown()) ? wxSOLID : wxTRANSPARENT);
    dc.SetBrush(brush);
    dc.DrawRectangle(Logical2DeviceX(m_IconStart), y, m_LabelY, m_LabelY);

    // plot lap name
    dc.DrawText(lap->GetName(), Logical2DeviceX(m_LabelStart), y);

    if (m_Distance < 0) continue;
    if (!lap->IsShown()) continue; // do not plot data from hidden laps

    // plot dashboard
    if (m_ShowDashboard) {
      // curve value
      if (m_Graph != NULL) {
        wxString value = m_Graph->GetCurveValueAt(lap, m_Distance);
        ::DrawTextRightAligned(dc, value, Logical2DeviceX(m_CurveValueEnd), y);
      }

      // steer gauge
      DrawSymmetricGauge(dc,
          wxPoint(Logical2DeviceX(m_SteerStart), y),
          wxSize(STEER_WIDTH, m_LabelY),
          MAX_STEER, -lap->GetLogDataAt(m_Distance, LOGTYPE_STEER, WHEELTYPE_FRONT_LEFT),
          *wxBLUE, 1);

      // pedal gauges
      int pedalheight = m_LabelY / 2 + 1;
      DrawAsymmetricGauge(dc,
          wxPoint(Logical2DeviceX(m_PedalStart), y),
          wxSize(PEDAL_WIDTH, pedalheight),
          100.0f, lap->GetLogDataAt(m_Distance, LOGTYPE_THROTTLE, WHEELTYPE_FRONT_LEFT),
          *wxGREEN, 4);
      DrawAsymmetricGauge(dc,
          wxPoint(Logical2DeviceX(m_PedalStart), y + pedalheight - 1),
          wxSize(PEDAL_WIDTH, pedalheight),
          100.0f, lap->GetLogDataAt(m_Distance, LOGTYPE_BRAKE, WHEELTYPE_FRONT_LEFT),
          *wxRED, 4);

      // gear gauge
      wxString gear;
      int gearValue = (int)(lap->GetLogDataAt(m_Distance, LOGTYPE_GEAR, WHEELTYPE_FRONT_LEFT));
      switch (gearValue) {
        case -1 : gear = _T("r"); break;
        case 0  : gear = _T("n"); break;
        default : gear.Printf(_T("%d"), gearValue);
      }
      ::DrawTextCenteredH(dc, gear, Logical2DeviceX(m_GearMid), y);

      // speed gauge
      wxString speed = wxString::Format(_T("%.1f"),
          lap->GetLogDataAt(m_Distance, LOGTYPE_SPEED, WHEELTYPE_FRONT_LEFT));
      ::DrawTextRightAligned(dc, speed, Logical2DeviceX(m_SpeedEnd), y);
    }
  }

  // plot the selected line
  if (m_SelectedLine >= 0) {
    dc.SetPen(cLegend::s_GridPen);
    DrawSelectionRectangle(dc, m_SelectedLine);
  }
}

//-----------------------------------------------------------------------------
// Draw a gauge for an asymmetric (= only positive-valued) value
// - dc = device context to draw in
// - pos = coordinates for topleft corner
// - size = size of gauge
// - range = range of possible values
// - value = actual gauge value to display
// - steps = number of gid subdivisions

void cLegend::DrawAsymmetricGauge(wxDC& dc, wxPoint pos, wxSize size, float range, float value, wxColour colour, int steps)
{
  wxASSERT(range > 0);
  wxASSERT(value >= 0);
  wxASSERT(steps > 0);

  // draw grid
  DrawGaugeGrid(dc, pos, size, steps);

  // take inner rectangle
  pos.x += 1;
  pos.y += 1;
  size.DecBy(2, 2);

  int width = (int)((float)size.GetWidth() * value / range); // width of bar

  // correction for values outside range
  if (width > size.GetWidth()) {
    // too far to the right
    width = size.GetWidth();
  }

  dc.SetPen(wxPen(colour));
  dc.SetBrush(colour);
  dc.DrawRectangle(pos.x, pos.y, width, size.GetHeight());
}

//-----------------------------------------------------------------------------
// Draw a gauge for a symmetric (= both positive- and negative-valued) value
// - dc = device context to draw in
// - pos = coordinates for topleft corner
// - size = size of gauge
// - range = range of possible values
// - value = actual gauge value to display
// - steps = number of gid subdivisions

void cLegend::DrawSymmetricGauge(wxDC& dc, wxPoint pos, wxSize size, float range, float value, wxColour colour, int steps)
{
  wxASSERT(range > 0);
  wxASSERT(steps > 0);
  wxASSERT(size.GetWidth() % 2 == 1); // for proper display, width should be an odd number

  // draw grid
  DrawGaugeGrid(dc, pos, size, 2 * steps);

  // take inner rectangle
  pos.x += 1;
  pos.y += 1;
  size.DecBy(2, 2);

  int start = (size.GetWidth() / 2); // distance of start of marker from left side
  start += (int)((float)size.GetWidth() * value / (2.0f * range));
  start -= (GAUGE_MARKER_WIDTH / 2);
  int width = GAUGE_MARKER_WIDTH;

  // correction for values outside range
  if (start < 0) {
    // too far to the left
    width += start;
    start = 0;
  }
  if (start + width > size.GetWidth()) {
    // too far to the right
    width = size.GetWidth() - start;
  }

  if (width > 0) {
    dc.SetPen(wxPen(colour));
    dc.SetBrush(colour);
    dc.DrawRectangle(pos.x + start, pos.y, width, size.GetHeight());
  }
}

//-----------------------------------------------------------------------------
// Draw the grid for a dashboard gauge
// - dc = device context to draw in
// - pos = coordinates for topleft corner
// - size = size of gauge
// - steps = number of subdivisions

void cLegend::DrawGaugeGrid(wxDC& dc, wxPoint pos, wxSize size, int steps)
{
  wxASSERT(steps > 0);

  dc.SetPen(cLegend::s_GridPen);
  dc.SetBrush(*wxTRANSPARENT_BRUSH);
  dc.DrawRectangle(pos.x, pos.y, size.GetWidth(), size.GetHeight());

  for (int i = 1; i < steps; i++) {
    wxCoord lx = pos.x + (size.GetWidth() * i) / steps;
    dc.DrawLine(lx, pos.y, lx, pos.y + size.GetHeight());
  }
}

//-----------------------------------------------------------------------------

void cLegend::OnMouseLeftClick(wxMouseEvent& event)
{
  event.Skip(); // always pass left-clicks to default event handler

  int line = DecodeMouse(event);
  if (line < 0) return;

  cLap* lap = MGR->GetLap(line);
  ::ShowLap_Send(lap, !lap->IsShown()); // show/hide lap
}

//-----------------------------------------------------------------------------

void cLegend::OnMouseRightClick(wxMouseEvent& event)
{
  m_Context_ClickedLine = DecodeMouse(event);

  cLap* lap = NULL;
  wxString lapName;
  if (m_Context_ClickedLine >= 0) {
    lap = MGR->GetLap(m_Context_ClickedLine);
    lapName = lap->GetName();
  }

  wxString lbl = _TT(ID_MENU_CLOSE_LAP, "") + _T(" \"") + lapName + _T("\"");
  m_Context.SetLabel(ID_MENU_CLOSE_LAP, lbl);

  if ((lap != NULL) && (lap->IsShown())) {
    m_Context.SetLabel(ID_MENU_HIDE_LAP, _TT(ID_MENU_HIDE_LAP, "Hide"));
  }
  else {
    m_Context.SetLabel(ID_MENU_HIDE_LAP, _TT(ID_TXT_LG_SHOW_LAP, "Show"));
  }

  bool hasLap = (lap != NULL);
  bool hasOthers = hasLap && (MGR->GetLapCount() > 1);
  m_Context.Enable(ID_MENU_CLOSE_LAP, hasLap);
  m_Context.Enable(ID_MENU_HIDE_LAP, hasLap);
  m_Context.Enable(ID_MENU_LG_EXPORT, hasLap);
  m_Context.Enable(ID_MENU_CLOSE_OTHERS, hasOthers);
  m_Context.Enable(ID_MENU_HIDE_OTHERS, hasOthers);
  m_Context.Enable(ID_MENU_SHOW_ALL, (MGR->GetLapCount() > 0));
  m_Context.Check(ID_MENU_DASHBOARD, m_ShowDashboard);

  m_MenuNames->Check(ID_MENU_LAPNAME_FILE, MGR->GetLapNaming() == LAPNAME_FILE);
  m_MenuNames->Check(ID_MENU_LAPNAME_TIME, MGR->GetLapNaming() == LAPNAME_TIME);
  m_MenuNames->Check(ID_MENU_LAPNAME_TIME_DRIVER, MGR->GetLapNaming() == LAPNAME_TIME_DRIVER);
  m_MenuNames->Check(ID_MENU_LAPNAME_TIME_CAR_DRIVER, MGR->GetLapNaming() == LAPNAME_TIME_CAR_DRIVER);

  PopupMenu(&m_Context);
}

//-----------------------------------------------------------------------------

void cLegend::OnMouseWheel(wxMouseEvent& event)
{
  // scroll window
  int startX, startY;
  GetViewStart(&startX, &startY);
  if (event.GetWheelRotation() > 0) {
    Scroll(-1, startY - 1);
  }
  else {
    Scroll(-1, startY + 1);
  }
}

//-----------------------------------------------------------------------------

void cLegend::OnMouseMove(wxMouseEvent& event)
{
  SelectLine(DecodeMouse(event));
}

void cLegend::OnMouseEntering(wxMouseEvent& event)
{
  SelectLine(DecodeMouse(event));
}

void cLegend::OnMouseLeaving(wxMouseEvent& WXUNUSED(event))
{
  SelectLine(-1);
}

//-----------------------------------------------------------------------------
// Set the 'selected' line
// - line = line number (-1 = none)

void cLegend::SelectLine(int line)
{
  wxASSERT((line == -1) || ((size_t)line < MGR->GetLapCount())); // line number must be a valid line, or negative

  if (line == m_SelectedLine) return; // no change

  wxClientDC dc(this);

  if (m_SelectedLine >= 0) {
    // remove current selection
    dc.SetPen(*wxWHITE_PEN);
    DrawSelectionRectangle(dc, m_SelectedLine);
  }

  m_SelectedLine = line;

  if (m_SelectedLine >= 0) {
    // draw new selection
    dc.SetPen(cLegend::s_GridPen);
    DrawSelectionRectangle(dc, m_SelectedLine);
  }
}

//-----------------------------------------------------------------------------
// Draw the 'selected' box around a lap, with the current pen
// - dc = current device context
// - line = line number

void cLegend::DrawSelectionRectangle(wxDC& dc, int line)
{
  wxASSERT((size_t)line < MGR->GetLapCount()); // line number must be a valid line

  wxCoord x = Logical2DeviceX(m_IconStart - BASE_MARGIN);
  wxCoord y = Line2CoordY(line) - BASE_MARGIN;
  int width = m_LabelEnd - m_IconStart + 2 * BASE_MARGIN;
  int height = m_LineSpacing;

  dc.SetBrush(*wxTRANSPARENT_BRUSH);
  dc.DrawRoundedRectangle(x, y, width, height, BASE_MARGIN - 1);
}

//-----------------------------------------------------------------------------
// Handle a click in the context menu

void cLegend::OnMenuClick(wxCommandEvent& event)
{
  cLap* clickedLap = NULL;
  if (m_Context_ClickedLine >= 0) clickedLap = MGR->GetLap(m_Context_ClickedLine);

  switch (event.GetId()) {
    case ID_MENU_HIDE_LAP :
      ::ShowLap_Send(clickedLap, !clickedLap->IsShown());
      break;

    case ID_MENU_CLOSE_LAP :
      ::DeleteLap_Send(clickedLap);
      break;

    case ID_MENU_LG_EXPORT :
      ExportLap(clickedLap);
      break;

    case ID_MENU_HIDE_OTHERS :
      ::ShowLap_Send(NULL, false); // hide all
      ::ShowLap_Send(clickedLap, true);
      break;

    case ID_MENU_CLOSE_OTHERS :
      for (int l = MGR->GetLapCount() - 1; l >= 0; l--) {
        cLap* lap = MGR->GetLap(l);
        if (lap != clickedLap) ::DeleteLap_Send(lap);
      }
      break;

    case ID_MENU_SHOW_ALL :
      ::ShowLap_Send(NULL, true);
      break;

    case ID_MENU_DASHBOARD :
      m_ShowDashboard = !m_ShowDashboard;
      UpdateAll();
      ::LayoutMainFrame_Send(); // minimum size has changed
      break;

    case ID_MENU_LAPNAME_FILE :
      MGR->SetLapNaming(LAPNAME_FILE);
      ::LayoutMainFrame_Send();
      break;

    case ID_MENU_LAPNAME_TIME :
      MGR->SetLapNaming(LAPNAME_TIME);
      ::LayoutMainFrame_Send();
      break;

    case ID_MENU_LAPNAME_TIME_DRIVER :
      MGR->SetLapNaming(LAPNAME_TIME_DRIVER);
      ::LayoutMainFrame_Send();
      break;

    case ID_MENU_LAPNAME_TIME_CAR_DRIVER :
      MGR->SetLapNaming(LAPNAME_TIME_CAR_DRIVER);
      ::LayoutMainFrame_Send();
      break;

    default :
      wxFAIL;
  }
}

//-----------------------------------------------------------------------------

wxSize cLegend::DoGetBestSize() const
{
  size_t laps = MGR->GetLapCount();
  if (laps < 2) laps = 2; // best size can show at least 2 laps
  return wxSize(m_MinWidth, Line2CoordY(laps + 1));
}

//-----------------------------------------------------------------------------

void cLegend::DoSetTrackCursorPos(cGraphView* view, float distance, int WXUNUSED(type))
{
  CHECK_PTR_NULL(view);
  wxASSERT(m_Graph == NULL);

  if (MGR->IsEmpty()) return; // empty display
  if (!m_ShowDashboard) return;

  // repaint the dashboard
  m_Distance = distance;
  if (view != NULL) m_Graph = view->GetGraph();
  wxPoint rectStart = wxPoint(Logical2DeviceX(m_LabelEnd), Line2CoordY(0));
  RefreshRect(wxRect(rectStart.x, rectStart.y,
      Logical2DeviceX(m_SpeedEnd) - rectStart.x, Line2CoordY(MGR->GetLapCount()) - rectStart.y));
  Update();
  m_Graph = NULL;
}

//-----------------------------------------------------------------------------

void cLegend::ShowDashboard(bool show)
{
  m_ShowDashboard = show;
  Refresh();
}

//-----------------------------------------------------------------------------
// Loading and saving the configuration settings

void cLegend::LoadConfig(wxRegConfig* config, const wxString& key)
{
  config->Read(key + _T("/dashboard"), &m_ShowDashboard);

  int naming = MGR->GetLapNaming();
  if (config->Read(key + _T("/lapnames"), &naming)) MGR->SetLapNaming((lapname_t)naming);
}


void cLegend::SaveConfig(wxRegConfig* config, const wxString& key)
{
  config->Write(key + _T("/dashboard"), m_ShowDashboard);
  config->Write(key + _T("/lapnames"), MGR->GetLapNaming());
}

//-----------------------------------------------------------------------------
// Translation

void cLegend::TranslateTexts()
{
  cLang::TranslateMenu(&m_Context);
}

//-----------------------------------------------------------------------------

void cLegend::DoSetTrackSelection(float start, float end)
{
  m_SelStart = start;
  m_SelEnd = end;
}

//-----------------------------------------------------------------------------
//

void cLegend::ExportLap(cLap* lap)
{
  CHECK_PTR(lap);

  // open file dialog
  wxString caption = _TT(ID_TXT_LG_EXPORT_CAPTION, "Export lap to CSV");
  wxString wildcard = _TT(ID_TXT_LG_EXPORT_WILDCARD , "CSV files") + _T(" (*.csv)|*.csv");
  wxFileDialog fileDlg(NULL, caption, APP->GetExportFolder(), wxEmptyString, wildcard,
      (wxFD_SAVE | wxFD_OVERWRITE_PROMPT | wxCHANGE_DIR));
  if (fileDlg.ShowModal() != wxID_OK) return;

  // get user choices
  APP->SetExportFolder(fileDlg.GetDirectory());
  wxString fileName = fileDlg.GetPath();

  fileDlg.Close();

  lap->Export(fileName, m_SelStart, m_SelEnd);
}
