#ifndef _CLAP_H_
#define _CLAP_H_

#include "global.h"
#include "cfile.h"
#include "ccarstate.h"
#include "ccar.h"
#include <wx/colour.h>

//-----------------------------------------------------------------------------

typedef enum {
  LAPNAME_NONE = 0,
  LAPNAME_FILE,
  LAPNAME_TIME,
  LAPNAME_TIME_DRIVER,
  LAPNAME_TIME_CAR_DRIVER
} lapname_t;

typedef enum {
  LAPSTATUS_CREATED = 0,
  LAPSTATUS_LOADING,
  LAPSTATUS_LOAD_OK,
  LAPSTATUS_LOAD_FAIL
} lapstatus_t;

//-----------------------------------------------------------------------------
// A lap, read from a RAF file

class cLap : public cFile
{
private:
  wxString m_Name;          // the current display name
  lapname_t m_Naming;       // the naming scheme used for the display name
  bool m_IsComplete;        // is this a complete lap? (= run till finish line)

  // file loading
  wxString m_FileName;      // the pathname of the RAF file
  lapstatus_t m_Status;     // status of loading
  wxString m_LoadError;     // error message from loading
  wxInt16 m_HeaderSize, m_BlockSize, m_WheelBlockSize, m_WheelBlockOffset; // sizes of data blocks
  wxInt32 m_BlockCount;     // the number of data blocks (= car states)
  wxInt32 m_BlockIndex;     // the index of the currently read block
  float m_TimeInterval;     // time interval between car states (in seconds)
  float m_PrevTime;         // time-in-lap of previous state
  size_t m_Wheelmap[4];     // maps position of wheels in the file to position on the car

  // lap data, read from the file
  wxString m_TrackName;     // the track name
  wxString m_ConfigName;    // the track configuration name
  wxString m_FullTrackName; // the track + config name
  wxString m_TrackCode;     // the track code (empty for LFS S2 patch W and earlier)
  float m_TrackLength;      // track length (in meters)
  cCar m_Car;               // the car data

  float m_MinX, m_MaxX, m_MinY, m_MaxY; // extremes of the car position

  wxColour m_Colour;        // colour to be used in graphs
  bool m_Shown;             // is the lap shown in graphs?

  wxString m_PlayerName;    // player name
  wxString m_Weather;       // weather
  wxString m_LfsVersion;    // LFS version
  wxInt8 m_PlayerFlags;     // player flags
  wxInt8 m_Hlvc;            // HLVC status
  wxInt8 m_SplitCount;      // actual number of splits
  wxInt32 m_Split[4];       // split times (in ms) including lap time
  float m_AvgTemp[4];       // average tyre temp
  cCarStateArray m_Log;     // logged car states in this lap

  bool LoadFailed(wxString error);

public:
  friend class cProperties;
  ENABLE_OBJCHECKER

  cLap(const wxString& fileName);
  virtual ~cLap();

  bool IsComplete() const { return m_IsComplete; }
  inline wxColour GetColour() { return m_Colour; }

  wxString GetName() const { return GetName(m_Naming); }
  wxString GetName(lapname_t naming) const;
  void SetName(lapname_t naming);
  void SetFileName(const wxString& newName);

  inline const wxString& GetFileName() const { return m_FileName; }
  inline const wxString& GetTrackName() const { return m_TrackName; }
  inline const wxString& GetConfigName() const { return m_ConfigName; }
  inline const wxString& GetFullTrackName() const { return m_FullTrackName; }
  inline const wxString& GetTrackCode() const { return m_TrackCode; }
  inline float GetTrackLength() const { return m_TrackLength; }
  inline const cCar& GetCar() const { return m_Car; }

  inline float GetLaptime() const { return GetSplit(m_SplitCount - 1); }
  inline int GetSplitCount() const { return m_SplitCount; }
  inline float GetSplit(int index) const { return m_Split[index] * 0.001f; }

  bool IsShown() const { return m_Shown; }
  void Show(bool show) { m_Shown = show; }

  bool LoadStart();
  bool LoadNext(int ms);
  inline lapstatus_t GetStatus() const { return m_Status; }
  inline bool IsLoaded() const { return m_Status == LAPSTATUS_LOAD_OK; }
  inline const wxString& GetLoadError() const { return m_LoadError; }

  inline float GetMinX() { return m_MinX; }
  inline float GetMaxX() { return m_MaxX; }
  inline float GetMinY() { return m_MinY; }
  inline float GetMaxY() { return m_MaxY; }

  inline size_t GetStateCount() const { CHECK_THIS; return m_Log.GetCount(); }
  inline const cCarState* GetState(int index) const { return &(m_Log[index]); }

  size_t FindStateAt(float distance) const;
  void GetPositionAt(float distance, float& x, float& y) const;
  float GetLogDataAt(float distance, int log, int wheel) const;
  float GetDistanceAt(float time);
  inline float GetTimeAt(float distance) { return GetLogDataAt(distance, LOGTYPE_TIMEINLAP, 0); }

  bool Export(const wxString& fileName, float start, float end);
};

//-----------------------------------------------------------------------------

WX_DECLARE_OBJARRAY(cLap, cLapArray);

#endif
