#include "ccurve_h.h"
#include "cgraph.h"

// size of the average-indicator
#define AVERAGE_INDICATOR_SIZE 6

//-----------------------------------------------------------------------------

cCurveH::cCurveH(cLap* lap)
: cCurve(lap)
{
  wxASSERT(m_Lap->GetStateCount() > 0);

  m_Logtype = LOGTYPE_FIRST;
  m_Wheel = WHEELTYPE_FIRST;

  m_BinStart = IMPOSSIBLY_HIGH_VALUE;
  m_BinEnd = IMPOSSIBLY_LOW_VALUE;

  m_Average = 0.0f;

  AdjustBbox();
}

//-----------------------------------------------------------------------------

cCurveH::~cCurveH()
{
}

//-----------------------------------------------------------------------------
// Prepare the curve's data for plotting
// - start, end = start and end of the part of the lap to be plotted
// - maxPoints = maximum number of plot points

void cCurveH::Rewind(float start, float end, size_t WXUNUSED(maxPoints))
{
  wxASSERT(end > start);
  wxASSERT(m_BinStart < m_BinEnd);

  // clear bin values
  m_Bin.Empty();
  m_Bin.Add(0, m_BinCount);
  m_SampleCount = 0;
  m_SampleSum = 0.0f;

  // count samples into bins
  float width = (m_BinEnd - m_BinStart) / m_BinCount;
  for (size_t s = 0; s < m_Lap->GetStateCount(); s++) {
    const cCarState* state = m_Lap->GetState(s);
    if (state->GetDistance() < start) continue;
    if (state->GetDistance() > end) break;

    float val = state->GetLogData(m_Logtype, m_Wheel);
    wxASSERT(val >= m_BinStart);
    int index = (int)((val - m_BinStart) / width);
    if (index >= m_BinCount) index = m_BinCount - 1;
    m_Bin[index] += 1;
    m_SampleCount += 1;
    m_SampleSum += val;
  }

  // prepare for plotting first non-empty bin
  m_PlotBin = 0;
  while ((m_PlotBin < m_BinCount) && (m_Bin[m_PlotBin] == 0)) m_PlotBin += 1;
  m_PlotPoint = 0;
}

//-----------------------------------------------------------------------------
// Get the next point on the curve
// x, y = receive the coordinates of the point

bool cCurveH::GetNextPoint(float& x, float& y)
{
  if (m_PlotBin >= m_BinCount) return false; // reached end of bins

  float width = (m_BinEnd - m_BinStart) / m_BinCount;

  // basic coordinates: topleft corner of bar
  x = m_BinStart + width * m_PlotBin;
  y = 100.0 * ((float)m_Bin[m_PlotBin]) / ((float)m_SampleCount);

  if (m_PlotPoint == 1) {
    // bottomright corner
    x += width;
    y = 0.0f;
  }

  // prepare for next point
  if (m_PlotPoint == 0) {
    // go to opposite corner
    m_PlotPoint = 1;
  }
  else {
    // go to next non-empty bin
    m_PlotBin += 1;
    while ((m_PlotBin < m_BinCount) && (m_Bin[m_PlotBin] == 0)) m_PlotBin += 1;
    m_PlotPoint = 0;
  }

  return true;
}

//-----------------------------------------------------------------------------
// Plot the curve in the graph
// - dc = current device context
// - graph = graph to plot in
// - start, end = start and end of the part of the lap to be plotted

void cCurveH::Plot(wxDC& dc, cGraph& graph, float start, float end)
{
  if (!IsShown()) return;

  Rewind(start, end, graph.GetSize().GetWidth() / 2);

  // store the value of the average now, because Rewind() is called at other occasions
  m_Average = (m_SampleCount == 0) ? 0.0f : (m_SampleSum / m_SampleCount);

  wxPoint avg; // coordinates of the average-indicator
  avg.x = graph.Value2CoordX(m_Average);
  avg.y = -1;

  dc.SetPen(m_Pen);
  wxBrush brush(m_Pen.GetColour(), wxBDIAGONAL_HATCH);

  float X, Y;
  wxPoint tl, br; // coordinates of topleft and bottomright corner
  while (GetNextPoint(X, Y)) {
    tl.x = graph.Value2CoordX(X);
    tl.y = graph.Value2CoordY(Y);

    if (!GetNextPoint(X, Y)) break;
    br.x = graph.Value2CoordX(X) + 1;
    br.y = graph.Value2CoordY(Y);

    // draw a white-filled rectangle (to erase gridlines etc.)
    dc.SetBrush(*wxWHITE_BRUSH);
    dc.DrawRectangle(tl.x, tl.y, br.x - tl.x, br.y - tl.y);
    // draw again, but with a hatch pattern
    dc.SetBrush(brush);
    dc.DrawRectangle(tl.x, tl.y, br.x - tl.x, br.y - tl.y);

    // get starting point for the average-indicator
    if ((tl.x <= avg.x) && (avg.x < br.x)) avg.y = tl.y;
  }

  // draw the average-indicator
  dc.SetPen(wxPen(m_Pen.GetColour(), 1, wxDOT));
  dc.DrawLine(avg.x, avg.y - BASE_MARGIN, avg.x, 0); // a vertical line
  wxPoint points[3];
  wxCoord triangle_y = graph.Value2CoordY(100.0f);
  points[0] = wxPoint(avg.x, triangle_y + AVERAGE_INDICATOR_SIZE);
  points[1] = wxPoint(avg.x - AVERAGE_INDICATOR_SIZE / 2, triangle_y);
  points[2] = wxPoint(avg.x + AVERAGE_INDICATOR_SIZE / 2, triangle_y);
  dc.SetPen(m_Pen);
  dc.SetBrush(wxBrush(m_Pen.GetColour()));
  dc.DrawPolygon(3, points); // a triangle at the top of the graph
}

//-----------------------------------------------------------------------------
// Set the type of data that is shown
// - log = type of log-data
// - wheel = index of the wheel for which the data must be shown

void cCurveH::SetLogtype(int log, int wheel)
{
  wxASSERT(log >= LOGTYPE_FIRST);
  wxASSERT(log < LOGTYPE_LAST);
  wxASSERT(wheel >= WHEELTYPE_FIRST);
  wxASSERT(wheel < WHEELTYPE_LAST);

  m_Logtype = log;
  m_Wheel = wheel;
  AdjustBbox();
}

//-----------------------------------------------------------------------------
// Re-calculate the bouding box

void cCurveH::AdjustBbox()
{
  m_MinY = 0.0f; // fixed value
  m_MaxY = 100.0f;

  // scan log data for extremes in X dimension
  m_MinX = IMPOSSIBLY_HIGH_VALUE;
  m_MaxX = IMPOSSIBLY_LOW_VALUE;
  for (size_t s = 0; s < m_Lap->GetStateCount(); s++) {
    float val = m_Lap->GetState(s)->GetLogData(m_Logtype, m_Wheel);
    ::Minimise(m_MinX, val);
    ::Maximise(m_MaxX, val);
  }
}

//-----------------------------------------------------------------------------
// Set the range for the histogram and the number of bins
// - bins = number of bins

void cCurveH::SetBins(float start, float end, int bins)
{
  wxASSERT(start <= m_MinX);
  wxASSERT(end >= m_MaxX);

  m_BinStart = start;
  m_BinEnd = end;
  m_BinCount = bins;
}
