use std::{
    io::{Read},
    net::*,
    process::*,
    time::{ self, Duration},
};
use clap::Parser;

use lfs_async_InSim as LFS;

/// LFS InSim video player...
///
/// Requires ffmpeg to function properly, and optiononaly ffplay,
/// both findable on official site, as of time of writing: https://ffmpeg.org/
#[derive(Parser, Debug)]
#[command(version, about, long_about)]
struct Args {
    /// Address to InSim protocol TCP IPv4 port.
    #[arg(short='a', long, default_value="127.0.0.1:11000")]
    insim_address: SocketAddrV4,
    
    /// Password to lfs instance.
    #[arg(short, long)]
    password: String,
    
    /// Video playback rate.
    ///
    /// This value now works so quirky, you just have to test your case.
    #[arg(short, long, default_value_t = 30)]
    rate: u64,
    
    /// Path to ffmpeg.
    #[arg(short, long, name="FFMPEG_PATH", default_value = "ffmpeg")]
    ffmpeg: Box<std::path::Path>,
    
    /// Path to ffplay.
    ///
    /// If given, there's a chance sound will play along.
    #[arg(long, name="FFPLAY_PATH")]
    ffplay: Option< Box<std::path::Path> >,
    
    /// Path to video.
    ///
    /// Later on supplied to ffmpeg.
    #[arg(short, long, name="VIDEO_PATH")]
    video: Box<std::path::Path>,
    
    /// Skip forward video. FFmpeg time syntax.
    ///
    /// The equivlaent of pre-input -ss ffmpeg param.
    #[arg(short)]
    s: Option<String>,
    
    /// Duration of playback. FFmpeg time syntax.
    ///
    /// The equivlaent of post-inputs -t ffmpeg param.
    #[arg(short)]
    t: Option<String>,
    
    /// Determines display outlook.
    ///
    /// More specificly, how the grayscaled picture pixels
    /// will translate to button text.
    ///
    /// From lowest luminace to highest.
    #[arg(long, name="CHARACTER/S", num_args=4,
        default_values_t=[
            "^0_".to_string(),
            "^3_".to_string(),
            "^6_".to_string(),
            "^7_".to_string(),
        ]
    )]
    subpixels: Vec<String>,
    
    /// Optional delay between program start and playback start.
    ///
    /// Applicable if you're afraid that you won't ALT+TAB fast enought
    /// to see show right as it starts.
    #[arg(long, name="microseconds")]
    delay: Option<u64>,
}

#[derive(Clone, Debug)]
struct Subpix(String, String, String, String);

#[tokio::main]
async fn main() -> Result< (), Box<dyn std::error::Error> > {
    let args = Args::parse();
    const WIDTH: usize = 80;
    //const HEIGHT: usize = (WIDTH / 4) * 3;
    const HEIGHT: usize = (WIDTH as f64 / 2.6) as usize;
    let rate: u64 = args.rate;
    let file: Box<std::path::Path> = args.video;
    let file = file.to_str().unwrap();
    let mut subpixels = args.subpixels.into_iter();
    let subpix = Subpix(
        subpixels.next().unwrap(),
        subpixels.next().unwrap(),
        subpixels.next().unwrap(),
        subpixels.next().unwrap(),
    );
    
    let mut lfs_server = LFS::Server {
        server_address: args.insim_address,
        keep_alive: LFS::Keep_Alive::Auto,
    }.initialise( LFS::IS::ISI::new()
        .set_ReqI( 1 )
        .set_Flags( match args.insim_address.ip() {
            &Ipv4Addr::LOCALHOST => &[LFS::ISF::LOCAL],
            _ => &[],
        } )?
        .set_Admin( &args.password )?
        .set_IName( "Bad Apple" )?
    ).await?;
    
    if let Some( delay_ms ) = args.delay {
        println!( "Prepare yourself, we're startin' in!" );
        tokio::time::sleep( Duration::from_millis( delay_ms ) ).await;
    }
    
    let vf = format!("scale={WIDTH}:{HEIGHT},fps={rate},format=gray");
    let mut ffmpeg_handle =
        std::process::Command::new( args.ffmpeg.to_str().unwrap() )
            .args( {
                let mut vec = vec![
                    "-re",
                    "-i", file,
                    "-vf", &vf,
                    //"-r", format!( "{rate}" ).as_str(),
                    "-f", "rawvideo",
                    "-",
                ];
                let mut offset = 0;
                if let Some( ref ss ) = args.s {
                    vec.insert(1, "-ss");
                    vec.insert(2, ss);
                    offset += 2;
                }
                if let Some( ref t ) = args.t {
                    vec.insert(3 + offset, "-t");
                    vec.insert(4 + offset, t);
                }
                vec
            } )
            .stdin( Stdio::null() )
            .stdout( Stdio::piped() )
            .spawn()?;
    
    let ffmpeg_stdout = ffmpeg_handle.stdout.take();
    
    let mut ffplay_handle = None;
    if let Some( path ) = args.ffplay {
        ffplay_handle = Some(
            std::process::Command::new( path.to_str().unwrap() ).args({
                let mut vec = vec![
                    "-i", file,
                    "-vn",
                    "-showmode", "0",
                ];
                let mut offset = 0;
                if let Some( ref ss ) = args.s {
                    vec.insert(0, "-ss");
                    vec.insert(1, ss);
                    offset += 2;
                }
                if let Some( ref t ) = args.t {
                    vec.insert(2 + offset, "-t");
                    vec.insert(3 + offset, t);
                }
                vec
            } )
            .stderr( Stdio::null() )
            .stdout( Stdio::null() )
            .stdin( Stdio::null() )
            .spawn()?
        );
    }
    
    let mut osim = String::with_capacity( subpix.0.len() * WIDTH * HEIGHT );
    let mut prev_osim = String::with_capacity( osim.capacity() );
    for _ in 0..HEIGHT { prev_osim += "\n" }
    if let Some( mut ffmpeg_sout) = ffmpeg_stdout {
        println!( "Runnin'" );
        let mut total_read_bytes = 0;
        const BUFLEN: usize = WIDTH * HEIGHT;
        let mut buf = [0; BUFLEN];
        // print!("\x1B[2J\x1B[3J\x1B[1;1H");
        // use std::io::Write;
        // std::io::stdout().flush();
        
        let mut btn = LFS::IS::BTN {
            Size: LFS::IS::BTN::SizeQuarter() as u8,
            Type: LFS::ISP::BTN,
            ReqI: 1,
            UCID: 255,
            ClickID: 0,
            Inst: 0,
            BStyle: [ LFS::ISBS::DARK ][..].try_into()?,
            TypeIn: 0,
            L: 0,
            T: 0,
            W: 240,
            H: 200 / HEIGHT as u8,
            Text: [0;240],
        };
        
        let frame_time_check = Duration::from_millis( 930 / rate );
        let mut skipped_frames = 0usize;
        let mut unskipped_frames = 0usize;
        let mut buttons = Vec::with_capacity( osim.capacity() );
        
        //Weird if let usage, but somehow from my various testings, it's often faster than just if.
        loop {
            let elapsed_on_read = time::Instant::now();
            let read_bytes = ffmpeg_sout.read( &mut buf )?;
            if let true = read_bytes != 0 {
                if let true = elapsed_on_read.elapsed() < frame_time_check { skipped_frames += 1; continue; }
                unskipped_frames += 1;
                // Should check would be nice, or even proper buffering,
                // but it would slow down the loop too much, not to
                // mention time consumption on implementation. At least
                // this solution can give some fun side-effects on LFS
                // Display, hiehieh.
                //if read_bytes < BUFLEN { panic!() }
                
                btn.ClickID = 0;
                btn.T = 0;
                
                osim.clear();
                //print!("\x1B[1;1H");
                total_read_bytes += read_bytes;
                let mut avg = ( unsafe { buf.get_unchecked(..read_bytes) }
                    .iter().fold( 0u64, |acc, pix| acc + *pix as u64) / read_bytes as u64) as u8;
                if avg < 16 { avg = 64 };
                for chunk in unsafe{ buf.get_unchecked( ..read_bytes ) }.chunks(WIDTH) {
                    for &pixel in chunk.iter() {
                        if let true = pixel < avg {
                            if let true = pixel < avg / 2 { osim += &subpix.0; continue; }
                            else { osim += &subpix.1; continue; }
                        } else if let true = pixel < ( (u8::MAX - avg) / 2 + avg )
                        { osim += &subpix.2; continue; }
                        else { osim += &subpix.3; }
                    }
                    osim += "\n";
                }
                //println!( "{osim}" );
                
                let mut po_iter = prev_osim.lines();
                for hline in osim.lines() {
                    btn.ClickID += 1;
                    if let true = hline != unsafe { po_iter.next().unwrap_unchecked() } {
                        btn.fast_text_exchange( hline );
                        buttons.push( btn.clone() );
                    }
                    btn.T += 200 / HEIGHT as u8;
                }
                
                if let false = buttons.is_empty() {
                    unsafe { lfs_server.unsafe_multiple_packet_send( &buttons ).await?; }
                }
                prev_osim = osim.clone();
                buttons.clear();
            } else { break; }
        }
        let read_bytes = format!( "Read bytes: {}kB, Skipped {skipped_frames} frames ({}%)",
            total_read_bytes / 1024,
            { let total_frames = unskipped_frames + skipped_frames;
            skipped_frames * 100 / total_frames }
            );
        lfs_server.packet_send( LFS::IS::MST::new( &read_bytes )? ).await?;
        tokio::time::sleep( Duration::from_millis( 500 ) ).await;
        println!( "{read_bytes}" );
    }
    
    ffmpeg_handle.kill()?;
    if let Some( mut hndl ) = ffplay_handle { hndl.kill()? };
    
    Ok(())
}
