#![allow(non_snake_case, illegal_floating_point_literal_pattern)]

use std::{
    io::{Read},
    net::*,
    process::*,
    time::{ self, Duration},
};
use clap::Parser;

use lfs_async_InSim as LFS;

/// LFS InSim video player... color version.
///
/// Requires ffmpeg to function properly, and optiononaly ffplay,
/// both findable on official site, as of time of writing: https://ffmpeg.org/
#[derive(Parser, Debug)]
#[command(version, about, long_about)]
struct Args {
    /// Address to InSim protocol TCP IPv4 port.
    #[arg(short='a', long, default_value="127.0.0.1:11000")]
    insim_address: SocketAddrV4,
    
    /// Password to lfs instance.
    #[arg(short, long)]
    password: String,
    
    /// Video playback rate.
    ///
    /// This value now works so quirky, you just have to test your case.
    #[arg(short, long, default_value_t = 30)]
    rate: u64,
    
    /// Path to ffmpeg.
    #[arg(short, long, name="FFMPEG_PATH", default_value = "ffmpeg")]
    ffmpeg: Box<std::path::Path>,
    
    /// Path to ffplay.
    ///
    /// If given, there's a chance sound will play along.
    #[arg(long, name="FFPLAY_PATH")]
    ffplay: Option< Box<std::path::Path> >,
    
    /// Path to video.
    ///
    /// Later on supplied to ffmpeg.
    #[arg(short, long, name="VIDEO_PATH")]
    video: Box<std::path::Path>,
    
    /// Skip forward video. FFmpeg time syntax.
    ///
    /// The equivlaent of pre-input -ss ffmpeg param.
    #[arg(short)]
    s: Option<String>,
    
    /// Duration of playback. FFmpeg time syntax.
    ///
    /// The equivlaent of post-inputs -t ffmpeg param.
    #[arg(short)]
    t: Option<String>,
    
    
    /// Optional delay between program start and playback start.
    ///
    /// Applicable if you're afraid that you won't ALT+TAB fast enought
    /// to see show right as it starts.
    #[arg(long, name="microsecunds")]
    delay: Option<u64>,
}

#[tokio::main]
async fn main() -> Result< (), Box<dyn std::error::Error> > {
    let args = Args::parse();
    const WIDTH: usize = 80;
    //const HEIGHT: usize = (WIDTH / 4) * 3;
    const HEIGHT: usize = (WIDTH as f64 / 2.6) as usize;
    let rate: u64 = args.rate;
    let file: Box<std::path::Path> = args.video;
    let file = file.to_str().unwrap();
    
    let mut lfs_server = LFS::Server {
        server_address: args.insim_address,
        keep_alive: LFS::Keep_Alive::Auto,
    }.initialise( LFS::IS::ISI::new()
        .set_ReqI( 1 )
        .set_Flags( match args.insim_address.ip() {
            &Ipv4Addr::LOCALHOST => &[LFS::ISF::LOCAL],
            _ => &[],
        } )?
        .set_Admin( &args.password )?
        .set_IName( "Bad Apple" )?
    ).await?;
    
    if let Some( delay_ms ) = args.delay {
        println!( "Prepare yourself, we're startin' in!" );
        tokio::time::sleep( Duration::from_millis( delay_ms ) ).await;
    }
    
    let vf = format!("scale={WIDTH}:{HEIGHT},fps={rate},format=rgb24");
    let mut ffmpeg_handle =
        std::process::Command::new( args.ffmpeg.to_str().unwrap() )
            .args( {
                let mut vec = vec![
                    //"-loop", "1",
                    "-re",
                    "-i", file,
                    "-vf", &vf,
                    //"-r", format!( "{rate}" ).as_str(),
                    "-f", "rawvideo",
                    "-",
                ];
                let mut offset = 0;
                if let Some( ref ss ) = args.s {
                    vec.insert(1, "-ss");
                    vec.insert(2, ss);
                    offset += 2;
                }
                if let Some( ref t ) = args.t {
                    vec.insert(3 + offset, "-t");
                    vec.insert(4 + offset, t);
                }
                vec
            } )
            .stdin( Stdio::null() )
            .stderr( Stdio::null() )
            .stdout( Stdio::piped() )
            .spawn()?;
    
    let ffmpeg_stdout = ffmpeg_handle.stdout.take();
    
    let mut ffplay_handle = None;
    if let Some( path ) = args.ffplay {
        ffplay_handle = Some(
            std::process::Command::new( path.to_str().unwrap() ).args({
                let mut vec = vec![
                    file,
                    "-vn",
                    "-showmode", "0",
                ];
                if let Some( ref ss ) = args.s {
                    vec.push( "-ss" );
                    vec.push( ss );
                }
                if let Some( ref t ) = args.t {
                    vec.push( "-t" );
                    vec.push( t );
                }
                vec
            } )
            .stderr( Stdio::null() )
            .stdout( Stdio::null() )
            .stdin( Stdio::null() )
            .spawn()?
        );
    }
    
    let mut osim = String::with_capacity( 3 * WIDTH * HEIGHT );
    let mut prev_osim = String::with_capacity( osim.capacity() );
    for _ in 0..HEIGHT { prev_osim += "\n" }
    if let Some( mut ffmpeg_sout) = ffmpeg_stdout {
        println!( "Runnin'" );
        let mut total_read_bytes = 0;
        const BUFLEN: usize = WIDTH * 3 * HEIGHT;
        let mut buf = [0; BUFLEN];
        // print!("\x1B[2J\x1B[3J\x1B[1;1H");
        // use std::io::Write;
        // std::io::stdout().flush();
        
        let mut btn = LFS::IS::BTN {
            Size: LFS::IS::BTN::SizeQuarter() as u8,
            Type: LFS::ISP::BTN,
            ReqI: 1,
            UCID: 255,
            ClickID: 0,
            Inst: 0,
            BStyle: [ LFS::ISBS::DARK ][..].try_into()?,
            TypeIn: 0,
            L: 0,
            T: 0,
            W: 240,
            H: 200 / HEIGHT as u8,
            Text: [0;240],
        };
        
        let frame_time_check = Duration::from_millis( 930 / rate );
        let mut skipped_frames = 0usize;
        let mut unskipped_frames = 0usize;
        let mut buttons = Vec::with_capacity( osim.capacity() );
        
        //Weird if let usage, but somehow from my various testings, it's often faster than just if.
        loop {
            let elapsed_on_read = time::Instant::now();
            let read_bytes = ffmpeg_sout.read( &mut buf )?;
            if let true = read_bytes != 0 {
                if let true = elapsed_on_read.elapsed() < frame_time_check { skipped_frames += 1; continue; }
                unskipped_frames += 1;
                // Should check would be nice, or even proper buffering,
                // but it would slow down the loop too much, not to
                // mention time consumption on implementation. At least
                // this solution can give some fun side-effects on LFS
                // Display, hiehieh.
                //if read_bytes < BUFLEN { panic!() }
                
                btn.ClickID = 0;
                btn.T = 0;
                
                osim.clear();
                //print!("\x1B[1;1H");
                total_read_bytes += read_bytes;
                
                enum ColorComponent {
                    RED,
                    GREEN,
                    BLUE,
                }
                
                struct SubPix {
                    value: usize,
                    component: ColorComponent,
                }
                
                for chunk in unsafe{ buf.get_unchecked( ..read_bytes ) }.chunks(WIDTH * 3) {
                    use ColorComponent::*;
                    for pixel_org in chunk.chunks(3) {
                        let mut iter = pixel_org.iter();
                        let mut pixel: [SubPix;3] = [
                            SubPix{ value: *iter.next().unwrap() as usize, component: RED},
                            SubPix{ value: *iter.next().unwrap() as usize, component: GREEN},
                            SubPix{ value: *iter.next().unwrap() as usize, component: BLUE},
                        ];
                        
                        pixel.sort_by_key( |tg| tg.value );
                        let lightness = (pixel[2].value + pixel[0].value) / 2;
                        
                        let chroma = pixel[2].value - pixel[0].value;
                        let mut hue = 0.0;
                        if let true = chroma != 0 {
                            hue = 60.0 * match pixel[2].component {
                                RED => {
                                    ( pixel_org[1] as f64 - pixel_org[2] as f64 ) / chroma as f64 % 6.0
                                },
                                GREEN => {
                                    ( pixel_org[2] as f64 - pixel_org[0] as f64 ) / chroma as f64 + 2.0
                                },
                                BLUE => {
                                    ( pixel_org[0] as f64 - pixel_org[1] as f64 ) / chroma as f64 + 4.0
                                },
                            }
                        }
                        
                        //dbg!( &hue );
                        
                        let saturation = match lightness {
                            0 | 255 => 0.0,
                            L => chroma as f64 / ( 255.0 - (2*L).abs_diff(255) as f64 ),
                        };
                        
                        //dbg!( saturation );
                        
                        match lightness {
                            ..=49 => osim += "^0_",
                            50..=220 => {
                                match saturation {
                                    //fuck gray
                                    //..=0.2 => osim += "^8_",
                                    ..=0.22 => osim += "^0_",
                                    _ => {
                                        match hue {
                                            ..=0.0 => osim += "^5_",
                                            0.0..=35.0 => osim += "^1_",
                                            35.0..=73.0 => osim += "^3_",
                                            73.0..=140.0 => osim += "^2_",
                                            140.0..=197.0 => osim += "^6_",
                                            197.0..=260.0 => osim += "^4_",
                                            260.0.. => osim += "^5_",
                                            any => panic!("How? {any}"),
                                        }
                                    }
                                }
                                //osim += "^8_"
                            },
                            221.. => osim += "^7_",
                        }
                        // if pixel[0] > 12 {
                        //     osim += "^7_";
                        // } else {
                        //     osim += "^0_";
                        // }
                    }
                    osim += "\n";
                }
                //println!( "{osim}" );
                
                let mut po_iter = prev_osim.lines();
                for hline in osim.lines() {
                    btn.ClickID += 1;
                    if let true = hline != unsafe { po_iter.next().unwrap_unchecked() } {
                        btn.fast_text_exchange( hline );
                        buttons.push( btn.clone() );
                    }
                    btn.T += 200 / HEIGHT as u8;
                }
                
                if let false = buttons.is_empty() {
                    unsafe { lfs_server.unsafe_multiple_packet_send( &buttons ).await?; }
                }
                prev_osim = osim.clone();
                buttons.clear();
            } else { break; }
        }
        let read_bytes = format!( "Read bytes: {}kB, Skipped {skipped_frames} frames ({}%)",
            total_read_bytes / 1024,
            { let total_frames = unskipped_frames + skipped_frames;
            skipped_frames * 100 / total_frames }
            );
        lfs_server.packet_send( LFS::IS::MST::new( &read_bytes )? ).await?;
        tokio::time::sleep( Duration::from_millis( 500 ) ).await;
        println!( "{read_bytes}" );
    }
    
    ffmpeg_handle.kill()?;
    if let Some( mut hndl ) = ffplay_handle { hndl.kill()? };
    
    Ok(())
}
