#![allow(unused_imports, dead_code)]

use std::io::{Read, stderr};
use std::net::{Ipv4Addr, SocketAddrV4};
use std::process::{ChildStderr, ChildStdin, ChildStdout, Stdio};
use std::time::Duration;
use tokio::time::sleep;
use clap::Parser;

use lfs_async_InSim as LFS;

/// LFS InSim video player...
#[derive(Parser, Debug)]
#[command(about, long_about)]
struct Args {
    /// Address to InSim protocol TCP IPv4 port.
    #[arg(short='a', long, default_value="127.0.0.1:11000")]
    insim_address: SocketAddrV4,
    
    /// Password to lfs instance.
    #[arg(short, long)]
    password: String,
    
    /// Video playback rate.
    ///
    /// More = more throttling.
    ///
    /// On my machine, 16Hz seemed to be max I could get locally,
    /// and even that was not achivable on all LFS screens. Somehow the
    /// menu screens are the worst, much better to play it when on track.
    ///
    /// Multiplayer wise; 6Hz was max I could get on freehosting + my LTE internet.
    #[arg(short, long, default_value_t = 10)]
    rate: u64,
    
    /// Path to ffmpeg.
    #[arg(short, long, name="FFMPEG_PATH", default_value = "ffmpeg")]
    ffmpeg: Box<std::path::Path>,
    
    /// Path to ffplay.
    ///
    /// If given, there's a chance sound will play along.
    #[arg(long, name="FFPLAY_PATH")]
    ffplay: Option< Box<std::path::Path> >,
    
    /// Path to video.
    ///
    /// Later on supplied to ffmpeg.
    #[arg(short, long, name="VIDEO_PATH")]
    video: Box<std::path::Path>,
    
    /// Determines display outlook.
    ///
    /// More specificly, how the grayscaled picture pixels
    /// will translate to button text.
    ///
    /// From lowest luminace to highest.
    #[arg(long, name="CHARACTER/S", num_args=4,
        default_values_t=[
            "^0_".to_string(),
            "^3_".to_string(),
            "^6_".to_string(),
            "^7_".to_string(),
        ]
    )]
    subpixels: Vec<String>,
    
    /// Optional delay between program start and playback start.
    ///
    /// Applicable if you're afraid that you won't ALT+TAB fast enought
    /// to see show right as it starts.
    #[arg(long, name="microsecunds")]
    delay: Option<u64>,
}

#[derive(Debug)]
struct Ffmpeg {
    stdin: Option<ChildStdin>,
    stderr: Option<ChildStderr>,
    stdout: Option<ChildStdout>,
}

#[derive(Clone, Debug)]
struct Subpix(String, String, String, String);

#[tokio::main]
async fn main() -> Result< (), Box<dyn std::error::Error> > {
    let args = Args::parse();
    const WIDTH: usize = 80;
    //const HEIGHT: usize = (WIDTH / 4) * 3;
    const HEIGHT: usize = (WIDTH as f64 / 2.6) as usize;
    let rate: u64 = args.rate;
    let file: Box<std::path::Path> = args.video;
    let file = file.to_str().unwrap();
    let mut subpixels = args.subpixels.into_iter();
    let subpix = Subpix(
        subpixels.next().unwrap(),
        subpixels.next().unwrap(),
        subpixels.next().unwrap(),
        subpixels.next().unwrap(),
    );
    
    let mut lfs_server = LFS::Server {
        server_address: args.insim_address,
        keep_alive: LFS::Keep_Alive::Auto,
    }.initialise( LFS::IS::ISI::new()
        .set_ReqI( 1 )
        .set_Flags( match args.insim_address.ip() {
            &Ipv4Addr::LOCALHOST => &[LFS::ISF::LOCAL],
            _ => &[],
        } )?
        .set_Admin( &args.password )?
        .set_IName( "Bad Apple" )?
    ).await?;
    
    if let Some( delay_ms ) = args.delay {
        println!( "Prepare yourself, we're startin' in!" );
        tokio::time::sleep( Duration::from_millis( delay_ms ) ).await;
    }
    
    let mut handle =
        std::process::Command::new( args.ffmpeg.to_str().unwrap() )
            .args( [
                "-re",
                //"-ss", "3:38",
                "-i", file,
                "-vf", format!("scale={WIDTH}:{HEIGHT},format=gray").as_str(),
                "-r", format!( "{rate}" ).as_str(),
                "-f", "rawvideo",
                "-",
            ] )
            .stdin( Stdio::null() )
            .stderr( Stdio::piped() )
            .stdout( Stdio::piped() )
            .spawn()?;
    
    let mut handle2 = None;
    if let Some( path ) = args.ffplay {
        handle2 = Some(
            std::process::Command::new( path.to_str().unwrap() ).args(
                [
                    "-i", file,
                    "-vn",
                    "-showmode", "0",
                ]
            ).spawn()?
        );
    }
    
    let ffmpeg = Ffmpeg {
        stdin: handle.stdin.take(),
        stderr: handle.stderr.take(),
        stdout: handle.stdout.take(),
    };
    
    let mut osim = String::new();
    if let Some( mut stdout ) = ffmpeg.stdout {
        println!( "Runnin'" );
        let mut total_read_bytes = 0;
        const BUFLEN: usize = WIDTH * HEIGHT;
        let mut buf = [0; BUFLEN];
        // print!("\x1B[2J\x1B[3J\x1B[1;1H");
        // use std::io::Write;
        // std::io::stdout().flush();
        
        let mut btn = LFS::IS::BTN {
            Size: LFS::IS::BTN::SizeQuarter() as u8,
            Type: LFS::ISP::BTN,
            ReqI: 1,
            UCID: 255,
            ClickID: 0,
            Inst: 0,
            BStyle: LFS::ISBS_computed( LFS::ISBS::DARK as u8 ),
            TypeIn: 0,
            L: 0,
            T: 0,
            W: 240,
            H: 200 / HEIGHT as u8,
            Text: [0;240],
        };
    
        loop {
            let time_instance = std::time::Instant::now();
            let read_bytes = stdout.read( &mut buf )?;
            if read_bytes == 0 { break; }
            // Should check would be nice, or even proper buffering,
            // but it would slow down the loop too much, not to
            // mention time consumption on implementation. At least
            // this solution can give some fun side-effects on LFS
            // Display, hiehieh.
            //if read_bytes < BUFLEN { panic!() }
            
            btn.ClickID = 0;
            btn.T = 0;
            
            osim.clear();
            //print!("\x1B[1;1H");
            total_read_bytes += read_bytes;
            let mut avg = ( unsafe { buf.get_unchecked(..read_bytes) }
                .iter().fold( 0u64, |acc, pix| acc + *pix as u64) / read_bytes as u64) as u8;
            if avg < 16 { avg = 64 };
            for chunk in unsafe{ buf.get_unchecked( ..read_bytes ) }.chunks(WIDTH) {
                for pixel in chunk.iter() {
                    if *pixel < avg / 2 { osim += &subpix.0; continue; } else
                    if *pixel < avg { osim += &subpix.1; continue; } else
                    if *pixel < ( (u8::MAX - avg) / 2 + avg ) { osim += &subpix.2; continue; } else {
                        osim += &subpix.3;
                    }
                }
                osim += "\n";
            }
            //println!( "{osim}" );
            
            for hline in osim.lines() {
                btn.ClickID += 1;
                btn.fast_text_exchange( hline );
                lfs_server.packet_send(btn).await?;
                btn.T += 200 / HEIGHT as u8;
            }
            
            //let mut no_throttle = 0usize;
            loop {
                if time_instance.elapsed() > Duration::from_millis( 1000 / rate ) {
                    //if let 0 = no_throttle { eprintln!("Throttling..."); }
                    break;
                }
                //else { no_throttle += 1; }
                
            }
            //std::thread::sleep( Duration::from_millis( 1000 / rate ) );
        }
        let read_bytes = format!( "Read bytes: {}kB", total_read_bytes / 1024 );
        lfs_server.packet_send( LFS::IS::MST::new( &read_bytes )? ).await?;
        tokio::time::sleep( Duration::from_millis( 500 ) ).await;
        println!( "{read_bytes}" );
    }
    
    handle.kill()?;
    if let Some( mut hndl ) = handle2 { hndl.kill()? };
    
    Ok(())
}
