"""
An InSim library for the Python programming language.

"""

import socket
import thread
import struct
import math
import re

# Connection Type
INSIM_TCP = socket.SOCK_STREAM
INSIM_UDP = socket.SOCK_DGRAM

# Current InSim version    
INSIM_VERSION = 4     

# Enum for packet-types
ISP_NONE = 0
ISP_ISI = 1
ISP_VER = 2
ISP_TINY = 3
ISP_SMALL = 4
ISP_STA = 5
ISP_SCH = 6
ISP_SFP = 7
ISP_SCC = 8
ISP_CPP = 9
ISP_ISM = 10
ISP_MSO = 11
ISP_III = 12
ISP_MST = 13
ISP_MTC = 14
ISP_MOD = 15
ISP_VTN = 16
ISP_RST = 17
ISP_NCN = 18
ISP_CNL = 19
ISP_CPR = 20
ISP_NPL = 21
ISP_PLP = 22
ISP_PLL = 23
ISP_LAP = 24
ISP_SPX = 25
ISP_PIT = 26
ISP_PSF = 27
ISP_PLA = 28
ISP_CCH = 29
ISP_PEN = 30
ISP_TOC = 31
ISP_FLG = 32
ISP_PFL = 33
ISP_FIN = 34
ISP_RES = 35
ISP_REO = 36
ISP_NLP = 37
ISP_MCI = 38
ISP_MSX = 39
ISP_MSL = 40
ISP_CRS = 41
ISP_BFN = 42
ISP_AXI = 43
ISP_AXO = 44
ISP_BTN = 45
ISP_BTC = 46
ISP_BTT = 47

COMPCAR = 255

# Enum for IS_TINY sub-type
TINY_NONE = 0
TINY_VER = 1
TINY_CLOSE = 2
TINY_PING = 3
TINY_REPLY = 4
TINY_VTC = 5
TINY_SCP = 6
TINY_SST = 7
TINY_GTH = 8
TINY_MPE = 9
TINY_ISM = 10
TINY_REN = 11
TINY_CLR = 12
TINY_NCN = 13
TINY_NPL = 14
TINY_RES = 15
TINY_NLP = 16
TINY_MCI = 17
TINY_REO = 18
TINY_RST = 19
TINY_AXI = 20
TINY_AXC = 21

# Enum for IS_SMALL sub-type
SMALL_NONE = 0
SMALL_SSP = 1
SMALL_SSG = 2
SMALL_VTA = 3
SMALL_TMS = 4
SMALL_STP = 5
SMALL_RTP = 6
SMALL_NLI = 7 

# Flags for ISI Flags
ISF_RES_0 = 1
ISF_RES_1 = 2
ISF_LOCAL = 4
ISF_MSO_COLS = 8
ISF_NLP	= 16
ISF_MCI	= 32

# Enum for User Values
MSO_SYSTEM = 0
MSO_USER = 1
MSO_PREFIX = 2
MSO_O = 3

# Enum for message sounds
SND_SILENT = 0
SND_MESSAGE = 1
SND_SYSMESSAGE = 2
SND_INVALIDKEY = 3
SND_ERROR = 4

# Enum for voting
VOTE_NONE = 0
VOTE_END = 1
VOTE_RESTART = 2
VOTE_QUALIFY = 3

# Enum for pitlane
PITLANE_EXIT = 0
PITLANE_ENTER = 1
PITLANE_NO_PURPOSE = 2
PITLANE_DT = 3
PITLANE_SG = 4

# Enum for view identifiers
VIEW_FOLLOW = 0
VIEW_HELI = 1
VIEW_CAM = 2
VIEW_DRIVER = 3
VIEW_CUSTOM = 4
VIEW_ANOTHER = 255

# Enum for leave reasons
LEAVR_DISCO = 0
LEAVR_TIMEOUT = 1
LEAVR_LOSTCONN = 2
LEAVR_KICKED = 3
LEAVR_BANNED = 4
LEAVR_SECURITY = 5

# Enum for Penalty values
PENALTY_NONE = 0		
PENALTY_DT = 1
PENALTY_DT_VALID = 2
PENALTY_SG = 3
PENALTY_SG_VALID = 4
PENALTY_30 = 5
PENALTY_45 = 6

# Enum for Penalty reasons
PENR_UNKNOWN = 1
PENR_ADMIN = 2
PENR_WRONG_WAY = 3
PENR_FALSE_START = 4
PENR_SPEEDING = 5
PENR_STOP_SHORT = 6
PENR_STOP_LATE = 7

# Enum for Tyre compounds
TYRE_R1 = 1
TYRE_R2 = 2
TYRE_R3 = 3
TYRE_R4 = 4
TYRE_ROAD_SUPER = 5
TYRE_ROAD_NORMAL = 6
TYRE_HYBRID = 7
TYRE_KNOBBLY = 8
TYRE_NOT_CHANGED = 255

# ISS state flags
ISS_GAME = 1
ISS_REPLAY = 2
ISS_PAUSED = 4
ISS_SHIFTU = 8
ISS_SHIFTU_HIGH	= 16
ISS_SHIFTU_FOLLOW = 32
ISS_SHIFTU_NO_OPT = 64
ISS_SHOW_2D = 128
ISS_FRONT_END = 256
ISS_MULTI = 512
ISS_MPSPEEDUP =	1024
ISS_WINDOWED = 2048
ISS_SOUND_MUTE = 4096
ISS_VIEW_OVERRIDE = 8192
ISS_VISIBLE = 16384

# Pitwork flags
PSE_NOTHING = 1
PSE_STOP = 2
PSE_FR_DAM = 4
PSE_FR_WHL = 8
PSE_LE_FR_DAM = 16,
PSE_LE_FR_WHL = 32,
PSE_RI_FR_DAM = 64,
PSE_RI_FR_WHL = 128,
PSE_RE_DAM = 256,
PSE_RE_WHL = 512,
PSE_LE_RE_DAM = 1024,
PSE_LE_RE_WHL = 2048,
PSE_RI_RE_DAM = 4096
PSE_RI_RE_WHL = 8192
PSE_BODY_MINOR = 16384
PSE_BODY_MAJOR = 32768
PSE_SETUP = 65536
PSE_REFUEL = 131072
PSE_NUM = 262144

# Player flags
PIF_SWAPSIDE = 1
PIF_RESERVED_2 = 2
PIF_RESERVED_4 = 4
PIF_AUTOGEARS = 8
PIF_SHIFTER = 16
PIF_RESERVED_32 = 32
PIF_HELP_B = 64
PIF_AXIS_CLUTCH = 128
PIF_INPITS = 256
PIF_AUTOCLUTCH = 512
PIF_MOUSE = 1024
PIF_KB_NO_HELP = 2048
PIF_KB_STABILISED = 4096
PIF_CUSTOM_VIEW = 8192

# Confirmation flags
CONF_MENTIONED = 1
CONF_CONFIRMED = 2
CONF_PENALTY_DT = 4
CONF_PENALTY_SG = 8
CONF_PENALTY_30 = 16
CONF_PENALTY_45 = 32
CONF_DID_NOT_PIT = 64
CONF_DISQ = (CONF_PENALTY_DT | CONF_PENALTY_SG | CONF_DID_NOT_PIT)
CONF_TIME = (CONF_PENALTY_30 | CONF_PENALTY_45)

# Race flags
HOSTF_CAN_VOTE = 1
HOSTF_CAN_SELECT = 2
HOSTF_MID_RACE = 32
HOSTF_MUST_PIT = 64
HOSTF_CAN_RESET = 128
HOSTF_FCV = 256
HOSTF_CRUISE = 512

# Button Recommended Area
IS_X_MIN = 0
IS_X_MAX = 110
IS_Y_MIN = 30
IS_Y_MAX = 170

# Enum for BFN SubT
BFN_DEL_BTN = 0
BFN_CLEAR = 1
BFN_USER_CLEAR = 2
BFN_REQUEST = 3

INST_ALWAYS_ON = 128

# Flags for IS_BTN BStyle
ISB_C1 = 1
ISB_C2 = 2
ISB_C4 = 4
ISB_CLICK = 8
ISB_LIGHT = 16
ISB_DARK = 32
ISB_LEFT = 64
ISB_RIGHT = 128

# Flags for BTN CFlags
ISB_LMB	= 1
ISB_RMB = 2
ISB_CTRL = 4
ISB_SHIFT = 8

# Default values for packet definitions.
_typeDefaults = {"B": 0, "h": 0, "H": 0, "s": "", "f": 0.0, "i": 0, "I": 0}

# Packet definitions 
_packetDefs = {}

# Initalisation
_packetDefs[ISP_ISI] = [["Size", "B"], 
                        ["Type", "B"],     
                        ["ReqI", "B"],                     
                        ["Zero", "B"], 
                        ["UDPPort", "H"], 
                        ["Flags", "H"], 
                        ["Sp0", "B"], 
                        ["Prefix", "B"],
                        ["Interval", "H"],
                        ["Admin", "16s"], 
                        ["IName", "16s"]]

# General Purpose
_packetDefs[ISP_TINY] = [["Size", "B"], 
                         ["Type", "B"],     
                         ["ReqI", "B"],                     
                         ["SubT", "B"]]

_packetDefs[ISP_SMALL] = [["Size", "B"], 
                          ["Type", "B"],     
                          ["ReqI", "B"],                     
                          ["SubT", "B"],
                          ["UVal", "H"]]

# Version Check
_packetDefs[ISP_VER] = [["Size", "B"],
                        ["Type", "B"],
                        ["ReqI", "B"],
                        ["Zero", "B"],
                        ["Version", "8s"],
                        ["Product", "6s"],
                        ["InSimVer", "H"]]

# State Reporting
_packetDefs[ISP_STA] = [["Size", "B"],
                        ["Type", "B"],
                        ["ReqI", "B"],
                        ["Zero", "B"],
                        ["ReplaySpeed", "f"],
                        ["Flags", "H"],
                        ["InGameCam", "B"],
                        ["ViewPLID", "B"],
                        ["NumP", "B"],
                        ["NumConns", "B"],
                        ["NumFinished", "B"],
                        ["RaceInProg", "B"],
                        ["QualMins", "B"],
                        ["RaceLaps", "B"],
                        ["Spare2", "B"],
                        ["Spare3", "B"],
                        ["Track", "6s"],
                        ["Weather", "B"],
                        ["Wind", "B"]]

_packetDefs[ISP_SFP] = [["Size", "B"],
                        ["Type", "B"],
                        ["ReqI", "B"],
                        ["Zero", "B"],
                        ["Flag", "H"],
                        ["OffOn", "B"],
                        ["Sp3", "B"]]

# Screen Mode
_packetDefs[ISP_MOD] = [["Size", "B"],
                        ["Type", "B"],
                        ["ReqI", "B"],
                        ["Zero", "B"],
                        ["Bits16", "i"],
                        ["RR", "i"],
                        ["Width", "i"],
                        ["Height", "i"]]

# Messages out (from LFS)
_packetDefs[ISP_MSO] = [["Size", "B"],
                        ["Type", "B"],
                        ["ReqI", "B"],
                        ["Zero", "B"],
                        ["UCID", "B"],
                        ["PLID", "B"],
                        ["UserType", "B"],
                        ["TextStart", "B"],
                        ["Msg", "128s"]]

_packetDefs[ISP_III] = [["Size", "B"],
                        ["Type", "B"],
                        ["ReqI", "B"],
                        ["Zero", "B"],
                        ["UCID", "B"],
                        ["PLID", "B"],
                        ["Sp2", "B"],
                        ["Sp3", "B"],
                        ["Msg", "64s"]]

# Messages in (to LFS)
_packetDefs[ISP_MST] = [["Size", "B"],
                        ["Type", "B"],
                        ["ReqI", "B"],
                        ["Zero", "B"],
                        ["Msg", "64s"]]

_packetDefs[ISP_MSX] = [["Size", "B"],
                        ["Type", "B"],
                        ["ReqI", "B"],
                        ["Zero", "B"],
                        ["Msg", "96s"]]

_packetDefs[ISP_MSL] = [["Size", "B"],
                        ["Type", "B"],
                        ["ReqI", "B"],
                        ["Sound", "B"],
                        ["Msg", "128s"]]

_packetDefs[ISP_MTC] = [["Size", "B"],
                        ["Type", "B"],
                        ["ReqI", "B"],
                        ["Zero", "B"],
                        ["UCID", "B"],
                        ["PLID", "B"],
                        ["Sp2", "B"],
                        ["Sp3", "B"],
                        ["Msg", "64s"]]

_packetDefs[ISP_SCH] = [["Size", "B"],
                        ["Type", "B"],
                        ["ReqI", "B"],
                        ["Zero", "B"],
                        ["CharB", "B"],
                        ["Flags", "B"],
                        ["Spare2", "B"],
                        ["Spare3", "B"]]

# Multiplayer Notification.
_packetDefs[ISP_ISM] = [["Size", "B"],
                        ["Type", "B"],
                        ["ReqI", "B"],
                        ["Zero", "B"],
                        ["Host", "B"],
                        ["Sp1", "B"],
                        ["Sp2", "B"],
                        ["Sp3", "B"],
                        ["HName", "32s"]]

# Voting
_packetDefs[ISP_VTN] = [["Size", "B"],
                        ["Type", "B"],
                        ["ReqI", "B"],
                        ["Zero", "B"],
                        ["UCID", "B"],
                        ["Action", "B"],
                        ["Spare2", "B"],
                        ["Spare3", "B"]]

# Race Tracking
_packetDefs[ISP_RST] = [["Size", "B"],
                        ["Type", "B"],
                        ["ReqI", "B"],
                        ["Zero", "B"],
                        ["RaceLaps", "B"],
                        ["QualMins", "B"],
                        ["NumP", "B"],
                        ["Spare", "B"],
                        ["Track", "6s"],
                        ["Weather", "B"],
                        ["Wind", "B"],
                        ["Flags", "H"],
                        ["NumNodes", "H"],
                        ["Finish", "H"],
                        ["Split1", "H"],
                        ["Split2", "H"],
                        ["Split3", "H"]]

_packetDefs[ISP_NCN] = [["Size", "B"],
                        ["Type", "B"],
                        ["ReqI", "B"],
                        ["UCID", "B"],
                        ["UName", "24s"],
                        ["PName", "24s"],
                        ["Admin", "B"],
                        ["Total", "B"],
                        ["Flags", "B"],
                        ["Sp3", "B"]]

_packetDefs[ISP_CNL] = [["Size", "B"],
                        ["Type", "B"],
                        ["ReqI", "B"],
                        ["UCID", "B"],
                        ["Reason", "B"],
                        ["Total", "B"],
                        ["Sp2", "B"],
                        ["Sp3", "B"]]

_packetDefs[ISP_CPR] = [["Size", "B"],
                        ["Type", "B"],
                        ["ReqI", "B"],
                        ["UCID", "B"],
                        ["PName", "24s"],
                        ["Plate", "8s"]]

_packetDefs[ISP_NPL] = [["Size", "B"],
                        ["Type", "B"],
                        ["ReqI", "B"],
                        ["PLID", "B"],
                        ["UCID", "B"],
                        ["PType", "B"],
                        ["Flags", "H"],
                        ["PName", "24s"],
                        ["Plate", "8s"],
                        ["CName", "4s"],
                        ["SName", "16s"],
                        ["Tyres1", "B"],
                        ["Tyres2", "B"],
                        ["Tyres3", "B"],
                        ["Tyres4", "B"],
                        ["H_Mass", "B"],
                        ["H_TRes", "B"],
                        ["Model", "B"],
                        ["Pass", "B"],
                        ["Spare", "i"],
                        ["Sp0", "B"],
                        ["NumP", "B"],
                        ["Sp2", "B"],
                        ["Sp3", "B"]]

_packetDefs[ISP_PLP] = [["Size", "B"],
                        ["Type", "B"],
                        ["ReqI", "B"],
                        ["PLID", "B"]]

_packetDefs[ISP_PLL] = [["Size", "B"],
                        ["Type", "B"],
                        ["ReqI", "B"],
                        ["PLID", "B"]]

_packetDefs[ISP_CRS] = [["Size", "B"],
                        ["Type", "B"],
                        ["ReqI", "B"],
                        ["PLID", "B"]]

_packetDefs[ISP_LAP] = [["Size", "B"],
                        ["Type", "B"],
                        ["ReqI", "B"],
                        ["PLID", "B"],
                        ["LTime", "I"],
                        ["ETime", "I"],
                        ["LapsDone", "H"],
                        ["Flags", "H"],
                        ["Sp0", "B"],
                        ["Penalty", "B"],
                        ["NumStops", "B"],
                        ["Sp3", "B"]]

_packetDefs[ISP_SPX] = [["Size", "B"],
                        ["Type", "B"],
                        ["ReqI", "B"],
                        ["PLID", "B"],
                        ["STime", "I"],
                        ["ETime", "I"],
                        ["Split", "B"],
                        ["Penalty", "B"],
                        ["NumStops", "B"],
                        ["Sp3", "B"]]

_packetDefs[ISP_PIT] = [["Size", "B"],
                        ["Type", "B"],
                        ["ReqI", "B"],
                        ["PLID", "B"],
                        ["LapsDone", "H"],
                        ["Flags", "H"],
                        ["Sp0", "B"],
                        ["Penalty", "B"],
                        ["NumStops", "B"],
                        ["Sp3", "B"],
                        ["Tyres", "4B"],
                        ["Work", "I"],
                        ["Spare", "I"]]

_packetDefs[ISP_PSF] = [["Size", "B"],
                        ["Type", "B"],
                        ["ReqI", "B"],
                        ["PLID", "B"],
                        ["STime", "I"],
                        ["Spare", "I"]]

_packetDefs[ISP_PLA] = [["Size", "B"],
                        ["Type", "B"],
                        ["ReqI", "B"],
                        ["PLID", "B"],
                        ["Fact", "B"],
                        ["Sp1", "B"],
                        ["Sp2", "B"],
                        ["Sp3", "B"]]

_packetDefs[ISP_CCH] = [["Size", "B"],
                        ["Type", "B"],
                        ["ReqI", "B"],
                        ["PLID", "B"],
                        ["Camera", "B"],
                        ["Sp1", "B"],
                        ["Sp2", "B"],
                        ["Sp3", "B"]]

_packetDefs[ISP_PEN] = [["Size", "B"],
                        ["Type", "B"],
                        ["ReqI", "B"],
                        ["PLID", "B"],
                        ["OldPen", "B"],
                        ["NewPen", "B"],
                        ["Reason", "B"],
                        ["Sp3", "B"]]

_packetDefs[ISP_TOC] = [["Size", "B"],
                        ["Type", "B"],
                        ["ReqI", "B"],
                        ["PLID", "B"],
                        ["OldUCID", "B"],
                        ["NewUCID", "B"],
                        ["Sp2", "B"],
                        ["Sp3", "B"]]

_packetDefs[ISP_FLG] = [["Size", "B"],
                        ["Type", "B"],
                        ["ReqI", "B"],
                        ["PLID", "B"],
                        ["OffOn", "B"],
                        ["Flag", "B"],
                        ["CarBehind", "B"],
                        ["Sp3", "B"]]

_packetDefs[ISP_PFL] = [["Size", "B"],
                        ["Type", "B"],
                        ["ReqI", "B"],
                        ["PLID", "B"],
                        ["Flags", "H"],
                        ["Spare", "H"]]

_packetDefs[ISP_FIN] = [["Size", "B"],
                        ["Type", "B"],
                        ["ReqI", "B"],
                        ["PLID", "B"],
                        ["TTime", "I"],
                        ["BTime", "I"],
                        ["SpA", "B"],
                        ["NumStops", "B"],
                        ["Confirm", "B"],
                        ["SpB", "B"],
                        ["LapsDone", "H"],
                        ["Flags", "H"]]

_packetDefs[ISP_RES] = [["Size", "B"],
                        ["Type", "B"],
                        ["ReqI", "B"],
                        ["PLID", "B"],
                        ["UName", "24s"],
                        ["PName", "24s"],
                        ["Plate", "8s"],
                        ["CName", "4s"],
                        ["TTime", "I"],
                        ["BTime", "I"],
                        ["SpA", "B"],
                        ["NumStops", "B"],
                        ["Confirm", "B"],
                        ["SpB", "B"],
                        ["LapsDone", "H"],
                        ["Flags", "H"],
                        ["ResultNum", "B"],
                        ["NumRes", "B"],
                        ["PSeconds", "H"]]

_packetDefs[ISP_REO] = [["Size", "B"],
                        ["Type", "B"],
                        ["ReqI", "B"],
                        ["NumP", "B"],
                        ["PLID", "32B"]]

# Autocross
_packetDefs[ISP_AXI] = [["Size", "B"],
                        ["Type", "B"],
                        ["ReqI", "B"],
                        ["Zero", "B"],
                        ["AXStart", "B"],
                        ["NumCP", "B"],
                        ["NumO", "H"],
                        ["LName", "32s"]]

_packetDefs[ISP_AXO] = [["Size", "B"],
                        ["Type", "B"],
                        ["ReqI", "B"],
                        ["PLID", "B"]]

#struct IS_MCI // Multi Car Info - if more than 8 in race then more than one of these is sent
#{
	#byte	Size;		// 4 + NumP * 28
	#byte	Type;		// ISP_MCI
	#byte	ReqI;		// 0 unless this is a reply to an TINY_MCI request
	#byte	NumC;		// number of valid CompCar structs in this packet

	#CompCar	Info[8];	// car info for each player, 1 to 8 of these (NumC)
#};

#struct CompCar // Car info in 28 bytes - there is an array of these in the MCI (below)
#{
	#word	Node;		// current path node
	#word	Lap;		// current lap
	#byte	PLID;		// player's unique id
	#byte	Position;	// current race position : 0 = unknown, 1 = leader, etc...
	#byte	Info;		// flags and other info - see below
	#byte	Sp3;
	#int		X;			// X map (65536 = 1 metre)
	#int		Y;			// Y map (65536 = 1 metre)
	#int		Z;			// Z alt (65536 = 1 metre)
	#word	Speed;		// speed (32768 = 100 m/s)
	#word	Direction;	// direction of car's motion : 0 = world y direction, 32768 = 180 deg
	#word	Heading;	// direction of forward axis : 0 = world y direction, 32768 = 180 deg
	#short	AngVel;		// signed, rate of change of heading : (16384 = 360 deg/s)
#};

# Car Tracking goes here
_packetDefs[ISP_MCI] = [["Size", "B"],
                        ["Type", "B"],
                        ["ReqI", "B"],
			["NumC", "B"]]

_packetDefs[COMPCAR] = [["Node", "H"],
                        ["Lap", "H"],
                        ["PLID", "B"],
                        ["Position", "B"],
                        ["Info", "B"],
                        ["Sp3", "B"],
                        ["X", "i"],
                        ["Y", "i"],
                        ["Z", "i"],
                        ["Speed", "H"],
                        ["Direction", "H"],
                        ["Heading", "H"],
                        ["AngVel", "h"]]

# Camera Control
_packetDefs[ISP_SCC] = [["Size", "B"],
                        ["Type", "B"],
                        ["ReqI", "B"],
                        ["Zero", "B"],
                        ["ViewPLID", "B"],
                        ["InGameCam", "B"],
                        ["Sp2", "B"],
                        ["Sp3", "B"]]

# ISP_CCP goes here

# Buttons
_packetDefs[ISP_BFN] = [["Size", "B"],
                        ["Type", "B"],
                        ["ReqI", "B"],
                        ["SubT", "B"],
                        ["UCID", "B"],
                        ["ClickID", "B"],
                        ["Inst", "B"],
                        ["Sp3", "B"]]

_packetDefs[ISP_BTN] = [["Size", "B"],
                        ["Type", "B"],
                        ["ReqI", "B"],
                        ["UCID", "B"],
                        ["ClickID", "B"],
                        ["Inst", "B"],
                        ["BStyle", "B"],
                        ["TypeIn", "B"],
                        ["L", "B"],
                        ["T", "B"],
                        ["W", "B"],
                        ["H", "B"],
                        ["Text", "240s"]]

_packetDefs[ISP_BTC] = [["Size", "B"],
                        ["Type", "B"],
                        ["ReqI", "B"],
                        ["UCID", "B"],
                        ["ClickID", "B"],
                        ["Inst", "B"],
                        ["CFlags", "B"],
                        ["Sp3", "B"]]

_packetDefs[ISP_BTT] = [["Size", "B"],
                        ["Type", "B"],
                        ["ReqI", "B"],
                        ["UCID", "B"],
                        ["ClickID", "B"],
                        ["Inst", "B"],
                        ["TypeIn", "B"],
                        ["Sp3", "B"],
                        ["Text", "96s"]]

# Helper functions.
_PacketType = lambda data: struct.unpack("B", data[1:2])[0]
_TinyType = lambda data: struct.unpack("B", data[3:4])[0]
_EatNullChars = lambda str: str.replace("\000", "")
StripColours = lambda str: re.sub(re.compile("\^[0-9]"), "", str)

class _Buffer:
    def __init__(self):
        self.__buffer = ""
        
    def __repr__(self):
        return self.__buffer
    
    def __len__(self):
        return len(self.__buffer)
    
    def Append(self, data):
        self.__buffer = self.__buffer + data
        
    def NextPacketSize(self):
        if len(self.__buffer) > 0:
            return struct.unpack("B", self.__buffer[0:1])[0]
        return 0
    
    def NextPacket(self):
        packet = None
        size = self.NextPacketSize()
        if len(self.__buffer) > 0 and len(self.__buffer) >= size:
            packet = self.__buffer[:size]
            self.__buffer = self.__buffer[size:] 
        return packet

class InSim:
    BufferSize = 256
    
    def __init__(self, connType=INSIM_TCP):
        self.__socket = socket.socket(socket.AF_INET, connType)        
        self.__callbacks = {}
        self.__buffer = _Buffer()

    def Connect(self, ip="127.0.0.1", port=29999):
        self.__socket.connect((ip, port))
        self.Connected = True
        thread.start_new_thread(self.__ReceiveThread, ())

    def Close(self):
        self.__socket.close()
        self.Connected = False

    def Send(self, packet):
        self.__socket.send(packet)              

    def __ReceiveThread(self, *args):
        while self.Connected:
            data = self.__socket.recv(self.BufferSize)
            if len(data) == 0:
                self.Close()
            else:
                self.__buffer.Append(data)              
                while True:
                    packet = self.__buffer.NextPacket()
                    if packet == None:
                        break;
                    else:
                        self.__KeepAlive(packet)
                        self.__RaisePacketEvent(packet)
                        
    def __RaisePacketEvent(self, data):
        key = _PacketType(data)
        if self.__callbacks.has_key(key):
	    if key == ISP_MCI:
		packet = _MciPacket(data)
	    else:
		packet = Packet(data=data)
            self.__callbacks[key](packet)
        
    def __KeepAlive(self, data):
        if _PacketType(data) == ISP_TINY:
            if _TinyType(data) == TINY_NONE:
                self.Send(data)
                    
    def RegisterPackets(self, callbacks):
        for k, v in callbacks.iteritems():
            self.__callbacks[k] = v  
            
    def UnregisterPackets(self, packets):
        for packet in packets:
            del self.__callbacks[packet]      
            
class Packet:
    def __init__(self, packetType=ISP_NONE, data=None):
        self.__values = {}
        self.__packStr = ""   
        values = []    
        
        if data == None:
            self.__packetType = packetType
        else:
            self.__packetType = _PacketType(data) 
            
        for packetDef in _packetDefs[self.__packetType]:
            self.__packStr = self.__packStr + packetDef[1]
            
        if data == None:
            for packetDef in _packetDefs[self.__packetType]:
                values.append(_typeDefaults[packetDef[1][-1:]])
        else:
            values = struct.unpack(self.__packStr, data)
            
	for i in range(len(_packetDefs[self.__packetType])):
	    if data != None and _packetDefs[self.__packetType][i][1][-1:] == "s":
		self.__values[_packetDefs[self.__packetType][i][0]] = _EatNullChars(values[i])
	    else:
		self.__values[_packetDefs[self.__packetType][i][0]] = values[i]
	    
        self.__values["Size"] = struct.calcsize(self.__packStr)
        self.__values["Type"] = self.__packetType       
 	
    def __getitem__(self, key):
        return self.__values[key]
    
    def __setitem__(self, key, value):
        self.__values[key] = value
        
    def __repr__(self):
        data = []
        for packetDef in _packetDefs[self.__packetType]:
            data.append(self.__values[packetDef[0]])
        return str(data)
    
    def __cmp__(self, packet):
        return cmp(packet["Type"], self.__values["Type"])           
    
    def __len__(self):
        return self.__values["Size"]
    
    def Pack(self):
        values = []
        for packetDef in _packetDefs[self.__packetType]:
            values.append(self.__values[packetDef[0]])
        return struct.pack(self.__packStr, *values)      
  
_packetDefs[ISP_MCI] = [["Size", "B"],
                        ["Type", "B"],
                        ["ReqI", "B"],
			["NumC", "B"]]

_packetDefs[COMPCAR] = [["Node", "H"],
                        ["Lap", "H"],
                        ["PLID", "B"],
                        ["Position", "B"],
                        ["Info", "B"],
                        ["Sp3", "B"],
                        ["X", "i"],
                        ["Y", "i"],
                        ["Z", "i"],
                        ["Speed", "H"],
                        ["Direction", "H"],
                        ["Heading", "H"],
                        ["AngVel", "h"]]    
    
class _MciPacket():
    def __init__(self, data):
	self.__values = {}
	mciPack = ""
	for packetDef in _packetDefs[ISP_MCI]:
	    mciPack = mciPack + packetDef[1]
	mciSize = struct.calcsize(mciPack)
	mciValues = struct.unpack(mciPack, data[:mciSize])
	for i in range(len(_packetDefs[ISP_MCI])):
	    self.__values[_packetDefs[ISP_MCI][i][0]] = mciValues[i]	
	    
	ccPack = ""
	for packetDef in _packetDefs[COMPCAR]:
	    ccPack = ccPack + packetDef[1]
	ccSize = struct.calcsize(ccPack)
	offset = mciSize
	self.__values["CompCars"] = []
	for i in range(self.__values["NumC"]):
	    ccValues = struct.unpack(ccPack, data[offset:ccSize + offset])
	    compcar = {}
	    for i in range(len(_packetDefs[COMPCAR])):
		compcar[_packetDefs[COMPCAR][i][0]] = ccValues[i]  
	    self.__values["CompCars"].append(compcar)
	    offset = offset + ccSize
	
    def __getitem__(self, key):
	return self.__values[key]
	    
    def __len__(self):
	return self.__values["NumC"]
	    
    def __repr__(self):
	return str(self.__values)
	    
if __name__ == "__main__": 
    pack = "BBBBHHBBBBiiiHHHh"
    str = [4, ISP_MCI, 0, 1, 
	   0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12]
    data = struct.pack(pack, *str)
    print len(data[4:28 + 4])
    mci = _MciPacket(data)
    
    #def CarPositionUpdate(mci):
	#print mci["NumC"]
    
    #insim = InSim()
    #insim.RegisterPackets({ISP_MCI: CarPositionUpdate})    
    
    #insim.Connect("127.0.0.1", 29999)
    
    #isi = Packet(ISP_ISI)
    #isi["Admin"] = "Pass"
    #isi["IName"] = "UnitTest"
    #isi["Interval"] = 1000
    #isi["Flags"] = isi["Flags"] | ISF_LOCAL
    #isi["Flags"] = isi["Flags"] | ISF_MCI
    #insim.Send(isi.Pack())
    
    #while insim.Connected:
	#import time
	#time.sleep(1)
