import ConfigParser
import urllib
import time
import Pyinsim
import struct

# Constants
CACHE_FILE = "Fuel.txt"
CONFIG_FILE = "Config.ini"
PROGRAM_NAME = "FuelCalculator"
PROGRAM_VERSION = 0.1

# Class to manage querying and caching LFSWorld data
class FuelCalculator:
    def __init__(self, pubStatUrl, identKey, username, cacheSeconds):
        self.__pubStatUrl = pubStatUrl
        self.__identKey = identKey
        self.__username = username
        self.__lastCache = 0
        self.__cacheSeconds = cacheSeconds

    def __CreatePubStatUrl(self):
        return self.__pubStatUrl + "?version=1.4&idk=" + \
               self.__identKey + "&action=fuel&racer=" + self.__username
        
    def __CacheFuelUseage(self):
        try:
            self.__lastCache = time.time()
            url = self.__CreatePubStatUrl()
            file = open(CACHE_FILE, "w")
            for line in urllib.urlopen(url).readlines():
                file.write(line)
        except IOError, (exMsg):
            print "Error caching fuel:", exMsg
        finally:
            file.close()        
    
    def CacheFuelUseage(self):
        if self.__cacheSeconds == 0:
            self.__CacheFuelUseage()
        else:
            timestamp = time.time()
            if (timestamp - self.__lastCache) > self.__cacheSeconds:
                self.__CacheFuelUseage()
                self.__lastCache = timestamp     
        
    def CalculateFuelUseage(self, track, car, laps):
        try:
            file = open(CACHE_FILE, "r")
            for line in file.readlines():
                tokens = line.split()
                if len(tokens) == 3:
                    if tokens[0] == track and tokens[1] == car:
                        return (laps * float(tokens[2]), float(tokens[2]))       
        except IOError, (exMsg):
            print "Error reading cache:", exMsg
        finally:
            file.close()   

# Class to manage connections list and a few other user-related things.
class Connections:
    def __init__(self):
        self.__connections = {}
        self.__players = {}    
        
    def ConnectionJoined(self, ncn):
        self.__connections[ncn["UCID"]] = ncn
        print ncn
    
    def ConnectionLeft(self, cnl):
        del self.__connections[cnl["UCID"]]
        
    def PlayerJoined(self, npl):
        self.__players[npl["PLID"]] = npl
        print npl
        
    def PlayerLeft(self, pll):
        del self.__players[pll["PLID"]]
        
    def GetPlayersUsername(self, plid):
        npl = self.__players[plid]
        ncn = self.__connections[npl["UCID"]]
        return ncn["UName"]
    
    def GetPlayersCar(self, plid):
        return self.__players[plid]["CName"]    
    
    def CheckValidUser(self, plid, username):
        npl = self.__players[plid]
        ncn = self.__connections[npl["UCID"]]
        if ncn["UName"] == username:
            return True
        return False
    
    def GetPlayer(self, plid):
        return self.__players[plid]
    
    def GetConnection(self, ucid):
        return self.__connections[ucid]

# Load config.
config = ConfigParser.ConfigParser()
config.read(CONFIG_FILE)
try:
    hostAddress = config.get("InSim", "HostAddress")
    insimPort = int(config.get("InSim", "InSimPort"))
    adminPass = config.get("InSim", "AdminPass")
    prefix = config.get("InSim", "Prefix")
    username = config.get("LFSWorld", "UserName")
    pubStatUrl = config.get("LFSWorld", "PubStatUrl")
    identKey = config.get("LFSWorld", "IdentKey")
    cacheSeconds = int(config.get("LFSWorld", "CacheSeconds"))
except Exception, (exMsg):
    print "Config Error:", exMsg
    exit()    
    
# Init globals
insim = Pyinsim.InSim()
fuel = FuelCalculator(pubStatUrl, identKey, username, cacheSeconds)
conns = Connections()
currentTrack = None
isMultiplayer = False
tracks = {"BL1" : "000",
          "BL1R": "001",
          "BL2" : "010",
          "BL2R": "011",
          "BL3" : "020",
          "BL3R": "021",
          "SO1" : "100",
          "SO1R": "101",
          "SO2" : "110",
          "SO2R": "111",
          "SO3" : "120",
          "SO3R": "121",
          "SO4" : "130",
          "SO4R": "131",    
          "SO5" : "140",
          "SO5R": "141", 
          "SO6" : "150",
          "SO6R": "151",            
          "SO7" : "160",
          "SO7R": "161",
          "FE1" : "200",
          "FE1R": "201",
          "FE2" : "210",
          "FE2R": "211",
          "FE3" : "220",
          "FE3R": "221",
          "FE4" : "230",
          "FE4R": "231",
          "FE5" : "240",
          "FE5R": "241",
          "FE6" : "250",
          "FE6R": "251",
          "AU1" : "300",
          "AU2" : "310",
          "AU3" : "320",
          "AU4" : "330",
          "KY1" : "400",
          "KY1R": "401",
          "KY2" : "410",
          "KY2R": "411",
          "KY3" : "420",
          "KY3R": "421",
          "WE1" : "500",
          "WE1R": "501",
          "AS1" : "600",
          "AS1" : "601",
          "AS2" : "610",
          "AS2" : "611",
          "AS3" : "620",
          "AS3" : "621",
          "AS4" : "630",
          "AS4" : "631",
          "AS5" : "640",
          "AS5" : "641",
          "AS6" : "650",
          "AS6" : "651",
          "AS7" : "660",
          "AS7" : "661"}

# Functions
def RequestInfo():
    tiny = Pyinsim.Packet(Pyinsim.ISP_TINY)
    tiny["ReqI"] = 1
    tiny["SubT"] = Pyinsim.TINY_NCN
    insim.Send(tiny.Pack())
    tiny["SubT"] = Pyinsim.TINY_NPL
    insim.Send(tiny.Pack())  
    tiny["SubT"] = Pyinsim.TINY_SST
    insim.Send(tiny.Pack())    

def SendMessageConnection(msg, ucid=0, plid=0):
    mst = Pyinsim.Packet(Pyinsim.ISP_MST)
    mst["UCID"] = ucid
    mst["PLID"] = plid
    mst["Msg"] = msg
    insim.Send(mst.Pack())

def SendMessageLocal(msg, sound=Pyinsim.SND_SILENT):
    msl = Pyinsim.Packet(Pyinsim.ISP_MSL)
    msl["Sound"] = sound
    msl["Msg"] = msg
    insim.Send(msl.Pack())

def RequestFuelUseage(laps, id):
    car = conns.GetPlayersCar(id)    
    track = tracks[currentTrack]
    fuel.CacheFuelUseage()
    avgFuel = fuel.CalculateFuelUseage(track, car, laps)
    msg = ""
    if avgFuel == None:
        msg = "^3Fuel Est: No data"
    else:
        msg = "^3Fuel Est: " + str(laps) + " * " + str(avgFuel[1]) + " = " + str(avgFuel[0]) + "%"
    SendMessageLocal(msg)
            
# Events
def CheckVersion(var):
    if var["InSimVer"] != Pyinsim.INSIM_VERSION:
        print "Invalid version!"
        insim.Close()
        exit()

def StateChanged(sta):
    global currentTrack
    global isMultiplayer
    if sta["Track"] != currentTrack:
        currentTrack = sta["Track"]
    if (sta["Flags"] & Pyinsim.ISS_MULTI) == Pyinsim.ISS_MULTI:
        isMultiplayer = True
    else:
        isMultiplayer = False

def MessageReceived(mso):
    if isMultiplayer:
        if mso["PLID"] != 0 and mso["UCID"] != 0:
            ply = conns.GetPlayer(mso["PLID"])
            conn = conns.GetConnection(ply["UCID"])
            if conn["UName"] == username:    
                msg = mso["Msg"][mso["TextStart"]:]
                tokens = msg.split()
                if len(tokens) == 2:
                    if tokens[0] == (prefix + "fuel"):
                        if conns.CheckValidUser(mso["PLID"], username) == True:
                            RequestFuelUseage(int(tokens[1]), mso["PLID"])
    else:
        msg = mso["Msg"][mso["TextStart"]:]
        tokens = msg.split()
        if len(tokens) == 2:
            if tokens[0] == (prefix + "fuel"):
                RequestFuelUseage(int(tokens[1]), mso["PLID"])   
                
def MessageReceivedEx(mso):
    if mso["PLID"] == 0:
        return 
    if isMultiplayer == True:
        ply = conns.GetPlayer(mso["PLID"])
        conn = conns.GetConnection(ply["UCID"])
        if conn["UName"] != username:
            return
    msg = mso["Msg"][mso["TextStart"]:]
    tokens = msg.split()
    if len(tokens) == 2:
        if tokens[0] == (prefix + "fuel"):
            RequestFuelUseage(int(tokens[1]), mso["PLID"])
    
def MultiplayerJoined(ism):
    RequestInfo()
    print "join"
            
insim.RegisterPackets({Pyinsim.ISP_VER: CheckVersion, 
                       Pyinsim.ISP_NCN: conns.ConnectionJoined,
                       Pyinsim.ISP_CNL: conns.ConnectionLeft,
                       Pyinsim.ISP_NPL: conns.PlayerJoined,
                       Pyinsim.ISP_PLL: conns.PlayerLeft,
                       Pyinsim.ISP_STA: StateChanged,
                       Pyinsim.ISP_MSO: MessageReceivedEx,
                       Pyinsim.ISP_ISM: MultiplayerJoined})

# Initailise InSim
print PROGRAM_NAME, PROGRAM_VERSION
try:
    insim.Connect(hostAddress, insimPort)
    isi = Pyinsim.Packet(Pyinsim.ISP_ISI)
    isi["ReqI"] = 1
    isi["Admin"] = adminPass
    isi["IName"] = PROGRAM_NAME
    isi["Prefix"] = ord(prefix)
    insim.Send(isi.Pack())
    RequestInfo()
except:
    print "Connection failed"
else:
    print "InSim connected"
    while insim.Connected:
        time.sleep(1)
    print "Bye!"
