#include "cpresets.h"
#include <wx/config.h>

// the separator on a button between the shortcut and the name
#define LABEL_SEPARATOR ':'

// special characters in a preset name (must be escaped when storing)
#define SPECIAL_CHARS _T("|=;")

// the name of the default preset
#define DEFAULT_PRESET_NAME _T("Default")

// the ID of the button for the default preset
#define ID_DEFAULT_BUTTON 0

//-----------------------------------------------------------------------------
// Event table

BEGIN_EVENT_TABLE(cPresets, cPane)
  EVT_BUTTON(wxID_ANY, cPresets::OnButton)
END_EVENT_TABLE()

//-----------------------------------------------------------------------------

cPresets::cPresets(wxWindow* parent)
: cPane(parent)
{
  m_TopSizer = new wxBoxSizer(wxHORIZONTAL);

  // create dummy button, to calculate minimum size
  wxButton* button = new wxButton(this, wxID_ANY, _T("Dummy"));
  SetMinSize(wxSize(100, button->GetSize().GetHeight() + 2 * BASE_MARGIN));
  button->Destroy();

  SetSizer(m_TopSizer);
}

//-----------------------------------------------------------------------------

cPresets::~cPresets()
{
}

//-----------------------------------------------------------------------------

const wxString cPresets::GetPresetName(size_t index) const
{
  wxString name = ::UnEscape(m_Names[index], SPECIAL_CHARS);
  return name;
}


int cPresets::Find(const wxString& name)
{
  return m_Names.Index(::Escape(name, SPECIAL_CHARS));
}

//-----------------------------------------------------------------------------
// Add a preset
// - name = preset name
// - panes, graph = settings for the panes and for the graphs

size_t cPresets::AddPreset(const wxString& name, const wxString& panes, const wxString& graphs, int id)
{
  wxASSERT(!name.IsEmpty());
  m_Names.Add(::Escape(name, SPECIAL_CHARS));
  m_Panes.Add(panes);
  m_Graphs.Add(graphs);

  // determine shortcut (= highest existing + 1)
  int shortcut = -1;
  for (size_t i = 0; i < m_Shortcuts.GetCount(); i++) {
    if (shortcut < m_Shortcuts[i]) shortcut = m_Shortcuts[i];
  }
  shortcut += 1;
  if (shortcut > 9) shortcut = -1;
  m_Shortcuts.Add(shortcut);

  // add button to window
  wxString label;
  if (shortcut >=0) {
    label.Printf(_T("%d%c%s"), shortcut, LABEL_SEPARATOR, name.c_str());
  }
  else {
    label = name;
  }

  wxButton* button = new wxButton(this, id, label, wxDefaultPosition, wxDefaultSize, wxBU_EXACTFIT);
  m_TopSizer->Add(button, wxSizerFlags(1).Proportion(0).Left().Border(wxTOP | wxBOTTOM | wxLEFT, BASE_MARGIN));

  Layout();
  Refresh();

  return shortcut;
}

//-----------------------------------------------------------------------------
// Add the default preset

size_t cPresets::AddDefault(const wxString& panes, const wxString& graphs)
{
  wxASSERT(m_Shortcuts.GetCount() == 0); // must be the first preset
  return AddPreset(_TT(ID_TXT_PRESET_DEFAULT, DEFAULT_PRESET_NAME), panes, graphs, ID_DEFAULT_BUTTON);
}

//-----------------------------------------------------------------------------
// Replace the contents of a preset
// - index = index of the preset
// - panes, graph = settings for the panes and for the graphs

void cPresets::ReplacePreset(size_t index, const wxString& panes, const wxString& graphs)
{
  m_Panes[index] = panes;
  m_Graphs[index] = graphs;
}

//-----------------------------------------------------------------------------
// Delete a preset
// - index = index of the preset

void cPresets::DeletePreset(size_t index)
{
  m_Names.RemoveAt(index);
  m_Panes.RemoveAt(index);
  m_Graphs.RemoveAt(index);
  m_Shortcuts.RemoveAt(index);

  wxSizerItem* item = m_TopSizer->GetItem(index);
  wxASSERT(item != NULL);
  wxButton* button = (wxButton*)item->GetWindow();
  wxASSERT(button != NULL);

  m_TopSizer->Remove(index);
  button->Destroy();
  Layout();
  Refresh();
}

//-----------------------------------------------------------------------------

void cPresets::OnButton(wxCommandEvent& event)
{
  wxButton* button = (wxButton*)wxWindow::FindWindowById(event.GetId(), this);
  wxASSERT(button != NULL);

  // get name of preset
  wxString name = button->GetLabel();
  if ((name.Length() > 2) && (name[1] == LABEL_SEPARATOR)) name = name.Mid(2);

  // send event to select the preset
  ::Preset_Send(1, name);
}

//-----------------------------------------------------------------------------
// Loading and saving the configuration settings

void cPresets::LoadConfig(wxRegConfig* config, const wxString& key)
{
  int p = 0;
  while (true) {
    wxString subkey;
    subkey.Printf(_T("%s/preset%d"), key.c_str(), p);
    wxString name, panes, graphs;
    if (!(config->Read(subkey + _T("/name"), &name))) break;
    if (!(config->Read(subkey + _T("/panes"), &panes))) break;
    if (!(config->Read(subkey + _T("/graphs"), &graphs))) break;
    AddPreset(name, panes, graphs);
    p += 1;
  }
}


void cPresets::SaveConfig(wxRegConfig* config, const wxString& key)
{
  // save presets (omit the default preset)
  config->DeleteGroup(key);
  for (size_t p = 1; p < m_Names.GetCount(); p++) {
    wxString subkey;
    subkey.Printf(_T("%s/preset%d"), key.c_str(), p - 1); // numbering starts from 0, for backward compatibility
    config->Write(subkey + _T("/name"), m_Names[p]);
    config->Write(subkey + _T("/panes"), m_Panes[p]);
    config->Write(subkey + _T("/graphs"), m_Graphs[p]);
  }
}

//-----------------------------------------------------------------------------

void cPresets::TranslateTexts()
{
  // translate name of default preset
  if (m_Shortcuts.GetCount() == 0) return;
  m_Names[0] = _TT(ID_TXT_PRESET_DEFAULT, DEFAULT_PRESET_NAME);

  // set new name in button
  wxButton* button = (wxButton*)(wxWindow::FindWindowById(ID_DEFAULT_BUTTON, this));
  button->SetLabel(wxString::Format(_T("0%c%s"), LABEL_SEPARATOR, m_Names[0].c_str()));

  Layout(); // button size may change because of translation
}
