#include "cmainframe.h"
#include "capp.h"
#include "cgraphview.h"
#include "clegend.h"
#include "cproperties.h"
#include "cdrivingline.h"
#include "ctrackmap.h"
#include "cmgr.h"
#include "cfiledlg.h"
#include "clog.h"
#include "cpresets.h"
#include <wx/msgdlg.h>
#include <wx/filedlg.h>
#include <wx/menu.h>
#include <wx/config.h>
#include <wx/statusbr.h>
#include <wx/textdlg.h>
#include <wx/choicdlg.h>
#include <wx/utils.h>
#include <wx/aboutdlg.h>

// the time (in ms) spent loading (the next part of) a RAF file, before returning to the event loop
#define FILELOAD_TIMESLICE 10

//-----------------------------------------------------------------------------
// Event table

BEGIN_EVENT_TABLE(cMainFrame, wxFrame)

  // misc events
  EVT_CLOSE(cMainFrame::OnClose)
  EVT_IDLE(cMainFrame::OnIdle)
  EVT_MOVE(cMainFrame::OnMove)

  // menu events
  EVT_MENU_OPEN(cMainFrame::OnMenu)
  EVT_MENU(ID_MENU_OPEN, cMainFrame::OnOpen)
  EVT_MENU(ID_MENU_ADD, cMainFrame::OnOpen)
  EVT_MENU_RANGE(wxID_FILE1, wxID_FILE9, cMainFrame::OnOpenRecent)
  EVT_MENU(ID_MENU_CLOSEALL, cMainFrame::OnCloseAll)
  EVT_MENU(ID_MENU_EXIT, cMainFrame::OnExit)
  EVT_MENU(ID_MENU_ADDGRAPH_Y, cMainFrame::OnAddGraph)
  EVT_MENU(ID_MENU_ADDGRAPH_XY, cMainFrame::OnAddGraph)
  EVT_MENU(ID_MENU_ADDGRAPH_H, cMainFrame::OnAddGraph)
  EVT_MENU(ID_MENU_ZOOM_RESET_ALL, cMainFrame::OnZoom)
  EVT_MENU(ID_MENU_ZOOM_FIT, cMainFrame::OnZoom)
  EVT_MENU_RANGE(ID_MENU_VIEWFLAG_FIRST, ID_MENU_VIEWFLAG_LAST, cMainFrame::OnChangeView)
  EVT_MENU(ID_MENU_HIDEALLPANES, cMainFrame::OnChangeView)
  EVT_MENU(ID_MENU_ADD_PRESET, cMainFrame::OnAddPreset)
  EVT_MENU(ID_MENU_DELETE_PRESET, cMainFrame::OnDeletePreset)
  EVT_MENU(ID_MENU_PRESET_TOOLBAR, cMainFrame::OnChangeView)
  EVT_MENU_RANGE(ID_MENU_PRESET_FIRST, ID_MENU_PRESET_LAST, cMainFrame::OnSelectPresetFromMenu)
  EVT_MENU_RANGE(ID_MENU_OPTIONS_FIRST, ID_MENU_OPTIONS_LAST, cMainFrame::OnSetOption)
  EVT_MENU_RANGE(ID_MENU_HELP_FIRST, ID_MENU_HELP_LAST, cMainFrame::OnHelp)
  EVT_MENU_RANGE(ID_MENU_LANGUAGE_FIRST, ID_MENU_LANGUAGE_LAST, cMainFrame::OnSelectLanguage)

  // custom events
  EVT_COMMAND(wxID_ANY, EVT_LRA_CLOSEVIEW, cMainFrame::OnCloseView)
  EVT_COMMAND(wxID_ANY, EVT_LRA_DELETELAP, cMainFrame::OnDeleteLap)
  EVT_COMMAND(wxID_ANY, EVT_LRA_EQUALHEIGHTGRAPHS, cMainFrame::OnEqualHeightGraphs)
  EVT_COMMAND(wxID_ANY, EVT_LRA_LAYOUTFRAME, cMainFrame::OnLayoutMainFrame)
  EVT_COMMAND(wxID_ANY, EVT_LRA_MOVEGRAPH, cMainFrame::OnMoveGraph)
  EVT_COMMAND(wxID_ANY, EVT_LRA_OPENFILE, cMainFrame::OnOpenFile)
  EVT_COMMAND(wxID_ANY, EVT_LRA_PRESET, cMainFrame::OnPresetEvent)
  EVT_COMMAND(wxID_ANY, EVT_LRA_SETTRACKCURSORPOS, cMainFrame::OnSetTrackCursorPos)
  EVT_COMMAND(wxID_ANY, EVT_LRA_SETTRACKSELECTION, cMainFrame::OnSetTrackSelection)
  EVT_COMMAND(wxID_ANY, EVT_LRA_SHOWLAP, cMainFrame::OnShowLap)
  EVT_COMMAND(wxID_ANY, EVT_LRA_SYNCGRAPHS, cMainFrame::OnSyncGraphs)

END_EVENT_TABLE()

//-----------------------------------------------------------------------------
// title = frame caption
// pos = screen position
// size = window size
// infile = RAF file to load at startup

cMainFrame::cMainFrame(const wxString& title, const wxPoint& pos, const wxSize& size, const wxString& infile)
  : wxFrame((wxFrame *)NULL, -1, title, pos, size)
{
  m_Position = pos;
  m_CursorDist = -1.0f;
  m_SelStart = 0.0f;
  m_SelEnd = 1.0f;
  m_AllPanesHidden = false;

  // set the frame icon
  SetIcon(wxICON(lra));

  // load the selected language
  cLang::SetFolder(APP->GetAppFolder() + wxFILE_SEP_PATH + LANGUAGE_FOLDER);
  wxString lang;
  CONFIG->Read(_T("/options/language"), &lang);
  cLang::SetLanguage(lang);

  // ----- panes and AUI setup -----

  m_Legend = new cLegend(this);
  m_Panes.Add(m_Legend);
  m_Properties = new cProperties(this);
  m_Panes.Add(m_Properties);
  m_TrackMap = new cTrackMap(this);
  m_Panes.Add(m_TrackMap);
  m_DrivingLine = new cDrivingLine(this);
  m_Panes.Add(m_DrivingLine);
  m_Presets = new cPresets(this);
  m_Panes.Add(m_Presets);
  m_Log = new cLog(this);
  m_Panes.Add(m_Log);

  m_Graphs = new cGraphContainer(this,
      m_AuiMgr.GetArtProvider()->GetColour(wxAUI_DOCKART_BORDER_COLOUR),
      m_AuiMgr.GetArtProvider()->GetColour(wxAUI_DOCKART_BACKGROUND_COLOUR));

  // add panes to AUI  manager
  m_AuiMgr.SetManagedWindow(this);
  m_AuiMgr.AddPane(m_Graphs, wxCENTER);
  m_AuiMgr.AddPane(m_TrackMap, wxRIGHT);
  m_AuiMgr.AddPane(m_DrivingLine, wxRIGHT);
  m_AuiMgr.AddPane(m_Legend, wxRIGHT);
  m_AuiMgr.AddPane(m_Properties, wxBOTTOM);
  m_AuiMgr.AddPane(m_Presets,
      wxAuiPaneInfo().ToolbarPane().Top().PaneBorder(false).Floatable(false).LeftDockable(false).RightDockable(false));
  m_AuiMgr.AddPane(m_Log, wxBOTTOM);

  // set names of panes, so they will be restored when loading the perspective
  m_AuiMgr.GetPane(m_Graphs).Name(_T("graphs"));
  m_AuiMgr.GetPane(m_Legend).Name(_T("legend"));
  m_AuiMgr.GetPane(m_TrackMap).Name(_T("trackmap"));
  m_AuiMgr.GetPane(m_DrivingLine).Name(_T("drivingline"));
  m_AuiMgr.GetPane(m_Properties).Name(_T("properties"));
  m_AuiMgr.GetPane(m_Presets).Name(_T("presets"));
  m_AuiMgr.GetPane(m_Log).Name(_T("log"));

  // some panes are initially NOT shown
  ShowPane(m_DrivingLine, false);
  ShowPane(m_Properties, false);
  ShowPane(m_Presets, false);
  ShowPane(m_Log, false);

  // styling
  m_AuiMgr.GetArtProvider()->SetMetric(wxAUI_DOCKART_GRADIENT_TYPE, wxAUI_GRADIENT_NONE);

  // ----- menus -----

  // load file history from config
  for (int i = 9; i >= 0; i--) {
    // (add item #0 last, so it will be at the top of the menu)
    wxString key;
    key.Printf(_T("/locations/file_history/file%d"), i);
    wxString file;
    if (CONFIG->Read(key, &file)) m_FileHistory.AddFileToHistory(file);
  }

  // File history submenu
  m_FileHistoryMenu = new wxMenu;
  m_FileHistory.UseMenu(m_FileHistoryMenu);
  m_FileHistory.AddFilesToMenu();

  // 'File' menu
  wxMenu* menuFile = new wxMenu;
  cLang::AppendMenuItem(menuFile, ID_MENU_OPEN, _T("&Open...\tCtrl+O"), _T("Open new RAF file(s)"));
  cLang::AppendMenuItem(menuFile, ID_MENU_ADD, _T("&Add...\tCtrl+A"), _T("Open additional RAF file(s)"));
  cLang::AppendMenuItem(menuFile, ID_MENU_OPENRECENT, _T("Add &recent"), m_FileHistoryMenu);
  cLang::AppendMenuItem(menuFile, ID_MENU_CLOSEALL, _T("&Close all\tCtrl+W"), _T("Close all files"));
  menuFile->AppendSeparator();
  cLang::AppendMenuItem(menuFile, ID_MENU_EXIT, _T("E&xit\tCtrl+Q"), _T("Exit " APPLICATION_NAME));

  // 'Panes' submenu
  wxMenu* menuPanes = new wxMenu;
  cLang::AppendCheckItem(menuPanes, ID_MENU_LEGEND, _T("&Legend\tF5"),
      _T("Show or hide the legend"));
  cLang::AppendCheckItem(menuPanes, ID_MENU_TRACKMAP, _T("&Track map\tF6"),
      _T("Show or hide the track map"));
  cLang::AppendCheckItem(menuPanes, ID_MENU_DRIVINGLINE, _T("&Driving line\tF7"),
      _T("Show or hide the driving line display"));
  cLang::AppendCheckItem(menuPanes, ID_MENU_PROPERTIES, _T("&Properties\tF8"),
      _T("Show or hide the properties pane"));
  cLang::AppendCheckItem(menuPanes, ID_MENU_LOG, _T("&Messages\tShift+F8"),
      _T("Show or hide the messages log"));
  menuPanes->AppendSeparator();
  cLang::AppendMenuItem(menuPanes, ID_MENU_HIDEALLPANES, _T("Hide all\tF2"),
      _T("Show or hide all panes and toolbars"));

  // 'Zoom' submenu
  wxMenu* menuZoom = new wxMenu;
  cLang::AppendMenuItem(menuZoom, ID_MENU_ZOOM_RESET_ALL, _T("Reset\tCtrl+Z"), _T("Reset the zoom in all graphs"));
  cLang::AppendMenuItem(menuZoom, ID_MENU_ZOOM_FIT, _T("Fit\tCtrl+F"));

  // 'View' menu
  wxMenu* menuView = new wxMenu;
  cLang::AppendMenuItem(menuView, ID_MENU_ADDGRAPH_Y, _T("Add &graph\tF4"),
      _T("Add a normal graph"));
  cLang::AppendMenuItem(menuView, ID_MENU_ADDGRAPH_XY, _T("Add &xy plot\tShift+F4"),
      _T("Add an XY plot"));
  cLang::AppendMenuItem(menuView, ID_MENU_ADDGRAPH_H, _T("Add &histogram"),
      _T("Add a histogram"));
  cLang::AppendMenuItem(menuView, ID_MENU_PANES, _T("&Panes"), menuPanes);
  menuView->AppendSeparator();
  cLang::AppendCheckItem(menuView, ID_MENU_CROSSHAIR, _T("&Crosshair\tF9"),
      _T("Draw a crosshair in graphs"));
  cLang::AppendMenuItem(menuView, ID_MENU_ZOOM, _T("&Zoom"), menuZoom);
  menuView->AppendSeparator();
  cLang::AppendCheckItem(menuView, ID_MENU_FULLSCREEN, _T("&Full screen\tF11"),
      _T("Resize the window to full screen, or vice versa"));

  // 'Presets' menu
  m_MenuPreset = new wxMenu;
  cLang::AppendMenuItem(m_MenuPreset, ID_MENU_ADD_PRESET, _T("&Save view as preset..."),
      _T("Save the current graph and pane settings as a preset"));
  cLang::AppendMenuItem(m_MenuPreset, ID_MENU_DELETE_PRESET, _T("&Delete preset..."),
      _T("Delete an existing preset"));
  cLang::AppendCheckItem(m_MenuPreset, ID_MENU_PRESET_TOOLBAR, _T("Show &toolbar"),
      _T("Show or hide the presets toolbar"));
  m_MenuPreset->AppendSeparator();

  // 'Units' submenu
  wxMenu* menuUnits = new wxMenu;
  cLang::AppendCheckItem(menuUnits, ID_MENU_SPEED_KMH, _T("&km/h"), _T("Display speeds in km/h"));
  cLang::AppendCheckItem(menuUnits, ID_MENU_SPEED_MPH, _T("&mph"), _T("Display speeds in mph"));

  // 'Language' submenu
  m_MenuLang = new wxMenu;
  wxArrayString langList;
  cLang::GetLanguageList(langList);
  for (size_t i = 0; i < langList.GetCount(); i++) {
    cLang::AppendCheckItem(m_MenuLang, ID_MENU_LANGUAGE_FIRST + i, langList[i]);
  }

  // 'Options' menu
  wxMenu* menuOptions = new wxMenu;
  cLang::AppendMenuItem(menuOptions, ID_MENU_UNITS, _T("&Units"), menuUnits);
  menuOptions->AppendSeparator();
  cLang::AppendCheckItem(menuOptions, ID_MENU_XRULERS, _T("&X rulers"),
      _T("Show or hide the rulers for the X axis"));
  cLang::AppendCheckItem(menuOptions, ID_MENU_TOOLTIPS, _T("&Tooltips\tShift+F1"),
      _T("Show tooltips on the graphs"));
  cLang::AppendCheckItem(menuOptions, ID_MENU_STATUSBAR, _T("&Status bar"),
      _T("Show or hide the status bar"));
  menuOptions->AppendSeparator();
  cLang::AppendCheckItem(menuOptions, ID_MENU_SESSION, _T("&Restore session"),
      _T("At startup, re-load the graphs and files of the previous session"));
  cLang::AppendCheckItem(menuOptions, ID_MENU_CUSTOM_FILE_DLG, _T("Custom &file dialog"),
      _T("Use the custom dialog for selecting RAF files, instead of the standard one"));
  cLang::AppendMenuItem(menuOptions, ID_MENU_LANGUAGE, _T("&Language"), m_MenuLang);

  // 'Help' menu
  wxMenu* menuHelp = new wxMenu;
  cLang::AppendMenuItem(menuHelp, ID_MENU_MANUAL, _T("Manual\tF1"), _T("View the LRA manual"));
  m_ManualFile = APP->GetAppFolder() + wxFILE_SEP_PATH + LRA_MANUAL;
  menuHelp->Enable(ID_MENU_MANUAL, wxFileExists(m_ManualFile));
  cLang::AppendMenuItem(menuHelp, ID_MENU_WEBSITE, _T("Visit LRA &web page"),
      _T("Go to the forum pages for LRA, to check for updates or to give feedback"));
  menuHelp->AppendSeparator();
  cLang::AppendMenuItem(menuHelp, ID_MENU_ABOUT, _T("&About..."));

  // menu bar
  m_MenuBar = new wxMenuBar();
  m_MenuBar->Append(menuFile, _TT(ID_MENU_FILE, "&File"));
  m_MenuBar->Append(menuView, _TT(ID_MENU_VIEW, "&View"));
  m_MenuBar->Append(m_MenuPreset, _TT(ID_MENU_PRESETS, "&Presets"));
  m_MenuBar->Append(menuOptions, _TT(ID_MENU_OPTIONS, "&Options"));
  m_MenuBar->Append(menuHelp, _TT(ID_MENU_HELP, "&Help"));
  SetMenuBar(m_MenuBar);

  // ----- misc config settings -----

  // directories
  CONFIG->Read(_T("/locations/file_folder"), &m_FileOpenDir);
  wxString smxDir;
  CONFIG->Read(_T("/locations/smx_folder"), &smxDir);
  MGR->SetSmxDir(smxDir);

  // load status of toolbar from config
  bool toolbar;
  CONFIG->Read(_T("/view/presets_toolbar"), &toolbar, false);
  ShowPane(m_Presets, toolbar);

  // view settings and options
  CONFIG->Read(_T("/view/crosshair"), &m_CrossHairVisible, false);
  ShowCrossHair(m_CrossHairVisible);

  CONFIG->Read(_T("/options/xrulers"), &m_Xrulers, true);
  ShowXrulers(m_Xrulers);

  CONFIG->Read(_T("/options/tooltips"), &m_ToolTips, false);
  m_Graphs->EnableToolTips(m_ToolTips);

  CONFIG->Read(_T("/options/speed_mph"), &cCarState::s_SpeedInMph, false);

  bool statusbar;
  CONFIG->Read(_T("/options/statusbar"), &statusbar, true);
  if (statusbar) CreateStatusBar();

  CONFIG->Read(_T("/options/file_dialog"), &m_CustomFileDialog, true);

  // custom file dialog
  m_FileDialog = new cFileDlg(this);
  m_FileDialog->LoadConfig(CONFIG, _T("/windows/file_dialog"));

  // create default graph
  AddGraph(GRAPHTYPE_Y, DEFAULT_PROPORTION, false);

  // save current perspective and graphs as default preset
  UpdateAll();
  m_Presets->AddDefault(m_AuiMgr.SavePerspective(), SaveGraphs());

  // config settings for panes
  m_Legend->LoadConfig(CONFIG, _T("/panes/legend"));
  m_DrivingLine->LoadConfig(CONFIG, _T("/panes/drivingline"));
  m_TrackMap->LoadConfig(CONFIG, _T("/panes/trackmap"));
  m_Presets->LoadConfig(CONFIG, _T("/presets"));

  // show the user-defined presets in the menu
  RebuildPresetsMenu();

  // redirect scrollwheel events to the window at the pointer
  RedirectEvents(this, NULL, true, false);

  // restore the session from the config
  CONFIG->Read(_T("/session/restore"), &m_RestoreSession, false);
  if (m_RestoreSession) {
    // restore perspective
    wxString perspective;
    CONFIG->Read(_T("/session/perspective"), &perspective);
    if (!perspective.IsEmpty()) LoadPerspective(perspective);

    // restore graphs
    wxString graphs;
    CONFIG->Read(_T("/session/graphs"), &graphs);
    if (!graphs.IsEmpty()) LoadGraphs(graphs);

    // restore files (unless a file was given as command line parameter)
    if (IsEmpty(infile)) {
      int i = 0;
      while (true) {
        wxString key;
        key.Printf(_T("/session/files/file%d"), i);
        wxString file;
        if (!CONFIG->Read(key, &file)) break;
        RequestFileLoad(file);
        i += 1;
      }
    }
  }

  // restore 'maximized' state
  bool maximized = false;
  CONFIG->Read(_T("/windows/main/maximized"), &maximized);
  Maximize(maximized);

  UpdateAll();

  // load RAF file from command line (if any)
  if (!IsEmpty(infile)) RequestFileLoad(infile);
}

//-----------------------------------------------------------------------------
// Application exit

void cMainFrame::OnExit(wxCommandEvent& WXUNUSED(event))
{
  Close(TRUE);
}

void cMainFrame::OnClose(wxCloseEvent& WXUNUSED(event))
{
  RestorePanes(); // restore pane states so they can be stored correctly

  CONFIG->Write(_T("version"), APPLICATION_VERSION); // useful to solve later compatibility problems

  CONFIG->Write(_T("/windows/main/pos/x"), m_Position.x);
  CONFIG->Write(_T("/windows/main/pos/y"), m_Position.y);
  CONFIG->Write(_T("/windows/main/maximized"), IsMaximized());

  // record size if it is 'normal'
  if (!IsMaximized() && !IsIconized() && !IsFullScreen()) {
    CONFIG->Write(_T("/windows/main/size/x"), GetSize().GetWidth());
    CONFIG->Write(_T("/windows/main/size/y"), GetSize().GetHeight());
  }

  CONFIG->Write(_T("/session/restore"), m_RestoreSession);
  wxString perspective = m_AuiMgr.SavePerspective();
  if (m_RestoreSession) {
    // save session
    CONFIG->Write(_T("/session/perspective"), perspective);
    CONFIG->Write(_T("/session/graphs"), SaveGraphs());
    CONFIG->DeleteGroup(_T("/session/files"));
    for (size_t i = 0; i < MGR->GetLapCount(); i++) {
      wxString key;
      key.Printf(_T("/session/files/file%d"), i);
      CONFIG->Write(key, MGR->GetLap(i)->GetFileName());
    }
  }

  CONFIG->Write(_T("/view/crosshair"), m_CrossHairVisible);
  CONFIG->Write(_T("/view/autoscroll"), m_AutoScroll);
  CONFIG->Write(_T("/view/presets_toolbar"), m_AuiMgr.GetPane(m_Presets).IsShown());

  // other options
  CONFIG->Write(_T("/options/speed_mph"), cCarState::s_SpeedInMph);
  CONFIG->Write(_T("/options/xrulers"), m_Xrulers);
  CONFIG->Write(_T("/options/tooltips"), m_ToolTips);
  CONFIG->Write(_T("/options/statusbar"), (GetStatusBar() != NULL));
  CONFIG->Write(_T("/options/language"), cLang::GetLanguage());

  // file dialog settings
  CONFIG->Write(_T("/options/file_dialog"), m_CustomFileDialog);
  m_FileDialog->SaveConfig(CONFIG, _T("/windows/file_dialog"));

  CONFIG->Write(_T("/locations/file_folder"), m_FileOpenDir);
  wxString smxDir = MGR->GetSmxDir();
  CONFIG->Write(_T("/locations/smx_folder"), smxDir);

  for (size_t h = 0; h < m_FileHistory.GetCount(); h++) {
    wxString key;
    key.Printf(_T("/locations/file_history/file%d"), h);
    CONFIG->Write(key, m_FileHistory.GetHistoryFile(h));
  }

  // config settings for panes
  m_Legend->SaveConfig(CONFIG, _T("/panes/legend"));
  m_DrivingLine->SaveConfig(CONFIG, _T("/panes/drivingline"));
  m_TrackMap->SaveConfig(CONFIG, _T("/panes/trackmap"));
  m_Presets->SaveConfig(CONFIG, _T("/presets"));

#ifdef __WXDEBUG__
  cLang::DumpDefault();
#endif

  m_AuiMgr.UnInit();

  Destroy();
}

//-----------------------------------------------------------------------------
// Delete all lap data

void cMainFrame::CloseAll()
{
  if (MGR->IsEmpty()) return; // nothing to do

  // cancel loading
  CancelFileLoads();

  // clear graphs
  for (size_t v = 0; v < m_Graphs->GetCount(); v++) {
    m_Graphs->GetGraph(v)->DeleteAllLaps();
  }

  // delete laps
  MGR->ClearAll();

  // reset view
  DoSetTrackSelection(0.0f, 1.0f); // reset track selection
  DoSetTrackCursorPos(NULL, -1.0f, true); // remove track cursors
  DoZoomReset();

  // clear error log
  m_Log->Clear();

  UpdateAll();
}

//-----------------------------------------------------------------------------
// Handle clicks in the 'Help' menu

void cMainFrame::OnHelp(wxCommandEvent& event)
{
  wxString version;

  switch (event.GetId()) {
    case ID_MENU_MANUAL :
      wxLaunchDefaultBrowser(_T("file://") + m_ManualFile);
      return;

    case ID_MENU_ABOUT :
    {
      wxAboutDialogInfo info;
      info.SetName(_T(APPLICATION_NAME));
      version = APPLICATION_VERSION;
#ifdef __WXDEBUG__
      version += _T(" (debug)");
#endif
      info.SetVersion(version);
      info.SetCopyright(_T("(c) Dick Alstein <dick@djkb.pilmo.nl>"));
      wxAboutBox(info);
      return;
    }

    case ID_MENU_WEBSITE :
      wxLaunchDefaultBrowser(LRA_WEBSITE);
      return;

    default :
      wxFAIL;
  }
}

//-----------------------------------------------------------------------------
// Open RAF file(s) from a dialog

void cMainFrame::OnOpen(wxCommandEvent& event)
{
  bool addFiles = (event.GetId() == ID_MENU_ADD) && (!MGR->IsEmpty()); // add files, or open new ones?

  // create file dialog
  wxString caption;
  if (addFiles) {
    caption = _TT(ID_TXT_MF_ADDFILES, "Add RAF file(s)");
  }
  else {
    caption = _TT(ID_TXT_MF_OPENFILES, "Open RAF file(s)");
  }
  wxArrayString fileNames; // will receive the selected filename(s)
  if (m_CustomFileDialog) {
    // custom dialog
    if (m_FileDialog->DoShowModal(caption, m_FileOpenDir, addFiles) != wxID_OK) return;

    // get user choices
    m_FileOpenDir = m_FileDialog->GetDirectory();
    m_FileDialog->GetPaths(fileNames);

    m_FileDialog->Close();
  }
  else {
    // default Windows dialog
    wxFileDialog fileDlg(this, caption, m_FileOpenDir,
        _T(""), // no default file name
        _T("RAF files (*.raf)|*.raf"), // wildcard
        (wxOPEN | wxMULTIPLE | wxFILE_MUST_EXIST | wxCHANGE_DIR) // style
       );
    if (fileDlg.ShowModal() != wxID_OK) return;

    // get user choices
    m_FileOpenDir = fileDlg.GetDirectory();
    fileDlg.GetPaths(fileNames);

    fileDlg.Close();
  }

  if (fileNames.IsEmpty()) return; // nothing selected
  if (!addFiles) CloseAll(); // close existing files

  Update(); // remove file dialog and redraw main window

  // load the file(s)
  wxBusyCursor cursor; // activate hourglass
  Enable(false);
  for (size_t f = 0; f < fileNames.GetCount(); f++) {
    RequestFileLoad(fileNames[f]);
  }
  Enable(true);
}

//-----------------------------------------------------------------------------
// Open a file from the file history

void cMainFrame::OnOpenRecent(wxCommandEvent& event)
{
  wxString file = m_FileHistory.GetHistoryFile(event.GetId() - wxID_FILE1);
  RequestFileLoad(file);
}

//-----------------------------------------------------------------------------
// Update menu contents, prior to opening the menu

void cMainFrame::OnMenu(wxMenuEvent& WXUNUSED(event))
{
  EnableMenuItems();
}

//-----------------------------------------------------------------------------
// Enable/disable/check menu items

void cMainFrame::EnableMenuItems()
{
  // File menu
  m_MenuBar->Enable(ID_MENU_OPENRECENT, (m_FileHistory.GetCount() > 0));

  // View menu
  m_MenuBar->Check(ID_MENU_CROSSHAIR, m_CrossHairVisible);
  m_MenuBar->Check(ID_MENU_LEGEND, m_AuiMgr.GetPane(m_Legend).IsShown());
  m_MenuBar->Check(ID_MENU_TRACKMAP, m_AuiMgr.GetPane(m_TrackMap).IsShown());
  m_MenuBar->Check(ID_MENU_DRIVINGLINE, m_AuiMgr.GetPane(m_DrivingLine).IsShown());
  m_MenuBar->Check(ID_MENU_PROPERTIES, m_AuiMgr.GetPane(m_Properties).IsShown());
  m_MenuBar->Check(ID_MENU_LOG, m_AuiMgr.GetPane(m_Log).IsShown());
  m_MenuBar->Check(ID_MENU_FULLSCREEN, IsFullScreen());

  if (m_AllPanesHidden) {
    m_MenuBar->SetLabel(ID_MENU_HIDEALLPANES, _TT(ID_TXT_MF_SHOWALLPANES, "Show all\tF2"));
  }
  else {
    m_MenuBar->SetLabel(ID_MENU_HIDEALLPANES, _TT(ID_MENU_HIDEALLPANES, wxEmptyString));
  }

  // Preset menu
  m_MenuBar->Enable(ID_MENU_ADD_PRESET, !m_AllPanesHidden);
  m_MenuBar->Enable(ID_MENU_DELETE_PRESET, (m_Presets->GetCount() > 1));
  m_MenuBar->Check(ID_MENU_PRESET_TOOLBAR, m_AuiMgr.GetPane(m_Presets).IsShown());

  // Options menu
  m_MenuBar->Check(ID_MENU_SPEED_KMH, !cCarState::s_SpeedInMph);
  m_MenuBar->Check(ID_MENU_SPEED_MPH, cCarState::s_SpeedInMph);
  m_MenuBar->Check(ID_MENU_XRULERS, m_Xrulers);
  m_MenuBar->Check(ID_MENU_TOOLTIPS, m_ToolTips);
  m_MenuBar->Check(ID_MENU_STATUSBAR, (GetStatusBar() != NULL));
  m_MenuBar->Check(ID_MENU_SESSION, m_RestoreSession);
  m_MenuBar->Check(ID_MENU_CUSTOM_FILE_DLG, m_CustomFileDialog);

  // Languages submenu
  for (size_t i = 0; i < m_MenuLang->GetMenuItemCount(); i++) {
    wxMenuItem* item = m_MenuLang->FindItem(ID_MENU_LANGUAGE_FIRST + i);
    wxASSERT(item != NULL);
    item->Check(item->GetItemLabelText() == cLang::GetLanguage());
  }
}

//-----------------------------------------------------------------------------
// Toggle visibility of panes, and the other view settings

void cMainFrame::OnChangeView(wxCommandEvent& event)
{
  cPane* pane = NULL;

  switch (event.GetId()) {
    case ID_MENU_LEGEND :
      pane = m_Legend;
      break;

    case ID_MENU_TRACKMAP :
      pane = m_TrackMap;
      break;

    case ID_MENU_DRIVINGLINE :
      pane = m_DrivingLine;
      break;

    case ID_MENU_PROPERTIES :
      pane = m_Properties;
      break;

    case ID_MENU_LOG :
      pane = m_Log;
      break;

    case ID_MENU_CROSSHAIR :
      ShowCrossHair(!m_CrossHairVisible);
      return;

    case ID_MENU_FULLSCREEN :
      // toggle between normal window and full screen
      DoSetTrackCursorPos(NULL, -1.0f, true); // remove track cursors
      ShowFullScreen(!IsFullScreen(),
          wxFULLSCREEN_NOBORDER | wxFULLSCREEN_NOCAPTION | wxFULLSCREEN_NOSTATUSBAR);
      return;

    case ID_MENU_PRESET_TOOLBAR :
      pane = m_Presets;
      break;

    case ID_MENU_HIDEALLPANES :
      ToggleAllPanes();
      return;
  }

  wxASSERT(pane != NULL);
  if (pane == NULL) return; // defensive

  ShowPane(pane, !pane->IsShown());
  m_AuiMgr.Update();
}

//-----------------------------------------------------------------------------

void cMainFrame::ShowCrossHair(bool visible)
{
  m_CrossHairVisible = visible;
  for (size_t v = 0; v < m_Graphs->GetCount(); v++) {
    m_Graphs->GetGraph(v)->ShowCrossHair(m_CrossHairVisible);
  }
}

//-----------------------------------------------------------------------------

void cMainFrame::ShowXrulers(bool visible)
{
  m_Xrulers = visible;
  for (size_t v = 0; v < m_Graphs->GetCount(); v++) {
    m_Graphs->GetGraph(v)->ShowXruler(m_Xrulers);
  }
}

//-----------------------------------------------------------------------------
// Change the zoom for the graphs

void cMainFrame::OnZoom(wxCommandEvent& event)
{
  switch (event.GetId()) {
    case ID_MENU_ZOOM_RESET_ALL :
      DoZoomReset();
      break;

    case ID_MENU_ZOOM_FIT :
      for (size_t v = 0; v < m_Graphs->GetCount(); v++) {
        m_Graphs->GetGraph(v)->FitSelection();
      }
      break;

    default :
      wxFAIL;
  }

  // update panes
  wxASSERT(m_Graphs->GetCount() > 0);
  m_Graphs->GetGraph(0)->RequestSync();
}

void cMainFrame::DoZoomReset()
{
  for (size_t v = 0; v < m_Graphs->GetCount(); v++) {
    m_Graphs->GetGraph(v)->ZoomReset();
  }

  // reset track selection (to the whole track)
  DoSetTrackSelection(0.0f, MGR->GetTrackLength());
}

//-----------------------------------------------------------------------------

void cMainFrame::OnMove(wxMoveEvent& event)
{
  // record last 'normal' position
  if (!IsMaximized() && !IsIconized() && !IsFullScreen()) m_Position = GetPosition();
  event.Skip();
}

//-----------------------------------------------------------------------------
// Handle a "close view" event

void cMainFrame::OnCloseView(wxCommandEvent& event)
{
  wxWindow* view;
  ::CloseView_Decode(event, view);

  int index;
  index = m_Graphs->Index((cGraphView*)view);
  if (index != wxNOT_FOUND) {
    // close graph
    DeleteGraph(index);
  }

  index = m_Panes.Index((cPane*)view);
  if (index != wxNOT_FOUND) {
    // close pane, but only if it is floating
    wxAuiPaneInfo* pane = &m_AuiMgr.GetPane(view);
    if ((pane->IsShown()) && (pane->IsFloating())) {
      ShowPane(m_Panes[index], false);
      m_AuiMgr.Update();
    }
  }
}

//-----------------------------------------------------------------------------
// Handle a "delete lap" event

void cMainFrame::OnDeleteLap(wxCommandEvent& event)
{
  cLap* lap;
  ::DeleteLap_Decode(event, lap);

  for (size_t v = 0; v < m_Graphs->GetCount(); v++) {
    m_Graphs->GetGraph(v)->DeleteLap(lap);
  }

  MGR->DeleteLap(lap);

  if (MGR->IsEmpty()) {
    // no more laps - reset to defaults
    DoSetTrackSelection(0.0f, 1.0f); // reset track selection
    DoSetTrackCursorPos(NULL, -1.0f, true); // remove track cursors
    DoZoomReset();
  }

  UpdateAll();
}

//-----------------------------------------------------------------------------
// Handle a "show/hide lap" event

void cMainFrame::OnShowLap(wxCommandEvent& event)
{
  cLap* lap;
  bool show;
  ::ShowLap_Decode(event, lap, show);

  // update the lap(s)
  if (lap == NULL) {
    // show/hide all laps
    for (size_t l = 0; l < MGR->GetLapCount(); l++) {
      MGR->GetLap(l)->Show(show);
    }
  }
  else {
    // show/hide only this lap
    lap->Show(show);
  }

  // update graphs
  for (size_t v = 0; v < m_Graphs->GetCount(); v++) {
    m_Graphs->GetGraph(v)->DoShowLap(lap, show);
  }

  // update panes
  for (size_t p = 0; p < m_Panes.GetCount(); p++) {
    m_Panes[p]->DoShowLap(lap, show);
  }
}

//-----------------------------------------------------------------------------
// Handle a "synchronise graphs" event

void cMainFrame::OnSyncGraphs(wxCommandEvent& event)
{
  cGraphView* view = NULL;
  ::SyncGraphs_Decode(event, view);

  // update graphs
  for (size_t v = 0; v < m_Graphs->GetCount(); v++) {
    m_Graphs->GetGraph(v)->DoSynchronise(view);
  }
}

//-----------------------------------------------------------------------------
// Handle a "set track-cursor position" event

void cMainFrame::OnSetTrackCursorPos(wxCommandEvent& event)
{
  float distance;
  bool scroll;
  ::SetTrackCursorPos_Decode(event, distance, scroll);

  // find the graphview under the mouse pointer
  wxPoint pt;
  wxWindow* wap = wxFindWindowAtPointer(pt);
  if (wap != NULL) wap = wap->GetParent();
  cGraphView* view = NULL;
  if (m_Graphs->Index(wap) != wxNOT_FOUND) view = (cGraphView*)wap;

  // pass it on to the graphs and panes
  DoSetTrackCursorPos(view, distance, scroll);
}

void cMainFrame::DoSetTrackCursorPos(cGraphView* view, float distance, bool scroll)
{
  m_CursorDist = distance;

  // maintain consistency between cursor and selection
  if (scroll && (m_CursorDist >= 0.0f) &&
      ((m_CursorDist < m_SelStart) || (m_CursorDist > m_SelEnd))) {
    // cursor pos is outside selected part - move selection
    float delta = m_CursorDist - ((m_SelStart + m_SelEnd) / 2.0f); // amount to move
    if ((m_SelStart + delta) < 0.0f) delta = -m_SelStart;
    if ((m_SelEnd + delta) > MGR->GetTrackLength()) delta = MGR->GetTrackLength() - m_SelEnd;

    if (fabs(delta) > EPSILON) {
      // actually needs some scrolling
      DoSetTrackSelection(m_SelStart + delta, m_SelEnd + delta);
    }
  }

  for (size_t v = 0; v < m_Graphs->GetCount(); v++) {
    m_Graphs->GetGraph(v)->DoSetTrackCursorPos(view, distance, scroll);
  }

  for (size_t p = 0; p < m_Panes.GetCount(); p++) {
    m_Panes[p]->DoSetTrackCursorPos(view, distance, scroll);
  }
}

//-----------------------------------------------------------------------------
// Update the contents

void cMainFrame::UpdateAll()
{
  // show the track name in the title bar and in the trackmap
  wxString frameTitle(_T(APPLICATION_NAME));
  wxString mapTitle(_TT(ID_CAPTION_TRACKMAP, "Track map"));
  if (!MGR->IsEmpty()) {
    frameTitle += _T(" @ ") + MGR->GetTrackName();
    mapTitle += _T(" (") + MGR->GetTrackCode() + _T(")");
  }
  SetTitle(frameTitle);
  m_AuiMgr.GetPane(m_TrackMap).Caption(mapTitle);

  // set the captions of the other panes
  m_AuiMgr.GetPane(m_DrivingLine).Caption(_TT(ID_CAPTION_DRIVINGLINE, "Driving line"));
  m_AuiMgr.GetPane(m_Legend).Caption(_TT(ID_CAPTION_LEGEND, "Legend"));
  m_AuiMgr.GetPane(m_Properties).Caption(_TT(ID_CAPTION_PROPERTIES, "Properties"));
  m_AuiMgr.GetPane(m_Log).Caption(_TT(ID_CAPTION_MESSAGES, "Messages"));

  // update 'ticked' and 'enabled' state of menu items
  EnableMenuItems();

  // update pane contents and sizes
  for (size_t p = 0; p < m_Panes.GetCount(); p++) {
    cPane* pane = m_Panes[p];
    pane->UpdateAll();
    m_AuiMgr.GetPane(pane).MinSize(pane->GetMinSize());
    m_AuiMgr.GetPane(pane).BestSize(pane->GetBestSize());
  }

  m_AuiMgr.Update();

  for (size_t v = 0; v < m_Graphs->GetCount(); v++) {
    m_Graphs->GetGraph(v)->UpdateAll();
  }

  m_Graphs->Layout();

  DoSetTrackSelection(m_SelStart, m_SelEnd);
  DoSetTrackCursorPos(NULL, m_CursorDist, true);

  Refresh();
}

//-----------------------------------------------------------------------------

void cMainFrame::OnAddGraph(wxCommandEvent& event)
{
  switch (event.GetId()) {
    case ID_MENU_ADDGRAPH_Y :
      AddGraph(GRAPHTYPE_Y, DEFAULT_PROPORTION);
      break;
    case ID_MENU_ADDGRAPH_XY :
      AddGraph(GRAPHTYPE_XY, DEFAULT_PROPORTION);
      break;
    case ID_MENU_ADDGRAPH_H :
      AddGraph(GRAPHTYPE_H, DEFAULT_PROPORTION);
      break;
    default :
      wxFAIL;
  }
}

//-----------------------------------------------------------------------------
// Add a new graph
// - type = graph type
// - size = relative size (proportion) of new graph
// - update = update immediately?
// - settings = graph settings (may be empty)

void cMainFrame::AddGraph(graph_t type, int size, bool update, wxString settings)
{
  cGraphView* view = m_Graphs->AddGraph(type, settings, size);

  // copy general view settings to the new graph
  view->ShowCrossHair(m_CrossHairVisible);
  view->ShowXruler(m_Xrulers);
  view->EnableToolTips(m_ToolTips);

  if (update) UpdateAll();
}

//-----------------------------------------------------------------------------
// Delete an existing graph
// - index = index of graph
// - update = update immediately?

void cMainFrame::DeleteGraph(size_t index, bool update)
{
  wxASSERT(m_Graphs->GetCount() > index);
  cGraphView* view = m_Graphs->GetGraph(index);
  m_Graphs->DeleteGraph(view);

  if (update) UpdateAll();
}

//-----------------------------------------------------------------------------
// Handle a "set track selection" event

void cMainFrame::OnSetTrackSelection(wxCommandEvent& event)
{
  float start, end;
  ::SetTrackSelection_Decode(event, start, end);
  DoSetTrackSelection(start, end);
}

void cMainFrame::DoSetTrackSelection(float start, float end)
{
  m_SelStart = start;
  m_SelEnd = end;

  // pass it on to the graphs and panes
  for (size_t v = 0; v < m_Graphs->GetCount(); v++) {
    m_Graphs->GetGraph(v)->DoSetTrackSelection(start, end);
  }
  for (size_t p = 0; p < m_Panes.GetCount(); p++) {
    m_Panes[p]->DoSetTrackSelection(start, end);
  }

  // maintain consistency between cursor and selection
  if ((m_CursorDist >= 0.0f) && ((m_CursorDist < m_SelStart) || (m_CursorDist > m_SelEnd))) {
    // cursor pos is outside selected part - set it halfway in it
    DoSetTrackCursorPos(NULL, (m_SelStart + m_SelEnd) / 2.0f, true);
  }
}

//-----------------------------------------------------------------------------
// Settings from the Options menu

void cMainFrame::OnSetOption(wxCommandEvent& event)
{
  switch (event.GetId()) {
    case ID_MENU_SPEED_KMH :
    case ID_MENU_SPEED_MPH :
      cCarState::s_SpeedInMph = (event.GetId() == ID_MENU_SPEED_MPH);
      for (size_t v = 0; v < m_Graphs->GetCount(); v++) {
        m_Graphs->GetGraph(v)->UpdateAll();
      }
      return;

    case ID_MENU_XRULERS :
      ShowXrulers(!m_Xrulers);
      return;

    case ID_MENU_TOOLTIPS :
      m_ToolTips = !m_ToolTips;
      m_Graphs->EnableToolTips(m_ToolTips);
      return;

    case ID_MENU_STATUSBAR :
      // toggle status bar
      if (GetStatusBar() == NULL) {
        CreateStatusBar();
      }
      else {
        GetStatusBar()->Hide();
        SetStatusBar(NULL);
      }
      UpdateAll();
      return;

    case ID_MENU_SESSION :
       m_RestoreSession = !m_RestoreSession;
       break;

    case ID_MENU_CUSTOM_FILE_DLG :
      m_CustomFileDialog = !m_CustomFileDialog;
      return;
  }
}

//-----------------------------------------------------------------------------
// Helper function to load a perspective

void cMainFrame::LoadPerspective(const wxString& perspective)
{
  // undo 'all panes hidden' status
  bool panesWereHidden = m_AllPanesHidden;
  RestorePanes();

  // save the state of the 'special' panes
  // these must not be changed by loading the perspective
  wxString presets = m_AuiMgr.SavePaneInfo(m_AuiMgr.GetPane(m_Presets));
  wxString log = m_AuiMgr.SavePaneInfo(m_AuiMgr.GetPane(m_Log));

  // load the perspective
  m_AuiMgr.LoadPerspective(perspective, false);

  // restore the state of the 'special' panes
  m_AuiMgr.LoadPaneInfo(presets, m_AuiMgr.GetPane(m_Presets));
  m_AuiMgr.LoadPaneInfo(log, m_AuiMgr.GetPane(m_Log));

  // reset the 'should be hidden' flags
  for (size_t p = 0; p < m_Panes.GetCount(); p++) {
    m_Panes[p]->SetShouldBeHidden(!(m_AuiMgr.GetPane(m_Panes[p]).IsShown()));
  }

  m_AuiMgr.GetPane(m_Graphs).Show(); // defensive - the graph container is always open

  // re-apply 'all panes hidden' status
  if (panesWereHidden) ToggleAllPanes();

  // set input focus to first graph
  m_Graphs->GetGraph(0)->SetFocus();
}

//-----------------------------------------------------------------------------
// Save the current graph settings to a string

wxString cMainFrame::SaveGraphs() const
{
  wxString graphs;

  graphs = _T("format2|");
  for (size_t v = 0; v < m_Graphs->GetCount(); v++) {
    graphs += wxString::Format(_T("type=%d;"), m_Graphs->GetGraph(v)->GetType());
    graphs += wxString::Format(_T("size=%d;"), m_Graphs->GetProportion(v));
    graphs += m_Graphs->GetGraph(v)->SaveSettings();
    graphs += _T("|");
  }

  return graphs;
}

//-----------------------------------------------------------------------------
// Restore the graph settings

void cMainFrame::LoadGraphs(const wxString& graphs)
{
  wxArrayString settings;
  ::SplitStringList(graphs, _T('|'), settings);

  if (settings.GetCount() < 2)
      return; // no graphs

  if ((settings[0] != _T("format1")) && (settings[0] != _T("format2")))
      return; // wrong format version

  // delete all current graphs
  while (m_Graphs->GetCount() > 0) DeleteGraph(0, false);

  for (size_t i = 1; i < settings.GetCount(); i++) {
    // default values
    graph_t type = GRAPHTYPE_Y;
    int size = DEFAULT_PROPORTION;

    wxArrayString keys;
    wxArrayString values;
    ::DecodeKeyValueList(settings[i], keys, values);
    for (size_t k = 0; k < keys.GetCount(); k++) {
      if (keys[k] == _T("type")) type = (graph_t)wxAtoi(values[k]);
      if (keys[k] == _T("size")) size = wxAtoi(values[k]);
    }

    // create graph
    AddGraph(type, size, false, settings[i]);
  }

  wxASSERT(m_Graphs->GetCount() > 0);
  return;
}

//-----------------------------------------------------------------------------
// Handling presets

void cMainFrame::RebuildPresetsMenu()
{
  // remove existing items from menu
  for (size_t i = 0; i < MAX_PRESETS; i++) {
    wxMenuItem* item = m_MenuPreset->FindItem(ID_MENU_PRESET_FIRST + i);
    if (item == NULL) break;
    m_MenuPreset->Delete(item);
  }

  // add presets to menu
  for (size_t i = 0; i < m_Presets->GetCount(); i++) {
    wxString label = m_Presets->GetPresetName(i);
    int shortcut = m_Presets->GetShortcut(i);
    if (shortcut >= 0) label += wxString::Format(_T("\t%d"), shortcut);
    m_MenuPreset->Append(ID_MENU_PRESET_FIRST + i, label);
  }
}


void cMainFrame::OnAddPreset(wxCommandEvent& WXUNUSED(event))
{
  wxTextEntryDialog dlg(this,
      _TT(ID_TXT_MF_ADDPRESET_PROMPT, "Name for new preset"),
      _TT(ID_TXT_MF_ADDPRESET_CAPTION, "Save view as preset"));
  if (dlg.ShowModal() == wxID_CANCEL) return;
  wxString name = dlg.GetValue();
  name.Trim(true);
  name.Trim(false);

  if (name.IsEmpty()) return; // no name entered

  // check if name already exists
  int index = m_Presets->Find(name);
  if (index == wxNOT_FOUND) {
    // add preset
    if (m_Presets->GetCount() == MAX_PRESETS) return;
    m_Presets->AddPreset(name, m_AuiMgr.SavePerspective(), SaveGraphs());
  }
  else {
    // overwrite preset (after confirm)
    if (wxMessageBox(
        _TT(ID_TXT_MF_OVERWRITEPRESET_PROMPT, "Overwrite existing preset?"),
        _TT(ID_TXT_MF_OVERWRITEPRESET_CAPTION, "Preset already exists"), wxOK | wxCANCEL, this) == wxCANCEL)
        return;
    m_Presets->ReplacePreset(index, m_AuiMgr.SavePerspective(), SaveGraphs());
  }

  RebuildPresetsMenu();
}


void cMainFrame::OnDeletePreset(wxCommandEvent& WXUNUSED(event))
{
  wxASSERT(m_Presets->GetCount() > 0);

  wxArrayString names;
  // build list of names, but omit default preset from choice
  for (size_t i = 1; i < m_Presets->GetCount(); i++) names.Add(m_Presets->GetPresetName(i));
  wxSingleChoiceDialog dlg(this,
      _TT(ID_TXT_MF_DELETEPRESET_PROMPT, "Select preset to delete"),
      _TT(ID_TXT_MF_DELETEPRESET_CAPTION, "Delete preset"), names);
  if (dlg.ShowModal() == wxID_CANCEL) return;

  int index = dlg.GetSelection();
  if (index < 0) return;
  index += 1; // correction for omitted default preset

  // delete preset
  m_Presets->DeletePreset(index);

  RebuildPresetsMenu();
}


void cMainFrame::OnSelectPresetFromMenu(wxCommandEvent& event)
{
  int index = event.GetId() - ID_MENU_PRESET_FIRST;
  wxASSERT(index >= 0);
  wxASSERT(index < (int)m_Presets->GetCount());

  SelectPreset(index);
}


void cMainFrame::OnPresetEvent(wxCommandEvent& event)
{
  int action;
  wxString name;
  ::Preset_Decode(event, action, name);

  size_t index;
  switch (action) {
    case 0 : // reload presets
      RebuildPresetsMenu();
      break;

    case 1 : // select preset
      index = m_Presets->Find(name);
      wxASSERT(index != wxNOT_FOUND);
      SelectPreset(index);
      break;

    default :
      wxFAIL;
  }
}


void cMainFrame::SelectPreset(size_t index)
{
  Freeze();
  LoadPerspective(m_Presets->GetPanes(index));
  LoadGraphs(m_Presets->GetGraphs(index));
  Thaw();
  UpdateAll();
}

//-----------------------------------------------------------------------------
// Handle requests to open a file

void cMainFrame::OnOpenFile(wxCommandEvent& event)
{
  wxString filename;
  ::OpenFile_Decode(event, filename);
  RequestFileLoad(filename);

  // activate window
  if (IsIconized()) Maximize(false); // un-iconize
  Raise();
}

//-----------------------------------------------------------------------------
// Initiate the loading of a RAF file
// - fileName = pathname of file

void cMainFrame::RequestFileLoad(const wxString& fileName)
{
  if (fileName.IsEmpty()) return;       // ignore empty names
  if (!wxFileExists(fileName)) return;  // ignore directories and nonexistent files
  if (MGR->IsLoaded(fileName)) return;  // avoid loading a file twice

  wxBeginBusyCursor();

  cLap* lap = new cLap(fileName);
  m_LapsToLoad.Add(lap);
}

//-----------------------------------------------------------------------------
// Cancel all file loading

void cMainFrame::CancelFileLoads()
{
  for (size_t i = 0; i < m_LapsToLoad.GetCount(); i++) {
    delete m_LapsToLoad[i];
    wxEndBusyCursor();
  }
  m_LapsToLoad.Clear();
}

//-----------------------------------------------------------------------------
// Idle event handler, used for file loading

void cMainFrame::OnIdle(wxIdleEvent& event)
{
  if (m_LapsToLoad.IsEmpty()) return; // nothing to do

  cLap* lap = m_LapsToLoad[0]; // get the first file in the queue
  switch (lap->GetStatus()) {

    case LAPSTATUS_CREATED :
      // start loading the lap
      if (GetStatusBar() != NULL) {
        SetStatusText(
            wxString::Format(_TT(ID_TXT_MF_STATUS_LOADING, "Loading \"%s\"..."),
            lap->GetName(LAPNAME_FILE).c_str()));
      }
      lap->LoadStart();
      break;

    case LAPSTATUS_LOADING :
      // continue loading
      lap->LoadNext(FILELOAD_TIMESLICE);
      break;

    case LAPSTATUS_LOAD_OK :
    {
      // handle success of loading
      wxEndBusyCursor();
      m_LapsToLoad.Remove(lap);

      // try to add loaded lap
      wxString error;
      if (MGR->AddLap(lap, error)) {
        // warn if lap is not complete
        if (!lap->IsComplete()) {
          Report(_TT(ID_TXT_MF_INCOMPLETELAP, "file contains an incomplete lap"),
              wxString::Format(
                  _TT(ID_TXT_MF_ADDING, "adding \"%s\""),
                  lap->GetName(LAPNAME_FILE).c_str()), false);
        }

        // update display
        for (size_t v = 0; v < m_Graphs->GetCount(); v++) {
          m_Graphs->GetGraph(v)->AddLap(lap);
        }
        if (MGR->GetLapCount() == 1) {
          // reset track selection (to the whole track)
          DoSetTrackSelection(0.0f, MGR->GetTrackLength());
        }
        m_FileHistory.AddFileToHistory(lap->GetFileName());
        UpdateAll();
      }
      else {
        // adding failed
        Report(error, wxString::Format(_TT(ID_TXT_MF_ADDING, "adding \"%s\""), lap->GetName(LAPNAME_FILE).c_str()));
        delete lap;
      }
      break;
    }

    case LAPSTATUS_LOAD_FAIL :
      // handle failure of loading
      wxEndBusyCursor();
      m_LapsToLoad.Remove(lap);
      Report(
          lap->GetLoadError(),
          wxString::Format(_TT(ID_TXT_MF_LOADING, "loading \"%s\""), lap->GetName(LAPNAME_FILE).c_str()));
      delete lap;
      break;
  }

  // check if we're done yet
  if (m_LapsToLoad.IsEmpty()) {
    // notify user of end of loading
    if (GetStatusBar() != NULL) SetStatusText(wxEmptyString);
  }
  else {
    event.RequestMore(); // need more idle events
  }
}

//-----------------------------------------------------------------------------
// Report a message in the log window
// - desc = message contents
// - activity = activity during which the message was generated
// - error = is this an error or an informational?

void cMainFrame::Report(const wxString& desc, const wxString& activity, bool error)
{
  // assemble message text
  wxString msg = error ? _TT(ID_TXT_MF_ERROR, "Error") : _TT(ID_TXT_MF_WARNING, "Warning");
  if (!activity.IsEmpty()) msg += _T(" ") + activity;
  msg += _T(": ") + desc;

  // open log pane, if needed
  if (error && (!m_AuiMgr.GetPane(m_Log).IsShown())) {
    m_Log->Clear(); // delete old contents
    wxAuiPaneInfo* pane = &m_AuiMgr.GetPane(m_Log);
    pane->Float();
    wxPoint pos = GetPosition();
    pos.x += 100;
    pos.y += 100;
    pane->FloatingPosition(pos);
    pane->FloatingSize(m_Log->GetMinSize());
    pane->Show();
    m_AuiMgr.Update();
  }

  // show message
  m_Log->AddMessage(msg, error);
}

//-----------------------------------------------------------------------------
// Handle requests to move a graph

void cMainFrame::OnMoveGraph(wxCommandEvent& event)
{
  cGraphView* view;
  int change;
  MoveGraph_Decode(event, view, change);
  m_Graphs->MoveGraph(view, change);
}

//-----------------------------------------------------------------------------
// Select a language from the menu

void cMainFrame::OnSelectLanguage(wxCommandEvent& event)
{
  wxMenuItem* item = m_MenuLang->FindItem(event.GetId());
  if (item == NULL) return; // defensive

  cLang::SetLanguage(item->GetItemLabelText());

  // re-translate the menus
  wxMenuBar* bar = GetMenuBar();
  for (size_t m = 0; m < bar->GetMenuCount(); m++) {
    cLang::TranslateMenu(bar->GetMenu(m));
  }
  bar->SetMenuLabel(0, cLang::TranslateText(ID_MENU_FILE, wxEmptyString));
  bar->SetMenuLabel(1, cLang::TranslateText(ID_MENU_VIEW, wxEmptyString));
  bar->SetMenuLabel(2, cLang::TranslateText(ID_MENU_PRESETS, wxEmptyString));
  bar->SetMenuLabel(3, cLang::TranslateText(ID_MENU_OPTIONS, wxEmptyString));
  bar->SetMenuLabel(4, cLang::TranslateText(ID_MENU_HELP, wxEmptyString));

  // re-translate the strings in the dialogs
  cGraphTools::ResetTexts();
  m_FileDialog->TranslateTexts();

  // re-translate the strings in the panes
  for (size_t p = 0; p < m_Panes.GetCount(); p++) {
    m_Panes[p]->TranslateTexts();
  }

  // re-translate the strings in the graphs
  m_Graphs->TranslateTexts();

  UpdateAll(); // also re-translates the pane captions
}

//-----------------------------------------------------------------------------
// Hiding/Showing panes

void cMainFrame::ToggleAllPanes()
{
  if (m_AllPanesHidden) {
    // show all panes that do not have the 'hidden' flag set
    for (size_t p = 0; p < m_Panes.GetCount(); p++) {
      if (!m_Panes[p]->ShouldBeHidden()) ShowPane(m_Panes[p], true);
    }
  }
  else {
    // hide all panes (but do NOT set the 'hidden' flag
    for (size_t p = 0; p < m_Panes.GetCount(); p++) {
      m_AuiMgr.GetPane(m_Panes[p]).Show(false);
    }
  }
  UpdateAll();
  m_AllPanesHidden = !m_AllPanesHidden;
}


void cMainFrame::ShowPane(cPane* pane, bool shown)
{
  wxASSERT(m_Panes.Index(pane) != wxNOT_FOUND); // must be a valid pane

  wxAuiPaneInfo* paneInfo = &m_AuiMgr.GetPane(pane);
  if (shown) {
    if (!paneInfo->IsShown()) {
      // show unopened pane (always in docked position)
      paneInfo->Show();
      paneInfo->Dock();
    }
  }
  else {
    // close opened pane
    if (paneInfo->IsShown()) paneInfo->Hide();
  }
  pane->SetShouldBeHidden(!paneInfo->IsShown());
}
