#include "cgraphcontainer.h"
#include "cgraphview.h"
#include <wx/dcclient.h>

// width of the spacer between graphs (in pixels)
#define SPACER_SIZE 4

// minimum graph size when resizing
#define MIN_GRAPH_SIZE 75

//-----------------------------------------------------------------------------
// Event table

BEGIN_EVENT_TABLE(cGraphContainer, wxWindow)
  EVT_LEFT_DOWN(cGraphContainer::OnMouseLeftClick)
  EVT_LEFT_UP(cGraphContainer::OnMouseLeftRelease)
  EVT_ENTER_WINDOW(cGraphContainer::OnMouseEntering)
  EVT_LEAVE_WINDOW(cGraphContainer::OnMouseLeaving)
  EVT_MOTION(cGraphContainer::OnMouseMove)
  EVT_RIGHT_DOWN(cGraphContainer::OnMouseRightClick)
  EVT_MENU(wxID_ANY, cGraphContainer::OnMenuClick)
  EVT_PAINT(cGraphContainer::OnPaint)
  EVT_SIZE(cGraphContainer::OnSize)
END_EVENT_TABLE()

//-----------------------------------------------------------------------------

cGraphContainer::cGraphContainer(wxWindow* parent, wxColour border, wxColour background)
: wxWindow(parent, -1, wxDefaultPosition, wxDefaultSize, wxNO_BORDER)
{
  m_DragMode = false;

  SetBackgroundColour(*wxWHITE);

  m_BackgroundColour = background;
  m_BorderColour = border;

  cLang::AppendMenuItem(&m_Context, ID_MENU_EQUAL_HEIGHT, _T("Equal height graphs"));

  m_TopSizer = new wxBoxSizer(wxVERTICAL);
  SetSizer(m_TopSizer);

  RedirectEvents(this, NULL, true, false); // redirect scrollwheel events to the window at the pointer
}

//-----------------------------------------------------------------------------

cGraphContainer::~cGraphContainer()
{
}

//-----------------------------------------------------------------------------

void cGraphContainer::OnSize(wxSizeEvent& event)
{
  Layout();
  Refresh();
  event.Skip();
}

//-----------------------------------------------------------------------------
// Add a graph (and take ownership of it)

cGraphView* cGraphContainer::AddGraph(graph_t type, const wxString& settings, int size)
{
  wxString set = settings;
  if (set.IsEmpty()) {
    // clone settings from existing graph (of same type)
    for (size_t i = 0; i < m_Graphs.GetCount(); i++) {
      if (m_Graphs[i]->GetType() == type) set = m_Graphs[i]->SaveSettings();
    }
  }

  cGraphView* view = new cGraphView(this, type, set);
  wxASSERT(view != NULL);

  // put spacer above graph (unless it's the first)
  if (m_Graphs.GetCount() > 0) m_TopSizer->AddSpacer(SPACER_SIZE);

  m_Graphs.Add(view);
  m_TopSizer->Add(view, wxSizerFlags(1).Expand().Proportion(size));
  TellGraphPositions();

  // NB: Layout() is not called!

  return view;
}

//-----------------------------------------------------------------------------
// Delete a graph

void cGraphContainer::DeleteGraph(cGraphView* view)
{
  CHECK_PTR(view);
  wxASSERT(view != NULL);
  wxASSERT(m_TopSizer->GetItem(view) != NULL);

  int index = m_Graphs.Index(view);
  wxASSERT(index != wxNOT_FOUND);
  if (index == wxNOT_FOUND) return; // defensive

  view->Hide();

  // remove spacer
  if (index == 0) {
    // first graph: remove spacer after it (if any)
    if (m_Graphs.GetCount() > 1) m_TopSizer->Remove(1);
  }
  else {
    // other graphs: remove spacer before it
    m_TopSizer->Remove(2 * index - 1);
  }

  // remove graph
  m_Graphs.Remove(view);
  m_TopSizer->Detach(view);
  view->Destroy();

  TellGraphPositions();

  // NB: Layout() is not called!

  // if one graph remains then reset its size to default
  if (m_Graphs.GetCount() == 1) SetProportion(0, DEFAULT_PROPORTION);
}

//-----------------------------------------------------------------------------
// Get the relative size of a graph

int cGraphContainer::GetProportion(size_t index) const
{
  wxASSERT(m_Graphs.GetCount() > index);
  wxSizerItem * item = m_TopSizer->GetItem(m_Graphs[index]);
  wxASSERT(item != NULL);
  if (item == NULL) return DEFAULT_PROPORTION; // defensive

  return item->GetProportion();
}

//-----------------------------------------------------------------------------
// Set the relative size of a graph

void cGraphContainer::SetProportion(size_t index, int proportion)
{
  wxASSERT(m_Graphs.GetCount() > index);
  wxSizerItem * item = m_TopSizer->GetItem(m_Graphs[index]);
  wxASSERT(item != NULL);
  if (item == NULL) return; // defensive

  return item->SetProportion(proportion);
}

//-----------------------------------------------------------------------------

void cGraphContainer::OnMouseLeftClick(wxMouseEvent& event)
{
  wxASSERT(m_Graphs.GetCount() >= 2); // clicked on a spacer, so there must be > 1 graph

  event.Skip(); // always pass left-clicks to default event handler

  if (m_DragMode) return; // already dragging

  // get logical position of mouse
  wxClientDC dc(this);
  wxPoint pos = event.GetLogicalPosition(dc);

  // get index of spacer that is being dragged
  for (size_t i = 1; i < m_Graphs.GetCount(); i++) {
    if (m_Graphs[i]->GetPosition().y > pos.y) {
      m_DragIndex = i - 1;

      // enter dragging mode
      m_DragMode = true;
      CaptureMouse();
      return;
    }
  }
}

//-----------------------------------------------------------------------------

void cGraphContainer::OnMouseMove(wxMouseEvent& event)
{
  if (!m_DragMode) return;

  // get logical position of mouse
  wxClientDC dc(this);
  wxPoint pos = event.GetLogicalPosition(dc);

  // avoid 'jitter' when mouse is held at same position
  if (pos == m_LastDragPos) return;
  m_LastDragPos = pos;

  // calculate new sizes for graphs above and below spacer
  wxASSERT(m_Graphs.GetCount() > m_DragIndex + 1);
  wxCoord size1 = pos.y - m_Graphs[m_DragIndex]->GetPosition().y;
  wxCoord size2 = m_Graphs[m_DragIndex + 1]->GetPosition().y +
      m_Graphs[m_DragIndex + 1]->GetSize().GetHeight() - pos.y;
  // enforce minimum size
  if (size1 < MIN_GRAPH_SIZE) return;
  if (size2 < MIN_GRAPH_SIZE) return;

  // resize graphs
  int total = GetProportion(m_DragIndex) + GetProportion(m_DragIndex + 1);
  int prop1 = (total * size1) / (size1 + size2);
  int prop2 = total - prop1;
  m_Graphs[m_DragIndex]->Freeze(); // freezing and thawing the graphs prevents 'flicker' of scrollbars
  m_Graphs[m_DragIndex + 1]->Freeze();
  SetProportion(m_DragIndex, prop1);
  SetProportion(m_DragIndex + 1, prop2);
  Layout();
  m_Graphs[m_DragIndex]->Thaw();
  m_Graphs[m_DragIndex + 1]->Thaw();

  // redraw spacer
  pos = m_Graphs[m_DragIndex + 1]->GetPosition();
  int width = m_Graphs[m_DragIndex + 1]->GetSize().GetWidth();
  RefreshRect(wxRect(pos.x, pos.y - SPACER_SIZE, width, SPACER_SIZE));
  Update();
}

//-----------------------------------------------------------------------------

void cGraphContainer::OnMouseLeftRelease(wxMouseEvent& event)
{
  event.Skip(); // always pass left-clicks to default event handler

  if (m_DragMode) {
    // end drag
    if (HasCapture()) ReleaseMouse();
    m_DragMode = false;
  }
}

//-----------------------------------------------------------------------------

void cGraphContainer::OnPaint(wxPaintEvent& WXUNUSED(event))
{
  wxPaintDC dc(this);

  // draw spacers
  dc.SetPen(wxPen(m_BorderColour));
  dc.SetBrush(wxBrush(m_BackgroundColour));
  for (size_t i = 1; i < m_Graphs.GetCount(); i++) {
    // draw rectangle in spacer
    wxPoint pos = m_Graphs[i]->GetPosition();
    int width = m_Graphs[i]->GetSize().GetWidth();
    dc.DrawRoundedRectangle(pos.x + 2, pos.y - SPACER_SIZE, width - 4, SPACER_SIZE, SPACER_SIZE / 2);
 }
}

//-----------------------------------------------------------------------------

void cGraphContainer::OnMouseRightClick(wxMouseEvent& WXUNUSED(event))
{
  // show context menu
  PopupMenu(&m_Context);
}

//-----------------------------------------------------------------------------
// Handle a click in the context menu

void cGraphContainer::OnMenuClick(wxCommandEvent& event)
{
  switch (event.GetId()) {
    case ID_MENU_EQUAL_HEIGHT :
      EqualHeight();
      break;

    default :
      wxFAIL;
  }
}

//-----------------------------------------------------------------------------
// Make all graphs equal height

void cGraphContainer::EqualHeight()
{
  for (size_t i = 0; i < m_Graphs.GetCount(); i++) {
    SetProportion(i, DEFAULT_PROPORTION);
  }
  Layout();
  Refresh();
}

//-----------------------------------------------------------------------------
// Move a graph up or down
// - view = graphview to move
// - change = displacement (< 0 = move up, > 0 = move down)

void cGraphContainer::MoveGraph(cGraphView* view, int change)
{
  CHECK_PTR(view);
  int index = m_Graphs.Index(view);
  wxASSERT(index != wxNOT_FOUND);
  if (index == wxNOT_FOUND) return;

  // sanity checks
  if (index + change < 0) change = -index;
  if (index + change >= (int)m_Graphs.GetCount()) change = m_Graphs.GetCount() - 1 - index;
  if (change == 0) return; // no change

  // temporarily save the proportions
  wxArrayInt props;
  for (size_t i = 0; i < m_Graphs.GetCount(); i++) {
    props.Add(GetProportion(i));
  }

  // move the graph inside the data structures
  m_Graphs.RemoveAt(index);
  m_Graphs.Insert(view, index + change);
  int proportion = props[index];
  props.RemoveAt(index);
  props.Insert(proportion, index + change);

  // clear and re-populate the sizer
  m_TopSizer->Clear();
  for (size_t i = 0; i < m_Graphs.GetCount(); i++) {
    if (i > 0) m_TopSizer->AddSpacer(SPACER_SIZE);
    m_TopSizer->Add(m_Graphs[i], wxSizerFlags(1).Expand().Proportion(props[i]));
  }

  TellGraphPositions();
  Layout();
  Refresh();
}

//-----------------------------------------------------------------------------
// Let each graph know what its position is

void cGraphContainer::TellGraphPositions()
{
  for (size_t i = 0; i < m_Graphs.GetCount(); i++) {
    m_Graphs[i]->GraphPositionIs(i, m_Graphs.GetCount());
  }
}

//-----------------------------------------------------------------------------
// Enable or disable the tooltips in all graphs

void cGraphContainer::EnableToolTips(bool flag)
{
  for (size_t i = 0; i < m_Graphs.GetCount(); i++) {
    m_Graphs[i]->EnableToolTips(flag);
  }
}

//-----------------------------------------------------------------------------
// Translation

void cGraphContainer::TranslateTexts()
{
  cLang::TranslateMenu(&m_Context);

  for (size_t i = 0; i < m_Graphs.GetCount(); i++) {
    m_Graphs[i]->TranslateTexts();
  }
}
