#include "cgraph.h"
#include "cgraphview.h"
#include "cruler.h"
#include "cmgr.h"
#include <wx/dcclient.h>

// scroll speed: after SCROLL_STEPS clicks on the arrows, a full screen has scrolled by
#define SCROLL_STEPS 100

// number of scroll units per notch of the scroll wheel
#define WHEEL_SPEED 10

// maximum zoom factor
#define MAX_ZOOM 100

// minimum size of rubberband-box for zooming in
#define MIN_RBB_BOX_SIZE 10

//-----------------------------------------------------------------------------
// Event table

BEGIN_EVENT_TABLE(cGraph, wxWindow)
  EVT_IDLE(cGraph::OnIdle)
  EVT_KEY_DOWN(cGraph::OnKey)
  EVT_LEFT_DOWN(cGraph::OnMouseLeftClick)
  EVT_LEFT_UP(cGraph::OnMouseLeftRelease)
  EVT_LEFT_DCLICK(cGraph::OnMouseLeftDclick)
  EVT_MENU(wxID_ANY, cGraph::OnMenuClick)
  EVT_MIDDLE_DOWN(cGraph::OnMouseMiddleClick)
  EVT_MOTION(cGraph::OnMouseMove)
  EVT_MOUSEWHEEL(cGraph::OnMouseWheel)
  EVT_RIGHT_DOWN(cGraph::OnMouseRightClick)
  EVT_SIZE(cGraph::OnSize)
END_EVENT_TABLE()

//-----------------------------------------------------------------------------
// Static members

wxPen cGraph::s_GridPen(wxColour(LIGHT_GREY, LIGHT_GREY, LIGHT_GREY));
wxPen cGraph::s_AxisPen(wxColour(MEDIUM_GREY, MEDIUM_GREY, MEDIUM_GREY));

//-----------------------------------------------------------------------------

cGraph::cGraph(cGraphView* parent, graph_t type)
: wxWindow(parent, -1, wxDefaultPosition, wxDefaultSize,
    wxFULL_REPAINT_ON_RESIZE | wxSIMPLE_BORDER | wxWANTS_CHARS)
{
  m_Type = type;
  m_Parent = parent;
  m_RulerX = NULL;
  m_RulerY = NULL;

  m_ToolTips = false;
  m_ToolTip = new wxToolTip(wxEmptyString);
  SetToolTip(m_ToolTip);

  m_SelStart = 0.0f;
  m_SelEnd = 1.0f;
  m_CursorDist = -1.0f;

  m_Connect = true;
  m_ZoomX = 1.0f;
  m_ZoomY = 1.0f;
  m_ScaleX = 1.0f;
  m_ScaleY = 1.0f;

  m_FitDimensions = wxHORIZONTAL | wxVERTICAL;

  m_NeedToSync = false;

  m_CrossHairMode = false;
  m_CrossHairVisible = false;
  m_RbbMode = RBBMODE_NONE;

  cLang::AppendMenuItem(&m_Context, ID_MENU_FIT, _T("Fit"));
  cLang::AppendMenuItem(&m_Context, ID_MENU_ZOOM_RESET_Y, _T("Reset zoom Y"));
  cLang::AppendMenuItem(&m_Context, ID_MENU_ZOOM_RESET, _T("Reset zoom"));

  if (m_Type == GRAPHTYPE_Y) {
    m_Context.AppendSeparator();
    cLang::AppendMenuItem(&m_Context, ID_MENU_GR_EXPORT, _T("Export to CSV..."));
  }

  m_Context.AppendSeparator();
  cLang::AppendMenuItem(&m_Context, ID_MENU_MOVE_UP, _T("Move graph up"));
  cLang::AppendMenuItem(&m_Context, ID_MENU_MOVE_DOWN, _T("Move graph down"));
  m_Context.AppendSeparator();
  cLang::AppendMenuItem(&m_Context, ID_MENU_EQUAL_HEIGHT, _T("Equal height graphs"));
  cLang::AppendMenuItem(&m_Context, ID_MENU_CLOSE_GRAPH, _T("Close graph"));

  SetBackgroundColour(*wxWHITE);

  // make scrolling actions of the graph view work on this window
  m_Parent->SetTargetWindow(this);
}

//-----------------------------------------------------------------------------

cGraph::~cGraph()
{
  if (HasCapture()) ReleaseMouse();
}

//-----------------------------------------------------------------------------
// Set the child windows

void cGraph::SetRulerX(cRuler* ruler)
{
  m_RulerX = ruler;
  if (m_RulerX != NULL) m_RulerX->SetGraph(this);
}

void cGraph::SetRulerY(cRuler* ruler)
{
  m_RulerY = ruler;
  if (m_RulerY != NULL) m_RulerY->SetGraph(this);
}

//-----------------------------------------------------------------------------
// Re-calculate the bounding box

void cGraph::AdjustBbox()
{
  CHECK_THIS;

  if (m_Curves.IsEmpty()) {
    // no curves - set defaults
    m_MinX = 0.0f;
    m_MinY = 0.0f;
    m_MaxX = 1.0f;
    m_MaxY = 1.0f;
  }
  else {
    // set the bbox so that all curves fit in it
    m_MinX = m_Curves[0].GetMinX();
    m_MaxX = m_Curves[0].GetMaxX();
    m_MinY = m_Curves[0].GetMinY();
    m_MaxY = m_Curves[0].GetMaxY();

    for (size_t c = 1; c < m_Curves.GetCount(); c++) {
      ::Minimise(m_MinX, m_Curves[c].GetMinX());
      ::Maximise(m_MaxX, m_Curves[c].GetMaxX());
      ::Minimise(m_MinY, m_Curves[c].GetMinY());
      ::Maximise(m_MaxY, m_Curves[c].GetMaxY());
    }
  }

  wxASSERT(m_MinX <= m_MaxX);
  wxASSERT(m_MinY <= m_MaxY);

  // avoid bbox of zero size
  if (m_MinX == m_MaxX) {
    m_MinX -= 1.0f;
    m_MaxX += 1.0f;
  }
  if (m_MinY == m_MaxY) {
    m_MinY -= 1.0f;
    m_MaxY += 1.0f;
  }

  OnAdjustBbox();

  AdjustDimensions();
}

//-----------------------------------------------------------------------------

void cGraph::OnSize(wxSizeEvent& event)
{
  AdjustDimensions();
  event.Skip();
}

//-----------------------------------------------------------------------------
// Change the user zoom
// - factor = multiplication factor

void cGraph::ZoomX(float factor)
{
  wxASSERT(factor > 0);

  float orig = GetGraphOriginX();

  if (!SetZoomX(m_ZoomX * factor)) return;

  AdjustDimensions();
  SetGraphOriginX(orig);
  Refresh();
}

void cGraph::ZoomY(float factor)
{
  wxASSERT(factor > 0);

  float orig = GetGraphOriginY();

  if (!SetZoomY(m_ZoomY * factor)) return;

  AdjustDimensions();
  SetGraphOriginY(orig);
  Refresh();
}

//-----------------------------------------------------------------------------
// Reset the user zoom factor
// - dimension = which dimension to reset (wxHORIZONTAL and/or wxVERTICAL)

void cGraph::ZoomReset(int dimension)
{
  if ((m_ZoomX == 1.0f) && (m_ZoomY == 1.0f)) return;

  if (dimension & wxHORIZONTAL) m_ZoomX = 1.0f;
  if (dimension & wxVERTICAL) m_ZoomY = 1.0f;
  AdjustDimensions();
  Refresh();
}

//-----------------------------------------------------------------------------
// Helper functions to set the user zoom factor
// - zoom = new zoom
// Returns true if zoom factor changed

bool cGraph::SetZoomX(float zoom)
{
  if (zoom < 1.0f) zoom = 1.0f;
  if (zoom > MAX_ZOOM) zoom = MAX_ZOOM;
  if (zoom == m_ZoomX) return false;
  m_ZoomX = zoom;
  return true;
}


bool cGraph::SetZoomY(float zoom)
{
  if (zoom < 1.0f) zoom = 1.0f;
  if (zoom > MAX_ZOOM) zoom = MAX_ZOOM;
  if (zoom == m_ZoomY) return false;
  m_ZoomY = zoom;
  return true;
}

//-----------------------------------------------------------------------------
// Calculate the step for the grid and the tick count (X or Y dimension)
// - range = difference between min and max values
// - size = the window's size in this dimension
// - integer = are the values in this dimension integers?
// - grid = receives the grid step
// - tick = receives the tick count

void cGraph::CalcStep(float range, int size, bool integer, float& grid, int& tick)
{
  wxASSERT(range > 0);

  grid = ::FloorOneSD(range / 8.0f); // through this calculation, there will be between 8 and 15 grid squares

  tick = ::FirstSD(grid);
  if (tick == 1) tick = 10;

  // correction for small window size: less tick marks, so they won't 'touch' when plotted
  float pixPerSquare = (float)size / (range / grid); // number of pixels per grid square
  while (3 * tick > pixPerSquare) {
    switch (tick) {
      case 4 : case 6 : case 10 : tick = 2; break;
      case 8 :                    tick = 4; break;
      case 9 :                    tick = 3; break;
      default :                   tick = 1; break;
    }
    if (tick == 1) break; // reached bottom
  }

  // correction for integer values
  if (integer) {
    grid = floor(grid);
    if (grid < 1.0f) grid = 1.0f;
    if (tick > grid) tick = (int)grid;
  }
}

//-----------------------------------------------------------------------------
// Re-calculate the graph dimensioning

void cGraph::AdjustDimensions()
{
  int width, height;
  GetClientSize(&width, &height);
  if ((width == 0) || (height == 0)) return; // defensive

  m_ScaleX = width / (m_MaxX - m_MinX);
  m_ScaleY = height / (m_MaxY - m_MinY);
  SetVirtualSize((int)(width * m_ZoomX), (int)(height * m_ZoomY));

  int rateX = width / SCROLL_STEPS;
  if (rateX < 1) rateX = 1;
  int rateY = height / SCROLL_STEPS;
  if (rateY < 1) rateY = 1;
  m_Parent->SetScrollRate(rateX, rateY);

  GetClientSize(&width, &height);
  cGraph::CalcStep((m_MaxX - m_MinX) / m_ZoomX, width, IsIntegerX(), m_GridStepX, m_RulerTickX);
  cGraph::CalcStep((m_MaxY - m_MinY) / m_ZoomY, height, IsIntegerY(), m_GridStepY, m_RulerTickY);

  ClearToolTip();
}

//-----------------------------------------------------------------------------

void cGraph::OnKey(wxKeyEvent& event)
{
  switch (event.GetKeyCode()) {
    case WXK_ESCAPE :
      if (m_RbbMode == RBBMODE_NONE) {
        event.Skip();
        return;
      }
      // finish rubberband-drawing mode without action
      DrawRubberbandBox(); // un-draw the box
      if (HasCapture()) ReleaseMouse();
      m_RbbMode = RBBMODE_NONE;
      break;

    default:
      event.Skip();
      return;
  }
}

//-----------------------------------------------------------------------------

void cGraph::OnMouseWheel(wxMouseEvent& event)
{
  if (event.CmdDown()) {
    // Ctrl pressed - zoom graph
    float factor = (event.GetWheelRotation() > 0) ? GRAPH_ZOOM_FACTOR : (1.0f / GRAPH_ZOOM_FACTOR);
    if (!event.ShiftDown()) {
      ZoomX(factor);
      RequestSync(); // zoom the other graphs
    }
    else {
      ZoomY(factor);
    }
  }
  else {
    // Ctrl not pressed - scroll graph
    RemoveCrossHair();
    RemoveTrackCursor();
    int scrollX, scrollY;
    m_Parent->GetViewStart(&scrollX, &scrollY);
    int amount = (event.GetWheelRotation() > 0) ? (- WHEEL_SPEED) : WHEEL_SPEED;
    if (!event.ShiftDown()) {
      // X: scroll all graphs
      m_Parent->Scroll(scrollX + amount, -1);
      RequestSync();
    }
    else {
      // Y: scroll only this graph
      m_Parent->Scroll(-1, scrollY + amount);
    }
  }
}

//-----------------------------------------------------------------------------

void cGraph::OnMouseMiddleClick(wxMouseEvent& WXUNUSED(event))
{
  ZoomReset();
  RequestSync();
}

//-----------------------------------------------------------------------------

void cGraph::OnMouseMove(wxMouseEvent& event)
{
  // get mouse position
  wxClientDC dc(this);
  wxPoint pos = event.GetLogicalPosition(dc);

  // set tooltip
  if (m_ToolTips) {
    wxString valX = FormatValueX(Coord2ValueX(pos.x));
    wxString valY = FormatValueY(Coord2ValueY(pos.y));
    m_ToolTip->SetTip(wxString::Format(_T("x = %s  y = %s"), valX.c_str(), valY.c_str()));
  }

  if (!event.LeftIsDown()) return; // not drag-selecting

  if (m_RbbMode == RBBMODE_NONE) return; // not ready for drag-selecting

  if (m_RbbMode == RBBMODE_PREPARE) {
    // enter rubberband-drawing mode
    rbbmode_t newmode = GetRbbMode(event);
    wxASSERT(newmode != RBBMODE_PREPARE);
    if (newmode == RBBMODE_NONE) return; // graph does not allow drag-selecting

    RemoveCrossHair();
    CaptureMouse();
    m_RbbMode = newmode;
    m_RbbDestination = m_RbbOrigin;
    if (m_RbbMode == RBBMODE_XONLY) {
      // box spans full height of window
      m_RbbOrigin.y = GetClientSize().GetHeight();
      m_RbbDestination.y = 0;
    }
  }
  else {
    // already in rubberband-drawing mode
    DrawRubberbandBox(); // un-draw the old box

    // get new position
    m_RbbDestination = pos;
    if (m_RbbMode == RBBMODE_XONLY) m_RbbDestination.y = 0;
  }

  // draw the box at its new position
  DrawRubberbandBox();
}

//-----------------------------------------------------------------------------
// Handling rubberband mode

void cGraph::OnMouseLeftClick(wxMouseEvent& event)
{
  event.Skip(); // always pass left-clicks to the default event handler

  // prepare for rubberband-drawing mode
  m_RbbMode = RBBMODE_PREPARE;
  wxClientDC dc(this);
  m_RbbOrigin = event.GetLogicalPosition(dc);
}


void cGraph::DrawRubberbandBox()
{
  wxClientDC dc(this);
  dc.SetPen(*wxWHITE_PEN);
  dc.SetBrush(*wxTRANSPARENT_BRUSH);
  dc.SetLogicalFunction(wxINVERT);

  dc.DrawRectangle(m_RbbOrigin.x, m_RbbOrigin.y,
      m_RbbDestination.x - m_RbbOrigin.x,
      m_RbbDestination.y - m_RbbOrigin.y
      );
}


void cGraph::OnMouseLeftRelease(wxMouseEvent& event)
{
  if (m_RbbMode == RBBMODE_NONE) return; // no preceding click detected

  if (m_RbbMode == RBBMODE_PREPARE) {
    // click + release at same mouse position
    wxClientDC dc(this);
    OnMouseClickRelease(event.GetLogicalPosition(dc)); // handling depends on graph type
    m_RbbMode = RBBMODE_NONE;
    return;
  }

  // finish rubberband-drawing mode
  DrawRubberbandBox(); // un-draw the box
  if (HasCapture()) ReleaseMouse();
  m_RbbMode = RBBMODE_NONE;

  // limit box to window
  int winWidth, winHeight;
  GetClientSize(&winWidth, &winHeight);
  if (m_RbbDestination.x < 0) m_RbbDestination.x = 0;
  if (m_RbbDestination.x > winWidth) m_RbbDestination.x = winWidth;
  if (m_RbbDestination.y < 0) m_RbbDestination.y = 0;
  if (m_RbbDestination.y > winHeight) m_RbbDestination.y = winHeight;

  // get size of rubberband box
  wxCoord boxWidth = m_RbbDestination.x - m_RbbOrigin.x;
  wxCoord boxHeight = m_RbbDestination.y - m_RbbOrigin.y;

  if ((abs(boxWidth) < MIN_RBB_BOX_SIZE) || (abs(boxHeight) < MIN_RBB_BOX_SIZE)) {
    // box is too small - ignore
    return;
  }

  // zoom & scroll into the box
  float origX, origY; // values of bottom-left corner of box
  if (boxWidth < 0) {
    // dragged right-to-left
    boxWidth = -boxWidth;
    origX = Coord2ValueX(m_RbbDestination.x);
  }
  else {
    // dragged left-to-right
    origX = Coord2ValueX(m_RbbOrigin.x);
  }
  if (boxHeight < 0) {
    // box was dragged top-to-bottom
    boxHeight = -boxHeight;
    origY = Coord2ValueY(m_RbbOrigin.y);
  }
  else {
    origY = Coord2ValueY(m_RbbDestination.y);
  }

  SetZoomX(m_ZoomX * (float)winWidth / (float)boxWidth);
  SetZoomY(m_ZoomY * (float)winHeight / (float)boxHeight);
  AdjustDimensions();
  SetGraphOriginX(origX);
  SetGraphOriginY(origY);

  Refresh();
  RequestSync();
}

//-----------------------------------------------------------------------------
// Convert graph values to window coordinates and vice versa

wxCoord cGraph::Value2CoordX(float value)
{
  int logicalPosX = (int)((value - m_MinX) * (m_ScaleX * m_ZoomX));
  int devicePosX, devicePosY;
  m_Parent->CalcScrolledPosition(logicalPosX, 0, &devicePosX, &devicePosY);
  return devicePosX;
}


float cGraph::Coord2ValueX(wxCoord coord)
{
  int logicalPosX, logicalPosY;
  m_Parent->CalcUnscrolledPosition(coord, 0, &logicalPosX, &logicalPosY);
  return ((float)logicalPosX) / (m_ScaleX * m_ZoomX) + m_MinX;
}


wxCoord cGraph::Value2CoordY(float value)
{
  int logicalPosY = (int)((m_MaxY - value) * (m_ScaleY * m_ZoomY));
  int devicePosX, devicePosY;
  m_Parent->CalcScrolledPosition(0, logicalPosY, &devicePosX, &devicePosY);
  return devicePosY;
}


float cGraph::Coord2ValueY(wxCoord coord)
{
  int logicalPosX, logicalPosY;
  m_Parent->CalcUnscrolledPosition(0, coord, &logicalPosX, &logicalPosY);
  return m_MaxY - ((float)logicalPosY) / (m_ScaleY * m_ZoomY);
}

//-----------------------------------------------------------------------------
// Get/Set the origin (= graph values for the bottomleft corner of the visible part)

float cGraph::GetGraphOriginX()
{
  return Coord2ValueX(0);
}

float cGraph::GetGraphOriginY()
{
  return Coord2ValueY(GetClientSize().GetHeight());
}


void cGraph::SetGraphOriginX(float value)
{
  RemoveCrossHair();
  wxCoord coordX = Value2CoordX(value);

  int sppuX, sppuY;
  m_Parent->GetScrollPixelsPerUnit(&sppuX, &sppuY);

  int scrollX, scrollY;
  m_Parent->GetViewStart(&scrollX, &scrollY);
  scrollX += (coordX + sppuX / 2) / sppuX;
  m_Parent->Scroll(scrollX, scrollY);
}

void cGraph::SetGraphOriginY(float value)
{
  RemoveCrossHair();
  wxCoord coordY = Value2CoordY(value);

  int sppuX, sppuY;
  m_Parent->GetScrollPixelsPerUnit(&sppuX, &sppuY);

  int scrollX, scrollY;
  m_Parent->GetViewStart(&scrollX, &scrollY);
  scrollY += (coordY - GetClientSize().GetHeight() + sppuY / 2) / sppuY;
  m_Parent->Scroll(scrollX, scrollY);
}

//-----------------------------------------------------------------------------
// Draw the crosshair
// - pos = screen coordinate at which to draw the crosshair

void cGraph::DrawCrossHair(wxPoint pos)
{
  if (!m_CrossHairMode) return;
  if (!((m_RbbMode == RBBMODE_NONE) || (m_RbbMode == RBBMODE_PREPARE))){
    return; // ignore if there is also a rubberband box
  }

  wxClientDC dc(this);
  dc.SetPen(*wxWHITE_PEN);
  dc.SetLogicalFunction(wxINVERT);
  dc.CrossHair(pos.x, pos.y);

  m_CrossHairVisible = true;
  m_CrossHairPos = pos;
}

//-----------------------------------------------------------------------------
// Draw the crosshair at the current mouse position
// - pos = screen coordinate at which to draw the crosshair

void cGraph::DrawCrossHairAtPointer()
{
  wxPoint pos = ScreenToClient(wxGetMousePosition());
  if ((pos.x < 0) || (pos.y < 0) ||
      (pos.x > GetClientSize().GetWidth()) ||
      (pos.y > GetClientSize().GetHeight())) {
    // mouse is not over window
    return;
  }
  DrawCrossHair(pos);
}

//-----------------------------------------------------------------------------
// Remove the crosshair

void cGraph::RemoveCrossHair()
{
  if (!m_CrossHairVisible) return;
  DrawCrossHair(m_CrossHairPos);
  m_CrossHairVisible = false;
}

//-----------------------------------------------------------------------------
// Show or hide the crosshair

void cGraph::ShowCrossHair(bool show)
{
  if (!show) RemoveCrossHair();
  m_CrossHairMode = show;
}

//-----------------------------------------------------------------------------
// Check if sync'ing is needed

void cGraph::OnIdle(wxIdleEvent& WXUNUSED(event))
{
  if (!m_NeedToSync) return;
  m_NeedToSync = false;
  DoSync();
}

//-----------------------------------------------------------------------------

void cGraph::OnMouseRightClick(wxMouseEvent& WXUNUSED(event))
{
  if (HasCapture()) return; // don't show context menu while a drag is going on

  // count non-hidden curves
  int nonhidden = 0;
  for (size_t c = 0; c < m_Curves.GetCount(); c++) {
    if (m_Curves[c].IsShown()) nonhidden += 1;
  }

  // create context menu
  m_Context.Enable(ID_MENU_ZOOM_RESET, (m_ZoomX != 1.0f) || (m_ZoomY != 1.0f));
  m_Context.Enable(ID_MENU_ZOOM_RESET_Y, (m_ZoomY != 1.0f));
  m_Context.Enable(ID_MENU_FIT, (nonhidden > 0));
  if (m_Type == GRAPHTYPE_Y) m_Context.Enable(ID_MENU_GR_EXPORT, (nonhidden > 0));
  PopupMenu(&m_Context);
}

//-----------------------------------------------------------------------------
// Handle a click in the context menu

void cGraph::OnMenuClick(wxCommandEvent& event)
{
  switch (event.GetId()) {
    case ID_MENU_ZOOM_RESET :
      ZoomReset();
      RequestSync();
      break;

    case ID_MENU_ZOOM_RESET_Y :
      ZoomReset(wxVERTICAL);
      RequestSync();
      break;

    case ID_MENU_FIT :
      FitSelection();
      break;

    case ID_MENU_GR_EXPORT :
      ExportGraph();
      break;

    case ID_MENU_MOVE_UP :
      ::MoveGraph_Send(m_Parent, -1);
      break;

    case ID_MENU_MOVE_DOWN :
      ::MoveGraph_Send(m_Parent, 1);
      break;

    case ID_MENU_EQUAL_HEIGHT :
      ::EqualHeightGraphs_Send();
      break;

    case ID_MENU_CLOSE_GRAPH :
      ::CloseView_Send(m_Parent);
      break;

    default :
      wxFAIL;
  }
}

//-----------------------------------------------------------------------------
// zoom & scroll Y so the selected part fits

void cGraph::FitSelection()
{
  CHECK_THIS;
  if (m_Curves.IsEmpty()) return; // empty display

  // get the minimum and maximum Y values in the selected part of the track
  float minX = IMPOSSIBLY_HIGH_VALUE;
  float maxX = IMPOSSIBLY_LOW_VALUE;
  float minY = IMPOSSIBLY_HIGH_VALUE;
  float maxY = IMPOSSIBLY_LOW_VALUE;
  // scan all curves
  for (size_t c = 0; c < m_Curves.GetCount(); c++) {
    if (!m_Curves[c].IsShown()) continue; // curve is hidden
    float cminX, cmaxX, cminY, cmaxY;
    m_Curves[c].GetBbox(GetSelectionStart(), GetSelectionEnd(), cminX, cmaxX, cminY, cmaxY);
    if (cminX < minX) minX = cminX;
    if (cmaxX > maxX) maxX = cmaxX;
    if (cminY < minY) minY = cminY;
    if (cmaxY > maxY) maxY = cmaxY;
  }

  if (maxY < minY) return; // all curves were hidden

  // avoid zero-size bounding box
  if (maxX - minX < EPSILON) {
    minX -= 1.0f;
    maxX += 1.0f;
  }
  if (maxY - minY < EPSILON) {
    minY -= 1.0f;
    maxY += 1.0f;
  }

  // zoom & scroll
  if (m_FitDimensions & wxHORIZONTAL) SetZoomX((m_MaxX - m_MinX) / (maxX - minX));
  if (m_FitDimensions & wxVERTICAL) SetZoomY((m_MaxY - m_MinY) / (maxY - minY));
  AdjustDimensions();

  if (m_FitDimensions & wxHORIZONTAL) SetGraphOriginX(minX);
  if (m_FitDimensions & wxVERTICAL) SetGraphOriginY(minY);
  Refresh();
  RequestSync();
}

//-----------------------------------------------------------------------------

void cGraph::OnMouseLeftDclick(wxMouseEvent& WXUNUSED(event))
{
  ZoomReset();
  RequestSync();
}

//-----------------------------------------------------------------------------

void cGraph::UpdateAll()
{
  CHECK_THIS;

  // update display
  AdjustBbox();
  Refresh();
}

//-----------------------------------------------------------------------------
// Draw or remove the track cursor

void cGraph::DrawTrackCursor(bool scroll)
{
  wxPoint pos = ScreenToClient(wxGetMousePosition());
  ::SetTrackCursorPos_Send(Coord2ValueX(pos.x), scroll);
}

void cGraph::RemoveTrackCursor()
{
  ::SetTrackCursorPos_Send(-1, false);
}

//-----------------------------------------------------------------------------
// Set the "connect" attirbute

void cGraph::SetConnect(bool connect)
{
  m_Connect = connect;

  // pass the setting to all curves
  for (size_t c = 0; c < m_Curves.GetCount(); c++) {
    m_Curves[c].SetConnect(m_Connect);
  }

  Refresh();
}

//-----------------------------------------------------------------------------
// Tell the graph what its position is

void cGraph::GraphPositionIs(int index, int total)
{
  wxASSERT(index >= 0);
  wxASSERT(total >= 1);
  wxASSERT(index < total);

  m_Context.Enable(ID_MENU_MOVE_UP, (index > 0));
  m_Context.Enable(ID_MENU_MOVE_DOWN, (index < total - 1));
  m_Context.Enable(ID_MENU_EQUAL_HEIGHT, (total > 1));
  m_Context.Enable(ID_MENU_CLOSE_GRAPH, (total > 1));
}

//-----------------------------------------------------------------------------
// Find the curve for a certain lap (wxNOT_FOUND if none)

int cGraph::FindCurve(cLap* lap) const
{
  for (size_t c = 0; c < m_Curves.GetCount(); c++) {
    if (m_Curves[c].GetLap() == lap) return c;
  }

  return wxNOT_FOUND;
}

//-----------------------------------------------------------------------------
// Print a graph value with the correct number of decimals

wxString cGraph::FormatValueX(float val)
{
  wxString format;
  int decimals = 0;
  if (!IsIntegerX()) {
    int power = (int)(floor(log10(m_GridStepX)));
    if (power < 0) decimals = -power;
    if (power < 3) decimals += 1;
  }
  format.Printf(_T("%%.%df"), decimals);

  return wxString::Format(format, val);
}

wxString cGraph::FormatValueY(float val)
{
  wxString format;
  int decimals = 0;
  if (!IsIntegerY()) {
    int power = (int)(floor(log10(m_GridStepY)));
    if (power < 0) decimals = -power;
    if (power < 3) decimals += 1;
  }
  format.Printf(_T("%%.%df"), decimals);

  return wxString::Format(format, val);
}

//-----------------------------------------------------------------------------
// Enable or disable the tooltips

void cGraph::EnableToolTips(bool flag)
{
  m_ToolTips = flag;
  if (!m_ToolTips) ClearToolTip();
  Refresh();
}
