#include "cgraph_xy.h"
#include "cgraphview.h"
#include "ccurve_xy.h"
#include "cruler.h"
#include "cmgr.h"
#include <wx/dcclient.h>

// diameter of the track cursor
#define CURSOR_DIAMETER 3

//-----------------------------------------------------------------------------
// Event table

BEGIN_EVENT_TABLE(cGraphXY, cGraph)
  EVT_MOTION(cGraphXY::OnMouseMove)
  EVT_ENTER_WINDOW(cGraphXY::OnMouseEntering)
  EVT_LEAVE_WINDOW(cGraphXY::OnMouseLeaving)
  EVT_PAINT(cGraphXY::OnPaint)
END_EVENT_TABLE()

//-----------------------------------------------------------------------------

cGraphXY::cGraphXY(cGraphView* parent)
: cGraph(parent, GRAPHTYPE_XY)
{
  m_LogtypeX = LOGTYPE_FIRST;
  m_LogtypeY = LOGTYPE_FIRST;
  m_Wheel = WHEELTYPE_FIRST;

  AdjustBbox(); // init bounding box
}

//-----------------------------------------------------------------------------

cGraphXY::~cGraphXY()
{
}

//-----------------------------------------------------------------------------
// Get the rbb mode to start select-dragging (RBBMODE_NONE = veto)
// - event = event that signals start of drag

rbbmode_t cGraphXY::GetRbbMode(wxMouseEvent& WXUNUSED(event))
{
  return RBBMODE_BOTH;
}

//-----------------------------------------------------------------------------
// Add a curve to the graph
// - lap = lap for which to add a curve

void cGraphXY::AddCurve(cLap* lap)
{
  // add the curve
  cCurveXY* curve = new cCurveXY(lap);
  curve->SetLogtype(m_LogtypeX, m_LogtypeY, m_Wheel);
  curve->SetConnect(m_Connect);
  m_Curves.Add(curve);

  // update display
  SetZoomX(1.0f);
  SetZoomY(1.0f);
  AdjustBbox();
  Refresh();
}

//-----------------------------------------------------------------------------
// Delete a curve
// - lap = lap whose curve is to be deleted

void cGraphXY::DeleteCurve(cLap* lap)
{
  // find the curve for this lap
  for (size_t c = 0; c < m_Curves.GetCount(); c++) {
    if (m_Curves[c].GetLap() == lap) {
      // found it
      m_Curves.RemoveAt(c);
      AdjustBbox();
      Refresh();
      return;
    }
  }
}

//-----------------------------------------------------------------------------
// Delete all curves

void cGraphXY::DeleteAllCurves()
{
  m_Curves.Clear();
  m_SelStart = 0.0f;
  m_SelEnd = 1.0f;

  ZoomReset();
  AdjustBbox();
}

//-----------------------------------------------------------------------------
// Plot the graph

void cGraphXY::OnPaint(wxPaintEvent& WXUNUSED(event))
{
  wxPaintDC dc(this);

  int screenX, screenY;
  dc.GetSize(&screenX, &screenY);
  if ((screenX <= 0) || (screenY <= 0)) return;

  // plot the grid
  float value;
  wxCoord coord;
  // X grid (vertical lines)
  value = GetGridStartX();
  value -= fmod(value, m_GridStepX);
  value -= m_GridStepX;
  coord = 0;
  while (coord <= screenX) {
    if (fabs(value) < EPSILON) {
      dc.SetPen(cGraph::s_AxisPen);
    }
    else {
      dc.SetPen(cGraph::s_GridPen);
    }
    coord = Value2CoordX(value);
    dc.DrawLine(coord, 0, coord, screenY);
    value += m_GridStepX;
  }
  // Y grid (horizontal lines)
  value = GetGridEndY();
  value -= fmod(value, m_GridStepY);
  value += m_GridStepY;
  coord = 0;
  while (coord <= screenY) {
    if (fabs(value) < EPSILON) {
      dc.SetPen(cGraph::s_AxisPen);
    }
    else {
      dc.SetPen(cGraph::s_GridPen);
    }
    coord = Value2CoordY(value);
    dc.DrawLine(0, coord, screenX, coord);
    value -= m_GridStepY;
  }

  // plot each curve
  for (size_t c = 0; c < m_Curves.GetCount(); c++) {
    m_Curves[c].Plot(dc, *this, m_SelStart, m_SelEnd);

    // plot the cursor
    if (m_CursorDist >= 0.0) {
      float x, y;
      m_Curves[c].GetValueAt(m_CursorDist, x, y);
      if (m_Curves[c].IsShown()) {
        dc.SetPen(m_Curves[c].GetPen());
        dc.SetBrush(wxBrush(m_Curves[c].GetLap()->GetColour()));
        dc.DrawCircle(Value2CoordX(x), Value2CoordY(y), CURSOR_DIAMETER);
      }
    }
  }

  // update the rulers
  if (m_RulerX != NULL) m_RulerX->Refresh();
  if (m_RulerY != NULL) m_RulerY->Refresh();

  // plot the crosshair (if it was present)
  if (m_CrossHairVisible) {
    dc.SetPen(*wxBLACK_PEN);
    dc.CrossHair(m_CrossHairPos.x, m_CrossHairPos.y);
  }
}

//-----------------------------------------------------------------------------
// Set the type of data that is shown
// - log_x, log_y = type of log-data for X and Y dimension
// - wheel = index of the wheel for which the data must be shown

void cGraphXY::SetLogtype(int log_x, int log_y, int wheel)
{
  wxASSERT(log_x >= LOGTYPE_FIRST);
  wxASSERT(log_x < LOGTYPE_LAST);
  wxASSERT(log_y >= LOGTYPE_FIRST);
  wxASSERT(log_y < LOGTYPE_LAST);
  wxASSERT(wheel >= WHEELTYPE_FIRST);
  wxASSERT(wheel < WHEELTYPE_LAST);

  if ((m_LogtypeX == log_x) && (m_LogtypeY == log_y) && (m_Wheel == wheel)) return; // no change
  m_LogtypeX = log_x;
  m_LogtypeY = log_y;
  m_Wheel = wheel;

  // set it for each curve
  for (size_t c = 0; c < m_Curves.GetCount(); c++) {
    ((cCurveXY&)m_Curves[c]).SetLogtype(log_x, log_y, wheel);
  }

  // update display
  ZoomReset();
  AdjustBbox();
  Refresh();
}

//-----------------------------------------------------------------------------
// Set the visible ('selected') part of the lap

void cGraphXY::DoSetTrackSelection(float start, float end)
{
  m_SelStart = start;
  m_SelEnd = end;

  AdjustBbox();
  Refresh();
}

//-----------------------------------------------------------------------------
// Draw the cursor at a new position
// - distance = distance_in_lap (in m) that corresponds to the cursor position
// - scroll = scroll if needed to get the cursor in the visible part
// If distance < 0 then cursor is removed

void cGraphXY::DoSetTrackCursorPos(cGraphView* WXUNUSED(view), float distance, bool scroll)
{
  if (MGR->IsEmpty()) return; // empty display
  if (!scroll) return; // ignore "fast movements" of cursor

  // draw the cursor at its new position
  PutTrackCursor(distance);
}

//-----------------------------------------------------------------------------
// Loading and saving the configuration settings

wxString cGraphXY::SaveSettings() const
{
  wxString settings;

  settings += wxString::Format(_T("data_x=%s;"), cCarState::GetLogtypeCode(GetLogtypeX()).c_str());
  settings += wxString::Format(_T("data_y=%s;"), cCarState::GetLogtypeCode(GetLogtypeY()).c_str());
  settings += wxString::Format(_T("wheel=%s;"), cCarState::GetWheeltypeCode(GetWheeltype()).c_str());
  settings += wxString::Format(_T("connect=%d;"), (m_Connect) ? 1: 0);

  return settings;
}


void cGraphXY::LoadSettings(const wxString& settings)
{
  // get current settings
  int data_x = GetLogtypeX();
  int data_y = GetLogtypeY();
  int wheel = GetWheeltype();
  bool connect = m_Connect;

  // parse new settings
  wxArrayString keys;
  wxArrayString values;
  ::DecodeKeyValueList(settings, keys, values);
  for (size_t k = 0; k < keys.GetCount(); k++) {
    int val = wxAtoi(values[k]);
    if (keys[k] == _T("data_x")) {
      data_x = cCarState::FindLogtypeCode(values[k]);
    }
    else if (keys[k] == _T("data_y")) {
      data_y = cCarState::FindLogtypeCode(values[k]);
    }
    else if (keys[k] == _T("wheel")) {
      wheel = cCarState::FindWheeltypeCode(values[k]);
    }
    else if (keys[k] == _T("connect")) {
      connect = (val != 0);
    }
  }
  // fallback to current settings if new ones incorrect
  if (data_x == wxNOT_FOUND) data_x = GetLogtypeX();
  if (data_y == wxNOT_FOUND) data_y = GetLogtypeY();
  if (wheel == wxNOT_FOUND) wheel = GetWheeltype();

  // put new settings
  SetConnect(connect);
  SetLogtype(data_x, data_y, wheel);
  UpdateAll();
}

//-----------------------------------------------------------------------------
// Draw the cursors at a new position
// - distance = distance-in-lap for cursor position

void cGraphXY::PutTrackCursor(float distance)
{
  if (MGR->IsEmpty()) return; // nothing to do

  RemoveCrossHair();
  if (m_CursorDist >= 0.0f) RefreshRect(GetTrackCursorPosition()); // redraw lines at old position
  m_CursorDist = distance;
  if (m_CursorDist >= 0.0f) RefreshRect(GetTrackCursorPosition()); // draw lines at new position
}

//-----------------------------------------------------------------------------
// Get a bounding box for the cursor positions

wxRect cGraphXY::GetTrackCursorPosition()
{
  wxASSERT(!(MGR->IsEmpty()));
  wxASSERT(m_CursorDist >= 0.0f);

  wxCoord minX = 0;
  wxCoord maxX = 0;
  wxCoord minY = 0;
  wxCoord maxY = 0;
  for (size_t i = 0; i < m_Curves.GetCount(); i++) {
    float x, y;
    m_Curves[i].GetValueAt(m_CursorDist, x, y);
    wxCoord cursorX = Value2CoordX(x);
    wxCoord cursorY = Value2CoordY(y);

    if (i == 0) {
      minX = cursorX;
      maxX = cursorX;
      minY = cursorY;
      maxY = cursorY;
    }
    else {
      if (minX > cursorX) minX = cursorX;
      if (maxX < cursorX) maxX = cursorX;
      if (minY > cursorY) minY = cursorY;
      if (maxY < cursorY) maxY = cursorY;
    }
  }

  return wxRect(
      minX - CURSOR_DIAMETER,
      minY - CURSOR_DIAMETER,
      maxX - minX + 2 * CURSOR_DIAMETER,
      maxY - minY + 2 * CURSOR_DIAMETER);
}

//-----------------------------------------------------------------------------
// Handle a left-click + release at the same position
// - pos = logical position of mouse

void cGraphXY::OnMouseClickRelease(wxPoint pos)
{
  ::SetTrackCursorPos_Send(DecodeMouse(pos), true);
}

//-----------------------------------------------------------------------------
// Return the distance-in-lap that corresponds to a mouse position

float cGraphXY::DecodeMouse(wxPoint pos)
{
  if (MGR->IsEmpty()) return -1;

  // find state that is nearest to this position
  float nearest_dist = -1; // distance-in-lap of state that is nearest to mouse position
  float nearest_pix = IMPOSSIBLY_HIGH_VALUE; // square of distance in pixels to that state

  // scan all laps
  for (size_t i = 0; i < MGR->GetLapCount(); i++) {
    cLap* lap = MGR->GetLap(i);
    // scan the states
    for (size_t s = 0; s < lap->GetStateCount(); s++) {
      const cCarState* state = lap->GetState(s);
      if (state->GetDistance() < m_SelStart) continue;
      if (state->GetDistance() > m_SelEnd) break;

      float dX = Value2CoordX(state->GetLogData(m_LogtypeX, m_Wheel)) - pos.x;
      float dY = Value2CoordY(state->GetLogData(m_LogtypeY, m_Wheel)) - pos.y;
      float d = (dX * dX) + (dY * dY);
      if (d < nearest_pix) {
        nearest_pix = d;
        nearest_dist = state->GetDistance();
      }
    }
  }

  if (sqrt(nearest_pix) > CLICK_DETECT_THRESHOLD) return -1;

  return nearest_dist;
}

//-----------------------------------------------------------------------------

void cGraphXY::OnMouseMove(wxMouseEvent& event)
{
  event.Skip(); // always pass event to parent class (for rbb handling)

  // move the crosshair
  RemoveCrossHair();
  DrawCrossHairAtPointer();
}

//-----------------------------------------------------------------------------

void cGraphXY::OnMouseEntering(wxMouseEvent& WXUNUSED(event))
{
  DrawCrossHairAtPointer();
}

void cGraphXY::OnMouseLeaving(wxMouseEvent& WXUNUSED(event))
{
  RemoveCrossHair();
}
