#include "cruler.h"
#include "cgraph.h"
#include <wx/dcclient.h>
#include <wx/sizer.h>

// length in pixels of the ticks
#define LARGE_TICK_SIZE 6
#define MEDIUM_TICK_SIZE 4
#define SMALL_TICK_SIZE 2

// margin between label and tick
#define LABEL_MARGIN_X 0
#define LABEL_MARGIN_Y 2

//-----------------------------------------------------------------------------
// Event table

BEGIN_EVENT_TABLE(cRuler, wxWindow)
  EVT_ENTER_WINDOW(cRuler::OnMouseEvent)
  EVT_LEAVE_WINDOW(cRuler::OnMouseEvent)
  EVT_MOTION(cRuler::OnMouseEvent)
  EVT_PAINT(cRuler::OnPaint)
END_EVENT_TABLE()

//-----------------------------------------------------------------------------

cRuler::cRuler(wxWindow* parent, int side)
: wxWindow(parent, -1, wxDefaultPosition, wxDefaultSize, wxNO_BORDER)
{
  wxASSERT((side == wxLEFT) || (side == wxBOTTOM));

  m_Side = side;
  m_Graph = NULL;
  m_ToolTips = false;

  SetBackgroundColour(parent->GetBackgroundColour());

  // determine minimum window size
  wxClientDC dc(this);
  dc.SetFont(*wxNORMAL_FONT);
  wxCoord sizeX, sizeY;
  dc.GetTextExtent(LARGEST_VALUE_TEXT, &sizeX, &sizeY); // max label size
  switch (side) {
    case wxBOTTOM : // X axis
      SetMinSize(wxSize(sizeX, sizeY + LARGE_TICK_SIZE + LABEL_MARGIN_X));
      break;

    case wxLEFT : // Y axis
      SetMinSize(wxSize(sizeX + LARGE_TICK_SIZE + LABEL_MARGIN_Y, sizeY));
      break;
  }
}

//-----------------------------------------------------------------------------

cRuler::~cRuler()
{
}

//-----------------------------------------------------------------------------

void cRuler::SetGraph(cGraph* graph)
{
  wxASSERT((m_Graph == NULL) || (graph != NULL));
  m_Graph = graph;
}

//-----------------------------------------------------------------------------

void cRuler::OnPaint(wxPaintEvent& WXUNUSED(event))
{
  CHECK_PTR_NULL(m_Graph);

  wxPaintDC dc(this);
  dc.SetPen(*wxBLACK_PEN);
  dc.SetFont(*wxNORMAL_FONT);

  if (m_Graph == NULL) return; // defensive

  int screenX, screenY;
  dc.GetSize(&screenX, &screenY);
  if ((screenX <= 0) || (screenY <= 0)) return;

  // determine label format
  float step;
  if (m_Side == wxBOTTOM) {
    step = m_Graph->GetGridStepX();
  }
  else {
    step = m_Graph->GetGridStepY();
  }
  wxString labelFormat;
  int power = (int)floor(log10(step));
  labelFormat.Printf(_T("%%.%df"), (power < 0) ? (-power) : 0);

  // plot the stuff
  wxString label;
  wxString units;
  wxCoord coord = 0;        // coordinate at which to plot next label
  wxCoord gridCoord;        // coordinate where grid line is
  wxCoord tickCoord;        // coordinate where tick mark is
  wxCoord lastLabelEnd = 0; // end coordinate of last label plotted
  if (m_Side == wxBOTTOM) {
    // X axis
    float value = m_Graph->GetGridStartX();
    value -= fmod(value, step);
    value -= step;
    while (coord <= screenX) {
      if (fabs(value) < EPSILON) value = 0.0f; // prevents value being printed as "-0.0"
      label.Printf(labelFormat, value);
      wxCoord width, height;
      dc.GetTextExtent(label, &width, &height);

      gridCoord = m_Graph->Value2CoordX(value) + 1;
      coord = gridCoord - width / 2 + 1;

      // plot the tick marks
      int ticks = m_Graph->GetRulerTicksX();
      for (int t = 0; t < ticks; t++) {
        tickCoord = m_Graph->Value2CoordX(value + step * t / ticks) + 1;
        wxCoord size = cRuler::GetTickSize(t, ticks);
        dc.DrawLine(tickCoord, 0, tickCoord, size);
      }

      value += step; // prepare for next

      // if part of label is outside ruler limits, but value is inside, then pull label inside
      if (coord < 0) {
        if (gridCoord < 0) continue; // grid line not plotted, so skip plotting label
        coord = 0;
      }
      if (coord + width > screenX) {
        if (gridCoord > screenX) continue;
        coord = screenX - width;
      }

      // avoid overlapping labels
      if (coord < lastLabelEnd) continue;

      // plot label
      dc.DrawText(label, coord, LARGE_TICK_SIZE + LABEL_MARGIN_X);
      lastLabelEnd = coord + width;
    }

    units = m_Graph->GetUnitsX();
  }
  else {
    // Y axis
    float value = m_Graph->GetGridEndY();
    value -= fmod(value, step);
    value += step;
    while (coord <= screenY) {
      if (fabs(value) < EPSILON) value = 0.0f; // prevents value being printed as "-0.0"
      label.Printf(labelFormat, value);
      wxCoord width, height;
      dc.GetTextExtent(label, &width, &height);

      gridCoord = m_Graph->Value2CoordY(value) + 1;
      coord = gridCoord - height / 2;

      // plot the tick marks
      int ticks = m_Graph->GetRulerTicksY();
      for (int t = 0; t < ticks; t++) {
        tickCoord = m_Graph->Value2CoordY(value + step * t / ticks) + 1;
        wxCoord size = cRuler::GetTickSize(t, ticks);
        dc.DrawLine(screenX - size, tickCoord, screenX, tickCoord);
      }

      value -= step; // prepare for next

      // if part of label is outside ruler limits, but value is inside, then pull label inside
      if (coord < 0) {
        if (gridCoord < 0) continue;
        coord = 0;
      }
      if (coord + height > screenY) {
        if (gridCoord > screenY) continue;
        coord = screenY - height;
      }

      // avoid overlapping labels
      if (coord < lastLabelEnd) continue;

      // plot label
      dc.DrawText(label, screenX - width - LARGE_TICK_SIZE - LABEL_MARGIN_Y, coord);
      lastLabelEnd = coord + height;
    }

    units = m_Graph->GetUnitsY();
  }

  // set tooltip
  if (m_ToolTips) {
    wxString tip = _TT(ID_TXT_RL_UNITS, "units");
    if (units.IsEmpty()) {
      SetToolTip(tip + _T(": -"));
    }
    else {
      SetToolTip(tip + _T(": ") + units);
    }
  }
}

//-----------------------------------------------------------------------------
// Get the size of a tick mark
// - index = number of this tick mark (0 = grid line)
// - count = number of tick marks between grid lines

wxCoord cRuler::GetTickSize(int index, int count)
{
  wxASSERT(index >= 0);
  wxASSERT(index < count);
  wxASSERT(count <= 10);

  // first tick mark is always large
  if (index == 0) return LARGE_TICK_SIZE;

  // with 2 tick marks: the remaining one is always small
  if (count == 2) return SMALL_TICK_SIZE;

  // with 9 tick marks: 3rd and 6th tick marks are medium
  if (count == 9) return (index %3 == 0) ? MEDIUM_TICK_SIZE : SMALL_TICK_SIZE;

  // otherwise: one medium tick mark halfway (so if count is odd, all ticks are small)
  return (count == 2 * index) ? MEDIUM_TICK_SIZE : SMALL_TICK_SIZE;
}

//-----------------------------------------------------------------------------
// Pass some types of mouse event to the graph,
// to enable plotting the crosshair when the mouse is above the X ruler

void cRuler::OnMouseEvent(wxMouseEvent& event)
{
  // only pass event if this is the X ruler
  if ((m_Graph != NULL) && (m_Side == wxBOTTOM)) {
    m_Graph->GetEventHandler()->ProcessEvent(event);
  }
}

//-----------------------------------------------------------------------------
// Enable or disable the tooltips

void cRuler::EnableToolTips(bool flag)
{
  m_ToolTips = flag;
  if (!m_ToolTips) SetToolTip(wxEmptyString);
}
