using System;
using System.IO;
using System.Collections.Generic;
using System.Text;
using System.Windows.Forms;
using System.Diagnostics;
using System.Threading;
using LFS_External.InSim;

namespace LFSRestrictions
{
    public class Config
    {
        private Dictionary<string, string> keyValues
            = new Dictionary<string, string>();

        public Config() { }

        public Config(string path)
        {
            Load(path);
        }

        public void Load(string path)
        {
            using (StreamReader sr = new StreamReader(path))
            {
                string line = null;
                while ((line = sr.ReadLine()) != null)
                {
                    try
                    {
                        string[] kv = line.Split('=');
                        keyValues.Add(kv[0], kv[1]);
                    }
                    catch
                    {
                    }
                }
            }
        }

        public void Save(string path)
        {
            using (StreamWriter sw = new StreamWriter(path))
            {
                foreach (KeyValuePair<string, string> keyValue in keyValues)
                {
                    sw.WriteLine(
                        "{0}={1}",
                        keyValue.Key,
                        keyValue.Value);
                }
            }
        }

        public void Set(string key, string value)
        {
            if (keyValues.ContainsKey(key))
                keyValues[key] = value;
            else
                keyValues.Add(key, value);
        }

        public void Set(string key, bool value)
        {
            Set(key, value.ToString());
        }

        public void Set(string key, ushort value)
        {
            Set(key, value.ToString());
        }

        public void Set(string key, int value)
        {
            Set(key, value.ToString());
        }

        public string GetString(string key)
        {
            return keyValues[key];
        }

        public int GetInt32(string key)
        {
            return Int32.Parse(GetString(key));
        }

        public ushort GetUShort(string key)
        {
            return ushort.Parse(GetString(key));
        }

        public bool GetBool(string key)
        {
            return Boolean.Parse(GetString(key));
        }
    }

    public partial class LFSCarRestrictor : Form
    {
        // InSim objects
        InSimInterface InSim;
        ConnectionSettings cfg;

        // These are the main lists that contain all Players and Connections (Being maintained automatically)
        List<clsConnection> Connections = new List<clsConnection>();
        List<clsPlayer> Players = new List<clsPlayer>();

        public LFSCarRestrictor()
        {
            InitializeComponent();
            InitializeInSimEvents(); // Event handlers (Uncoment a line in here to enable a packet receive event)
        }

		// Form load
        private void LFSCarRestrictor_Load(object sender, EventArgs e)
		{
			try
			{
                // The path to the config file.
                string path = Environment.CurrentDirectory + "\\Restrictions.txt";

                Config config = new Config();
                config.Load(path);

                string IP = config.GetString("IP");
                ushort InSimPort = config.GetUShort("InSimPort");
                string Password = config.GetString("Password");
                string Name = config.GetString("Name");

                // Connect to InSim
				cfg = new ConnectionSettings(IP, InSimPort, 0, Flags.InSimFlags.ISF_MCI | Flags.InSimFlags.ISF_MSO_COLS, '!', 1000, Password, "InSim");
				InSim = new InSimInterface(cfg);
				InSim.Connect();

                // Leave this
				Thread.Sleep(500);

				// Request all players and connections
				InSim.Send_TINY(Enums.Tiny.TINY_NPL, 255);
				InSim.Send_TINY(Enums.Tiny.TINY_NCN, 255);

                InSim.Send_MST_Message("/msg IP: " + IP);
                InSim.Send_MST_Message("/msg InSimPort: " + InSimPort);

                InSim.Send_MST_Message("/msg " + Name + " running!");
            }
			catch (Exception ex)
			{
				Debug.WriteLine(ex.Message);
			}
		}

		// Close button
		private void btnClose_Click(object sender, EventArgs e)
		{
			InSim.Close();
			Application.Exit();
		}

        #region Default Methods
        // Uncomment a line in this method to enable its event.
        private void InitializeInSimEvents()
        {
            //LFS_External.InSim.Events.AXC_Received += new LFS_External.InSim.Events.AXC_EventHandler(AXC);
            //LFS_External.InSim.Events.AXI_Received += new LFS_External.InSim.Events.AXI_EventHandler(AXI);
            //LFS_External.InSim.Events.AXO_Received += new LFS_External.InSim.Events.AXO_EventHandler(AXO);
            //LFS_External.InSim.Events.BTC_Received += new LFS_External.InSim.Events.BTC_EventHandler(BTC);
            //LFS_External.InSim.Events.BTT_Received += new LFS_External.InSim.Events.BTT_EventHandler(BTT);
            //LFS_External.InSim.Events.BFN_Received += new LFS_External.InSim.Events.BFN_EventHandler(BFN);
            //LFS_External.InSim.Events.CCH_Received += new LFS_External.InSim.Events.CCH_EventHandler(CCH);
            //LFS_External.InSim.Events.CLR_Received += new LFS_External.InSim.Events.CLR_EventHandler(CLR);
            LFS_External.InSim.Events.CNL_Received += new LFS_External.InSim.Events.CNL_EventHandler(CNL);
            //LFS_External.InSim.Events.CPP_Received += new LFS_External.InSim.Events.CPP_EventHandler(CPP);
            LFS_External.InSim.Events.CPR_Received += new LFS_External.InSim.Events.CPR_EventHandler(CPR);
            //LFS_External.InSim.Events.CRS_Received += new LFS_External.InSim.Events.CRS_EventHandler(CRS);
            //LFS_External.InSim.Events.FIN_Received += new LFS_External.InSim.Events.FIN_EventHandler(FIN);
            //LFS_External.InSim.Events.FLG_Received += new LFS_External.InSim.Events.FLG_EventHandler(FLG);
            //LFS_External.InSim.Events.III_Received += new LFS_External.InSim.Events.III_EventHandler(III);
            //LFS_External.InSim.Events.ISM_Received += new LFS_External.InSim.Events.ISM_EventHandler(ISM);
            //LFS_External.InSim.Events.LAP_Received += new LFS_External.InSim.Events.LAP_EventHandler(LAP);
            //LFS_External.InSim.Events.MCI_Received += new LFS_External.InSim.Events.MCI_EventHandler(MCI);
            //LFS_External.InSim.Events.MPE_Received += new LFS_External.InSim.Events.MPE_EventHandler(MPE);
            LFS_External.InSim.Events.MSO_Received += new LFS_External.InSim.Events.MSO_EventHandler(MSO);
            LFS_External.InSim.Events.NCN_Received += new LFS_External.InSim.Events.NCN_EventHandler(NCN);
            //LFS_External.InSim.Events.NLP_Received += new LFS_External.InSim.Events.NLP_EventHandler(NLP);
            LFS_External.InSim.Events.NPL_Received += new LFS_External.InSim.Events.NPL_EventHandler(NPL);
            //LFS_External.InSim.Events.PEN_Received += new LFS_External.InSim.Events.PEN_EventHandler(PEN);
            //LFS_External.InSim.Events.PFL_Received += new LFS_External.InSim.Events.PFL_EventHandler(PFL);
            //LFS_External.InSim.Events.PIT_Received += new LFS_External.InSim.Events.PIT_EventHandler(PIT);
            //LFS_External.InSim.Events.PLA_Received += new LFS_External.InSim.Events.PLA_EventHandler(PLA);
            LFS_External.InSim.Events.PLL_Received += new LFS_External.InSim.Events.PLL_EventHandler(PLL);
            //LFS_External.InSim.Events.PLP_Received += new LFS_External.InSim.Events.PLP_EventHandler(PLP);
            //LFS_External.InSim.Events.PSF_Received += new LFS_External.InSim.Events.PSF_EventHandler(PSF);
            //LFS_External.InSim.Events.REN_Received += new LFS_External.InSim.Events.REN_EventHandler(REN);
            //LFS_External.InSim.Events.REO_Received += new LFS_External.InSim.Events.REO_EventHandler(REO);
            //LFS_External.InSim.Events.REPLY_Received += new LFS_External.InSim.Events.REPLY_EventHandler(REPLY);
            //LFS_External.InSim.Events.RES_Received += new LFS_External.InSim.Events.RES_EventHandler(RES);
            //LFS_External.InSim.Events.RST_Received += new LFS_External.InSim.Events.RST_EventHandler(RST);
            //LFS_External.InSim.Events.RTP_Received += new LFS_External.InSim.Events.RTP_EventHandler(RTP);
            //LFS_External.InSim.Events.SPX_Received += new LFS_External.InSim.Events.SPX_EventHandler(SPX);
            //LFS_External.InSim.Events.STA_Received += new LFS_External.InSim.Events.STA_EventHandler(STA);
            LFS_External.InSim.Events.TOC_Received += new LFS_External.InSim.Events.TOC_EventHandler(TOC);
            //LFS_External.InSim.Events.VER_Received += new LFS_External.InSim.Events.VER_EventHandler(VER);
            //LFS_External.InSim.Events.VTA_Received += new LFS_External.InSim.Events.VTA_EventHandler(VTA);
            //LFS_External.InSim.Events.VTC_Received += new LFS_External.InSim.Events.VTC_EventHandler(VTC);
            //LFS_External.InSim.Events.VTN_Received += new LFS_External.InSim.Events.VTN_EventHandler(VTN);
        }

        // Methods for automatically update Players[] and Connection[] lists (Called from NPL and NCN)
        private void RemoveFromConnectionsList(Packets.IS_CNL CNL)
        {
            // Copy of item to remove
            clsConnection RemoveItem = new clsConnection();

            // Check what item the connection had
            foreach (clsConnection Conn in Connections)
            {
                if (CNL.UCID == Conn.UniqueID)
                {
                    // Copy item (Can't delete it here)
                    RemoveItem = Conn;
                    continue;
                }
            }

            // Remove item
            Connections.Remove(RemoveItem);
        }
        private void AddToConnectionsList(Packets.IS_NCN NCN)
        {
            bool InList = false;

            // Check of connection is already in the list
            foreach (clsConnection Conn in Connections)
            {
                if (Conn.UniqueID == NCN.UCID)
                {
                    InList = true;
                    continue;
                }
            }



            // If not, add it
            if (!InList)
            {
                // Assign values of new connnnection.
                clsConnection NewConn = new clsConnection();
                NewConn.UniqueID = NCN.UCID;
                NewConn.Username = NCN.UName;
                NewConn.PlayerName = NCN.PName;
                NewConn.IsAdmin = NCN.Admin;
                NewConn.Flags = NCN.Flags;

                Connections.Add(NewConn);
            }
        }
        private void RemoveFromPlayersList(Packets.IS_PLL PLL)
        {
            // Copy of item to remove
            clsPlayer RemoveItem = new clsPlayer();

            // Check what item the player had
            foreach (clsPlayer Player in Players)
            {
                if (PLL.PLID == Player.PlayerID)
                {
                    // Copy item (Can't delete it here)
                    RemoveItem = Player;
                    continue;
                }
            }

            // Remove item
            Players.Remove(RemoveItem);
        }
        private bool AddToPlayersList(Packets.IS_NPL NPL)
        {
            bool InList = false;

            // Check if player is already in the list
            foreach (clsPlayer Player in Players)
            {
                if (Player.PlayerID == NPL.PLID)
                {
                    Player.AddedMass = NPL.H_Mass;
                    Player.CarName = NPL.CName;
                    Player.Flags = NPL.Flags;
                    Player.Passengers = NPL.Pass;
                    Player.Plate = NPL.Plate;
                    Player.PlayerType = (clsPlayer.enuPType)NPL.PType;
                    Player.SkinName = NPL.SName;
                    Player.Tyre_FL = NPL.Tyre_FL;
                    Player.Tyre_FR = NPL.Tyre_FR;
                    Player.Tyre_RL = NPL.Tyre_RL;
                    Player.Tyre_RR = NPL.Tyre_RR;
					Player.IntakeRestriction = NPL.H_TRes;
                    return true;
                }
            }

            // If not, add it
            if (!InList)
            {
                // Assign values of new player.
                clsPlayer NewPLayer = new clsPlayer();
                NewPLayer.AddedMass = NPL.H_Mass;
                NewPLayer.CarName = NPL.CName;
                NewPLayer.Flags = NPL.Flags;
                NewPLayer.Passengers = NPL.Pass;
                NewPLayer.Plate = NPL.Plate;
                NewPLayer.PlayerID = NPL.PLID;
                NewPLayer.UniqueID = NPL.UCID;
                NewPLayer.PlayerName = NPL.PName;
                NewPLayer.PlayerType = (clsPlayer.enuPType)NPL.PType;
                NewPLayer.SkinName = NPL.SName;
                NewPLayer.Tyre_FL = NPL.Tyre_FL;
                NewPLayer.Tyre_FR = NPL.Tyre_FR;
                NewPLayer.Tyre_RL = NPL.Tyre_RL;
                NewPLayer.Tyre_RR = NPL.Tyre_RR;

                Players.Add(NewPLayer);
            }

            return false;
        }

        // Returns an index value for Connections[] that corresponds with the UniqueID of a connection
        public int GetConnIdx(int UNID)
        {
            for (int i = 0; i < Connections.Count; i++)
            {
                if (Connections[i].UniqueID == UNID) { return i; }
            }
            return 0;
        }

        // Returns an index value for Players[] that corresponds with the UniqueID of a player
        public int GetPlyIdx(int PLID)
        {
            for (int i = 0; i < Players.Count; i++)
            {
                if (Players[i].PlayerID == PLID) { return i; }
            }
            return 0;
        }
        #endregion

        /**********************************/
        /********* PACKET ARRIVAL *********/			// (Uncomment an event in the InitializeInSimEvents method and create the corresponding method down here to enable an other event)
        /**********************************/
        private void NPL(Packets.IS_NPL NPL)
        {
            bool LeavesPits = AddToPlayersList(NPL);    // Update Players[] list (don't remove this line!)

            // The path to the config file.
            string path = Environment.CurrentDirectory + "\\Restrictions.txt";

            // Create a config
            Config config = new Config();

            // Saving a config.
            // config.Set("MyInt", 3635);
            // config.Set("MyString", "Hello, world!");
            // config.Set("MyBool", false);

            // config.Save(path);

            // Loading a config.
            config.Load(path);

            // int myInt = config.GetInt32("MyInt");
            // string myStr = config.GetString("MyString");
            // bool myBool = config.GetBool("MyBool");

            bool DoSpectate = false;

            int WeightRestrict = config.GetInt32(NPL.CName + " Weight");
            int IntakeRestrict = config.GetInt32(NPL.CName + " Intake");

            if (WeightRestrict != 0)
            {
                if (NPL.H_Mass != WeightRestrict)
                {
                    InSim.Send_MST_Message("/msg " + NPL.PName + "^2 " + NPL.CName + " weight restrict " + WeightRestrict + " kg.");
                    DoSpectate = true;
                }
            }

            if (IntakeRestrict != 0)
            {
                if (NPL.H_TRes != IntakeRestrict)
                {
                    InSim.Send_MST_Message("/msg " + NPL.PName + "^2 " + NPL.CName + " intake restrict " + IntakeRestrict + "%");
                    DoSpectate = true;
                }
            }

            if (DoSpectate)
            {
                    InSim.Send_MST_Message("/spec " + NPL.PName);
            }

        }

        private void NCN(Packets.IS_NCN NCN)
        {
            AddToConnectionsList(NCN);                  // Update Connections[] list (don't remove this line!)
        }

        private void CNL(Packets.IS_CNL CNL)
        {
            RemoveFromConnectionsList(CNL);             // Update Connections[] list (don't remove this line!)
        }

        private void MSO(Packets.IS_MSO MSO)
        {
            if (MSO.Msg.Contains("WTF") | MSO.Msg.Contains("wtf"))
            {
                InSim.Send_MST_Message("/msg " + Connections[GetConnIdx(MSO.UCID)].PlayerName + "... ^2Have a Drive-Through");
                InSim.Send_MST_Message("/p_dt " + Connections[GetConnIdx(MSO.UCID)].Username);
            }
        }

        private void CPR(Packets.IS_CPR CPR)
        {
			foreach (clsConnection c in Connections)    // Update Connections[] and Players[] list (Don't remove those 2 foreach loops!)
				if (c.UniqueID == CPR.UCID)
				{
					c.PlayerName = CPR.PName;
				}

			foreach (clsPlayer p in Players)
				if (p.UniqueID == CPR.UCID)
				{
					p.PlayerName = CPR.PName;
					p.Plate = CPR.Plate;
				}
        }

        private void PLL(Packets.IS_PLL PLL)
        {
            RemoveFromPlayersList(PLL);                 // Update Players[] list (don't remove this line!)
        }

        private void TOC(Packets.IS_TOC TOC)
        {
            Players[GetPlyIdx(TOC.OldUCID)].UniqueID = TOC.NewUCID;		// Update Players[] list (don't remove this line!)
            Players[GetPlyIdx(TOC.OldUCID)].PlayerID = TOC.PLID;		// Update Players[] list (don't remove this line!)
        }
    }
}