#include "cgraphtools.h"
#include "cgraph.h"
#include "cgraphview.h"
#include "cmgr.h"

// group separators for dropdown boxes
#define CHOICE_SEPARATOR_LOGTYPE _T("----------------------------------")
#define CHOICE_SEPARATOR_WHEELTYPE _T("------------")

//-----------------------------------------------------------------------------
// Static members

wxArrayString cGraphTools::s_LogTypeNames;   // initialisation in CreateLogtypeControl
wxArrayString cGraphTools::s_WheelTypeNames; // initialisation in CreateWheeltypeControl

//-----------------------------------------------------------------------------
// Event table

BEGIN_EVENT_TABLE(cGraphTools, wxWindow)
  EVT_BUTTON(wxID_CLOSE, cGraphTools::OnButtonClose)
  EVT_BUTTON(ID_BTN_FIT, cGraphTools::OnButtonFit)
  EVT_SIZE(cGraphTools::OnSize)
END_EVENT_TABLE()

//-----------------------------------------------------------------------------

cGraphTools::cGraphTools(cGraphView* parent, cGraph* graph)
: wxWindow(parent, -1, wxDefaultPosition, wxDefaultSize, wxNO_BORDER)
{
  m_Parent = parent;
  m_Graph = graph;

  SetBackgroundColour(parent->GetBackgroundColour());

  m_FitButton = new wxButton(this, ID_BTN_FIT,
      _T("Fit"), wxDefaultPosition, wxDefaultSize, wxBU_EXACTFIT);

  m_CloseButton = new wxButton(this, wxID_CLOSE,
      _T("Close"), wxDefaultPosition, wxDefaultSize, wxBU_EXACTFIT);

  m_TopSizer = new wxBoxSizer(wxHORIZONTAL);
  m_TopSizer->AddStretchSpacer();
  m_TopSizer->Add(m_FitButton,
          wxSizerFlags(1).Proportion(0).Left().Border(wxALL, BASE_MARGIN));
  m_TopSizer->Add(m_CloseButton,
          wxSizerFlags(1).Proportion(0).Left().Border(wxTOP | wxRIGHT | wxBOTTOM, BASE_MARGIN));
  SetSizer(m_TopSizer);
}

//-----------------------------------------------------------------------------

cGraphTools::~cGraphTools()
{
}

//-----------------------------------------------------------------------------

void cGraphTools::OnButtonClose(wxCommandEvent& WXUNUSED(event))
{
  ::CloseView_Send(m_Parent);
}

//-----------------------------------------------------------------------------
// enable/disable the controls

void cGraphTools::EnableControls()
{
  // hide the 'Fit' button if all curves are hidden
  int curvesShown = 0;
  for (size_t c = 0; c < m_Graph->GetCurveCount(); c++) {
    if (m_Graph->GetCurve(c)->IsShown()) curvesShown += 1;
  }
  m_FitButton->Enable(curvesShown > 0);

  // NB 'Close' button is enabled in GraphPositionIs

  // do subclass-specific enabling
  DoEnableControls();
}

//-----------------------------------------------------------------------------

void cGraphTools::OnButtonFit(wxCommandEvent& WXUNUSED(event))
{
  m_Graph->FitSelection();
}

//-----------------------------------------------------------------------------

void cGraphTools::GraphPositionIs(bool top, bool bottom)
{
  // hide the 'Close' button if it's the only graph
  m_CloseButton->Show(!(top && bottom));
  Layout();
}

//-----------------------------------------------------------------------------

void cGraphTools::OnSize(wxSizeEvent& event)
{
  Layout();
  event.Skip();
}

//-----------------------------------------------------------------------------
// Manage a wxChoice control for selecting a logtype

wxChoice* cGraphTools::CreateLogtypeControl(wxWindowID id)
{
  if (s_LogTypeNames.IsEmpty()) {
    // build list of logtypes
    s_LogTypeNames.Add(cCarState::GetLogtypeName(LOGTYPE_SPEED));
    s_LogTypeNames.Add(cCarState::GetLogtypeName(LOGTYPE_TIMEINLAP));
    s_LogTypeNames.Add(cCarState::GetLogtypeName(LOGTYPE_RPM));
    s_LogTypeNames.Add(cCarState::GetLogtypeName(LOGTYPE_GLONG));
    s_LogTypeNames.Add(cCarState::GetLogtypeName(LOGTYPE_GLAT));
    s_LogTypeNames.Add(cCarState::GetLogtypeName(LOGTYPE_GTOTAL));
    s_LogTypeNames.Add(cCarState::GetLogtypeName(LOGTYPE_SLIPANGLE));
    s_LogTypeNames.Add(CHOICE_SEPARATOR_LOGTYPE);
    s_LogTypeNames.Add(cCarState::GetLogtypeName(LOGTYPE_STEER));
    s_LogTypeNames.Add(cCarState::GetLogtypeName(LOGTYPE_THROTTLE));
    s_LogTypeNames.Add(cCarState::GetLogtypeName(LOGTYPE_BRAKE));
    s_LogTypeNames.Add(cCarState::GetLogtypeName(LOGTYPE_CLUTCH));
    s_LogTypeNames.Add(cCarState::GetLogtypeName(LOGTYPE_GEAR));
    s_LogTypeNames.Add(CHOICE_SEPARATOR_LOGTYPE);
    s_LogTypeNames.Add(cCarState::GetLogtypeName(LOGTYPE_CAMBER));
    s_LogTypeNames.Add(cCarState::GetLogtypeName(LOGTYPE_WHEELANGLE));
    s_LogTypeNames.Add(cCarState::GetLogtypeName(LOGTYPE_SUSPTRAVEL));
    s_LogTypeNames.Add(cCarState::GetLogtypeName(LOGTYPE_WHEELSPEED_VERT));
    s_LogTypeNames.Add(cCarState::GetLogtypeName(LOGTYPE_FORCE_LON));
    s_LogTypeNames.Add(cCarState::GetLogtypeName(LOGTYPE_FORCE_LAT));
    s_LogTypeNames.Add(cCarState::GetLogtypeName(LOGTYPE_TYRELOAD));
    s_LogTypeNames.Add(cCarState::GetLogtypeName(LOGTYPE_WHEELSPEED_ROT));
    s_LogTypeNames.Add(cCarState::GetLogtypeName(LOGTYPE_SLIPRATIO));
    s_LogTypeNames.Add(cCarState::GetLogtypeName(LOGTYPE_POWER));
  }

  return new wxChoice(this, id, wxDefaultPosition, wxDefaultSize, s_LogTypeNames);
}


bool cGraphTools::SetLogtypeSelection(wxChoice* control, int log)
{
  return control->SetStringSelection(cCarState::GetLogtypeName(log));
}


int cGraphTools::GetLogtypeSelection(wxChoice* control)
{
  return cCarState::FindLogtypeName(control->GetStringSelection());
}

//-----------------------------------------------------------------------------
// Manage a wxChoice control for selecting a wheeltype

wxChoice* cGraphTools::CreateWheeltypeControl(wxWindowID id)
{
  if (s_WheelTypeNames.IsEmpty()) {
    // build list of wheeltypes
    s_WheelTypeNames.Add(cCarState::GetWheeltypeName(WHEELTYPE_FRONT_LEFT));
    s_WheelTypeNames.Add(cCarState::GetWheeltypeName(WHEELTYPE_FRONT_RIGHT));
    s_WheelTypeNames.Add(cCarState::GetWheeltypeName(WHEELTYPE_REAR_LEFT));
    s_WheelTypeNames.Add(cCarState::GetWheeltypeName(WHEELTYPE_REAR_RIGHT));
    s_WheelTypeNames.Add(CHOICE_SEPARATOR_WHEELTYPE);
    s_WheelTypeNames.Add(cCarState::GetWheeltypeName(WHEELTYPE_AVG));
    s_WheelTypeNames.Add(cCarState::GetWheeltypeName(WHEELTYPE_AVG_FRONT));
    s_WheelTypeNames.Add(cCarState::GetWheeltypeName(WHEELTYPE_AVG_REAR));
    s_WheelTypeNames.Add(cCarState::GetWheeltypeName(WHEELTYPE_MIN));
    s_WheelTypeNames.Add(cCarState::GetWheeltypeName(WHEELTYPE_MIN_FRONT));
    s_WheelTypeNames.Add(cCarState::GetWheeltypeName(WHEELTYPE_MIN_REAR));
    s_WheelTypeNames.Add(cCarState::GetWheeltypeName(WHEELTYPE_MAX));
    s_WheelTypeNames.Add(cCarState::GetWheeltypeName(WHEELTYPE_MAX_FRONT));
    s_WheelTypeNames.Add(cCarState::GetWheeltypeName(WHEELTYPE_MAX_REAR));
    s_WheelTypeNames.Add(cCarState::GetWheeltypeName(WHEELTYPE_DELTA_ALL));
    s_WheelTypeNames.Add(cCarState::GetWheeltypeName(WHEELTYPE_DELTA_FRONT));
    s_WheelTypeNames.Add(cCarState::GetWheeltypeName(WHEELTYPE_DELTA_REAR));
  }

  return new wxChoice(this, id, wxDefaultPosition, wxDefaultSize, s_WheelTypeNames);
}


bool cGraphTools::SetWheeltypeSelection(wxChoice* control, int wheel)
{
  return control->SetStringSelection(cCarState::GetWheeltypeName(wheel));
}


int cGraphTools::GetWheeltypeSelection(wxChoice* control)
{
  return cCarState::FindWheeltypeName(control->GetStringSelection());
}

//-----------------------------------------------------------------------------
// Fill a wxChoice control for selecting a lap
// - control = existing control
// - current = currently selected lap, will be selected again afterwards (may be NULL)
// Returns the index of the selected item

int cGraphTools::FillLapSelection(wxChoice* control, cLap* current)
{
  int index = wxNOT_FOUND; // index of lap that will be selected

  // rebuild the list of laps
  control->Clear();
  for (size_t i = 0; i < MGR->GetLapCount(); i++) {
    cLap* lap = MGR->GetLap(i);
    control->Append(lap->GetName().c_str(), (void*)lap);

    if (lap == current) index = i;
  }

  // re-select the lap in the list
  if ((index == wxNOT_FOUND) && (!control->IsEmpty())) {
    control->SetSelection(0); // select first lap by default
  }
  else {
    control->SetSelection(index);
  }

  return index;
}

//-----------------------------------------------------------------------------
// Get the currently selected lap of wxChoice control for selecting a lap
// - control = existing control
// Returns NULL if none selected

cLap* cGraphTools::GetLapSelection(wxChoice* control)
{
  int selection = control->GetSelection();
  return (selection == wxNOT_FOUND) ? NULL : ((cLap*)control->GetClientData(selection));
}
