#include "cdrivingline.h"
#include "cmgr.h"
#include "clap.h"
#include "ctrackpath.h"
#include <wx/dcclient.h>
#include <wx/cursor.h>
#include <wx/config.h>

// default scale factor
#define DEFAULT_SCALE 1.0f

// default visible range (= #metres in visible part of window when at minimum size)
#define DEFAULT_RANGE 100.0f

// multiplication/division factor for zooming in/out (for mouse clicks and for mouse wheel)
#define ZOOM_FACTOR_BUTTON 1.5f
#define ZOOM_FACTOR_WHEEL 1.25f

// diameter of the track cursor
#define CURSOR_DIAMETER 3

//-----------------------------------------------------------------------------
// Menu identifiers

enum ENUM_MENU_IDS
{
  ID_MENU_ZOOM_IN = 100,
  ID_MENU_ZOOM_OUT,
  ID_MENU_RESETZOOM,
  ID_MENU_AUTOSCROLL,
  ID_MENU_LAST
};

//-----------------------------------------------------------------------------
// Event table

BEGIN_EVENT_TABLE(cDrivingLine, cPane)
  EVT_LEFT_DCLICK(cDrivingLine::OnMouseLeftDoubleClick)
  EVT_LEFT_DOWN(cDrivingLine::OnMouseLeftClick)
  EVT_LEFT_UP(cDrivingLine::OnMouseLeftRelease)
  EVT_MIDDLE_DOWN(cDrivingLine::OnMouseMiddleClick)
  EVT_MOTION(cDrivingLine::OnMouseMove)
  EVT_MOUSEWHEEL(cDrivingLine::OnMouseWheel)
  EVT_RIGHT_DCLICK(cDrivingLine::OnMouseRightDoubleClick)
  EVT_RIGHT_DOWN(cDrivingLine::OnMouseRightClick)
  EVT_MENU(wxID_ANY, cDrivingLine::OnMenuClick)
  EVT_PAINT(cDrivingLine::OnPaint)
  EVT_SIZE(cDrivingLine::OnSize)
  EVT_SCROLLWIN(cDrivingLine::OnScrollWin)
END_EVENT_TABLE()

//-----------------------------------------------------------------------------

cDrivingLine::cDrivingLine(wxWindow* parent)
: cPane(parent)
{
  m_Scale = DEFAULT_SCALE; // scale is fixed
  m_Zoom = 1.0f;

  m_CursorDist = -1.0f;
  m_AutoScroll = false;

  m_DragMode = false;
  m_ClickPos = -1;

  m_Path = NULL;
  m_PolygonsValid = false;
  m_Polygons = NULL;
  m_PolygonSize = NULL;
  m_PolygonCount = 0;

  SetBackgroundColour(*wxWHITE);

  m_Context.Append(ID_MENU_RESETZOOM, _T("Reset zoom"));
  m_Context.Append(ID_MENU_ZOOM_IN, _T("Zoom in"));
  m_Context.Append(ID_MENU_ZOOM_OUT, _T("Zoom out"));
  m_Context.AppendSeparator();
  m_Context.AppendCheckItem(ID_MENU_AUTOSCROLL, _T("Auto-scroll"));
}

//-----------------------------------------------------------------------------

cDrivingLine::~cDrivingLine()
{
  if (HasCapture()) ReleaseMouse();

  m_PolygonsValid = false;
  delete[] m_Polygons;
  delete[] m_PolygonSize;
}

//-----------------------------------------------------------------------------
// Plot the trackmap

void cDrivingLine::OnPaint(wxPaintEvent& WXUNUSED(event))
{
  wxPaintDC dc(this);
  dc.SetLogicalFunction(wxCOPY);

  if (MGR->IsEmpty()) return;

  // plot the track
  DrawTrack(dc);

  // plot the trajectory of each lap
  for (size_t i = 0; i < MGR->GetLapCount(); i++) {
    cLap* lap = MGR->GetLap(i);
    if (!lap->IsShown()) continue;

    dc.SetPen(wxPen(lap->GetColour(), 1, wxSOLID));

    // plot the cursor
    if (m_CursorDist >= 0.0) {
      float x, y;
      lap->GetPositionAt(m_CursorDist, x, y);
      dc.SetBrush(wxBrush(lap->GetColour()));
      dc.DrawCircle(Track2CoordX(x), Track2CoordY(y), CURSOR_DIAMETER);
    }

    // get screen coordinates of first state
    const cCarState* state = lap->GetState(0);
    wxCoord lastX = Track2CoordX(state->GetPosX());
    wxCoord lastY = Track2CoordY(state->GetPosY());

    // plot the lap's trajectory
    for (size_t s = 1; s < lap->GetStateCount(); s++) {
      state = lap->GetState(s);
      wxCoord stateX = Track2CoordX(state->GetPosX());
      wxCoord stateY = Track2CoordY(state->GetPosY());

      dc.DrawLine(lastX, lastY, stateX, stateY);

      lastX = stateX;
      lastY = stateY;
    }
  }
}

//-----------------------------------------------------------------------------
// Draw the track path

void cDrivingLine::DrawTrack(wxDC& dc)
{
  if (!m_PolygonsValid) {
    // need to renew the polygons
    delete[] m_Polygons;
    m_Polygons = NULL;
    delete[] m_PolygonSize;
    m_PolygonSize = NULL;
    m_PolygonCount = 0;
    m_PolygonsValid = true;

    m_Path = MGR->GetTrackPath();
    CHECK_PTR_NULL(m_Path);
    if (m_Path == NULL) return; // no track path loaded

    // allocate
    m_PolygonCount = m_Path->m_NodeCount;
    m_Polygons = new wxPoint[4 * m_PolygonCount];
    m_PolygonSize = new int[m_PolygonCount];

    // calculate coordinates
    for (int n = 0; n < m_PolygonCount; n ++) {
      m_PolygonSize[n] = 4;
      int nn = (n + 1) % m_PolygonCount; // next node
      m_Polygons[4*n].x = Track2CoordX(m_Path->m_RoadLeftX[n]);
      m_Polygons[4*n].y = Track2CoordY(m_Path->m_RoadLeftY[n]);
      m_Polygons[4*n+1].x = Track2CoordX(m_Path->m_RoadRightX[n]);
      m_Polygons[4*n+1].y = Track2CoordY(m_Path->m_RoadRightY[n]);
      m_Polygons[4*n+2].x = Track2CoordX(m_Path->m_RoadRightX[nn]);
      m_Polygons[4*n+2].y = Track2CoordY(m_Path->m_RoadRightY[nn]);
      m_Polygons[4*n+3].x = Track2CoordX(m_Path->m_RoadLeftX[nn]);
      m_Polygons[4*n+3].y = Track2CoordY(m_Path->m_RoadLeftY[nn]);
    }
  }

  if (m_Polygons == NULL) return; // drawing track not possible

  CHECK_PTR(m_Path);
  wxCoord offsetX = Track2CoordX(m_Path->m_RoadLeftX[0]) - m_Polygons[0].x;
  wxCoord offsetY = Track2CoordY(m_Path->m_RoadLeftY[0]) - m_Polygons[0].y;

  // draw the polygons
  wxColour col(220, 220, 220);
  dc.SetPen(wxPen(col));
  dc.SetBrush(wxBrush(col));
  dc.DrawPolyPolygon(m_PolygonCount, m_PolygonSize, m_Polygons, offsetX, offsetY);

  // draw the finish line
  dc.SetPen(*wxBLACK_PEN);
  dc.DrawLine(
      Track2CoordX(m_Path->m_RoadLeftX[m_Path->m_FinishNode]),
      Track2CoordY(m_Path->m_RoadLeftY[m_Path->m_FinishNode]),
      Track2CoordX(m_Path->m_RoadRightX[m_Path->m_FinishNode]),
      Track2CoordY(m_Path->m_RoadRightY[m_Path->m_FinishNode]));
}

//-----------------------------------------------------------------------------
// Draw the cursor at a new position
// - distance = distance_in_lap (in m) that corresponds to the cursor position
// - scroll = scroll if needed to get the cursor in the visible part
// If distance < 0 then cursor is removed

void cDrivingLine::DoSetTrackCursorPos(cGraphView* WXUNUSED(view), float distance, bool scroll)
{
  if (MGR->IsEmpty()) return; // empty display
  if ((!scroll) && (!m_AutoScroll)) return; // ignore "fast movements" of cursor

  // draw the cursor at its new position
  PutTrackCursor(distance);

  // check if we need to scroll the window to keep the cursor visible
  if (m_CursorDist >= 0.0) {
    wxCoord marginX = GetSize().GetWidth() / 3; // minimum distance to window border before scrolling occurs
    int dx = 0;                                 // amount (in pixels) that the centre needs to be moved
    wxRect rect = GetTrackCursorPosition();
    wxCoord centreX = rect.x + rect.width / 2;
    wxCoord centreY = rect.y + rect.height / 2;
    if (centreX < marginX)
        dx = marginX - centreX;
    if (centreX > GetClientSize().GetWidth() - marginX)
        dx = GetClientSize().GetWidth() - marginX - centreX;

    wxCoord marginY = GetSize().GetHeight() / 3;
    int dy = 0;
    if (centreY < marginY)
        dy = marginY - centreY;
    if (centreY > GetClientSize().GetHeight() - marginY)
        dy = GetClientSize().GetHeight() - marginY - centreY;

    ScrollView(dx, dy);
  }
}

//-----------------------------------------------------------------------------

void cDrivingLine::OnSize(wxSizeEvent& event)
{
  // force full redraw when window is resized
  Refresh();
  event.Skip();
}

//-----------------------------------------------------------------------------

void cDrivingLine::OnMouseLeftClick(wxMouseEvent& event)
{
  event.Skip(); // always pass left-clicks to default event handler
  if (MGR->IsEmpty()) return; // empty display

  if (event.CmdDown()) {
    // zoom in
    SetZoom(m_Zoom * ZOOM_FACTOR_BUTTON);
    m_ClickPos = -1;
  }
  else {
    // prepare for drag-scrolling
    wxClientDC dc(this);
    m_ClickPos = DecodeMouse(event.GetLogicalPosition(dc));
  }
}

//-----------------------------------------------------------------------------

void cDrivingLine::OnMouseMove(wxMouseEvent& event)
{
  if (MGR->IsEmpty()) return;

  // get logical position of mouse
  wxClientDC dc(this);
  wxPoint pos = event.GetLogicalPosition(dc);

  if (event.LeftIsDown()) {
    if (!m_DragMode) {
      // enter dragging mode
      m_DragMode = true;
      CaptureMouse();
      SetCursor(wxCursor(wxCURSOR_HAND));
    }
    else {
      // already dragging
      ScrollView(pos.x - m_MovePos.x, pos.y - m_MovePos.y);
    }

    m_MovePos = pos;
  }
}

//-----------------------------------------------------------------------------

void cDrivingLine::OnMouseLeftRelease(wxMouseEvent& event)
{
  event.Skip(); // always pass left-clicks to default event handler

  if (m_DragMode) {
    // end drag
    SetCursor(wxNullCursor);
    if (HasCapture()) ReleaseMouse();
    m_DragMode = false;
  }
  else {
    if (MGR->IsEmpty()) return; // empty display
    if (m_ClickPos < 0) return; // no valid click

    ::SetTrackCursorPos_Send(m_ClickPos, true);
  }
}

//-----------------------------------------------------------------------------

void cDrivingLine::OnMouseLeftDoubleClick(wxMouseEvent& event)
{
  if (event.CmdDown()) {
    // zoom in
    SetZoom(m_Zoom * ZOOM_FACTOR_BUTTON);
    m_ClickPos = -1;
  }
  else {
    ResetZoom();
  }
}

void cDrivingLine::OnMouseRightDoubleClick(wxMouseEvent& event)
{
  if (event.CmdDown()) {
    // zoom out
    SetZoom(m_Zoom / ZOOM_FACTOR_BUTTON);
  }
}

//-----------------------------------------------------------------------------

void cDrivingLine::SetZoom(float zoom)
{
  if (m_Zoom == zoom) return;
  m_Zoom = zoom;
  m_PolygonsValid = false;
  Refresh();
}

void cDrivingLine::ResetZoom()
{
  SetZoom(1.0);
}

//-----------------------------------------------------------------------------

void cDrivingLine::UpdateAll()
{
  if (m_Path != MGR->GetTrackPath()) {
    // just loaded a new track
    m_PolygonsValid = false;
  }

  if (MGR->GetLapCount() > 0) {
    // reset view centre to starting line
    const cCarState* state = MGR->GetLap(0)->GetState(0);
    m_CentreX = state->GetPosX();
    m_CentreY = state->GetPosY();
  }

  Refresh();

  int range = (int)(m_Scale * DEFAULT_RANGE);
  SetMinSize(wxSize(range, range));
}

//-----------------------------------------------------------------------------

void cDrivingLine::OnMouseWheel(wxMouseEvent& event)
{
  if (MGR->IsEmpty()) return;

  float factor = ((event.GetWheelRotation() > 0) ? ZOOM_FACTOR_WHEEL : (1.0f / ZOOM_FACTOR_WHEEL));
  SetZoom(m_Zoom * factor);
}

//-----------------------------------------------------------------------------
// Return the distance-in-lap that corresponds to a mouse position

float cDrivingLine::DecodeMouse(wxPoint pos)
{
  if (MGR->IsEmpty()) return -1;

  // find state that is nearest to this position
  float nearest_dist = -1; // distance-in-lap of state that is nearest to mouse position
  float nearest_pix = IMPOSSIBLY_HIGH_VALUE; // square of distance in pixels to that state

  // scan all laps
  for (size_t i = 0; i < MGR->GetLapCount(); i++) {
    cLap* lap = MGR->GetLap(i);
    // scan all states
    for (size_t s = 0; s < lap->GetStateCount(); s++) {
      const cCarState* state = lap->GetState(s);
      float dX = Track2CoordX(state->GetPosX()) - pos.x;
      float dY = Track2CoordY(state->GetPosY()) - pos.y;
      float d = (dX * dX) + (dY * dY);
      if (d < nearest_pix) {
        nearest_pix = d;
        nearest_dist = state->GetDistance();
      }
    }
  }

  if (sqrt(nearest_pix) > CLICK_DETECT_THRESHOLD) return -1;

  return nearest_dist;
}

//-----------------------------------------------------------------------------
// Scroll the display

void cDrivingLine::ScrollView(int dx, int dy)
{
  if ((dx == 0) && (dy == 0)) return;
  m_CentreX -= ((float)dx) / (m_Scale * m_Zoom);
  m_CentreY += ((float)dy) / (m_Scale * m_Zoom);
  ScrollWindow(dx, dy);
}

//-----------------------------------------------------------------------------

wxSize cDrivingLine::DoGetBestSize() const
{
  int range = (int)(2.0f * m_Scale * DEFAULT_RANGE);
  return wxSize(range, range);
}

//-----------------------------------------------------------------------------

void cDrivingLine::OnMouseRightClick(wxMouseEvent& event)
{
  if (event.CmdDown()) {
    // zoom out
    SetZoom(m_Zoom / ZOOM_FACTOR_BUTTON);
  }
  else {
    if (HasCapture()) return; // don't show context menu while a drag is going on

    // show context menu
    bool hasLaps = (MGR->GetLapCount() > 0);
    m_Context.Enable(ID_MENU_ZOOM_IN, hasLaps);
    m_Context.Enable(ID_MENU_ZOOM_OUT, hasLaps);
    m_Context.Enable(ID_MENU_RESETZOOM, hasLaps && (m_Zoom != 1.0f));
    m_Context.Check(ID_MENU_AUTOSCROLL, m_AutoScroll);
    PopupMenu(&m_Context);
  }
}

//-----------------------------------------------------------------------------
// Handle a click in the context menu

void cDrivingLine::OnMenuClick(wxCommandEvent& event)
{
  switch (event.GetId()) {
    case ID_MENU_ZOOM_IN :
      SetZoom(m_Zoom * ZOOM_FACTOR_BUTTON);
      break;

    case ID_MENU_ZOOM_OUT :
      SetZoom(m_Zoom / ZOOM_FACTOR_BUTTON);
      break;

    case ID_MENU_RESETZOOM :
      ResetZoom();
      break;

    case ID_MENU_AUTOSCROLL :
      m_AutoScroll = !m_AutoScroll;
      break;

    default :
      wxFAIL;
  }
}

//-----------------------------------------------------------------------------
// Draw the cursors at a new position
// - distance = distance-in-lap for cursor position

void cDrivingLine::PutTrackCursor(float distance)
{
  if (MGR->IsEmpty()) return; // nothing to do

  if (m_CursorDist >= 0.0f) RefreshRect(GetTrackCursorPosition()); // redraw lines at old position

  m_CursorDist = distance;

  if (m_CursorDist >= 0.0f) RefreshRect(GetTrackCursorPosition()); // draw lines at new position
}

//-----------------------------------------------------------------------------
// Get a bounding box for the cursor positions

wxRect cDrivingLine::GetTrackCursorPosition()
{
  wxASSERT(!(MGR->IsEmpty()));
  wxASSERT(m_CursorDist >= 0.0f);

  wxCoord minX = 0;
  wxCoord maxX = 0;
  wxCoord minY = 0;
  wxCoord maxY = 0;
  for (size_t i = 0; i < MGR->GetLapCount(); i++) {
    float x, y;
    MGR->GetLap(i)->GetPositionAt(m_CursorDist, x, y);
    wxCoord cursorX = Track2CoordX(x);
    wxCoord cursorY = Track2CoordY(y);

    if (i == 0) {
      minX = cursorX;
      maxX = cursorX;
      minY = cursorY;
      maxY = cursorY;
    }
    else {
      if (minX > cursorX) minX = cursorX;
      if (maxX < cursorX) maxX = cursorX;
      if (minY > cursorY) minY = cursorY;
      if (maxY < cursorY) maxY = cursorY;
    }
  }

  return wxRect(
      minX - CURSOR_DIAMETER,
      minY - CURSOR_DIAMETER,
      maxX - minX + 2 * CURSOR_DIAMETER,
      maxY - minY + 2 * CURSOR_DIAMETER);
}

//-----------------------------------------------------------------------------
// Loading and saving the configuration settings

void cDrivingLine::LoadConfig(wxRegConfig* config, const wxString& key)
{
  config->Read(key + _T("/autoscroll"), &m_AutoScroll);
  double zoom = m_Zoom;
  if (config->Read(key + _T("/zoom"), &zoom)) m_Zoom = zoom;
}


void cDrivingLine::SaveConfig(wxRegConfig* config, const wxString& key)
{
  config->Write(key + _T("/autoscroll"), m_AutoScroll);
  config->Write(key + _T("/zoom"), m_Zoom);
}
