#ifndef _CCARSTATE_H_
#define _CCARSTATE_H_

#include "global.h"
#include "cvector.h"
#include <wx/file.h>

class cCar;
class cCarState;

//-----------------------------------------------------------------------------

enum LOGTYPE_IDS {
  LOGTYPE_FIRST = 0,
  LOGTYPE_SPEED = LOGTYPE_FIRST,
  LOGTYPE_TIMEINLAP,
  LOGTYPE_RPM,
  LOGTYPE_GEAR,
  LOGTYPE_STEER,
  LOGTYPE_THROTTLE,
  LOGTYPE_BRAKE,
  LOGTYPE_SLIPRATIO,
  LOGTYPE_CAMBER,
  LOGTYPE_SUSPTRAVEL,
  LOGTYPE_FORCE_LON,
  LOGTYPE_FORCE_LAT,
  LOGTYPE_TYRELOAD,
  LOGTYPE_GLONG,
  LOGTYPE_GLAT,
  LOGTYPE_GTOTAL,
  LOGTYPE_WHEELANGLE,
  LOGTYPE_SLIPANGLE,
  LOGTYPE_CLUTCH,
  LOGTYPE_WHEELSPEED_VERT,
  LOGTYPE_WHEELSPEED_ROT,
  LOGTYPE_POWER,
  LOGTYPE_LAST
};

class cLogtype
{
private:
  friend class cCarState; // class can only be accessed from cCarState

  wxString m_Name;    // the display name
  wxString m_Unit;    // the unit in which the data is expressed
  wxString m_Code;    // the code in config settings
  bool m_IsPerWheel;  // does this type have data for each wheel?
  bool m_IsInteger;   // is it integer-valued?
  bool m_IsSymmetric; // can it have both positive and negative values?

  void Set(wxString name, wxString unit, wxString code, bool perwheel, bool integer, bool symmetric);
};

//-----------------------------------------------------------------------------

enum WHEELTYPE_IDS {
  WHEELTYPE_FIRST = 0,
  WHEELTYPE_FRONT_LEFT = WHEELTYPE_FIRST,
  WHEELTYPE_FRONT_RIGHT,
  WHEELTYPE_REAR_LEFT,
  WHEELTYPE_REAR_RIGHT,
  WHEELTYPE_AVG,
  WHEELTYPE_AVG_FRONT,
  WHEELTYPE_AVG_REAR,
  WHEELTYPE_MIN,
  WHEELTYPE_MIN_FRONT,
  WHEELTYPE_MIN_REAR,
  WHEELTYPE_MAX,
  WHEELTYPE_MAX_FRONT,
  WHEELTYPE_MAX_REAR,
  WHEELTYPE_DELTA_ALL,
  WHEELTYPE_DELTA_FRONT,
  WHEELTYPE_DELTA_REAR,
  WHEELTYPE_LAST
};

class cWheeltype
{
private:
  friend class cCarState; // class can only be accessed from cCarState

  wxString m_Name;    // the display name
  wxString m_Code;    // the code in config settings

  void Set(wxString name, wxString code);
};

//-----------------------------------------------------------------------------
// Contains data from a single log entry

class cCarState
{
private:
  // static members
  static cLogtype s_Logtypes[LOGTYPE_LAST];
  static cWheeltype s_Wheeltypes[WHEELTYPE_LAST];
  static bool s_TypesDefined;

  cCar* m_Car;            // static car data, from file header

  // car state data, read from the file
  cVector m_Pos;          // car position (in m)
  float m_Speed;          // car speed (in m/s)
  float m_TimeInLap;      // time since start of lap
  float m_Rpm;            // engine rpm
  wxInt8 m_Gear;          // gear (0=reverse, 1=neutral, 2=first, etc.)
  float m_Steer;          // steering input (in radians)
  float m_Throttle;       // throttle input (0 - 1)
  float m_Brake;          // brake input (0 - 1)
  float m_Clutch;         // clutch input (0 - 1)
  float m_Distance;       // distance in lap (in m)
  cVector m_Right;        // right-vector
  cVector m_Fwd;          // forward-vector

  // derived car state data
  float m_Radius;         // turn radius (> 0 for right turn, < 0 for left turn, 0 for no turn)
  float m_Glong;          // longitudinal G (in m/s2)
  float m_Glat;           // lateral G (in m/s2)
  float m_SlipAngle;      // car slip angle

  // wheel state data, read from the file
  // one value per wheel (index 0 = front left, 1 = front right, 2 = rear left, 3 = rear right)
  float m_Deflect[4];     // suspension deflect
  float m_ForceLat[4];    // X force (lateral)
  float m_ForceLong[4];   // Y force (longitudinal)
  float m_Load[4];        // vertical load
  float m_AngVel[4];      // angular velocity (radians/s)
  float m_Camber[4];      // lean rel. to road (radians a-c viewed from rear)
  float m_WheelAngle[4];  // steer, incl. Ackerman and toe

  // derived wheel state data
  float m_WheelSpeed[4];  // vertical wheel speed

  float GetActualWheelSpeed(int wheel) const;

public:
  friend class cLap;

  cCarState(cCar* car);
  virtual ~cCarState();

  static bool s_SpeedInMph; // return speed in miles per hour?

  static void DefineTypes();

  static wxString GetLogtypeName(int log);
  static wxString GetLogtypeCode(int log);
  static wxString GetLogtypeUnits(int log);
  static int FindLogtypeName(const wxString& name);
  static int FindLogtypeCode(const wxString& code);

  static wxString GetWheeltypeName(int wheel);
  static wxString GetWheeltypeCode(int wheel);
  static int FindWheeltypeName(const wxString& name);
  static int FindWheeltypeCode(const wxString& code);

  static bool IsInteger(int log)
      { wxASSERT(s_TypesDefined); return s_Logtypes[log].m_IsInteger; }
  static bool IsPerWheel(int log)
      { wxASSERT(s_TypesDefined); return s_Logtypes[log].m_IsPerWheel; }
  static bool IsSymmetric(int log, int wheel);

  inline float GetPosX() const { return m_Pos.x; }
  inline float GetPosY() const { return m_Pos.y; }
  inline float GetPosZ() const { return m_Pos.z; }
  inline float GetDistance() const { return m_Distance; }
  inline float GetTime() const { return m_TimeInLap; }
  float GetLogData(int log, int wheel) const;

  void CalculateDerivedData(const cCarState& prev, const cCarState& next);
};

WX_DECLARE_OBJARRAY(cCarState, cCarStateArray);

#endif
